/*
 *  Interface/wrapper for GStreamer GstPad
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_PAD_H
#define KDE_GST_PAD_H

#include <kde/gst/object.h>
#include <kde/gst/format.h>
#include <kde/gst/query.h>

#include <qvaluevector.h>


namespace KDE {
namespace GST {
	class Caps;
	class PadTemplate;
	class Event;
	class Element;

/**
 * Element are linked to each other via "pads", which are extremely 
 * light-weight generic links. After two pad are retrieved from an 
 * element with @ref Element#getPad(), the pads can be linked with 
 * @ref #link().
 * 
 * Pads are typically created from a @ref PadTemplate.
 *
 * Pad have #ref Caps attached to it to describe the media type they
 * are capable of dealing with. @ref #getCaps() and @ref #setCaps() are used to
 * manipulate the caps of the pads. 
 * Pads created from a padtemplate cannot set capabilities that are incompatible 
 * with the padtemplates capabilities.
 * 
 * @ref #getParent() will retrieve the @ref Element that owns the pad.
 *
 * To send an event on a pad, use @ref #sendEvent().
 *
 * @short The link between Elements
 * @see PadTemplate
 * @see Event
 * @see Element
 */
	class Pad : public Object {
	Q_OBJECT
	private:
                friend class GStreamer;
                static Wrapper* wrapperFactory(void *real);
		void *reserved;
		
	public:
		enum Direction {
		  DIRECTION_UNKNOWN,
		  DIRECTION_SRC,
		  DIRECTION_SINK
		};

		enum Flags {
		  FLAG_DISABLED		= Object::FLAG_LAST,
		  FLAG_EOS,
		  FLAG_FLAG_LAST		= Object::FLAG_LAST + 4
		};

/**
 * Creates a new Pad that wrapps the given GstElement. Usually
 * you really don't want to call this, use @ref #wrap instead.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstPad to be wrapped
 */
	        Pad(void *real);
	        virtual ~Pad();

/**
 * Creates a new Pad that wrapps the given GstPad. 
 * Unlike the Pad constructor this function also re-uses 
 * existing wrapper objects.
 *
 * @param real the GstPad to be wrapped
 * @return the wrapped Pad
 */
		static Pad* wrap(void *real);

/**
 * Create a new pad with given name.
 *
 * @param name name of new pad
 * @param direction either PAD_SRC or PAD_SINK
 */
		Pad(const QString &name, Direction direction);

/**
 * Create a new pad with given name from the given template.
 *
 * @param templ the pad template to use
 * @param name the name of the element
 */
		Pad(PadTemplate *templ, const QString &name);

/**
 * Get the direction of the pad.
 *
 * @return the direction of the pad
 */
		Direction getDirection();

/**
 * Try to set the caps on the given pad.
 *
 * @param caps the caps to set
 * @return TRUE if the caps could be set
 */
		bool trySetCaps(Caps *caps);

/**
 * Get the capabilities of this pad.
 *
 * @return the capabilities of this pad
 */
		Caps* getCaps();

/**
 * Check if two pads have compatible capabilities.
 *
 * @param sinkpad the sinkpad to check against
 * @return TRUE if they are compatible or the capabilities
 * could not be checked
 */
		bool checkCompatibility(Pad *sinkpad);

/**
 * Get the real parent object of this pad. If the pad
 * is a ghostpad, the actual owner of the real pad is
 * returned, as opposed to the getParent().
 *
 * @return the parent object
 */
		Element* getRealParent();

/**
 * Add a ghost pad to a pad.
 *
 * @param ghostpad the ghost pad to add
 */
		void addGhostPad(Pad *ghostpad);

/**
 * Remove a ghost pad from a pad.
 *
 * @param ghostpad the ghost pad to remove from the pad
 */
		void removeGhostPad(Pad *ghostpad);

/**
 * Get the ghost parents of this pad.
 *
 * @return a vector containing references to the ghost pads
 */
		QValueVector<Pad*> getGhostPadList();

/**
 * Get the peer pad of this pad.
 *
 * @return the peer pad
 */
		Pad* getPeer();

/**
 * Links the source pad to the sink pad.
 *
 * @param sinkpad the sink pad to link
 * @return TRUE if the pad could be linked, FALSE otherwise
 */
		bool link(Pad *sinkpad);

/**
 * Links the source pad to the sink pad. The filter indicates the media type
 * that should flow trought this link.
 *
 * @param sinkpad the sink pad to link
 * @param filtercaps the filter caps.
 * @return TRUE if the pad could be linked, FALSE otherwise
 */
		bool linkFiltered(Pad *sinkpad, Caps *filtercaps);

/**
 * Try to relink this pad and its peer with the specified caps. 
 *
 * @param sinkpad the sink pad to relink
 * @param filtercaps the capabilities to use in the relinking
 * @return a boolean indicating the peer pad could accept the caps.
 *    if FALSE is returned, the pads are unlinked.
 */
		bool relinkFiltered(Pad *sinkpad, Caps *filtercaps);

/**
 * Try to relink this pad and its peer with the specified caps
 *
 * @param sinkpad the sink pad to relink
 * @param filtercaps the capabilities to use in the relinking
 * @return a boolean indicating the peer pad could accept the caps.
 */
		bool tryRelinkFiltered(Pad *sinkpad, Caps *filtercaps);

/**
 * Unlinks the source pad from the sink pad.
 *
 * @param sinkpad the sink pad to unlink
 */
		void unlink(Pad *sinkpad);

/**
 * Checks if the source pad can be linked to the sink pad.
 *
 * @param sinkpad the sink pad to link
 * @return TRUE if the pads can be link, FALSE otherwise
 */
		bool canLink(Pad *sinkpad);

/**
 * Get the padtemplate object of this pad.
 *
 * @return the padtemplate object
 */
		PadTemplate* getPadTemplate();

/**
 * Get the capabilities of this pad.
 *
 * @return a list of the capabilities of this pad
 */
		Caps* getPadTemplateCaps();

/**
 * Get the caps of the allowed media types that can
 * go through this pad.
 *
 * @return the allowed caps, newly allocated
 */
		Caps* getAllowedCaps();

/**
 * Attempt to relink the pad to its peer through its filter, 
 * set with [re]linkFiltered. This function is useful when a
 * plugin has new capabilities on a pad and wants to notify the peer.
 *
 * @return TRUE on success, FALSE otherwise.
 */
		bool recalcAllowedCaps();

/**
 * Try to negotiate the pads.
 *
 * @param sinkpad a sinkpad 
 * @return a boolean indicating the pad succesfully negotiated.
 */
		bool performNegotiate(Pad *sinkpad);

/**
 * Send the event to the pad.
 *
 * @param event the event to send to the pad.
 * @return TRUE if the event was handled.
 */
		bool sendEvent(Event *event);

/**
 * Invoke the default query function on a pad. 
 *
 * @param type the type of query to perform
 * @param format a pointer to the format of the query
 * @param value a pointer to the result of the query
 * @return TRUE if the query could be performed.
 */
		bool queryDefault(QueryType type,
				  Format *format,  long long *value);

/**
 * Query a pad for one of the available PadQuery properties.
 *
 * @param type the type of query to perform
 * @param format a pointer to the format of the query
 * @param value a pointer to the result of the query
 * @return TRUE if the query could be performed.
 */
		bool query(QueryType type,
			   Format *format, long long *value);
	};
}
}

#endif
