/*
 *  Interface/wrapper for GStreamer GstPadTemplate
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_PADTEMPLATE_H
#define KDE_GST_PADTEMPLATE_H

#include <kde/gst/object.h>
#include <kde/gst/pad.h>

#include <qvaluevector.h>


namespace KDE {
namespace GST {
	class Caps;

/**
 * Padtemplates describe the possible media types a pad or an elementfactory 
 * can handle. 
 *
 * Pad and PadTemplates have @ref Caps attached to it to describe the media 
 * type they are capable of dealing with.  @ref #getCaps() is used to get the
 * caps of a padtemplate. It's not possible to modify the caps of a 
 * padtemplate after creation. 
 * 
 * A padtemplate can be used to create a pad or to add to an elementfactory.
 *
 * @short Describe the media type of a pad.
 * @see Pad
 * @see ElementFactory
 */
	class PadTemplate : public Object {
	Q_OBJECT
	private:
                friend class GStreamer;
                static Wrapper* wrapperFactory(void *real);
		void *reserved;
		
	public:
		enum Presence {
		  PAD_ALWAYS,
		  PAD_SOMETIMES,
		  PAD_REQUEST
		};

/**
 * Creates a new PadTemplate that wrapps the given GstElement. Usually
 * you really don't want to call this, use @ref #wrap instead.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstPadTemplate to be wrapped
 */
	        PadTemplate(void *real);
	        virtual ~PadTemplate();

/**
 * Creates a new PadTemplate that wrapps the given GstPadTemplate. 
 * Unlike the PadTemplate constructor this function also re-uses 
 * existing wrapper objects.
 *
 * @param real the GstPadTemplate to be wrapped
 * @return the wrapped PadTemplate
 */
		static PadTemplate* wrap(void *real);

/**
 * Creates a new padtemplate from the given arguments.
 * You must not create a object of this class on the stack, always 
 * use new.
 *
 * @param nameTemplate the name template
 * @param direction the direction for the template
 * @param presence the presence of the pad
 * @param caps1 capabilities for the template
 * @param caps2 more capabilities for the template (can be 0)
 * @param caps3 more capabilities for the template (can be 0)
 * @param caps4 more capabilities for the template (can be 0)
 * @param caps5 more capabilities for the template (can be 0)
 */
		PadTemplate(const QString &nameTemplate,
			    Pad::Direction direction, 
			    Presence presence,
			    Caps *caps1,
			    Caps *caps2 = 0,
			    Caps *caps3 = 0,
			    Caps *caps4 = 0,
			    Caps *caps5 = 0);

/**
 * Get the capabilities of the padtemplate
 *
 * @return a Caps*
 */
		Caps* getCaps();

/**
 * Get the capability with the given name from this padtemplate.
 *
 * @param name the name of the capability to get
 * @return a capability or NULL if not found
 */
		Caps* getCapsByName(const QString &name);

/**
 * Get the direction of the padtemplate.
 *
 * @return the direction
 */
		Pad::Direction direction();

/**
 * Get the nametemplate of the padtemplate.
 *
 * @return the nametemplate
 */
		QString nameTemplate();

/**
 * Get the presence of the padtemplate.
 *
 * @return the presence
 */
		Presence presence();

/**
 * Check if the properties of the padtemplate are fixed.
 *
 * @return true if the padtemplate is fixed
 */
		bool isFixed();
	};
}
}

#endif
