/*
 *  Wrapper base class for GstProps
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "props.h"
#include "helper.h"

extern "C" {
#include <gst/gst.h>
}

#include <qglobal.h>
#include <stdlib.h>
#include <string.h>

using namespace KDE::GST;


#define r() ((GstProps*)m_real)

Props::Props(void *real) :
	SimpleWrapper(real) {
}

Props::~Props() {
	if (m_real)
		gst_props_destroy(r());
}

Props* Props::wrap(void *real) {
	return (Props*) SimpleWrapper::wrap(real);
}

Props::Props() :
  SimpleWrapper(0) {
	wrap(gst_props_empty_new());
}

Props* Props::merge(Props *tomerge) {
	return wrap(gst_props_merge(r(), (GstProps*)tomerge->m_real));
}

Props* Props::copy () {
	return wrap(gst_props_copy(r()));
}

Props* Props::copyOnWrite() {
	return wrap(gst_props_copy_on_write(r()));
}

void Props::ref() {
	gst_props_ref(r());
}

void Props::unref() {
	gst_props_unref(r());
}

bool Props::checkCompatibility(Props *toprops) {
	return gst_props_check_compatibility(r(),(GstProps*)toprops->m_real);
}

bool Props::hasProperty(const QString &name) {
	return gst_props_has_property(r(), name.latin1());
}

QValueVector<PropsEntry*> Props::normalize() {
	return convertList<PropsEntry>(gst_props_normalize(r()));
}

Props* Props::set(const QString &name, int value) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_INT(value)));
}

Props* Props::setIntAny(const QString &name) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_INT_ANY));
}

Props* Props::setIntRange(const QString &name, int valuea, int valueb) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_INT_RANGE(valuea, valueb)));
}

Props* Props::setIntNegative(const QString &name) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_INT_NEGATIVE));
}

Props* Props::setFourCC(const QString &name, int value) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_FOURCC(value)));
}

Props* Props::setFourCC(const QString &name, const QString &value) {
	return wrap(gst_props_set(r(), name.latin1(), 
				  GST_PROPS_FOURCC(GST_STR_FOURCC(value.latin1()))));
}

Props* Props::setFourCC(const QString &name, 
			char a, char b, char c, char d) {
	return wrap(gst_props_set(r(), name.latin1(), 
				  GST_PROPS_FOURCC(GST_MAKE_FOURCC(a, b, c, d))));
}

Props* Props::set(const QString &name, bool value) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_BOOLEAN(CONVERT_GBOOL(value))));	
}

Props* Props::set(const QString &name, const QString &value) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_STRING(value.latin1())));
}

Props* Props::set(const QString &name, float value) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_FLOAT(value)));
}

Props* Props::set(const QString &name, float valuea, float valueb) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_FLOAT_RANGE(valuea, valueb)));
}

Props* Props::setIntPositive(const QString &name) {
	return wrap(gst_props_set(r(), name.latin1(), GST_PROPS_INT_POSITIVE));
}

int Props::getInt(const QString &name) {
	int i;
	if (gst_props_entry_get_int(gst_props_get_entry(r(), name.latin1()), &i))
		return i;
	else
		return 0;
}

bool Props::getIntRange(const QString &name, int &a, int &b) {
	int i1, i2;
	bool r = gst_props_entry_get_int_range(gst_props_get_entry(r(), name.latin1()), &i1, &i2);
	if (r) {
		a = i1;
		b = i2;
		return true;
	}
	return false;
}

float Props::getFloat(const QString &name) {
	float f;
	if (gst_props_entry_get_float(gst_props_get_entry(r(), name.latin1()), &f))
		return f;
	else
		return 0;
}

bool Props::getFloatRange(const QString &name, float &a, float &b) {
	float f1, f2;
	bool r = gst_props_entry_get_float_range(gst_props_get_entry(r(), name.latin1()), &f1, &f2);
	if (r) {
		a = f1;
		b = f2;
		return true;
	}
	return false;
}

int Props::getFourCCInt(const QString &name) {
	int i;
	if (gst_props_entry_get_int(gst_props_get_entry(r(), name.latin1()), &i))
		return i;
	else
		return 0;	
}

bool Props::getBool(const QString &name, bool &value) {
	gboolean b;
	if (gst_props_entry_get_int(gst_props_get_entry(r(), name.latin1()), &b)) {
		value = CONVERT_GBOOL(b);
		return true;
	}
	return false;
}

bool Props::getBool(const QString &name) {
	bool b = false;
	getBool(name, b);
	return b;
}

bool Props::hasFixedProperty(const QString &name) {
	return gst_props_has_fixed_property(r(), name.latin1());
}

bool Props::hasPropertyTyped(const QString &name, PropsType type) {
	return gst_props_has_property_typed(r(), 
					    name.latin1(), 
					    (GstPropsType) type);
}

Props* Props::intersect(Props *props2) {
	return wrap(gst_props_intersect(r(), (GstProps*)props2->m_real));
}

namespace {
	PropsEntry *createEntry(const GstPropsEntry *gpe) {
		return PropsEntry::wrap((void*)gpe);
	}
}

PropsEntry *PropsEntry::wrap(void *gpev) {
	const GstPropsEntry *gpe = (const GstPropsEntry *) gpev;
	PropsEntry *pe = new PropsEntry;
	pe->type = (Props::PropsType) gst_props_entry_get_type(gpe);
	bool s = true;
	switch (pe->type) {
	case Props::PROPS_INVALID_TYPE:
		return 0;
	case Props::PROPS_INT_TYPE:
		s = gst_props_entry_get_int(gpe, &pe->data.vInt);
		break;
	case Props::PROPS_FLOAT_TYPE:
		s = gst_props_entry_get_float(gpe, &pe->data.vFloat);
		break;
	case Props::PROPS_FOURCC_TYPE:
		s = gst_props_entry_get_fourcc_int(gpe, &pe->data.vFourcc);
		break;
	case Props::PROPS_BOOL_TYPE: {
		gboolean b = false;
		s = gst_props_entry_get_boolean(gpe, &b);
		pe->data.vBool = CONVERT_GBOOL(b);
		break;
	}
	case Props::PROPS_STRING_TYPE: {
		const gchar *str = 0;
		s = gst_props_entry_get_string(gpe, &str);
		if (s && str)
			pe->data.vString = strdup(str);
		else
			pe->data.vString = 0;
		break;
	}
	case Props::PROPS_FLOAT_RANGE_TYPE:
		s = gst_props_entry_get_float_range(gpe, 
						    &pe->data.vFloatRange.a, 
						    &pe->data.vFloatRange.b);
		break;
	case Props::PROPS_INT_RANGE_TYPE:
		s = gst_props_entry_get_int_range(gpe, 
						  &pe->data.vIntRange.a, 
						  &pe->data.vIntRange.b);
		break;
	case Props::PROPS_LIST_TYPE: {
		const GList *l;
		pe->data.vList = 0;
		s = gst_props_entry_get_list(gpe, &l);
		if (!s)
			break;
		
		const GList *f = g_list_first((GList*)l);
		const GList *last = g_list_last((GList*)l);
		
		pe->data.vList = new QValueVector<PropsEntry*>(g_list_length((GList*)l));
		if (!l)
			break;
		
		do {
			pe->data.vList->push_back(createEntry((GstPropsEntry*)f->data));
			if (f == last)
				break;
			f = g_list_next((GList*)f);
		} while (true);
	}
	default:
		qFatal("Unknown props entry type\n");
	}
	if (!s)
		qFatal("Unexpected props entry type\n");
	return pe;
}

const PropsEntry* Props::getEntry(const QString &name) {
	const GstPropsEntry *gpe = gst_props_get_entry(r(), name.latin1());
	if (!gpe)
		return 0;
	return createEntry(gpe);
}

PropsEntry::~PropsEntry() {
	if (type == Props::PROPS_LIST_TYPE) {
		QValueVector<PropsEntry*>::iterator i = data.vList->begin();
		while (i != data.vList->end()) {
			delete (*i);
			i++;
		}
		delete data.vList;
	}
	else if (type == Props::PROPS_STRING_TYPE) {
		if (data.vString)
			free(data.vString);
	}
}

