/*
 *  Wrapper base class for GObjects
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "wrapper.h"
#include "callbackmanager.h"

extern "C" {
#include <gst/gst.h>
}

using namespace KDE::GST;

QMap<void*,Wrapper*> Wrapper::instanceMap;
QMap<QString,KDE::GST::WrapObjectFactory*> Wrapper::classMap;

#define rg() G_OBJECT(m_real)

static void GObjectDisassociate(void *data, _GObject*) {
	Wrapper *o = (Wrapper*) data;
	o->disassociate(false);
}

// helper factory function for classMap. See used in gstreamer.cpp
Wrapper* Wrapper::wrapperFactory(void *real) {
	return new Wrapper(real);
}

Wrapper::Wrapper(void *real) :
	m_real(real),
	m_cbm(0),
	m_signalMap(0)
{
	if (m_real)
		registerReal();
}

Wrapper::~Wrapper() {
	if (m_real) {
		removeFromMap();
		g_object_weak_unref((_GObject*)m_real, 
				    ::GObjectDisassociate, 
				    this);
		if (g_type_is_a(G_OBJECT_TYPE(m_real), 
				g_type_from_name("GstObject")))
			gst_object_destroy(GST_OBJECT(m_real));
	}
	if (m_cbm)
		delete m_cbm;
	if (m_signalMap)
		delete m_signalMap;
}

void Wrapper::registerReal() {
	g_object_weak_ref((_GObject*)m_real,
			  ::GObjectDisassociate, 
			  this);
	addToMap();
}

CallbackManager *Wrapper::getCBManager() {
	if (!m_cbm)
		m_cbm = new CallbackManager();
	return m_cbm;
}

void *Wrapper::disassociate(bool weakUnref) {
	void *obj = m_real;
	removeFromMap();

	/** Disconnect wrapped callbacks **/
	if (m_cbm && m_cbm->callbackDataList) {
		CallbackData *cd = m_cbm->callbackDataList->first();
		while (cd) {
			if (cd->unregisterSignal)
				unregisterRawSignal(cd->name, 
						    cd->wrapCallback, 
						    cd);
			cd = m_cbm->callbackDataList->next();
		}
	}

	if (weakUnref) {
		/* Dont get notified when the object gets destroyed later */
		
		g_object_weak_unref((_GObject*)m_real, 
				    ::GObjectDisassociate, 
				    this);
	}

	/* Set to null, to the destructor does not delete the object*/
	m_real = 0;
	/* Delete this object, do not put any code after the delete */
	delete this;
	return obj;
}

void Wrapper::addToMap() {
	instanceMap.insert(m_real, this, false);
}

void Wrapper::removeFromMap() {
	instanceMap.remove(m_real);
}

Wrapper* Wrapper::getFromMap(void *real) {
	QMap<void*,Wrapper*>::iterator i = instanceMap.find(real);
	if (i == instanceMap.end())
		return 0;
	return i.data();
}

Wrapper* Wrapper::wrap(void *real) {
	if (!real)
		return 0;

	Wrapper *o = getFromMap(real);
	if (o)
		return o;
	return createWrapObject(real);
}


Wrapper* Wrapper::createWrapObject(void *real) {
	if (!real)
		return 0;

	GType t = G_OBJECT_TYPE(G_OBJECT(real));
	WrapObjectFactory *ocf = 0;
	while (t) {
		const gchar *n = g_type_name(t);
		QString name(n);

		if (classMap.contains(name)) {
			ocf = classMap[name];
			break;
		}
		t = g_type_parent(t);
	}
	if (!ocf)
		return 0;
	return (*ocf)(real);
}

void Wrapper::registerType(const QString &name, 
			   WrapObjectFactory f) {
	classMap[name] = f;
}

void *Wrapper::realObject() const {
	return m_real;
}

namespace {
	// mangle signal name + callback address into single string,
	// to simplify the map
	QString mangleCallback(const QString &name, 
			       void* callback, 
			       void *cookie) {
		return QString("%1#%2#%3").arg(name)
			.arg((ulong) callback)
			.arg((ulong) cookie);
	}
}

bool Wrapper::registerRawSignal(const QString &name,
				void *callback,
				void *cookie) {
	QString n = mangleCallback(name, callback, cookie);
	if (!m_signalMap)
		m_signalMap = new QMap<QString, unsigned long>();
	if (m_signalMap->contains(n))
		return true;

	gulong l = g_signal_connect(m_real,
				    name.latin1(),
				    (GCallback) callback,
				    cookie);
	if (!l)
		return false;

	m_signalMap->insert(n, l);
	return true;
}

void Wrapper::unregisterRawSignal(const QString &name,
				  void *callback,
				  void *cookie) {
	QString n = mangleCallback(name, callback, cookie);
	if (!m_signalMap)
		return;
	if (!m_signalMap->contains(n))
		return;

	gulong l = (*m_signalMap)[n];
	g_signal_handler_disconnect(m_real, l);
	m_signalMap->remove(n);
}


void Wrapper::setChar(const QString &name, char value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setUChar(const QString &name, unsigned char value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setBool(const QString &name, bool value) {
	g_object_set(rg(), name.latin1(), 
		     (gboolean) (value ? TRUE : FALSE), NULL);
}

void Wrapper::setInt(const QString &name, int value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setUInt(const QString &name, unsigned int value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setLong(const QString &name, long value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setULong(const QString &name, unsigned long value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setInt64(const QString &name, long long value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setUInt64(const QString &name, unsigned long long value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setFloat(const QString &name, float value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setDouble(const QString &name, double value) {
	g_object_set(rg(), name.latin1(), value, NULL);
}

void Wrapper::setString(const QString &name, const QString &value) {
	g_object_set(rg(), name.latin1(), value.latin1(), NULL);
}

void Wrapper::setPointer(const QString &name, void *pointer) {
	g_object_set(rg(), name.latin1(), pointer, NULL);
}

char Wrapper::getChar(const QString &name) {
	GValue value;
	memset(&value, 0, sizeof(value));
	g_value_init(&value, G_TYPE_CHAR);
	g_object_get_property(rg(), name.latin1(), &value);
	return g_value_get_char(&value);	
}

unsigned char Wrapper::getUChar(const QString &name) {
	GValue value;
	memset(&value, 0, sizeof(value));
	g_value_init(&value, G_TYPE_UCHAR);
	g_object_get_property(rg(), name.latin1(), &value);
	return g_value_get_uchar(&value);	
}

int Wrapper::getInt(const QString &name) {
	return gst_util_get_int_arg(rg(), name.latin1());
}

unsigned int Wrapper::getUInt(const QString &name) {
	GValue value;
	memset(&value, 0, sizeof(value));
	g_value_init(&value, G_TYPE_UINT);
	g_object_get_property(rg(), name.latin1(), &value);
	return g_value_get_uint(&value);	
}

long Wrapper::getLong(const QString &name) {
	return gst_util_get_long_arg(rg(), name.latin1());
}

unsigned long Wrapper::getULong(const QString &name) {
	GValue value;
	memset(&value, 0, sizeof(value));
	g_value_init(&value, G_TYPE_ULONG);
	g_object_get_property(rg(), name.latin1(), &value);
	return g_value_get_ulong(&value);
}

bool Wrapper::getBool(const QString &name) {
	return gst_util_get_bool_arg(rg(), name.latin1());
}

float Wrapper::getFloat(const QString &name) {
	return gst_util_get_float_arg(rg(), name.latin1());
}

double Wrapper::getDouble(const QString &name) {
	return gst_util_get_double_arg(rg(), name.latin1());
}

QString Wrapper::getString(const QString &name) {
	return QString(gst_util_get_string_arg(rg(), name.latin1()));
}

long long Wrapper::getInt64(const QString &name) {
	return gst_util_get_int64_arg(rg(), name.latin1());
}

unsigned long long Wrapper::getUInt64(const QString &name) {
	GValue value;
	memset(&value, 0, sizeof(value));
	g_value_init(&value, G_TYPE_UINT64);
	g_object_get_property(rg(), name.latin1(), &value);
	return g_value_get_uint64(&value);	
}

void *Wrapper::getPointer(const QString &name) {
	return gst_util_get_pointer_arg(rg(), name.latin1());
}
	
#include "wrapper.moc"

