/* $Id: net.c,v 1.92 2007/05/13 22:00:56 ekalin Exp $ */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <string.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <sys/time.h>
#include <libintl.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <zlib.h>

#ifdef __MINGW32__
#  include <windows.h>
#  include <winsock2.h>
#endif

#include "kildclient.h"
#include "perlscript.h"
#include "ansi.h"


/*************************
 * File global variables *
 *************************/
#define MAX_US 250000
static char *terminal_type = "KildClient";



/***************************
 * Telnet protocol defines *
 ***************************/
#define TELNET_IAC               255        /* 0xFF, 0377 */
#define TELNET_WILL              251        /* 0xFB, 0373 */
#define TELNET_WONT              252        /* 0xFC, 0374 */
#define TELNET_DO                253        /* 0xFD, 0375 */
#define TELNET_DONT              254        /* 0xFE, 0376 */
#define TELNET_SB                250        /* 0xFA, 0372 */
#define TELNET_SE                240        /* 0xF0, 0360 */
#define TELNET_GA                249        /* 0xF9, 0371 */

#define TELNET_OPT_ECHO          1          /* D'oh */
#define TELNET_OPT_NAWS          31         /* 0x1f, 0037 */

#define TELNET_OPT_TERMINAL_TYPE 24         /* 0x18, 0030 */
#define TELNET_TERMTYPE_IS       0
#define TELNET_TERMTYPE_SEND     1

#define TELNET_OPT_COMPRESS      85         /* 0x55, 0125, "U" */
#define TELNET_OPT_COMPRESS2     86         /* 0x56, 0126, "V" */


/***********************
 * Function prototypes *
 ***********************/
static void     disconnected_decide_action(World *world, gint action);
static gboolean flush_buffer(World *world, int action);
static void     process_recv_line(World *world,
                                  int start, int end,
                                  gboolean incomplete);
static void     process_telnet_iac(World *world, int pos, int end);
static void     start_mccp(World *world, int pos, int end);
static void     end_mccp(World *world);
static ssize_t  write_escaped(World *world, const void *buf, size_t count);
/* Send and receive data */
#ifdef HAVE_LIBGNUTLS
static ssize_t kc_recv(World *world, void *buf, size_t s);
static ssize_t kc_send(World *world, const void *buf, size_t s);
#else
#define kc_recv(w, b, s) (recv((w)->sock, (b), (s), 0))
#define kc_send(w, b, s) (send((w)->sock, (b), (s), 0))
#endif


gboolean
disconnect_world(gpointer data)
{
  /* This function is called as an idle function, so it must return FALSE. */
  World *world;
  char  *msg;
  gint   action;

  world = (World *) data;

  if (!world->connected) {
    return FALSE;
  }

  msg = close_connection(world);

  execute_hook(world, "OnDisconnect", NULL);

  currentWorld = NULL;

  /* Remove the tooltip */
  worldgui_destroy_tooltip_window(world->gui);
  if (world->gui->tooltip_timeout_id) {
    g_source_remove(world->gui->tooltip_timeout_id);
    world->gui->tooltip_timeout_id = 0;
  }

  /* Here we decide what to do */
  action = disconnected_msgbox(msg, TRUE);
  g_free(msg);

  disconnected_decide_action(world, action);

  return FALSE;
}


gboolean
reconnect_world(gpointer data)
{
  /* This function is called as an idle function, so it must return FALSE. */
  disconnected_decide_action((World *) data, RESP_RECONNECT);
  return FALSE;
}


gboolean
connect_another(gpointer data)
{
  /* This function is called as an idle function, so it must return FALSE. */
  disconnected_decide_action((World *) data, RESP_OTHER);
  return FALSE;
}


static
void
disconnected_decide_action(World *world, gint action)
{
  gchar *errmsg;

  while (action != GTK_RESPONSE_CLOSE) {
    if (action == RESP_OTHER) {
      gboolean offline_selected = FALSE;

      save_world_to_file(world);
      save_command_history(world);
      save_permanent_variables(world);

      open_new_world(world->gui, NULL, TRUE, &offline_selected);
      if (offline_selected) {
        mark_as_offline(world);
        return;
      }

      break;
    }
    if (action == RESP_OFFLINE) {
      mark_as_offline(world);
      return;
    }
    /* Else, which means reconnect */
    if (connect_to(world, &errmsg)) {
      return;
    }
    action = disconnected_msgbox(errmsg, TRUE);
    g_free(errmsg);
  }
  /* If we have gotten here, it means the current world must be closed,
     because either we selected close or connected to another one. */
  remove_world(world, action == GTK_RESPONSE_CLOSE);
}


gchar *
close_connection(World *world)
{
  /* Note: this function should only be called if the world is
     connected. */
  gchar *msg;

  execute_hook(world, "OnCloseConnected", NULL);

  flush_buffer(world, 2);
  if (world->err_no) {
    msg = g_strdup_printf(_("Disconnected from host %s: %s."),
                          world->host, g_strerror(world->err_no));
    world->err_no = 0;
  } else {
    msg = g_strdup_printf(_("Disconnected from host %s."), world->host);
  }
  ansitextview_append_string_with_fgcolor(world->gui, msg,
                                          globalPrefs.idxInfoMsgColor);
  ansitextview_append_string(world->gui, "\n");

  close(world->sock);
  world->connected = FALSE;
#ifdef USE_LIBGNUTLS
  if (world->use_tls) {
    gnutls_deinit(world->tls_session);
  }
#endif

  g_io_channel_unref(world->iochan);

  g_source_remove(world->io_watch_id);
  if (world->timeout_id) {
    g_source_remove(world->timeout_id);
    world->timeout_id = 0;
  }

  if (world->zstream) {
    end_mccp(world);
  }

  return msg;
}


gboolean
data_ready_cb(GIOChannel   *iochan,
              GIOCondition  cond,
              gpointer      data)
{
  World          *world;
  ssize_t         nread;
  unsigned char  *nlptr;
  int             start;
  int             nlpos;

  world = (World *) data;

  /* What if we have read a very long line and the buffer is full?
     It does happen in some muds. */
  if (world->pbuf_size == MAX_BUFFER) {
    flush_buffer(world, 1);
  }

  nread = kc_recv(world,
                  world->inbuffer,
                  MAX_BUFFER - world->pbuf_size);
  if (nread <= 0) {      /* Connection closed */
    if (nread == -1) {
      world->err_no = errno;
    } else {
      world->err_no = 0;
    }
    disconnect_world(world);
    return FALSE;
  }

  world->rawbytes += nread;
  if (rdumpFile) {
    fwrite(world->inbuffer, nread, 1, rdumpFile);
  }

  /* Prepare data to be uncompressed */
  if (world->zstream) {
    world->zstream->next_in  = world->inbuffer;
    world->zstream->avail_in = nread;
  }

  do {
    if (world->zstream) {
      if (world->zstream->avail_in) {
        gint status;

        /* What if we have read a very long line and buffer is full?
           It does happen in some muds. */
        if (world->pbuf_size == MAX_BUFFER) {
          flush_buffer(world, 1);
        }

        world->zstream->next_out  = world->dbuffer;
        world->zstream->avail_out = MAX_BUFFER - world->pbuf_size;
        status = inflate(world->zstream, Z_SYNC_FLUSH);
        if (status == Z_OK || status == Z_STREAM_END) {
          nread = MAX_BUFFER - world->pbuf_size - world->zstream->avail_out;
          if (status == Z_STREAM_END) {
            end_mccp(world);
          }
          /* If there was an error, we're in trouble.
             Let's simply ignore it. */
        }
      } else {
        break;
      }
    } else {
      memcpy(world->dbuffer, world->inbuffer, nread);
    }

    world->bytes += nread;
    if (dumpFile) {
      fwrite(world->dbuffer, nread, 1, dumpFile);
    }

    process_telnet_iac(world, 0, nread);

    start = 0;
    nlpos = -1;
    nlptr = (unsigned char *) memchr(world->pbuffer, '\n', world->pbuf_size);
    while (nlptr) {
      nlpos = nlptr - world->pbuffer;

      process_recv_line(world, start, nlpos + 1, FALSE);
      /* We set printed_until to zero to indicate that a full line
         has been printed and there is no more pending text. */
      world->printed_until = 0;
      start = nlpos + 1;
      nlptr = (unsigned char *) memchr(world->pbuffer + start,
                                       '\n',
                                       world->pbuf_size - start);
    }

    memmove(world->pbuffer,
            world->pbuffer + nlpos + 1,
            world->pbuf_size - nlpos - 1);
    world->pbuf_size = world->pbuf_size - nlpos - 1;
  } while (world->zstream);
  /* When data is not compressed, the end condition will be always
     false and the loop will execute only once. When data is
     compressed, the end condition will be always true, and the loop
     will run forever, until stopped by the break condition above. */

  /* Let other things happen before more data comes, including drawing
     the just received lines. */
  while (gtk_events_pending())
    gtk_main_iteration();

  /* Set up a timeout to print incomplete lines if no more data arrives. */
  if (world->pbuf_size && !world->timeout_id) {
    world->timeout_id = g_timeout_add_full(G_PRIORITY_LOW,
                                           150,
                                           flush_buffer_timeout_cb,
                                           world,
                                           NULL);
  }

  return TRUE;
}


gboolean
flush_buffer_timeout_cb(gpointer data)
{
  gboolean return_value;

  return_value = flush_buffer((World *) data, 0);
  if (!return_value) {
    ((World *) data)->timeout_id = 0;
  }
  return return_value;
}


static
gboolean
flush_buffer(World *world, int action)
{
  /*
   * If action != 0, flush it anyway.
   * If action = 0, only flush if last data written was more than
   *    200ms ago.
   * If action = 2, stop this callback aditionally.
   */
  gulong us_diff;

  /* Called when a command is sent, the line must be marked as new,
     even if there is nothing new. */
  if (action && world->pbuf_size) {
    process_recv_line(world, 0, world->pbuf_size,
                      FALSE);
    world->printed_until = 0;
    world->pbuf_size = 0;
  }

  /* If action is != 0, this will always be true */
  if (world->pbuf_size == world->printed_until) {
    return (action != 2);
  }

  /* This part below only executes if action == 0 */
  g_timer_elapsed(world->last_data_written, &us_diff);
  if (us_diff < MAX_US) {
    return TRUE;
  }

  /* action = 1 is when flushing because a command is sent. We should
     consider lines as finished in this case. */
  process_recv_line(world, world->printed_until, world->pbuf_size,
                    TRUE);
  world->printed_until = world->pbuf_size;

  return FALSE;
}


static
void
process_recv_line(World *world, int start, int end, gboolean incomplete)
{
  guchar      *converted;
  gsize        output_len;
  guchar      *stripped = NULL;
  GError      *error = NULL;
  gboolean     gag_output = FALSE;
  gboolean     gag_log    = FALSE;
  gint         dummy;
  GtkTextIter  line_start;
  GtkTextIter  line_end;

  /* Clear pending line if it gets completed */
  if (!incomplete && world->printed_until) {
    gtk_text_buffer_get_iter_at_mark(world->gui->txtBuffer,
                                     &line_start,
                                     world->gui->txtmark_linestart);
    gtk_text_buffer_get_iter_at_mark(world->gui->txtBuffer,
                                     &line_end,
                                     world->gui->txtmark_end);
    gtk_text_buffer_delete(world->gui->txtBuffer,
                           &line_start, &line_end);

    /* Since an ANSI sequence cannot be incomplete at the end of
       one line and continue at another (because \n is not valid
       inside ANSI strings), ANSI parsing at a line always starts
       in DATA mode. */
    world->gui->ta.ansifsm_state  = ANSIFSM_DATA;
    world->gui->ta.ansiseq_pos = 0;
    /* Furthermore, colors are restored to what they were in the
       end of the last complete line. */
    world->gui->ta.state = world->gui->ta.saved_state;
  }

  converted = (guchar *) g_convert((gchar *) (world->pbuffer + start),
                                   end - start,
                                   VTE_ENC, world->charset,
                                   NULL, &output_len,
                                   &error);
  if (converted) {
    if (!incomplete) {
      stripped = strip_ansi(converted, output_len);

      match_triggers(world,
                     &stripped, &converted, &output_len,
                     &gag_output, &gag_log,
                     &dummy, NULL);

      /* The line is written to the log file in match_triggers. */
    } else {
      ansitextview_append_ansi_string(world->gui,
                                      (gchar *) converted, output_len);
    }

    if (!gag_output) {
      /* The actual printing of the line is done in match_triggers. */
      if (!world->has_focus) {
        if (!world->has_unread_text) {
          world->has_unread_text = TRUE;
          gtk_label_set_markup(world->gui->lblNotebook, world->new_text_str);
          if (worlds_with_new_text != -1) {
            ++worlds_with_new_text;

            if (!window_has_focus) {
              adjust_window_title();
            }
          }
        }
      }
      if (!incomplete) {
        execute_hook(world, "OnReceivedText", (gchar *) converted);
      }
    }
    g_free(converted);
  } else {
    fprintf(stderr, "Error in charset conversion: %s\n", error->message);
  }

  g_timer_start(world->last_data_written);
}


static
void
process_telnet_iac(World *world, int pos, int end)
{
  /* This function processes Telnet IAC sequences and converts the
     newlines to the canonical \n. */
  unsigned char iac_response[4];
  time_t        now;

  while (pos < end) {
    switch (world->telnet_state) {
    case TELNETFSM_DATA:
      switch (world->dbuffer[pos]) {
      case TELNET_IAC:
        world->telnet_state = TELNETFSM_IAC;
        break;

      case '\n':
        world->pbuffer[world->pbuf_size++] = world->dbuffer[pos];
        world->telnet_state = TELNETFSM_DATA_NL;
        break;

      case '\r':
        world->telnet_state_ncrs = 1;
        world->telnet_state = TELNETFSM_DATA_CR;
        break;

      default:
        world->pbuffer[world->pbuf_size++] = world->dbuffer[pos];
      }
      break;

    case TELNETFSM_DATA_NL:
      if (world->dbuffer[pos] != '\r') {
        world->pbuffer[world->pbuf_size++] = world->dbuffer[pos];
        if (world->dbuffer[pos] != '\n') {
          world->telnet_state = TELNETFSM_DATA;
        }
      } else {
        world->telnet_state = TELNETFSM_DATA;
      }
      break;

    case TELNETFSM_DATA_CR:
      if (world->dbuffer[pos] == '\n') {
        world->pbuffer[world->pbuf_size++] = world->dbuffer[pos];
        world->telnet_state = TELNETFSM_DATA;
      } else if (world->dbuffer[pos] == '\r') {
        ++world->telnet_state_ncrs;
      } else {
        int i;

        /* Add the newlines. But check if there is space before. */
        if (world->pbuf_size + world->telnet_state_ncrs >=
            world->pbuf_alloc_size) {
          world->pbuf_alloc_size += world->telnet_state_ncrs;
          world->pbuffer = g_realloc(world->pbuffer,
                                     world->pbuf_alloc_size);
        }

        for (i = 0; i < world->telnet_state_ncrs; ++i) {
          world->pbuffer[world->pbuf_size++] = '\n';
        }
        world->pbuffer[world->pbuf_size++] = world->dbuffer[pos];
        world->telnet_state = TELNETFSM_DATA;
      }
      break;

    case TELNETFSM_IAC:
      switch (world->dbuffer[pos]) {
      case TELNET_IAC:
        world->pbuffer[world->pbuf_size++] = TELNET_IAC;
        world->telnet_state = TELNETFSM_DATA;
        break;

      case TELNET_DO:
        world->telnet_state = TELNETFSM_DO;
        break;

      case TELNET_DONT:
        world->telnet_state = TELNETFSM_DONT;
        break;

      case TELNET_WILL:
        world->telnet_state = TELNETFSM_WILL;
        break;

      case TELNET_WONT:
        world->telnet_state = TELNETFSM_WONT;
        break;

      case TELNET_SB:
        world->telnet_state = TELNETFSM_SB;
        break;

      default:
        world->telnet_state = TELNETFSM_DATA;
        break;
      }
      break;

    case TELNETFSM_DO:
      switch (world->dbuffer[pos]) {
      case TELNET_OPT_TERMINAL_TYPE:
        iac_response[0] = TELNET_IAC;
        iac_response[1] = TELNET_WILL;
        iac_response[2] = TELNET_OPT_TERMINAL_TYPE;
        kc_send(world, iac_response, 3);
        break;

      case TELNET_OPT_NAWS:
        iac_response[0] = TELNET_IAC;
        iac_response[1] = TELNET_WILL;
        iac_response[2] = TELNET_OPT_NAWS;
        kc_send(world, iac_response, 3);
        world->use_naws = TRUE;
        send_naws_size(world);
      }
      world->telnet_state = TELNETFSM_DATA;
      break;

    case TELNETFSM_DONT:
      switch (world->dbuffer[pos]) {
      case TELNET_OPT_NAWS:
        world->use_naws = FALSE;
        break;
      }
      world->telnet_state = TELNETFSM_DATA;
      break;

    case TELNETFSM_WILL:
      switch (world->dbuffer[pos]) {
      case TELNET_OPT_ECHO:
        world->noecho = TRUE;
        simo_combo_box_set_visibility(world->gui->cmbEntry, FALSE);
        iac_response[0] = TELNET_IAC;
        iac_response[1] = TELNET_DO;
        iac_response[2] = TELNET_OPT_ECHO;
        kc_send(world, iac_response, 3);
        break;

      case TELNET_OPT_COMPRESS2:
        now = time(NULL);
        if (world->mccp_behavior == MCCP_ALWAYS
            || (world->mccp_behavior == MCCP_AFTER_CONNECT
                && difftime(now, world->connected_time) <= 60)) {
          world->mccp_ver = 2;
          iac_response[0] = TELNET_IAC;
          iac_response[1] = TELNET_DO;
          iac_response[2] = TELNET_OPT_COMPRESS2;
          kc_send(world, iac_response, 3);
        }
        break;

      case TELNET_OPT_COMPRESS:
        /* Do nothing if version 2 is supported */
        now = time(NULL);
        if (world->mccp_ver <= 1
            && (world->mccp_behavior == MCCP_ALWAYS
                || (world->mccp_behavior == MCCP_AFTER_CONNECT
                    && difftime(now, world->connected_time) <= 60))) {
          world->mccp_ver = 1;
          iac_response[0] = TELNET_IAC;
          iac_response[1] = TELNET_DO;
          iac_response[2] = TELNET_OPT_COMPRESS;
          kc_send(world, iac_response, 3);
        }
        break;
      }
      world->telnet_state = TELNETFSM_DATA;
      break;

    case TELNETFSM_WONT:
      if (world->dbuffer[pos] == TELNET_OPT_ECHO) {
        world->noecho = FALSE;
        simo_combo_box_set_visibility(world->gui->cmbEntry, TRUE);
        iac_response[0] = TELNET_IAC;
        iac_response[1] = TELNET_DONT;
        iac_response[2] = TELNET_OPT_ECHO;
        kc_send(world, iac_response, 3);
      }
      world->telnet_state = TELNETFSM_DATA;
      break;

    case TELNETFSM_SB:
      if (world->dbuffer[pos] == TELNET_IAC) {
        world->telnet_state = TELNETFSM_SB_IAC;
      } else if (world->dbuffer[pos] == TELNET_OPT_TERMINAL_TYPE) {
        world->telnet_state = TELNETFSM_SB_IAC_TERMTYPE;
      } else if (world->dbuffer[pos] == TELNET_OPT_COMPRESS) {
        world->telnet_state = TELNETFSM_SB_IAC_COMPRESS;
      } else if (world->dbuffer[pos] == TELNET_OPT_COMPRESS2) {
        world->telnet_state = TELNETFSM_SB_IAC_COMPRESS2;
      }
      break;

    case TELNETFSM_SB_IAC:
      if (world->dbuffer[pos] == TELNET_SE) {
        world->telnet_state = TELNETFSM_DATA;
      } else {
        world->telnet_state = TELNETFSM_SB;
      }
      break;

    case TELNETFSM_SB_IAC_TERMTYPE:
      if (world->dbuffer[pos] == TELNET_TERMTYPE_SEND) {
        char *tmp;

        tmp = g_strdup_printf("%c%c%c%c%s%c%c",
                              TELNET_IAC, TELNET_SB,
                              TELNET_OPT_TERMINAL_TYPE, TELNET_TERMTYPE_IS,
                              terminal_type,
                              TELNET_IAC,
                              TELNET_SE);
        kc_send(world, tmp, strlen(terminal_type) + 7);
        g_free(tmp);
      }
      world->telnet_state = TELNETFSM_SB;
      break;

    case TELNETFSM_SB_IAC_COMPRESS:
      if (world->dbuffer[pos] == TELNET_WILL) {
        world->telnet_state = TELNETFSM_SB_IAC_COMPRESS_WILL;
      } else {
        /* This shouldn't happen, and if it does, it will probably mess
           up everything. But let's hope for the best and wait for an SE */
        world->telnet_state = TELNETFSM_SB_IAC;
      }
      break;

    case TELNETFSM_SB_IAC_COMPRESS2:
      if (world->dbuffer[pos] == TELNET_IAC) {
        world->telnet_state = TELNETFSM_SB_IAC_COMPRESS_WILL;
      } else {
        /* This shouldn't happen, and if it does, it will probably mess
           up everything. But let's hope for the best and wait for an SE */
        world->telnet_state = TELNETFSM_SB_IAC;
      }
      break;

    case TELNETFSM_SB_IAC_COMPRESS_WILL:
      if (world->dbuffer[pos] == TELNET_SE) {
        start_mccp(world, pos, end);
        world->telnet_state = TELNETFSM_DATA;
        return;
      } else {
      /* This shouldn't happen, and if it does, it will probably mess
         up everything. But let's hope for the best and wait for an SE */
        world->telnet_state = TELNETFSM_SB_IAC;
      }
    }
    ++pos;
  }
}


static
void
start_mccp(World *world, int pos, int end)
{
  if (world->mccp_ver > 0 && !world->zstream) {
    world->zstream = g_new0(z_stream, 1);
    if (inflateInit(world->zstream) == Z_OK) {
      gchar *status_text;

      memcpy(world->inbuffer, world->dbuffer + pos + 1, end - pos - 1);
      world->zstream->next_in   = world->inbuffer;
      world->zstream->avail_in  = end - pos - 1;
      world->zstream->next_out  = world->dbuffer;
      world->zstream->avail_out = MAX_BUFFER;

      status_text = g_strdup_printf(_("Connected%s to host %s (%s) port %d, MCCP version %d enabled."),
                                    world->use_tls ? _(" with SSL") : "",
                                    world->host, world->host_ip, world->port,
                                    world->mccp_ver);
      gtk_label_set_text(world->gui->lblStatus, status_text);
      g_free(status_text);
    } else {
      /* We're in trouble, but this shouldn't happen */
      g_free(world->zstream);
      world->zstream = NULL;
    }
  }
}


static
void
end_mccp(World *world)
{
  gchar *status_text;

  inflateEnd(world->zstream);
  g_free(world->zstream);
  world->zstream = NULL;

  status_text = g_strdup_printf(_("Connected%s to host %s (%s) port %d."),
                                world->use_tls ? _(" with SSL") : "",
                                world->host, world->host_ip, world->port);
  gtk_label_set_text(world->gui->lblStatus, status_text);
  g_free(status_text);
}


void
send_to_world(World *world, const char *cmd, gint len)
{
  /* cmd should be in UTF-8 */
  gchar    *locale_cmd;
  gsize     locale_len;
  GError   *error = NULL;
  gboolean  has_incomplete_line;

  has_incomplete_line = world->printed_until;
  if (!world->dont_flush) {
    flush_buffer(world, 1);
  }

  locale_cmd = g_convert(cmd, len,
                         world->charset, VTE_ENC,
                         NULL, &locale_len,
                         &error);
  if (error) {
    fprintf(stderr, "%s\n", error->message);
    return;
  }

  /* Flooding prevention */
  if (!world->last_command) {
    world->last_command = locale_cmd;
  } else {
    if (world->flood_prevention) {
      if (strcmp(world->last_command, locale_cmd) == 0) {
        if (++world->repeat_count == world->max_equal_commands) {
          send_to_world_no_check(world,
                                 world->flood_prevention_command,
                                 strlen(world->flood_prevention_command),
                                 NULL, 0,
                                 TRUE,
                                 has_incomplete_line);
          world->repeat_count = 1;
        }
      } else {
        world->repeat_count = 1;
      }
    }

    g_free(world->last_command);
    world->last_command = locale_cmd;
  }

  send_to_world_no_check(world,
                         cmd,
                         len,
                         locale_cmd,
                         locale_len,
                         TRUE,
                         has_incomplete_line);
}


void
send_to_world_no_check(World      *world,
                       const char *utf8_command,
                       int         utf8_len,
                       char       *locale_command,
                       gsize       locale_len,
                       gboolean    add_newline,
                       gboolean    has_incomplete_line)
{
  GError   *error = NULL;
  gboolean  needs_freeing = FALSE;

  if (!world->connected) {
    ansitextview_append_string_nl(world->gui, _("Not connected."));
    return;
  }

  if (!locale_command) {
    locale_command = g_convert(utf8_command, utf8_len,
                               world->charset, VTE_ENC,
                               NULL, &locale_len,
                               &error);
    if (error) {
      fprintf(stderr, "%s\n", error->message);
      return;
    }
    needs_freeing = TRUE;
  }

  write_escaped(world, locale_command, locale_len);
  if (add_newline) {
    kc_send(world, "\r\n", 2);
  }

  if (!world->itime_reset_activate) {
    world->last_command_time = time(NULL);
  }

  if (!world->noecho && world->cmd_echo) {
    ansitextview_append_string_with_fgcolor(world->gui, utf8_command,
                                            globalPrefs.idxCmdEchoColor);
    if (add_newline) {
      ansitextview_append_string(world->gui, "\n");
    }
  } else {
    /* When commands are not echoed, add a newline if there is an
       incomplete line --- most likely a prompt --- so that the
       output of the command appears in the next line. */
    if (has_incomplete_line) {
      ansitextview_append_string(world->gui, "\n");
    }
  }

  execute_hook(world, "OnSentCommand", utf8_command);

  if (needs_freeing) {
    g_free(locale_command);
  }
}


void
perform_auto_logon(World *world)
{
  gchar       *character = NULL;
  gchar       *password;
  gchar       *command = NULL;
  GtkTreeIter  iter;

  if (world->character_used == -1 || world->connection_style == NONE) {
    return;
  }

  gtk_tree_model_iter_nth_child(world->logon_characters,
                                &iter, NULL,
                                world->character_used);
  gtk_tree_model_get(world->logon_characters, &iter,
                     LOGON_CHAR, &character,
                     LOGON_PASS, &password,
                     -1);

  switch (world->connection_style) {
  case DIKU:
    command = g_strdup_printf("%s\r\n%s\r\n", character, password);
    break;

  case LPMUD:
    command = g_strdup_printf("connect %s %s\r\n", character, password);
    break;

  case NONE:
    /* Should never reach here. */
    break;
  }

  write_escaped(world, command, strlen(command));
  g_free(command);
  g_free(character);
  g_free(password);
}


static
ssize_t
write_escaped(World *world, const void *buf, size_t count)
{
  /*
   * Similar to write(), but escapes any IAC character sending
   * IAC IAC instead, as the specification tells us to do.
   */
  ssize_t  written = 0;
  size_t   size;
  void    *iac_pos;
  guchar   iac = TELNET_IAC;

  iac_pos = memchr(buf, TELNET_IAC, count);
  while (iac_pos) {
    size = iac_pos - buf;
    written += kc_send(world, buf, size + 1);
    kc_send(world, &iac, 1);
    count = count - size - 1;
    if (count == 0) {   /* Force exiting of loop */
      iac_pos = NULL;
      break;
    } else {
      buf = iac_pos + 1;
      iac_pos = memchr(buf, TELNET_IAC, count);
    }
  }
  if (count != 0) {
    written += kc_send(world, buf, count);
  }

  return written;
}


void
send_naws_size(World *world)
{
  /* 3 for start, 2 for end. Each value is 2 bytes, but may be 4
     if it happens to be 0xFFFF and must be IAC escaped. Thus 8 for both
     values. */
  guchar   buffer[3 + 8 + 2];
  int      bsize;
  uint16_t size;
  guint    cols;
  guint    rows;

  ansitextview_get_size(world->gui, &rows, &cols);

  buffer[0] = TELNET_IAC;
  buffer[1] = TELNET_SB;
  buffer[2] = TELNET_OPT_NAWS;
  bsize = 3;

  size = htons((uint16_t) cols);
  memcpy(buffer + bsize, &size, sizeof(size));
  /* Escape IAC */
  if (buffer[bsize] == TELNET_IAC) {
    buffer[bsize+2] = buffer[bsize+1];
    buffer[bsize+1] = TELNET_IAC;
    ++bsize;
  }
  ++bsize;
  if (buffer[bsize] == TELNET_IAC) {
    buffer[bsize+2] = buffer[bsize+1];
    buffer[bsize+1] = TELNET_IAC;
    ++bsize;
  }
  ++bsize;

  size = htons((uint16_t) rows);
  memcpy(buffer + bsize, &size, sizeof(size));
  /* Escape IAC */
  if (buffer[bsize] == TELNET_IAC) {
    buffer[bsize+2] = buffer[bsize+1];
    buffer[bsize+1] = TELNET_IAC;
    ++bsize;
  }
  ++bsize;
  if (buffer[bsize] == TELNET_IAC) {
    buffer[bsize+2] = buffer[bsize+1];
    buffer[bsize+1] = TELNET_IAC;
    ++bsize;
  }
  ++bsize;

  buffer[bsize++] = TELNET_IAC;
  buffer[bsize++] = TELNET_SE;
  kc_send(world, buffer, bsize);
}


#ifdef HAVE_LIBGNUTLS
static
ssize_t
kc_recv(World *world, void *buf, size_t s)
{
  if (world->use_tls) {
    return gnutls_record_recv(world->tls_session, buf, s);
  } else {
    return recv(world->sock, buf, s, 0);
  }
}


static
ssize_t
kc_send(World *world, const void *buf, size_t s)
{
  if (world->use_tls) {
    return gnutls_record_send(world->tls_session, buf, s);
  } else {
    return send(world->sock, buf, s, 0);
  }
}
#endif
