/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/* parseconf.c  

   author: Ron Lawrence <lawrence@missioncriticallinux.com>

*/
static const char *version __attribute__ ((unused)) = "$Id: parseconf.c,v 1.27 2000/11/17 14:15:11 lawrence Exp $";

#define USE_MALLOC
/*#define USE_HSEARCH*/

#include <parseconf.h>
#include "database.h"
#include "clu_lock.h"
#include <clusterdefs.h>
#include <search.h>
#include <stdio.h>    
#include <string.h>
#include <regex.h>    /* regular expressions */
#include <stdlib.h>
#include <string.h>
#include <fnmatch.h>
#include <time.h>

#include <sys/stat.h>
#include <unistd.h>
#include <msgsvc.h>
#include <clucfg.h>
#include <config.h>
#ifndef USE_MALLOC
#include "allocate.h"
#endif
#ifndef USE_HSEARCH
#include "hash.h"
#endif

int cluConfigChangeNotification(void);
int check_process_running(char *prog, pid_t *pid);

/* Implementation Private Material */

#define BUFFER_SIZE 1024
#define REGEX_STR_LEN 128

#define CFG_ZERO_LENGTH_FILE -99

static int initialized = 0;
static int loaded = 0;
static int cluster_locking = 1;

static const unsigned int MAX_LINE=512;
static const int MAX_REGEXP_STRING=1024;

static regex_t section_regex;
static regex_t assignment_regex;
static regex_t blank_line_regex;
static regex_t comment_line_regex;
static regex_t start_subsection_regex;
static regex_t end_subsection_regex;
static regex_t c_ident_regex;

static char * cfg_WriteBuffer(char *inbuffer, 
                              ssize_t *size);
static CFG_status cfg_ReadBuffer(char *buffer, 
                                 ssize_t size);

static off_t filesize(char *filename);
int initAlignedBufStuff(void);

static void cfg_UnescapeNewlines(char *str);
static char *cfg_EscapeNewlines(char *str, ssize_t size);


#ifndef USE_MALLOC
static struct MEM_pool * mp=NULL;

#define NEW(size) MEM_pool_allocate_string(mp,(size))
#define DELETE(p) MEM_pool_release_string(mp,(p))

#else

#define NEW(size) malloc((size))
#define DELETE(p) free((p))

#endif

#ifndef USE_HSEARCH
static struct hash_table * table = 0;
#endif 

static char separator = CLU_CONFIG_SEPARATOR;

struct path_elem {
    char * elem;
    struct path_elem* next;
};

struct path_elem path_root = { NULL, NULL };
struct path_elem *path_head=&path_root;

static void push_path_elem(char* new_elem) {
  struct path_elem *n;

  n = (struct path_elem *)
    NEW(sizeof(struct path_elem));
  if(NULL == n) {
    /* disaster */
    abort();
  }
  n->elem = NEW( strlen(new_elem) + 1);
  if(NULL== n->elem) abort();
  strcpy(n->elem, new_elem);
  n->next = path_head;
  path_head = n;
}

static void pop_path_elem(void) {
  struct path_elem *p;
  if(&path_root != path_head) {
    p = path_head->next;
    DELETE( path_head->elem);
    DELETE( (char*)path_head);
    path_head = p;
  } else {
    path_head->next = NULL;
  }
}

static void clear_path_stack(void) {
  while(&path_root != path_head)
    pop_path_elem();
}

static char* reverse_with_delimiters(char*str, char delim) {
  char *s0, *s1, *s2;

  char outbuffer[2048];

  s0 = str;
  s1 = strchr(str, delim);

  if(NULL == s1) {
    return(str);
  }
  strncpy(outbuffer, s0, s1-s0);
  outbuffer[s1-s0]=0;
  
  s2 = reverse_with_delimiters(s1+1,delim);
    
  sprintf(str, "%s%c%s", s2, delim, outbuffer);

  return str;
}

static char* get_path(char sep) {
  char *buffer=NULL;
  struct path_elem *p;
  int count=0;
  int position=0;
  /* pass one, count the charaters */
  p = path_head;
  if(&path_root == p) {
    return "";
  }
  while(NULL != p->next) {
    count += strlen(p->elem);
    count++; /* include space for the path sep */
    p = p->next;
  }
  /* pass two, create the string, reversed */
  buffer = NEW(count+1);
  if(NULL==buffer) {
#ifndef NDEBUG
    fprintf(stderr, "out of memory\n");
#endif
    return NULL;
  }
  p = path_head;
  sprintf(buffer ,"%s",p->elem);
  position += strlen(p->elem);
  p = p->next;
  while(NULL != p->next) {
    sprintf(buffer+position,"%c%s", sep, p->elem);
    position += strlen(p->elem)+1;
    p = p->next;
  }
  /* Get the string turned round the right way. */
  reverse_with_delimiters(buffer,sep);
  return buffer;
}

static char* top_path_elem(void) {
  if(path_head)
    return path_head->elem;
  else
    return "";
}

static void CFG_make_entry(char* ident, char* value) {
  char* path;
  char* private_path;
  char* private_value;
  char sepstr[2] = { separator, 0 };

  path = get_path(separator);
  
  private_path = NEW(strlen(path) + strlen(ident) + 3);
  if(NULL==private_path) {
#ifndef NDEBUG
    fprintf(stderr,"out of memory\n");
#endif
    return;
  }
  strcpy(private_path,path);
  if( path[0] != 0 ) 
    strcat(private_path, sepstr);
  strcat(private_path,ident);

  private_value = NEW(strlen(value)+1);
  if(NULL==private_value) {
#ifndef NDEBUG
    fprintf(stderr,"out of memory\n");
#endif
    return;
  }
  strcpy(private_value,value);

  CFG_Set(private_path, private_value);

  DELETE(private_value);
  DELETE(private_path);
  DELETE(path);
}

/* Zero out all whitespace chars from the end of the string backwards
   to the first non-white character.  Whitespace chars are assumed to
   be space or tab, only. 
*/
static char* str_trim_right(char *str) {
  int i;

  i = strlen(str) - 1;          /* Starting before the terminating null */
  for(; i>0; --i) {             /* Proceeding toward the beginning of
                                   the string. */
    if(' ' == str[i] ||         /* If char is a space or a tab. */
       '\t' == str[i] ) {
      str[i] = 0;               /* Set it to zero instead. */
    }
    else {
      break;                    /* Stop at the first non-whitespace char. */
    }
  }
  return str;                   /* Return the string, modified in place. */
}

static CFG_status CFG_parse_line(char *line, char *section) {
  int rc;
  regmatch_t match[3];
  rc = regexec(&section_regex, line, 3, match, 0);
  if(REG_NOMATCH != rc) {
    /* overwrite the section character string with the matched portion
       of the line.*/
    strncpy(section, 
            line + match[1].rm_so , 
            match[1].rm_eo - match[1].rm_so);
    /* terminate the line */
    *(section + (match[1].rm_eo - match[1].rm_so)) = 0;

    clear_path_stack();         /* Assume that we only see sections at
                                   top level. This is wrong, FIX ME!  */

    push_path_elem(section);
    return CFG_OK;
  }
  else {
    /* is it a blank line? */
    rc = regexec(&blank_line_regex, line, 3, match, 0);
    if(REG_NOMATCH != rc) {
      /* Skip it */
      return CFG_OK;
    }
    /* is it a comment line? */
    rc = regexec(&comment_line_regex, line, 3, match, 0);
    if(REG_NOMATCH != rc) {
      /* Skip it */
      return CFG_OK;
    }
    /* This must be an item line, or a subsection open or close. */
    /* try subsection open */
    rc = regexec(&start_subsection_regex, line, 2, match, 0);
    if(REG_NOMATCH != rc) {
      char* section_name;
      int length = match[1].rm_eo - match[1].rm_so;
      section_name = NEW(length + 1);
      if(NULL==section_name) {
#ifndef NDEBUG
        fprintf(stderr,"out of memory");
#endif
        return CFG_FAILED;
      }
      strncpy(section_name, line + match[1].rm_so, length);
      section_name[length] = 0;
      push_path_elem(section_name);
      /* path stack takes a copy, this one is no longer needed */
      DELETE( section_name);
      return CFG_OK;
    }

    /* try subsection close */
    rc = regexec(&end_subsection_regex, line, 2, match, 0);
    if(REG_NOMATCH != rc) {
      char* current_subsection = top_path_elem();
      
      if(NULL == current_subsection)
        current_subsection = "";
      *(line + match[1].rm_eo) = 0;
      if(!strcmp(current_subsection, line + match[1].rm_so)) {
        pop_path_elem();
        return CFG_OK;
      }
      else {
        /* parse error, improper nesting */
        return CFG_PARSE_FAILED;
      }
    }
    /* This is an assignment */
    {
      char *tmp;                /* ... but, check for a match in the
                                   first 4096 bytes of the string.
                                   Since tokens can't be more than 512
                                   bytes, this should generally work.
                                   This is done this way because
                                   regexec can lose if the string it
                                   is testing is too long. */
      tmp = alloca(4096);
      strncpy(tmp,line,4095);
      rc = regexec(&assignment_regex, tmp, 3, match, 0);
      if(REG_NOMATCH == rc) 
        return CFG_PARSE_FAILED;        
    } 
    {
      char *matched_name;
      char *matched_data;
      long length;

      matched_name = alloca(match[1].rm_eo - match[1].rm_so + 1);

      strncpy(matched_name,
              line + match[1].rm_so , 
              match[1].rm_eo - match[1].rm_so);
      /* Terminate the line */
      *(matched_name + (match[1].rm_eo - match[1].rm_so)) = 0;
      
      length = (long)strlen(line);
      matched_data = alloca(length - match[2].rm_so + 1);

      strncpy(matched_data,
              line + match[2].rm_so , 
              length - match[2].rm_so);
      /* Terminate the line, emphatically... */
      *(matched_data + (length - match[2].rm_so)) = '\0';
      /* Scrub off the newline---if there is one---on the end. */
      if('\n' == *(matched_data + (length - match[2].rm_so - 1)))
        *(matched_data + (length - match[2].rm_so - 1)) = '\0';
      str_trim_right(matched_data); /* The regex doesn't exclude white chars. */
      cfg_UnescapeNewlines(matched_data);
      CFG_make_entry(matched_name, matched_data);
      return CFG_OK;
    }
  } 
}
/* Tue Apr 25 16:05:24 2000 RAL made this a static function.  It is no
   longer available to code outside this file. 
*/
#ifdef USE_HSEARCH
static CFG_status cfg_init(char sep_char, int tablesize)
#else
static CFG_status cfg_init(char sep_char)
#endif
{
  if(initialized) {
    return CFG_OK;
  }
#ifndef USE_MALLOC
  if(NULL==mp) {
    mp = MEM_create_pool(MEM_BLOCK_SIZE);
  }
#endif
  separator = sep_char;
#ifdef USE_HSEARCH
  if(0 == hcreate(tablesize)) {
    return CFG_FAILED;
  } 
#else
  table = hash_new(hash_string, same_string);
#endif

  if(0 != sep_char) {
    separator = sep_char;
  }

  regcomp(&section_regex, 
          "^[ \t]*\\[[ \t]*([A-Za-z0-9_][A-Za-z_0-9]*)[ \t]*\\].*\n", 
          REG_EXTENDED);
  regcomp(&assignment_regex,
          "^[ \t]*([A-Za-z_0-9][A-Za-z_0-9]*)[ \t]*=[ \t]*(.*)\n",
          REG_EXTENDED);
  regcomp(&blank_line_regex,
          "^[ \t]*\n",
          REG_EXTENDED);
  regcomp(&comment_line_regex,
          "^[ \t]*#.*\n",
          REG_EXTENDED);
  regcomp(&start_subsection_regex,
          "^[ \t]*start ([A-Za-z_0-9][A-Za-z_0-9]*)[ \t]*\n",
          REG_EXTENDED | REG_ICASE);
  regcomp(&end_subsection_regex,
          "^[ \t]*end ([A-Za-z_0-9][A-Za-z_0-9]*)[ \t]*\n",
          REG_EXTENDED | REG_ICASE);
  regcomp(&c_ident_regex,
          "^([A-Za-z_][A-Za-z_0-9]*)(%[A-Za-z_][A-Za-z_0-9]*)*$",
          REG_EXTENDED );
  initialized = 1;
  atexit((void (*)(void))CFG_Destroy);
  return CFG_OK;
}

static CFG_status cfg_load(void) {
  CFG_status result;
  /* Read the config file from disk. */
  if(CFG_OK == (result = CFG_ReadFile(CLU_CONFIG_FILE))) {
    /* Clean out anything we don't need.  Initially, I am assuming
       that there isn't anything in members that I can get rid of.
       Services are more likely to change, so I clean them out
       entirely. I can remove individual entries from the members
       section, and other sections, but I need to know which ones are
       safe to delete. */
    CFG_RemoveMatch("services%*");
    /* Now overlay the contents of the shared database atop the
       database already loaded. */
    result = CFG_Read();
  } else if( CFG_ZERO_LENGTH_FILE == result ) {
    /* zero length database on local filesystem. */
    fprintf(stderr,
            "Empty config file, %s. Unable to get locking information. Exiting.\n", 
            CLU_CONFIG_FILE);
    exit(1);
    return CFG_FAILED;
  }
  if(CFG_OK == result) {
    initialized = 1;
    /* Now write a copy of the database that came from the quorum
       partition back to the disk based configuration file. */
    result = CFG_WriteFile(CLU_CONFIG_FILE);
  } else if(CFG_ZERO_LENGTH_FILE == result) {
    /* The cluster database on shared storage had zero length.  Write
       back the config file to the shared partition; at least member
       information will then be there. Service information will have
       been lost, but someone just did a diskutil -I, so, well,
       something dramatic should happen.  At least cluadmin won't dump
       core. */
    result = CFG_Write();
  }
  return result;
}

/* Public functions */

CFG_status CFG_Initialized() {
  if(initialized) 
    return CFG_OK;
  else
    return CFG_FALSE;
}

CFG_status CFG_Loaded() {
  if(loaded)
    return CFG_OK;
  else
    return CFG_FALSE;
}

static void CFG_RemoveAll(void) {
  CFG_RemoveMatch("*");
}

CFG_status CFG_Destroy(void) {
  static int called = 0;

  if(initialized) {
    /* De-allocate the regular expression matchers. */
    regfree(&section_regex);
    regfree(&assignment_regex);
    regfree(&blank_line_regex);
    regfree(&comment_line_regex);
    regfree(&start_subsection_regex);
    regfree(&end_subsection_regex);
    regfree(&c_ident_regex);

    /* De-allocate the hash tables. */
#ifdef USE_HSEARCH
    hdestroy();
#else
    if(called) {
      CFG_RemoveAll(); 
    } 
    else { 
      called = 1;
    }
    hash_destroy(table);
#endif

    path_head = &path_root;

#ifndef USE_MALLOC
    if(NULL!=mp) {
      MEM_destroy_pool(mp);
      mp=NULL;
    }
#endif
  }
  initialized = 0;
  loaded = 0;
  return CFG_OK;
}

/** CFG locking routines ************************************/

#define DATABASE_UNLOCKED 0
#define DATABASE_LOCKED_THIS_SESSION 1
#define DATABASE_LOCKED_OTHER_SESSION 2
#define DATABASE_LOCK_STATUS_UNKNOWN -1

#define LOCK_TIME_LIMIT (30L*60L) /* 1/2 hour. */

#define EXPIRED_YES 1
#define EXPIRED_NO  0
#define EXPIRED_UNK -1

static int   lock_status = 0;
static int   lock_expired = EXPIRED_UNK;
static ulong database_checksum = 0;

static ulong get_checksum(void) {
  ulong checksum = 0L;
  if (initAlignedBufStuff() < 0) {
    fprintf(stderr, __FUNCTION__ ": Unable to init rawio support.\n");
    return(-1);
  }

  if(cluster_locking)
    clu_lock();
  
  checksum = getDatabaseChecksum();

  if(cluster_locking)
    clu_un_lock();
  return checksum;  
}

static int parse_lock_entry(char   *entry,
                            char   *session_buffer, 
                            ssize_t  size, 
                            time_t *timestamp) {
  ssize_t len;
  int count;
  char ent[BUFFER_SIZE];
  len = strcspn(entry, " ");
  memcpy(session_buffer, entry, len > size ? size : len);
  session_buffer[len > size ? size : len] = 0;

  count = sscanf(entry, "%s %ld", 
                 ent, /* ignored */
                 timestamp);
  if(2 != count) {
    return CFG_FALSE;
  } 
  else {                        /* convert time to seconds */
    return CFG_OK;
  }
}

inline static int timestamp_expired(time_t timestamp) {
  return (time(NULL) - timestamp) > LOCK_TIME_LIMIT;
}

/* The lock entry is composed of the session string and a date/time
   stamp, ideally, one that is easy to parse. */

static int create_lock_string(char  *session, 
                              char  *buffer, 
                              ssize_t size) {
  time_t             curtime;
  struct tm         *loctime = NULL;
  char               date_buffer[BUFFER_SIZE];

  memset(date_buffer, 0, BUFFER_SIZE);
  curtime = time(NULL);
  loctime = localtime(&curtime);
  strftime(date_buffer, 
           BUFFER_SIZE, 
           "%s", 
           loctime);
  snprintf(buffer, size, "%s %s", session, date_buffer);
  return CFG_OK;
}

char *CFG_UserTimeString(char *timestring) {
  time_t t;
  struct tm         *loctime = NULL;
  static char date_buffer[BUFFER_SIZE];

  sscanf(timestring,"%ld", (long *)&t);

  loctime = localtime(&t);
  strftime(date_buffer, 
           BUFFER_SIZE, 
           "%a %b %d %R:%S %Y", /* Preferred date and time
                                   representation for current
                                   locale. */
           loctime);
  return date_buffer;
}

static void create_timestamp(char *buffer, ssize_t size)
{
  time_t             curtime;
  struct tm         *loctime = NULL;
  char               date_buffer[BUFFER_SIZE];

  memset(date_buffer, 0, BUFFER_SIZE);
  curtime = time(NULL);
  loctime = localtime(&curtime);
  strftime(date_buffer, 
           BUFFER_SIZE, 
           "%s", 
           loctime);
  snprintf(buffer, size, "%s", date_buffer);
}

/* read the lock information from the currently loaded cfg table.
   Return information through pointers passed in. */

static char* saved_lock_string=NULL;

char* CFG_LockString(void) {
  if(saved_lock_string)
    return saved_lock_string;
  else
    return "";
}

static void check_lock_status(char   *session,
                              int    *locked, 
                              char   *session_buffer,
                              ssize_t  session_buffer_size,
                              int    *expired,
                              time_t *timestamp) {
  CFG_status status;
  char *lock_entry;

  status = CFG_Get("session%lock", "none", &lock_entry);
  if( CFG_DEFAULT == status ) { /* No lock entry found */
    *locked = DATABASE_UNLOCKED;
    *expired = EXPIRED_UNK;
  }
  else if( CFG_OK == status ) {
    /* Save the lock string.  This is needed so we can prompt the user
       if the lock is held. */
    if(NULL != saved_lock_string) 
      free(saved_lock_string);
    saved_lock_string = strdup(lock_entry);

    if(!parse_lock_entry(lock_entry, 
                         session_buffer, 
                         session_buffer_size, 
                         timestamp)) {
      *locked = DATABASE_LOCK_STATUS_UNKNOWN;
      return;
    } 
    if(! strcmp(session, session_buffer)) {
      *locked = DATABASE_LOCKED_THIS_SESSION;
    } 
    else {
      *locked = DATABASE_LOCKED_OTHER_SESSION;
    }
    if( timestamp_expired(*timestamp) ) {
      *expired = EXPIRED_YES;
    }
    else {
      *expired = EXPIRED_NO;
    }
  }
  else {
    *locked = DATABASE_LOCK_STATUS_UNKNOWN;
  }
}

/* Seize the advisory session lock on the cluster database.  This
   should be called at the beginning of a sequence of commands that
   will potentially modify the configuration database. */

CFG_lockStatus CFG_Lock(char *session, 
                        int   forced) {
  struct hash_table *saved_table;
  int                saved_initialize;
  char               buffer[BUFFER_SIZE];
  ulong              first_checksum;
  ulong              second_checksum;
  time_t             timestamp;
  char               lock_session[BUFFER_SIZE];
  int                expired;
  CFG_lockStatus     result=0;

  saved_table = table;
  saved_initialize = initialized;
  table = hash_new(hash_string, same_string);
  CFG_Read();
  check_lock_status(session,
                    &lock_status, 
                    lock_session,
                    BUFFER_SIZE,
                    &expired,
                    &timestamp);
  if(DATABASE_LOCK_STATUS_UNKNOWN == lock_status) {
    result |= CFG_LOCK_FAILURE;
    goto exit;
  }

  first_checksum = get_checksum();
  if(forced) {
    lock_status = DATABASE_UNLOCKED;
  }
  if(DATABASE_UNLOCKED == lock_status) {
    /* Set a lock element into the in-memory version of the database */
    create_lock_string(session, buffer, BUFFER_SIZE);
    CFG_Set("session%lock", buffer);
    /* Get a checksum of the database, compare it with the earlier
       checksum.  If the checksum is different, something else has
       written the file.  */
    second_checksum = get_checksum();
    if(first_checksum == second_checksum) {
      /* Database has not changed. */
      CFG_Write();
      database_checksum = get_checksum();
      lock_status = DATABASE_LOCKED_THIS_SESSION;
      lock_expired = EXPIRED_NO;
      result |= CFG_LOCK_LOCKED_THIS_SESSION;
      goto exit;
    } 
    else {
      /* Database has changed from underneath us. */
      /* Return CFG_LOCK_FAILURE, punt, and let the caller try
         again. */
      result |=  CFG_LOCK_FAILURE;
      goto exit;
    }    
  } else if(DATABASE_LOCKED_THIS_SESSION == lock_status) { 
    /* compare checksums to see if the database file has changed on
       disk.  This should only happen if something has stolen, or
       ignored the lock. */
    second_checksum = get_checksum();
    if(database_checksum == second_checksum) {
      create_lock_string(session, buffer, BUFFER_SIZE);
      CFG_Set("session%lock", buffer);
      second_checksum = get_checksum();
      if(first_checksum == second_checksum) {
        /* Database has not changed. */
        CFG_Write();

        database_checksum = get_checksum();
        lock_status = DATABASE_LOCKED_THIS_SESSION;
        lock_expired = EXPIRED_NO;
        result |= CFG_LOCK_LOCKED_THIS_SESSION;
        goto exit;
      } 
      else {
        /* Database has changed from underneath us. */
        /* Return CFG_LOCK_FAILURE, punt, and let the caller try
           again. */
        result |=  CFG_LOCK_FAILURE;
        goto exit;
      }    
    } 
    else {
      /* If it has, the lock has, at least, been stolen, maybe
         worse. */
      result |= CFG_LOCK_LOST;
      goto exit;
    }
  } else if(DATABASE_LOCKED_OTHER_SESSION == lock_status) {
    /* We already think the database is locked by a different
       session. */
    if( timestamp_expired(timestamp) ) {
      result |= CFG_LOCK_LOCKED_OTHER_SESSION;
      result |= CFG_LOCK_EXPIRED;
      goto exit;
    }
    else {
      result |=  CFG_LOCK_LOCKED_OTHER_SESSION;
      goto exit;
    }
  } else {
    /* Some kind of error must have occurred, record it (someday), and
       return CFG_FAILED. */
    result |=  CFG_LOCK_FAILURE;
    goto exit;
  }
exit:
  CFG_RemoveMatch("*");
  hash_destroy(table);
  table = saved_table;
  initialized = saved_initialize;

  if ( result & CFG_LOCK_LOCKED_THIS_SESSION ) {
    CFG_Set("session%lock", buffer);
  }
  return result;
}

/* Release the advisory session lock on the cluster database.  This
   should be called at the end of a series of operations that might
   modify the configuration database. */

CFG_lockStatus CFG_Unlock(char *session, 
                          int forced) {
  struct hash_table *saved_table;
  int                saved_initialize;
  ulong              first_checksum;
  ulong              second_checksum;
  time_t             timestamp;
  char               lock_session[BUFFER_SIZE];
  int                expired;
  CFG_lockStatus     result=0;

  saved_table = table;
  saved_initialize = initialized;
  table = hash_new(hash_string, same_string);
  CFG_Read();

  check_lock_status(session,
                    &lock_status, 
                    lock_session,
                    BUFFER_SIZE,
                    &expired,
                    &timestamp);
  if(DATABASE_LOCK_STATUS_UNKNOWN == lock_status) {
    result |= CFG_LOCK_FAILURE;
    goto exit;
  }
  first_checksum = get_checksum();
  if(DATABASE_UNLOCKED == lock_status) {
    result |= CFG_LOCK_NOT_LOCKED;
    goto exit;
  } 
  else if(DATABASE_LOCKED_THIS_SESSION == lock_status) {
    CFG_Remove("session%lock");
    second_checksum = get_checksum();
    if(first_checksum == second_checksum) {
      CFG_Write();

      database_checksum = get_checksum();
      lock_status = DATABASE_UNLOCKED;
      lock_expired = EXPIRED_UNK;
      result |= CFG_LOCK_NOT_LOCKED;
      goto exit;
    } 
    else {
      /* Return CFG_LOCK_FAILURE, punt, and let the caller try
         again. */
      result |=  CFG_LOCK_FAILURE;
      goto exit;
    }
  }    
  else if(DATABASE_LOCKED_OTHER_SESSION == lock_status) {
    if(!forced && !timestamp_expired(timestamp)) {
      result |= CFG_LOCK_LOCKED_OTHER_SESSION;
      goto exit;
    }
    CFG_Remove("session%lock");
    second_checksum = get_checksum();
    if(first_checksum == second_checksum) {
      CFG_Write();

      database_checksum = get_checksum();
      lock_status = DATABASE_UNLOCKED;
      lock_expired = EXPIRED_UNK;
      result |= CFG_LOCK_NOT_LOCKED;
      goto exit;
    } 
    else {
      /* The database on disk has changed recently; return
         CFG_LOCK_FAILURE, punt, and let the caller try again. */
      result |=  CFG_LOCK_FAILURE;
      goto exit;
    }
  } else {
    /* Some kind of error must have occurred, record it, and return
       CFG_FAILED. */
    result |=  CFG_LOCK_FAILURE;
    goto exit;
  }
exit:
  CFG_RemoveMatch("*");
  hash_destroy(table);
  table = saved_table;
  initialized = saved_initialize;
  return result;
}

/* Check the condition of the advisory session lock.  Call this before
   calling CFG_Read, or CFG_Write to determine the current session
   lock status of the configuration database. 

   Returns a result, and also sets some file scoped variables for use
   of other cfg functions.  */

CFG_lockStatus CFG_CheckLock(char *session) {
  struct hash_table *saved_table;
  int                saved_initialize;
  char               lock_session[BUFFER_SIZE];
  time_t             timestamp;
  int                locked; 
  int                expired;
  CFG_lockStatus     result=0;

  saved_table = table;
  saved_initialize = initialized;
  table = hash_new(hash_string, same_string);
  CFG_Read();

  check_lock_status(session,
                    &locked, 
                    lock_session,
                    BUFFER_SIZE,
                    &expired,
                    &timestamp);

  CFG_RemoveMatch("*");
  hash_destroy(table);
  table = saved_table;
  initialized = saved_initialize;
  switch(locked) {              /* Eww. Fix me. */
    case DATABASE_UNLOCKED:
      result |= CFG_LOCK_NOT_LOCKED;
      break;
    case DATABASE_LOCKED_THIS_SESSION:
      result |= CFG_LOCK_LOCKED_THIS_SESSION;
      if(expired) {
        result |= CFG_LOCK_EXPIRED;
      }
      break;
    case DATABASE_LOCKED_OTHER_SESSION:
      result |= CFG_LOCK_LOCKED_OTHER_SESSION;
      if(expired) {
        result |= CFG_LOCK_EXPIRED;
      }
      break;
    case DATABASE_LOCK_STATUS_UNKNOWN:
      result |= CFG_LOCK_STATUS_UNKNOWN;
      break;
  }
  return result;
}

/* end of locking routines ***********************************/

/* Call stat to find out how big the file named by filename is. */

static off_t filesize(char *filename) {
  struct stat stat_buf;
  int res;
  res = stat(filename, &stat_buf);
  if(-1 == res)
    return (off_t) -1;
  else
    return stat_buf.st_size;
}

static off_t ffilesize(FILE *f) {
  struct stat stat_buf;
  int res;
  res = fstat(fileno(f), &stat_buf);
  if(-1 == res)
    return (off_t) -1;
  else
    return stat_buf.st_size;
}

/* Get the next line from a buffer of characters.  Lines are
   considered to be separated by newline characters, but newline
   characters may be escaped by a preceding backslash.  The line is
   copied into the destination buffer, which must be the address of an
   array of characters of at least length, len, that has been
   allocated using malloc.  Return a pointer into the source buffer
   one character past the just found, line-ending, newline.  This is
   to allow the caller to use this new pointer to read the next line
   from the buffer in a subsequent call to this function.  The source
   buffer is never modified. */

static char* next_line(char *source, char **destination, ssize_t len) {
  ssize_t k;
  int end_found=0;
  if(0 == source[0]) {
    return NULL;
  }
  k = strcspn(source, "\n");
  while(! end_found) {
    if(k && '\\' == source[k-1]) {
      /* escaped new-line */
      k = k + strcspn(source+k+1, "\n") + 1;
    } else {
      end_found = 1;
    }
  }
  if(k+1>len) {
    ssize_t m;
    /* The line is too long to fit in a buffer of length len.  The end
       of the line has been found; we now know how much space is
       needed.  Reallocate the given buffer, to ensure that it is big
       enough. */
    m = (k / len) + 1;
    *destination = (char *)realloc(*destination, m * len);
  }
  memcpy(*destination, source, k+1); /* Include the newline in the
                                       copy. */
  (*destination)[k+1] = 0;

  return source + k + 1;
}

/* Treat the passed buffer as though it were the contents of a config
   file, parse it and update the configuration database structures
   accordingly.  The passed buffer is not altered in any way.  The
   size argument is the length of the data in the passed buffer. If
   size is zero, the buffer is treated as a single, null terminated
   string. */

static CFG_status cfg_ReadBuffer(char *buffer, ssize_t size) {
  char *line = NULL;
  char section[MAX_LINE];
  ssize_t char_count = 0;

  line = (char*)malloc(MAX_LINE);

  if(NULL == line)
    return CFG_FAILED;

  if(NULL == buffer) {
    return CFG_FAILED;          /* Should be CFG_BAD_ARG, if there
                                   were such a thing. */
  }
  if(0 == size) {
    /* buffer is a null terminated string. */
    size = strlen(buffer);
  }
  if(!CFG_Initialized()) {
#ifdef USE_HSEARCH
    cfg_init(separator,1000);
#else
    cfg_init(separator);
#endif
  }
  memset(section, 0, MAX_LINE);
  clear_path_stack();
  loaded = 1;
  
  buffer = next_line(buffer, &line, MAX_LINE);
  while(line) {
    char_count += strlen(line); 
    CFG_parse_line(line, section);
    if(char_count < size) {
      buffer = next_line(buffer, &line, MAX_LINE);
      if(NULL == buffer) {	/* No more input */
	goto done;
      }
    }
    else {
      break;
    }
  }
  done:
  loaded = 1;
  free(line);
  return CFG_OK;  
}

/* Open the file of the given filename, read its contents into a
   dynamically allocated buffer, and parse it contents.  Reading the
   entire file into a buffer would not be a Good Idea(tm), if the file
   could be very large, but it shouldn't ever be. */

CFG_status CFG_ReadFile(char * filename) {
  FILE* f;
  off_t fsize=0;
  ssize_t charcount=0;
  char *contents=NULL;

  if(!CFG_Initialized()) {
#ifdef USE_HSEARCH
    cfg_init(separator,1000);
#else
    cfg_init(separator);
#endif
  }
  if(NULL == filename) {
    filename = CLU_CONFIG_FILE;
  }
  fsize = filesize(filename);
  if(-1 == fsize)
    return CFG_FAILED;
  else if(0 == fsize)
    return CFG_ZERO_LENGTH_FILE;
  contents = (char*)malloc(fsize + 2);
  memset(contents,0,fsize+2);
  if(0 == contents)
    return CFG_FAILED;
  f = fopen(filename,"r");
  if(0 == f)
    return CFG_FAILED;
  
  charcount = fread(contents,sizeof(char),fsize,f);
  if(charcount != (ssize_t)fsize) {
    fclose(f);
    return CFG_FAILED;
  }
  contents[fsize] = 0;
  strcat(contents,"\n");
  cfg_ReadBuffer(contents,0);
  fclose(f);
  free(contents);
  loaded = 1;
  return CFG_OK;
}

CFG_status CFG_ReadFD(FILE *f) {
  off_t fsize=0;
  ssize_t charcount=0;
  char *contents=NULL;

  if(!CFG_Initialized()) {
#ifdef USE_HSEARCH
    cfg_init(separator,1000);
#else
    cfg_init(separator);
#endif
  }
  fsize = ffilesize(f);
  if(-1 == fsize)
    return CFG_FAILED;
  contents = (char*)malloc(fsize + 2);
  memset(contents,0,fsize+2);
  if(0 == contents)
    return CFG_FAILED;
  
  charcount = fread(contents,sizeof(char),fsize,f);
  if(charcount != (ssize_t)fsize) {
    return CFG_FAILED;
  }
  contents[fsize] = 0;
  strcat(contents,"\n");
  cfg_ReadBuffer(contents,0);
  free(contents);
  loaded = 1;
  return CFG_OK;
}

/* Write the contents of the database structures to the cluster's
   shared disk partition. 

   N.B. for now, the cluster's global configu file is where we write
   to.  */

CFG_status CFG_Write(void) {
  char * buffer=NULL;
  ssize_t bsize;
  CFG_status result = CFG_OK;
  ssize_t written;
  char timestamp_buffer[BUFFER_SIZE];
  if (initAlignedBufStuff() < 0) {
    fprintf(stderr, __FUNCTION__ ": Unable to init rawio support.\n");
    return(-1);
  }

  if(cluster_locking)
    clu_lock();

  /* Set a new timestamp into the database. */
  create_timestamp(timestamp_buffer, BUFFER_SIZE);
  CFG_Set("cluster%timestamp",timestamp_buffer);

  buffer = (char*) malloc(BUFFER_SIZE);
  if(NULL != buffer) 
    bsize = BUFFER_SIZE;
  else {
    result = CFG_FAILED;
    goto failed;
  }
  buffer = cfg_WriteBuffer(buffer, &bsize);

  written = writeDatabase(buffer, bsize); 
  if(written == bsize) {
    result = CFG_OK;
  } 
  else if(-3 == written) {
    result = CFG_TOO_BIG;
    goto failed;
  }
  else {
    result = CFG_FAILED;
    goto failed;
  }
  free(buffer);
  buffer = NULL;
  if(cluster_locking)
    clu_un_lock();

  /* Also write the config file on disk. */
  CFG_WriteFile(CLU_CONFIG_FILE);
  /* Send a message to the quorumd */
  {
    pid_t pid;
    if(check_process_running("quorumd",&pid)) 
      cluConfigChangeNotification();
  }
  return result;

  failed:
  free(buffer);
  if(cluster_locking)
    clu_un_lock();
  return result;
}

/* Read the contents of the cluster's shared disk partition into the
   database structures.  Cleans the database structures prior to
   parsing the contents of the shared partition. 

   N.B. for now, the cluster's global configu file is where we write
   to.  */

CFG_status CFG_Read(void) {
  ssize_t length;
  char *buffer=NULL;
  ssize_t res;
  CFG_status result=CFG_OK;

  if (initAlignedBufStuff() < 0) {
    fprintf(stderr, __FUNCTION__ ": Unable to init rawio support.\n");
    return(-1);
  }

  if(cluster_locking)
    clu_lock();
  
  length = getDatabaseLength();

  if(length > 0) {
    buffer = (char*)malloc(length+2);
    memset(buffer, 0, length+2);
  }
  else {
    result= CFG_ZERO_LENGTH_FILE;
    goto failed;
  }
  if(NULL == buffer) {
    result = CFG_FAILED;
    goto failed;
  }
  res = readDatabase(buffer, length);
  
  if(res == length) {
    strcat(buffer,"\n");
    result = cfg_ReadBuffer(buffer, length);
  }
  else {
    result = CFG_FAILED;
    goto failed;
  }

  if(cluster_locking)
    clu_un_lock();
  free(buffer);
  return result;

  failed:
  if(cluster_locking)
    clu_un_lock();
  free(buffer);
  return result;
}

/* Copy the contents of the cluster's shared disk partition out to a
   backup file.  The backup file's name and location are specified in
   the global configuration file's "config" section in a "backupFile"
   entry. */

CFG_status CFG_Backup(void) {
  return CFG_WriteFile(CLU_CONFIG_FILE ".bak");
}

/* Restore a backup configuration file to the cluster's share disk
   partition. */

CFG_status CFG_Restore(void) {
  return CFG_ReadFile(CLU_CONFIG_FILE ".bak");
}

static int compare_keys(const void *a, const void *b) {
  return strcmp(*(char**)a,*(char**)b);
}

/* Some routines that treat strings as paths with embedded
   separators.  
*/

/* Compute the number of elements in the path.  This is one more than
   the number of occurances of the separator character; provided that
   the separator charactor is not allowed to appear at the beginning
   or the end of the string. 
*/
static int path_length(char* path, char sep) {
  int len=0;
  char *p=path;

  if(0 == path[0])
    return 0;

  while(p) {
    p = strchr(p, sep);
    if(p) {
      p++;
      len++;
    }
  }
  /* assumes the path doesn't begin or end with the sep. */
  return len + 1;
}

/* Return a pointer to the last element in the path. */
static char *path_tail(char *path) {
  char *p = path;
  p = strrchr(p, separator);
  if(p) 
    return ++p;
  else
    return NULL;
}

/* Given two paths, return a count of the number of elements that are
   the same starting at the beginning of the paths. 
*/
static int path_common_prefix_length(char *a, char *b) {
  char *p=a, *q=b;
  int pl, ql;
  int count = 0;
  char sepstring[2] = {0,0};
  sepstring[0] = separator;
  while( p && q ) {
    pl = strcspn(p, sepstring);
    ql = strcspn(q, sepstring);
    
    if(pl == ql && !memcmp(p,q,pl)) {
      count++;
      p += pl + 1;  q += ql + 1;
    }
    else {
      return count;
    }
  }
  return count;
}

/* Compare the lengths of two paths.  Return the difference as a
   signed number. 
*/
static int path_compare(char *a, char *b) {
  int alength, blength;

  alength = path_length(a, separator);
  blength = path_length(b, separator);

  return alength - blength;
}

/* Return a pointer to a path that consistes of the given path with
   its first element removed. 
*/
static char *path_rest(char* path) {
  char *p = path;
  if(!path[0]) 
    return path;
  p = strchr(p, separator);
  if(p) 
    return ++p;
  else
    return NULL;
}

/* Find the which'th element in path p, write its address in *res, and
   its length in *len. 
*/
static void path_index(char *p, int which, char sep,
                       char **res, int *len) {
  int i;
  char sepstring[2] = {0,0};
  sepstring[0] = sep;
  
  *res = p;
  *len=strcspn(*res, sepstring);
  for(i = 0; i < which; ++i) {
    *res = *res + *len + 1;
    *len=strcspn(*res, sepstring);
    if( *res + *len == 0 ) 
      break;
  }
}

/* This is a lot like memcpy.  I guess the difference will be that
   this function will re-allocate the buffer when we have run out of
   space to write in it.

   The caller must keep track of offsets, which is messy.  Destination
   is a pointer to the starting address of the buffer into which
   characters will be written.  The starting address is needed in case
   we must call realloc on the buffer.  Returns the offset after the
   last character written.  This should make it easier for the caller
   to keep track of where the next write should occur. 

   If it is necessary to re-allocate the buffer, the destination
   pointer will be updated with the new address, and dlen will be
   updated with the new buffer size.  
   
   The source, which must be a null-teminated string, is never
   modified. */

static ssize_t write_line(char *source, 
                         ssize_t offset, 
                         char **destination, 
                         ssize_t *dlen) {
  int src_len = strlen(source);
  int needed = offset + src_len + 1;
  if(needed > *dlen) {
    char * new_address = (char*)realloc(*destination, needed*2);
    if(NULL != new_address) {
      *destination = new_address;
      *dlen = needed*2;
    }
    else {
      return -1;                /* Failed to re-allocate the
                                   buffer. */
    }
  }
  memcpy(*destination + offset, source, src_len);
  (*destination)[offset+src_len] = 0; /* Null terminate the buffer. */

  return src_len + offset;
}

/* Write out the contents of the database structures to the given
   buffer, which must be exaclty size bytes long.  Returns the final
   address of the buffer if the contents can be successfully written
   to the buffer, returns NULL if, for any reason, the write cannot be
   completed.  If the passed buffer, which must be malloc'ed, is not
   large enough, it is re-alloc'ed until it is large enough.  The
   returned pointer may, as a consequence, be different from the
   passed pointer.  */

static char * cfg_WriteBuffer(char *inbuffer, 
                              ssize_t *size) {
  char **keys;
  int i=0, len;
  int count;
  char * output_buffer;
  int where = 0;
  ssize_t output_size = BUFFER_SIZE;
  char section[2][MAX_LINE];
  char buffer[2][BUFFER_SIZE];
  char *tmp_buffer;
  int tmp_buffer_size;
  int lengths[2] = {0,0};
  int common = 0;
  int current=0, previous=1;
  int path_change;
  char sepstring[2] = {0,0};

  tmp_buffer = (char*)malloc(BUFFER_SIZE);
  tmp_buffer_size = BUFFER_SIZE;

  sepstring[0] = separator; /* Make a 1 char separator string */
  buffer[current][0]=0;        buffer[previous][0] = 0; 
  section[current][0] = 0;     section[previous][0] = 0; 

  /* Get a list of keys from the hash table.  Keys is an array of
     pointers to character strings.  The array is allocated by the
     hash table, and must be free'd before we finish. */
  keys = hash_keys(table, &count);
  /* Sort the keys. */
  qsort(keys, count, sizeof(char*), compare_keys);

  /* Format and write them out. */

  output_buffer = inbuffer;

  for(i = 0; i < count+1; ++i) {
    if(i == count) 
      /* use an empty key string the last time through. */
      strcpy(buffer[current], "");
    else
      /* Make a local copy of the key string. */
      strcpy(buffer[current], keys[i]);

    /* Compute various useful lengths. */
    path_change = path_compare(
      path_rest(buffer[current]), 
      path_rest(buffer[previous]));
    common = path_common_prefix_length(
      path_rest(buffer[current]), 
      path_rest(buffer[previous]));

    lengths[current] = path_length(
      path_rest(buffer[current]), separator);

    len = strcspn(buffer[current], sepstring);
    memcpy(section[current], buffer[current], len);
    section[current][len]=0; 

    /* End any sub-sections that need to be ended. */
    if( lengths[previous] - 1 > common ) {
      char *name;
      int ii,k;
      char bf[MAX_LINE];
      for(ii=0; ii < lengths[previous] - common - 1; ++ii) {
        /* insert end section statements. */
        path_index(path_rest(buffer[previous]),
                   lengths[previous]-ii-2,
                   separator,
                   &name, 
                   &k);
        memcpy(bf, name, k);
        bf[k]=0;
        sprintf(tmp_buffer,"end %s\n",bf);
        where = write_line(tmp_buffer, where, &output_buffer, &output_size);
        if(where < 0) {
          free(output_buffer);
          free(tmp_buffer);
          return NULL;
        }
      }
    }
    /* decide whether this is a new section. */
    if(strcmp( section[current], section[previous]) &&
       i != count) {
      /* new section; print it. */
      sprintf(tmp_buffer, "\n[%s]\n", section[current]);
      where = write_line(tmp_buffer, where,
                         &output_buffer, &output_size);
      if(where < 0) {
        free(output_buffer);
        free(tmp_buffer);
        return NULL;
      }
    }
    /* Start any sub-sections that need to be started. */
    if( lengths[current] - 1 > common ) {
      char *name;
      int ii,k;
      char bf[MAX_LINE];
      for(ii=0; ii < lengths[current] - common - 1; ++ii ) {
        /* Insert begin section statements. */
        path_index(path_rest(buffer[current]),
                   common+ii,
                   separator,
                   &name, &k);
        memcpy(bf, name, k);
        bf[k]=0;
        sprintf(tmp_buffer,"start %s\n",bf); 
        where = write_line(tmp_buffer, where, &output_buffer, &output_size);
        if(where < 0) {
          free(output_buffer);
          free(tmp_buffer);
          return NULL;
        }
      }
    }
    if(i != count) {
      char *entry;
      char *buf;
      int bigness;

      entry = hash_find(table, keys[i]);
      if(entry) {
        buf = (char*)malloc(strlen(entry)+1);
        strcpy(buf,entry);
        entry = cfg_EscapeNewlines(buf, strlen(buf));
        bigness = strlen(entry) + strlen(path_tail(buffer[current])) + 7;
        if(tmp_buffer_size < bigness) {
          tmp_buffer = realloc(tmp_buffer, bigness);
          tmp_buffer_size = bigness;
        }
        sprintf(tmp_buffer, "  %s = %s\n", 
                path_tail(buffer[current]), 
                entry);
        where = write_line(tmp_buffer, where, &output_buffer, &output_size);
        free(entry);
        if(where < 0) {
          free(output_buffer);
          free(tmp_buffer);
          return NULL;
        }
      }
    }
    current ^= 1; previous ^= 1; /* Swap previous and current. */
  }
  sprintf(tmp_buffer,"\n");
  where = write_line(tmp_buffer, where, &output_buffer, &output_size);
  if(where < 0) {
    free(output_buffer);
    free(tmp_buffer);
    return NULL;
  }
  
  /* Free the pointers array! */
  free(keys);
  *size = output_size;
  free(tmp_buffer);
  return output_buffer;
}

/* Any commentary that was in the original file will not be present in
   the file that has been written out. 

   Always uses "start" and "end" for sub-section markers.

   Doesn't check that the keys are sensible.  If they aren't, who
   knows what might happen.

   This seems like a lot of code to do such a seemingly simple thing.  
*/
CFG_status CFG_WriteFile(char *filename) {
  FILE *f;
  char *buffer;
  ssize_t bsize;
  int written;
  CFG_status res = CFG_OK;

  buffer = (char*) malloc(BUFFER_SIZE);
  memset(buffer, 0 , BUFFER_SIZE);
  if(NULL != buffer) 
    bsize = BUFFER_SIZE;
  else
    return CFG_FAILED;

  f = fopen(filename,"w");
  if(NULL == f)
    return CFG_FAILED;

  buffer = cfg_WriteBuffer(buffer, &bsize);

  written = fwrite(buffer, 
                   sizeof(char), 
                   strlen(buffer), 
                   f);
  if(written != (int)strlen(buffer)) {
    res = CFG_FAILED;
  }
  free(buffer);
  fclose(f);
  return res;
}

/* Write out the key/value pairs to the open file pointed to by f */

CFG_status CFG_WriteFD(FILE *f) {
  char *buffer;
  ssize_t bsize;
  int written;
  CFG_status res = CFG_OK;

  buffer = (char*) malloc(BUFFER_SIZE);
  memset(buffer, 0 , BUFFER_SIZE);
  if(NULL != buffer) 
    bsize = BUFFER_SIZE;
  else
    return CFG_FAILED;

  buffer = cfg_WriteBuffer(buffer, &bsize);

  written = fwrite(buffer, 
                   sizeof(char), 
                   strlen(buffer), 
                   f);
  if(written != (int)strlen(buffer)) {
    res = CFG_FAILED;
  }
  free(buffer); 
  return res;
}

CFG_status CFG_Get(char* ident, char* dflt, char** result) {
#ifdef USE_HSEARCH
  ENTRY *hresult;
  ENTRY item;
  regmatch_t match[1];
  int rc;


  if(!CFG_Initialized()) {
    cfg_init(separator,1000);
  }
  if(!CFG_Loaded()) {
    cfg_load();
  }
  /* Check that the key is valid. */
  rc = regexec(&c_ident_regex, ident, 1, match, 0);
  if(REG_NOMATCH == rc) 
    return CFG_BAD_KEY;

  item.key = ident;
  hresult = hsearch(item, FIND);
  if(NULL == hresult) {
    *result = dflt;
    return CFG_DEFAULT;
  } else {
    *result = hresult->data;
  }
  return CFG_OK;
#else
  char * res = NULL;
  regmatch_t match[1];
  int rc;

  if(!CFG_Initialized()) {
    cfg_init(separator);
  }
  if(!CFG_Loaded()) {
    cfg_load();
  }
  /* Check that the key is valid. */
  rc = regexec(&c_ident_regex, ident, 1, match, 0);
  if(REG_NOMATCH == rc) 
    return CFG_BAD_KEY;

  res = hash_find(table,(void*)ident);
  if(NULL == res) {
    *result = dflt;
    return CFG_DEFAULT;
  }
  else {
    *result = (char*)res;
  }
  return CFG_OK;
#endif
}

CFG_status CFG_Set(char* ident, char* value) {
#ifdef USE_HSEARCH
  ENTRY *hresult;
  ENTRY item;
  char *ident_buffer, *value_buffer; 
  regmatch_t match[1];
  int rc;

  if(!CFG_Initialized()) {
    cfg_init(separator,1000);
  }
  if(!CFG_Loaded()) {
    cfg_load();
  }
  /* Check that the key is valid. */
  rc = regexec(&c_ident_regex, ident, 1, match, 0);
  if(REG_NOMATCH == rc) 
    return CFG_BAD_KEY;

  ident_buffer = NEW( strlen(ident)+1);
  if(NULL==ident_buffer) {
#ifndef NDEBUG
    fprintf(stderr,"out of memory");
#endif
    return CFG_FAILED;
  }
  if(ident_buffer)
    strcpy(ident_buffer,ident);
  item.key = ident_buffer;

  value_buffer = NEW( strlen(value)+1);
  if(NULL==value_buffer) {
#ifndef NDEBUG
    fprintf(stderr,"out of memory");
#endif
    return CFG_FAILED;
  }
  if(value_buffer)
    strcpy(value_buffer,value);
  item.data = value_buffer;

  hresult = hsearch(item, ENTER);
  
  return CFG_OK;
#else
  char *ident_buffer, *value_buffer; 
  regmatch_t match[1];
  int rc;

  if(!CFG_Initialized()) {
    cfg_init(separator);
  }
  if(!CFG_Loaded()) {
    cfg_load();
  }
  /* Check that the key is valid. */
  rc = regexec(&c_ident_regex, ident, 1, match, 0);
  if(REG_NOMATCH == rc) 
    return CFG_BAD_KEY;

  ident_buffer = NEW( strlen(ident)+1);
  if(NULL==ident_buffer) {
#ifndef NDEBUG
    fprintf(stderr,"out of memory");
#endif
    return CFG_FAILED;
  }
  if(ident_buffer)
    strcpy(ident_buffer,ident);

  value_buffer = NEW( strlen(value)+1);
  if(NULL==value_buffer) {
#ifndef NDEBUG
    fprintf(stderr,"out of memory");
#endif
    return CFG_FAILED;
  }
  if(value_buffer)
    strcpy(value_buffer,value);

  CFG_Remove(ident_buffer);
  hash_insert(table, ident_buffer, value_buffer);
  return CFG_OK;
#endif
}

CFG_status CFG_Remove(char *ident) {
  char *value=0;
  value = hash_find(table,(void*)ident);
  if(value) {
    DELETE(value);
    value = hash_remove(table, ident);
    if(value) 
      DELETE(value);
  }
  return CFG_OK;
}

CFG_status CFG_RemoveMatch(char *pattern) {
  struct CFG_Iter *it;
  char *key, *value;
  CFG_status res;

  CFG_CreateGlobIterator(pattern, &it);
  while(CFG_IteratorMore(it)) {
    res = CFG_IteratorNext(it, &key, &value);
    if(res) 
      CFG_Remove(key);
  }
  CFG_DestroyIterator(it);
  return CFG_OK;
}

/* Read the file named by filename, and insert its contents at the key
   given by ident.  When CFG_WriteFile writes out the config file, the
   entry for the file will have all of its trailing newlines escaped.
   The newline escapes are removed when the config file is later read
   in. */

CFG_status CFG_InsertEntryFromFile(char *ident, char *filename) {
  FILE* f;
  off_t fsize=0;
  ssize_t charcount=0;
  char *contents=NULL;
  CFG_status result = CFG_OK;

  if(!CFG_Initialized()) {
#ifdef USE_HSEARCH
    cfg_init(separator,1000);
#else
    cfg_init(separator);
#endif
  }
  if(NULL == filename) {
    return CFG_FAILED;
  }
  fsize = filesize(filename);
  if(-1 == fsize)
    return CFG_FAILED;
  else if(0 == fsize)
    return CFG_ZERO_LENGTH_FILE;
  contents = (char*)malloc(fsize + 2);
  memset(contents,0,fsize+2);
  if(0 == contents)
    return CFG_FAILED;
  f = fopen(filename,"r");
  if(0 == f)
    return CFG_FAILED;
  
  charcount = fread(contents,sizeof(char),fsize,f);
  if(charcount != (ssize_t)fsize) {
    fclose(f);
    return CFG_FAILED;
  }
  contents[fsize] = 0;
  result = CFG_Set(ident,contents);
  fclose(f);
  free(contents);
  return result;
}

/* Write the entry in the database at the key given by ident to the
   file given by filename.  If the file already exists it is
   overwritten.  */

CFG_status CFG_ExtractEntryToFile(char *ident, char *filename) {
  FILE *f;
  char *buffer;
  int written;
  CFG_status result = CFG_OK;

  result = CFG_Get(ident,0,&buffer);
  if(CFG_OK != result)
    return CFG_FAILED;

  f = fopen(filename,"w");
  if(NULL == f)
    return CFG_FAILED;

  written = fwrite(buffer, 
                   sizeof(char), 
                   strlen(buffer), 
                   f);
  if(written != (int)strlen(buffer)) {
    result = CFG_FAILED;
  }
  fclose(f);
  return result;
}

static int cfg_CountNewlines(char * str) {
  int count=0;

  while(*str) {
    if('\n' == *str) {
      count++;
    }
    str++;
  }
  return count;
}

/* Insert backslashes before every newline in the given string.  This
   is hard because the string may get longer, in which case a
   reallocation may be needed.  N.B. str must be a buffer allocated by
   malloc.  We may be realloc'ing it.  Beware. */

static char *cfg_EscapeNewlines(char *str, ssize_t size) {
  long count;
  long length = strlen(str);
  long w = 0;
  long mark = 0, where = 0;
  char *tbuffer;
  count = cfg_CountNewlines(str);
  if(count <= 1)
    return str;
  if(length + count + 1 > size) {
    str = realloc(str, length + count + 1);
    size = length + count;
  }
  tbuffer = (char*)malloc(size+1);
  strcpy(tbuffer,str);
  w = strcspn(tbuffer+mark, "\n");
  length = (long)strlen(tbuffer);
  while(where+w < length) 
  {
    memcpy(str+mark,tbuffer+where,w);
    mark += w;
    where += w;
    str[mark++] = '\\';
    str[mark++] = '\n';
    w = strcspn(tbuffer+(++where), "\n");
  }
  strcpy(str+mark, tbuffer+where);
  free(tbuffer);
  return str;
}

/* Remove any backslashes that immediately precede newlines.  This is
   easy, because the string always gets smaller. */

static void cfg_UnescapeNewlines(char *str) {
  long w=0;
  long mark=0, where=0;
  long length;

  w = strcspn(str+where, "\n");
  length = (long)strlen(str);
  while(where+w < length) {
    memcpy(str+mark,str+where,w);
    mark+=w;
    where+=w;
    if('\\' == str[mark-1]) {
      str[mark-1] = '\n';
    } else {
      str[mark++] = '\n';
    }
    w = strcspn(str+(++where), "\n");
  }
  strcpy(str+mark, str+where);
}

struct CFG_Iter {
    int  current;
    char **keys;
    int count;
};

CFG_status CFG_CreateIterator(struct CFG_Iter **iter) { 
  if(!CFG_Initialized()) {
#ifdef USE_HSEARCH
    cfg_init(separator,1000);
#else
    cfg_init(separator);
#endif
  }
  if(!CFG_Loaded()) {
    cfg_load();
  }
  *iter = (struct CFG_Iter *)malloc(sizeof(struct CFG_Iter));
  (*iter)->current = 0;
  (*iter)->keys = hash_keys(table, &((*iter)->count));
  qsort((*iter)->keys, ((*iter)->count), sizeof(char*), compare_keys);

  return CFG_OK;
}

CFG_status CFG_CreateGlobIterator(char *glob_pattern,
                                  struct CFG_Iter **iter) {
  if(!CFG_Initialized()) {
#ifdef USE_HSEARCH
    cfg_init(separator,1000);
#else
    cfg_init(separator);
#endif
  }
  if(!CFG_Loaded()) {
    cfg_load();
  }
  *iter = (struct CFG_Iter *)malloc(sizeof(struct CFG_Iter));
  (*iter)->current = 0;
  (*iter)->keys = hash_keys(table, &((*iter)->count));
  qsort((*iter)->keys, ((*iter)->count), sizeof(char*), compare_keys);
  {
    int i;
    for(i=0; i<(*iter)->count; ++i) {
      if(fnmatch(glob_pattern, (*iter)->keys[i], 0)) {
        (*iter)->keys[i] = NULL; /* Put a hole in the array. */
      }
    }
  }
  return CFG_OK;
}

CFG_status CFG_CreateFilterIterator(IterFilter filter,
                                    struct CFG_Iter **iter) {
  if(!CFG_Initialized()) {
#ifdef USE_HSEARCH
    cfg_init(separator,1000);
#else
    cfg_init(separator);
#endif
  }
  if(!CFG_Loaded()) {
    cfg_load();
  }
  *iter = (struct CFG_Iter *)malloc(sizeof(struct CFG_Iter));
  (*iter)->current = 0;
  (*iter)->keys = hash_keys(table, &((*iter)->count));
  qsort((*iter)->keys, ((*iter)->count), sizeof(char*), compare_keys);
  {
    int i;
    char *value;
    for(i=0; i<(*iter)->count; ++i) {
      CFG_Get((*iter)->keys[i],"",&value);
      if(!filter((*iter)->keys[i], value)) {
        (*iter)->keys[i] = NULL; /* Put a hole in the array. */
      }
    }
  }
  return CFG_OK;
}


CFG_status CFG_DestroyIterator(struct CFG_Iter *iter) {
  free(iter->keys);
  free(iter);
  return CFG_OK;
}


CFG_status CFG_IteratorMore(struct CFG_Iter *iter) {
  return (iter->current < iter->count) ? CFG_OK : CFG_FALSE;
}

CFG_status CFG_IteratorNext(struct CFG_Iter *iter,
                            char** key,
                            char** value) {
  *key = iter->keys[iter->current];
  iter->current++;
  /* The pattern match, or filter iterators will have holes in the
     keys array.  We need to make sure we skip over the holes. */
  while(!*key) {
    *key = iter->keys[iter->current];
    iter->current++;
    if(iter->current >= iter->count) 
      return CFG_FALSE;
  }    
  return CFG_Get(*key,"",value);
}

CFG_status CFG_IteratorRewind(struct CFG_Iter *iter) {
  
  iter->current = 0;
  return CFG_OK;
}

CFG_status CFG_IteratorCount(struct CFG_Iter *iter, int * count) {
  char *k, *v;
  *count = 0;
  CFG_IteratorRewind(iter);
  while(CFG_IteratorMore(iter)) {
    CFG_IteratorNext(iter, &k, &v);
    if(k) (*count)++;
  }
  return CFG_OK;
}
/* Turn off locking and unlocking of the cluster shared database.*/

CFG_status CFG_cluster_locking_off(void) {
  cluster_locking = CFG_FALSE;
  return CFG_OK;
}

/* Turn locking back on for the cluster shared database. */

CFG_status CFG_cluster_locking_on(void) {
  cluster_locking = CFG_OK;
  return CFG_OK;
}
