/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.

  
  Night/ware RPC100S  support contributed by
    Eric Z. Ayers  <eric.ayers@compgen.com> 
    Computer Generation, Incorporated
    
*/
#include <power.h>
#include <parseconf.h>
#include <string.h>
#include <stdlib.h>
#include <clucfg.h>

static const char *version __attribute__ ((unused)) = "$Id: power.c,v 1.9 2000/11/10 19:29:48 burke Exp $";

/* power.c

   Dispatch to code that works with the switch that is installed on
   the machine.

   author: Ron Lawrence <lawrence@missioncriticallinux.com>

*/

#define STREQ(s1,s2) (!strcmp((s1),(s2)))

#define SET(where, what)   (where |= (what))
#define RESET(where, what) (where &= ~(what))
#define ISSET(where, what) (where & (what))

#define DFLT_TMP_BUFFER_SZ 32

#define POWERD_SWITCH_TYPE "power%switchType" 

/* RPS10 decl's */

enum PWR_boolean PWR_RPS10_configure(char *config_file_name);
enum PWR_boolean PWR_RPS10_set(char* option, char* value);
enum PWR_boolean PWR_RPS10_get(char* option, char *dflt, char** value);
enum PWR_boolean PWR_RPS10_init(PWR_result *status);
enum PWR_boolean PWR_RPS10_release(void);
PWR_result  PWR_RPS10_status(void);
PWR_result  PWR_RPS10_reboot(void);
PWR_result  PWR_RPS10_off(void);
PWR_result  PWR_RPS10_on(void);

/* APC decl's */

enum PWR_boolean PWR_APC_configure(char *config_file_name);
enum PWR_boolean PWR_APC_set(char* option, char* value);
enum PWR_boolean PWR_APC_get(char* option, char *dflt, char** value);
enum PWR_boolean PWR_APC_init(PWR_result *status);
enum PWR_boolean PWR_APC_release(void);
PWR_result  PWR_APC_status(void);
PWR_result  PWR_APC_reboot(void);
PWR_result  PWR_APC_off(void);
PWR_result  PWR_APC_on(void);

/* Night/ware RPC100S decl's */

enum PWR_boolean PWR_NW_RPC100S_configure(char *config_file_name);
enum PWR_boolean PWR_NW_RPC100S_set(char* option, char* value);
enum PWR_boolean PWR_NW_RPC100S_get(char* option, char *dflt, char** value);
enum PWR_boolean PWR_NW_RPC100S_init(PWR_result *status);
enum PWR_boolean PWR_NW_RPC100S_release(void);
PWR_result  PWR_NW_RPC100S_status(void);
PWR_result  PWR_NW_RPC100S_reboot(void);
PWR_result  PWR_NW_RPC100S_off(void);
PWR_result  PWR_NW_RPC100S_on(void);
 
/* NONE decl's - not really a power switch, stub driver for no switch */

enum PWR_boolean PWR_NONE_configure(char *config_file_name);
enum PWR_boolean PWR_NONE_set(char* option, char* value);
enum PWR_boolean PWR_NONE_get(char* option, char *dflt, char** value);
enum PWR_boolean PWR_NONE_init(PWR_result *status);
enum PWR_boolean PWR_NONE_release(void);
PWR_result  PWR_NONE_status(void);
PWR_result  PWR_NONE_reboot(void);
PWR_result  PWR_NONE_off(void);
PWR_result  PWR_NONE_on(void);

static int switch_type = SWT_UNSPECIFIED;
static int determine_switch_type(void);

/* inline functions to expand in the following macro. */

inline static enum PWR_boolean RESULT_boolean(void) {
  return PWR_FALSE;
}

inline static PWR_result RESULT_result(void) 
{
  PWR_result result=0;
  SET(result, PWR_ERROR);
  return result;
}

/* Macro's can be hard to understand.  But, not this one. */

#define DISPATCH(type, cmd, args...)            \
do {                                            \
  switch(switch_type) {                         \
    case SWT_RPS10:                             \
      return PWR_RPS10_ ## cmd (args);          \
      break;                                    \
    case SWT_APC:                               \
      return PWR_APC_ ## cmd (args);            \
      break;                                    \
    case SWT_NONE:                              \
      return PWR_NONE_ ## cmd (args);           \
      break;                                    \
    case SWT_NW_RPC100S:                        \
      return PWR_NW_RPC100S_ ## cmd (args);     \
      break;                                    \
    default:                                    \
      return RESULT_ ## type ();                \
      break;                                    \
  }                                             \
} while(0) 

enum PWR_boolean PWR_configure(char *config_file_name) {
  DISPATCH(boolean,configure, config_file_name);
}

enum PWR_boolean PWR_set(char* option, char* value) {
  DISPATCH(boolean,set, option, value);
}

enum PWR_boolean PWR_get(char* option, char *dflt, char** value) {
  DISPATCH(boolean,get, option, dflt, value);
}

enum PWR_boolean PWR_init(PWR_result *status) {
  if (determine_switch_type() < 0) {
      *status = 0;
      SET(*status, PWR_ERROR);
  }
  DISPATCH(boolean,init, status);
}

/* Retrieves the switch type as specified in the configuration database,
   matches it against the known supported set of strings.  Upon a match it
   sets the static variable switch_type accordingly.
   Returns -1 on inability to read config file, 0 otherwise.
 */
static int determine_switch_type(void) {
  char *param;
  CluCfg              *cfg = (CluCfg *)NULL;
  char                *file = NULL;
  char sdflt[DFLT_TMP_BUFFER_SZ] = "";
  char *svalue;
  int  retval = 0;

  CFG_Get("power%type", sdflt, &svalue);
  if( STREQ(svalue,"") ) {
    cfg = get_clu_cfg(file);
    if (cfg == NULL) {
      switch_type = SWT_UNSPECIFIED;
      retval = -1;
    }
    else {
      param = cfg->nodes[cfg->lid].powerSwitchType;
      if(STREQ(PWR_TYPE_RPS10, param))
        switch_type = SWT_RPS10;
      else if(STREQ(PWR_TYPE_APC, param))
        switch_type = SWT_APC;
      else if(STREQ(PWR_TYPE_NW_RPC100S, param))
        switch_type = SWT_NW_RPC100S;	      
      else if(STREQ(PWR_TYPE_NONE, param))
        switch_type = SWT_NONE;
      else 
        switch_type = SWT_UNSPECIFIED;
    }
  } 
  else {
      if(STREQ(PWR_TYPE_RPS10, svalue))
        switch_type = SWT_RPS10;
      else if(STREQ(PWR_TYPE_APC, svalue))
        switch_type = SWT_APC;
      else if(STREQ(PWR_TYPE_NW_RPC100S, svalue))
        switch_type = SWT_NW_RPC100S;
      else if(STREQ(PWR_TYPE_NONE, svalue))
        switch_type = SWT_NONE;
      else 
        switch_type = SWT_UNSPECIFIED;
  }
  if (cfg != NULL) {
	free(cfg);
  }
  return(retval);
}

enum PWR_boolean PWR_release(void) {
  DISPATCH(boolean,release);
}

PWR_result  PWR_status(void) {
  DISPATCH(result,status);
}

PWR_result  PWR_reboot(void) {
  DISPATCH(result,reboot);
}

PWR_result  PWR_off(void) {
  DISPATCH(result,off);
}

PWR_result  PWR_on(void) {
  DISPATCH(result,on);
}

int  PWR_type(void) {
  if (switch_type == SWT_UNSPECIFIED) {
	determine_switch_type();
  }
  return(switch_type);
}

