

#  Copyright Mission Critical Linux, 2000

#  Kimberlite is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 2, or (at your option) any
#  later version.

#  Kimberlite is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with Kimberlite; see the file COPYING.  If not, write to the
#  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
#  MA 02139, USA.
#
# $Revision: 1.13 $
#
# Author: Gregory P. Myrdal <Myrdal@MissionCriticalLinux.Com>
#
#------------------------------------------------------------

#
# KornShell library for IP functions.
#

#
# inSameSubnet IPaddr1 IPaddr2 mask
#
# Given two IP addresses and a subnet mask determine if these IP
# addresses are in the same subnet.  If they are, return $YES, if
# not return $NO.  In case of an error, return $FAIL.
#
#
inSameIPsubnet ()
{
	typeset -i n
	typeset -i mask 
	typeset -i ip1 ip2		# IP addresses given
	typeset -i quad1 quad2		# calculated quad words

	if [ $# -ne 3 ]; then
	  print "Usage: ipSameSubnet IPaddr1 IPaddr2 mask"
	  return 1
	fi

	#
	# Remove '.' characters from dotted decimal notation and save
	# in arrays. i.e.
	#
	#	192.168.1.163 -> array[0] = 192
	#	                 array[1] = 168
	#	                 array[2] = 1
	#	                 array[3] = 163
	#
	set -A ip1 $(print $1 | sed 's/\./ /g')
	set -A ip2 $(print $2 | sed 's/\./ /g')
	set -A mask $(print $3 | sed 's/\./ /g')

	#
	# For each quad word, logically AND the IP address with the subnet
	# mask to get the network/subnet quad word.  If the resulting
	# quad words for both IP addresses are the same they are in the 
	# same IP subnet.
	#
	for n in 0 1 2 3
	do
	  ((quad1=ip1[n] & mask[n]))
	  ((quad2=ip2[n] & mask[n]))

	  if ((quad1 != quad2)) ; then
	    return $NO	# in different subnets
	  fi
	done

	return $YES		# in the same subnet, all quad words matched
}

#
# findInterface IPaddr
#
# Given a target IP address find the interface in which this address is
# configured.  If found return $SUCCESS, if not return $NOT_FOUND.  The
# interface name is returned to stdout.
#
findInterface()
{
	typeset line
	typeset intf
	typeset addr
	typeset state

	typeset target=$1

	ifconfig |&

	while read -p intf line
	do
	  while read -p line
	  do
	    if [ "$line" = "" ]; then	# go to next interface
	      continue 2
	    fi

	    set - $line

	    addr=
	    while [ $# -gt 0 ]; do
	      case $1 in
	        addr:*)
	          addr=${1##addr:}
	          if [ -n "$addr" -a $addr = $target ]; then
	            print $intf
	            return $SUCCESS
	          fi
	          ;;
	      esac
	      shift
	    done
	  done
	done

	return $NOT_FOUND
}

#
# findMacAddr
#
# Given an interface find the MAC addresses associated with it.
# Return $SUCCESS when found, return $NOT_FOUND if an interface 
# was not found and return $FAIL on error.
#
findMacAddr()
{
	typeset line
	typeset intf
	typeset addr
	typeset junk
	typeset state

	typeset target=$1

	ifconfig $target |&

	while read -p intf line
	do
	  set - $line

	  while [ $# -gt 0 ]; do
	    case $1 in
	      HWaddr)
	        print $2	 	# return MAC addr
	        return $SUCCESS 
	        ;;
 	    esac
	    shift
	  done
	done

	return $NOT_FOUND
}

#
# findNetmask
#
# Given an interface find the netmask addresses associated with it.
# Return $SUCCESS when found, return $NOT_FOUND if an interface 
# was not found and return $FAIL on error.
#
findNetmask()
{
	typeset line
	typeset intf
	typeset addr
	typeset junk
	typeset state
	typeset addr

	typeset target=$1
	typeset svc_name=$2
	typeset svc_id=$3
	typeset n=$4

	# 
	# If the user defined a network mask in the config file, use it.
	#
	addr=$(getSvcNetmask $DB $svc_id $n)
	case $? in
	  0) print $addr 	# found it
	     return $SUCCESS
	     ;;
	  2) addr= 		# not found
	     ;;
	  *) logAndPrint $LOG_ERR "\
Cannot find netmask $n for service $svc_name, err=$?"
	  return $FAIL ;;
	esac

	#
	# If no network mask is defined.  Find it from the interface
	# that this ip address is assigned to.
	#
	ifconfig $target |&

	while read -p line
	do
	  set - $line

	  while [ $# -gt 0 ]; do
	    case $1 in
	      Mask:*)
	        print ${1##*:}	 	# return netmask addr
	        return $SUCCESS 
	        ;;
 	    esac
	    shift
	  done
	done

	return $NOT_FOUND
}

#
# findBroadcast
#
# Given an interface find the broadcast addresses associated with it.
# Return $SUCCESS when found, return $NOT_FOUND if an interface 
# was not found and return $FAIL on error.
#
findBroadcast()
{
	typeset line
	typeset intf
	typeset addr
	typeset junk
	typeset state

	typeset target=$1
	typeset svc_name=$2
	typeset svc_id=$3
	typeset n=$4

	# 
	# If the user defined a broadcast address in the config file, use it.
	#
	addr=$(getSvcBroadcast $DB $svc_id $n)
	case $? in
	  0) print $addr 	# found it
	     return $SUCCESS
	     ;;
	  2) addr= 		# not found
	     ;;
	  *) logAndPrint $LOG_ERR "\
Cannot find broadcast $n for service $svc_name, err=$?"
	  return $FAIL ;;
	esac

	#
	# If no broadcast address is defined. Find it from the interface
	# that this ip address is assigned to.
	#
	ifconfig $target |&

	while read -p line
	do
	  set - $line

	  while [ $# -gt 0 ]; do
	    case $1 in
	      Bcast:*)
	        print ${1##*:}	 	# return broadcast addr
	        return $SUCCESS 
	        ;;
 	    esac
	    shift
	  done
	done

	return $NOT_FOUND
}

#
# findInterfaceInSubnet IPaddr
#
# Give a target IP address find the interface which is configured
# in this subnet and find a free interface name.  When found return
# $SUCCESS and print the interface name to stdout.  If not found return
# $NOT_FOUND, on error return $FAIL.
#
findInterfaceInSubnet ()
{
	typeset line
	typeset intf
	typeset addr
	typeset mask

	typeset target=$1

	ifconfig |&

	while read -p intf line
	do
	  while read -p line
	  do
	    if [ "$line" = "" ]; then	# go to next interface
	      continue 2
	    fi

	    set - $line

	    addr=
	    mask=
	    while [ $# -gt 0 ]; do
	      case $1 in
	        addr:*) addr=${1##addr:} ;;
	        Mask:*) mask=${1##Mask:} ;;
	      esac
	      shift
	    done

	    if [ -z "$addr" -o -z "$mask" ]; then
	      continue
	    fi
	  
	    inSameIPsubnet $target $addr $mask
	    if [ $? = $YES ]; then
	      print $intf
	      return $SUCCESS
	    fi
	  done
	done

	return $NOT_FOUND
}

#
# reserveNextFreeInterface IPaddr
#
# Given an IP address find an interface for it to be configured on.
# First we must find an interface in the same IP subnet as this IP
# address.  Then we look for a free instance of that interface.
# On return, print the name of the interface that can be used and
# return $SUCCESS.  On error or not found return $FAIL.
#
reserveNextFreeInterface()
{
	typeset used_interfaces used_intf
	typeset ipaddr
	typeset intf
	typeset -i n

	ipaddr=$1

	intf=$(findInterfaceInSubnet $ipaddr)
	if [ $? -ne $SUCCESS ]; then
	  #logAndPrint "Error: Cannot find interface for address $ipaddr"
	  return $FAIL
	fi

	lockFile $ifconfig_lockfile

	used_interfaces=$(ifconfig | grep "^${intf}:[0-9]" | sed 's/[ 	].*//')

	for n in 0 1 2 3 4 5 6 7 8 9
	do
	  for used_intf in $used_interfaces
	  do
	    if [ $used_intf = ${intf}:$n ]; then
	      continue 2	# its used, try next number
	    fi
	  done
  
	  #
	  # Claim this interface as used.  
	  #
	  # Note: The caller should apply the correct netmask and broadcast
	  # address and any other options.  This is just a place holder.
	  # Also the caller needs to 'ifconfig $intf down' if something 
	  # goes wrong.
	  #
	  ifconfig ${intf}:$n $ipaddr broadcast 0.0.0.0 \
	             netmask 255.255.255.255 > /dev/null 2>&1
	  print ${intf}:$n
	  unlockFile $ifconfig_lockfile
	  return $SUCCESS		# found a free interface
	done

	#logAndPrint "Error: Cannot find interface for address $ipaddr"
	unlockFile $ifconfig_lockfile
	return $NOT_FOUND		# could not find a free interface
}

#
#  startIP serviceID
#
# Given a service ID number configure all of its IP addresses.
#
startIP()
{
	typeset -i n=MIN_TOKEN_ID
	typeset ip_alias
	typeset mac_addr
	typeset intf
	typeset real_intf
	typeset ret_val
	typeset -i try=0
	typeset -i max_tries=10	# max tries to ifconfig address

	while : ; do
	  ((try=try+1))
	  ip_alias=$(getSvcIPaddress $DB $SVC_ID $n)
	  case $? in
	    0) : ;;		# found it
	    2) break ;;		# no more IP addresses found
	    *) logAndPrint $LOG_ERR "\
Cannot find IP address $n for service $SVC_NAME, err=$?"
	    return $FAIL ;;
	  esac

	  logAndPrint $LOG_INFO "Starting IP address $ip_alias"

	  intf=$(findInterface $ip_alias)
	  if [ $? -ne $SUCCESS ]; then

	    intf=$(reserveNextFreeInterface $ip_alias)
	    if [ $? -ne $SUCCESS ]; then
	      logAndPrint $LOG_ERR "Cannot find interface for $ip_alias"
	      return $FAIL
	    fi

	    real_intf=${intf%%:*}

	    netmask_addr=$(findNetmask $real_intf "$SVC_NAME" $SVC_ID $n)
	    if [ $? -ne $SUCCESS ]; then
	      logAndPrint $LOG_ERR "\
Cannot find network mask for interface '$real_intf'"

	      # remove entry added by reserveNextFreeInterface()
	      ifconfig $intf down

	      return $FAIL
	    fi

	    broadcast_addr=$(findBroadcast $real_intf "$SVC_NAME" $SVC_ID $n)
	    if [ $? -ne $SUCCESS ]; then
	      logAndPrint $LOG_ERR "\
Cannot find broadcast address for interface '$real_intf'"

	      # remove entry added by reserveNextFreeInterface()
	      ifconfig $intf down

	      return $FAIL
	    fi

	    ifconfig_cmd="ifconfig $intf $ip_alias broadcast $broadcast_addr netmask $netmask_addr"
	    logAndPrint $LOG_DEBUG $ifconfig_cmd

	    $ifconfig_cmd 2> /dev/null
	    ret_val=$?
	    if [ $ret_val -ne 0 -a $ret_val -ne 255 ]; then # 255 = already done
	      logAndPrint $LOG_ERR "\
Cannot ifconfig $ip_alias to $intf; err=$ret_val"
	      return $FAIL
	    fi
	  else
	    real_intf=${intf%%:*}
	  fi

	  intf=$(findInterface $ip_alias)
	  if [ $? -ne $SUCCESS ]; then
	    if [ $try -gt $max_tries ]; then
	      logAndPrint $LOG_ERR \
	      	"Cannot configure IP address $ip_alias; max attempts reached"
	      return $FAIL
	    fi
	    logAndPrint $LOG_WARNING \
	      "Cannot configure IP address $ip_alias; retrying"
	    continue
	  fi

	  logAndPrint $LOG_DEBUG "Adding host based route for $ip_alias"

	  route add -host $ip_alias dev $intf 2> /dev/null
	  ret_val=$?
	  if [ $ret_val -ne 0 -a $ret_val -ne 7 ]; then # 7 = already done
	    logAndPrint $LOG_ERR "\
Cannot route add $ip_alias to $intf; err=$ret_val"
	    return $FAIL
	  fi

	  mac_addr=$(findMacAddr $real_intf)
	  if [ $? -ne $SUCCESS ]; then
	    logAndPrint $LOG_ERR "\
Cannot find harware address for interface '$real_intf'"
	    return $FAIL
	  fi

	  logAndPrint $LOG_DEBUG "Adding ARP entry for $ip_alias ($mac_addr)"

	  arp -i $real_intf -s $ip_alias $mac_addr 2> /dev/null
	  ret_val=$?
	  if [ $ret_val -ne 0 -a $ret_val -ne 255 ]; then # 255 = already done
	    logAndPrint $LOG_ERR "\
Cannot add ARP entry for $ip_alias ($mac_addr); err=$ret_val"
	    return $FAIL
	  fi

	  logAndPrint $LOG_INFO "Sending Gratuitous arp for $ip_alias ($mac_addr)"
	  for i in 1 2 3
	  do
	    logAndPrint $LOG_DEBUG $GRATUITOUSARP $ip_alias $mac_addr $ip_alias ffffffffffff $real_intf
	    $GRATUITOUSARP $ip_alias $mac_addr $ip_alias ffffffffffff $real_intf
	    ret_val=$?
	    if [ $ret_val -ne 0 ]; then
	      logAndPrint $LOG_ERR "\
Cannot send gratuitous ARP for $ip_alias ($mac_addr); err=$ret_val"
	      return $FAIL
	    fi
	  done

	  ((n=n+1))
	done

	return $SUCCESS
}

#
#  stopIP serviceID
#
# Given a service ID number unconfigure all of its IP addresses.
#
stopIP()
{
	typeset -i n=MIN_TOKEN_ID
	typeset ip_alias
	typeset intf
	typeset ret_val

	while : ; do
	  ip_alias=$(getSvcIPaddress $DB $SVC_ID $n)
	  case $? in
	    0) : ;;		# found it
	    2) break ;;		# no more IP addresses found
	    *) logAndPrint $LOG_ERR "\
Cannot find IP address $n for service $SVC_NAME, err=$?"
	    return $FAIL ;;
	  esac

	  intf=$(findInterface $ip_alias)
	  if [ $? -ne $SUCCESS ]; then
	    logAndPrint $LOG_DEBUG "IP address $ip_alias already removed"
	  else
	    logAndPrint $LOG_INFO "Stopping IP address $ip_alias"

	    logAndPrint $LOG_DEBUG ifconfig $intf down

	    ifconfig $intf down 2> /dev/null
	    ret_val=$?
	    if [ $ret_val -ne 0 -a $ret_val -ne 255 ]; then # 255 = already done
	      logAndPrint $LOG_ERR "Cannot ifconfig $intf down"
	      return $FAIL
	    fi
	  fi

	  intf=$(findInterface $ip_alias)
	  if [ $? -ne $NOT_FOUND ]; then
	    logAndPrint $LOG_ERR "Cannot un-configure IP address $ip_alias"
	    return $FAIL
	  fi

	  logAndPrint $LOG_DEBUG "Removing host based route for $ip_alias"

	  route delete -host $ip_alias 2> /dev/null
	  ret_val=$?
	  if [ $ret_val -ne 0 -a $ret_val -ne 7 ]; then	# 7 means already done
	    logAndPrint $LOG_ERR "Cannot route delete $ip_alias; err=$ret_val"
	    return $FAIL
	  fi

	  logAndPrint $LOG_DEBUG "Removing ARP entry for $ip_alias"

	  arp -d $ip_alias 2> /dev/null
	  ret_val=$?
	  if [ $ret_val -ne 0 -a $ret_val -ne 255 ]; then # 255 = already done
	    logAndPrint $LOG_ERR "\
Cannot delete ARP entry for $ip_alias; err=$ret_val"
	    return $FAIL
	  fi

	  ((n=n+1))
	done

	return $SUCCESS
}
