/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/*
 * $Id: clustat.c,v 1.21 2000/11/13 14:33:56 burke Exp $
 *
 * Authors: Brian Stevens <stevens@missioncriticallinux.com>
 *          Jeff Moyer    <moyer@missioncriticallinux.com>
 *
 * Description: Command to query and display cluster statistics.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/param.h>
#include <string.h>

#include <clusterdefs.h>
#include <clucfg.h>
#include <msgsvc.h>
#include <svcmgr.h>
#include <parseconf.h>
#include <diskapis.h>
#include <disk_proto.h>
#include <svcmgr.h>
#include <power.h>

static const char *version __attribute__ ((unused)) = "$Revision: 1.21 $";

static SharedDiskNodeStates nodestates;

#define PSWITCH_STATUS_STRLEN  32
#define TIMESTR_LEN            128  /* The timestr we use is 28 bytes. */

int listen_fd=-1;

int main(
    int			argc,
    char		**argv)
{
    CluCfg		*cfg = (CluCfg *)NULL;    
    int			n, c, s;
    ChanState		*chan_state;
    msg_handle_t	con;
    ServiceBlock	svc;
    char		qstr[80];
    char		*node = NULL;
    char		*name;
    int			refresh = 0;
    char		opt;
    int			verbose = 0;
    int                 chan_sw_info = 0;
    int                 chan = 0;
    int			retval;
    int                 auth = 0;
    generic_msg_hdr     msghdr;
    PswitchStatusMsg    pswitch_msg;
    char                **pswitch_str = NULL;
    int			nodes_in_unknown_state;
    time_t              clk;
    struct tm           *tm;
    char                timestr[TIMESTR_LEN];
    int                 service_count, prior_service_count = 0;
    int 		myNodeNumber = cluGetLocalNodeId();

    while ((opt = getopt(argc, argv, "vi:c:sn")) != EOF) {
        switch (opt) {
        case 'v':
            verbose = 1;
            break;
        case 'i':
            refresh = atoi(optarg);
            break;
        case 'c':
            chan_sw_info = 1;
	    chan = atoi(optarg);
            break;
        case 's':
            chan_sw_info = 2;
            break;
        default:
            break;
        }
    }   

    /*
     * Only need to fill this in once.
     */
    msghdr.magic = GENERIC_HDR_MAGIC;

    if (geteuid() != (uid_t)0) {
    	    fprintf(stderr, "%s must be run as the user root\n", argv[0]);
    	    exit(1);    
    }

    cfg = get_clu_cfg(NULL);
    if (cfg == NULL) {
            fprintf(stderr, "error in get_clu_cfg\n");
	    perror("get_clu_cfg");
	    exit(1);
    }

    pswitch_str = (char **)malloc(sizeof(char*) * cfg->num_nodes);
    for (n = 0; n < cfg->num_nodes; n++) {
	    pswitch_str[n] = (char *)malloc(PSWITCH_STATUS_STRLEN);
    }

    while (1) {

	if (cfg != NULL) {
	    free(cfg);
	    cfg = (CluCfg *)NULL;
	}
        cfg = get_clu_cfg(NULL);
        if (cfg == NULL) {
            fprintf(stderr, "error in get_clu_cfg\n");
	    perror("get_clu_cfg");
	    exit(1);
        }

	/*
	 * Get the status of the channels
	 */

	chan_state = cluGetChanState();

        /*
         * When looking for channel state as specified in passed args
         */

	if (chan_sw_info == 1) {
	    if (chan_state) {
	        if ((chan >= 0) && (chan < cfg->num_chans)) {
                    /* return 1 or 0 depending on channel status */
		    if (chan_state[chan].state) {
		        fprintf (stdout, "%s\n", "Up");
			exit (0);
		    }
		    else {
		        fprintf (stdout, "%s\n", "Down");
		        exit (1);
		    }
                } else {
                    /* chan does not map to a working channel */
		    fprintf (stdout, "%s\n", "Unknown. Invalid channel number");
		    exit (2);
                }
            } else {
                /* could not retrieve any channel info */
		fprintf (stdout, "%s\n", "Unknown. Could not retrieve channel information");
		exit (2);
            }
        }

	/*
	 * And here comes the fun part.
	 */

	retval = PWR_type();
	for (n = 0; n < cfg->num_nodes; n++) {
	    if (retval == SWT_NONE) {
		snprintf(pswitch_str[n], PSWITCH_STATUS_STRLEN, "%s", "None");
	    }
	    else {
		con = msg_open(PROCID_POWERD, n);
		if (con < 0) {
			snprintf(pswitch_str[n], PSWITCH_STATUS_STRLEN, "%s", "Down");
			continue;
		} else {
			/*
			 * Construct a message to send to the power daemon.
			 */
			msghdr.command = PSWITCH_QUERY;
			msghdr.length = 0;
			if (msg_send(con, &msghdr, sizeof(msghdr)) < 0) {
				snprintf(pswitch_str[n], PSWITCH_STATUS_STRLEN, "%s", "Unknown");
				msg_close(con);
				continue;
			}
			retval = msg_receive_timeout(con, &pswitch_msg, 
					        sizeof(pswitch_msg), &auth, 2);
			if (retval != sizeof(pswitch_msg)) {
				snprintf(pswitch_str[n], PSWITCH_STATUS_STRLEN, "%s", "Unknown");
				msg_close(con);
				continue;
			}
			msg_close(con);
			if (pswitch_msg.status) {
				snprintf(pswitch_str[n], PSWITCH_STATUS_STRLEN, "%s", 
					strerror(pswitch_msg.status));
			} else {
				snprintf(pswitch_str[n], PSWITCH_STATUS_STRLEN, "%s", "Good");
			}
		}
	    }
	}

        /*
         * When looking for switch state only
         */

        if ( chan_sw_info == 2 ) {
            fprintf (stdout, "%s", pswitch_str[cfg->lid]);
            exit (0);
        } 

	retval = cluGetDiskNodeStates(&nodestates);
	if (retval < 0) {

            for (n = 0; n < cfg->num_nodes; n++) {
		if (n == myNodeNumber) {
	            nodestates.states[n] = NODE_DOWN;
		}
		else {
	            nodestates.states[n] = NODE_UNINITIALIZED;
	 	}
	    }
	    if (verbose)
		fprintf(stderr, "cluGetDiskNodeStates failed.\n");
	}

        if (refresh)
	    system("/usr/bin/clear");

        printf("Cluster Configuration (%s):\n\n", cfg->name);

	/*
	 * Put together a date and time string
	 */
	time(&clk);
	memset(timestr,0,TIMESTR_LEN);

	tm = localtime(&clk);
	if (tm) {
	    if (strftime(timestr, TIMESTR_LEN, "%a %b %d %H:%M:%S %Z %Y", tm)) {
		printf("%s\n", timestr);
	    }
	}
        printf("\nMember status:\n");
        printf("\n\t%-20s %-10s %-10s %-10s\n", "Member", "Id", "System", "Power");
        printf("\t%-20s %-10s %-10s %-10s\n", "", "", "Status", "Switch");
        printf("\t%-20s %-10s %-10s %-10s\n", "--------------------",
    	       "----------", "----------", "----------");
	nodes_in_unknown_state = 0;
        for (n = 0; n < cfg->num_nodes; n++) {
	  if (nodestates.states[n] == NODE_UNINITIALIZED) {
	    printf("\t%-20s %-10d %-10s %-10s\n", cfg->nodes[n].name, n, 
		   "Unknown", "Unknown");
	    nodes_in_unknown_state++;
	  }
	  else {
	    printf("\t%-20s %-10d %-10s %-10s\n", cfg->nodes[n].name, n,
	    	   nodeStateStrings[nodestates.states[n]], pswitch_str[n]);
	  }
        }

        printf("\nChannel status:\n");
        printf("\n\t%-30s %-10s %-10s\n", "Name", "Type", "Status");
        printf("\t%-30s %-10s %-10s\n", "------------------------------",
    	       "----------", "------------");
        for (c = 0; c < cfg->num_chans; c++) {
	    sprintf(qstr, "%s <--> %s", cfg->nodes[0].chans[c].dev.serial.name,
	    	    cfg->nodes[1].chans[c].dev.serial.name); 
	    printf("\t%-30s ", qstr);
	    if (cfg->nodes[0].chans[c].type == CHAN_NET) {
	        printf("%-10s ", "network");
	    } else {
	        if (cfg->nodes[0].chans[c].type == CHAN_SERIAL) {
	            printf("%-10s ", "serial");
	        } else {
	            printf("%-10s ", "unknown");
	        }
	    }
	    if (chan_state) {
	        if (chan_state[c].state) {
	            printf("%-20s\n", "ONLINE");
	        } else {
	            printf("%-20s\n", "OFFLINE");
	        }
	    }
	    else {
	            printf("%-20s\n", "Unknown");
	    }
        }
    
        printf("\nService status:\n");
        printf("\n\t%-20s %-10s %-20s\n", "Service", "Status", "Owner");
        printf("\t%-20s %-10s %-20s\n", "--------------------",
    	       "----------", "----------------------");
	service_count = 0;
        for (s = 0; s < MAX_SERVICES; s++) {
            if (getServiceStatus(s, &svc) < 0)
	        continue;
	    service_count++;
	    if (svc.owner == NODE_ID_NONE)
	        node = "None";
    	    else
	        node = cfg->nodes[svc.owner].name;

            /* Calling serviceExists before getSvcName avoids SIGSEGV
               when the service doesn't exist. */
	    if (serviceExists(svc.id) != YES) {
		/*
		 * Rather a kludge, but.... The service names are read in
		 * from metaconfig and cached in memory to avoid re-reading
		 * each iteration.  If we can't resolve a service name, then
		 * take that as an indication that a re-read of the config
		 * info is in order.  To do that, force an explicit close on
		 * the in-memory config which will then force a fresh read
		 * on the next access to configuration information.
		 */
        	CFG_Destroy();       
	    	if (serviceExists(svc.id) != YES)
	        	continue;
	      	continue;
	    }

	    if (getSvcName(svc.id, &name) != SUCCESS) {
	      	continue;
	    }
    
	    printf("\t%-20s %-10s %-20s\n", name,
	           serviceStateStrings[svc.state], node);
        }
	if (refresh) {
	    /* Force a re-read of the memory resident copy of the 
	     * configuration when the number of services changes.
	     */
	    if (service_count != prior_service_count) {
        	CFG_Destroy();       
	    }
	    prior_service_count = service_count;
	    sleep(refresh);
	} else {
	    break;
	}
    }
    if (cfg != NULL) {
        free(cfg);
    }
    exit(0);
}
