/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
#include <power.h>
#include <parseconf.h>
#include <stdio.h>
#include <argp.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>

static const char *version __attribute__ ((unused)) = "$Id: pswitch.c,v 1.10 2000/11/15 20:31:41 burke Exp $";

/* pswitch.c

   A shell interface for controlling the power switch. 

   author: Ron Lawrence <lawrence@missioncriticallinux.com>

*/

int cluster_member_check(void);

#define STREQ(s1,s2) (!strcmp((s1),(s2)))

/* #define PR_STAT(status, str, flag) \ */
/*    (printf( str "=%d:", ((status) & (flag)) != 0 )) */


/* static void print_status(PWR_result st) { */
/*   printf("status:"); */
/*   PR_STAT(st,"switch",PWR_SWITCH); */
/*   PR_STAT(st,"error",PWR_ERROR); */
/*   PR_STAT(st,"timeout",PWR_TIMEOUT); */
/*   PR_STAT(st,"initialized",PWR_INIT);   */
/*   printf("\n"); */
/* } */


static void print_status(PWR_result st) {

  printf("switch status:\n");

  printf("switch is: %s\n", ((( st & PWR_SWITCH ) != 0) ? "On" : "Off"));
  printf("error? %s\n", ((( st & PWR_ERROR ) != 0) ? "Yes" : "No"));
  printf("timedout? %s\n", ((( st & PWR_TIMEOUT ) != 0) ? "Yes" : "No"));
  printf("initialized? %s\n", ((( st & PWR_INIT ) != 0) ? "Yes" : "No"));
  printf("\n");
}


const char *argp_program_version = "pswitch $Id: pswitch.c,v 1.10 2000/11/15 20:31:41 burke Exp $";
     
/* Program documentation. */
static char doc[] = 
"pswitch - Operate the power switch." 
"\v"
"Pswitch must be run as root, and must not be run on a cluster member.\n"
"Recognized commands are: status, open, close, and reboot";
     
/* A description of the arguments we accept. */
static char args_doc[] = "COMMAND";

/* The options we understand. */
static struct argp_option options[] = {
  {"device",   'd', "DEVICE",   0,  "Specify a serial device", 0 },
  {"type",     't', "TYPE",     0,  "Specify switch type",     0 },
  {"silent",   's', 0,          0,  "Produce minimal output",  0 },
  {"verbose",  'v', 0,          0,  "Produce verbose output",  0 },
  {"version",  'V', 0,          0,  "Print version number",    0 },
  { 0,0,0,0,0,0 }
};

/* Used by `main' to communicate with `parse_opt'. */
struct arguments
{
  char *args[1];                /* 1 ARG */
  int verbose;
  char *device;
  char *type;
};

/* Parse a single option. */
static error_t
parse_opt (int key, char *arg, struct argp_state *state)
{
  /* Get the INPUT argument from `argp_parse', which we
     know is a pointer to our arguments structure. */
  struct arguments *arguments = state->input;
  
  switch (key)
  {
    case 'v':
      arguments->verbose = 1;
      break;
    case 's':
      arguments->verbose = 0;
      break;
    case 'V':
      fprintf(stdout, "%s\n", argp_program_version);
      exit(0);
      break;
    case 'd':
      arguments->device = arg;
      break;
    case 't':
      arguments->type = arg;
      break;
    case ARGP_KEY_ARG:
      if (state->arg_num >= 1)
        /* Too many arguments. */
        argp_usage (state);
      
      arguments->args[state->arg_num] = arg;
      
      break;
      
    case ARGP_KEY_END:
      if (state->arg_num < 1)
        /* Not enough arguments. */
        argp_usage (state);
      break;
      
    default:
      return ARGP_ERR_UNKNOWN;
  }
  return 0;
}

/* Our argp parser. */
static struct argp argp = { options, parse_opt, args_doc, doc,0,0,0 };
     
int main (int argc, char **argv)
{
  struct arguments arguments;
  int switch_type;

  PWR_result rc;

  if (geteuid() != (uid_t)0) {
    fprintf(stderr, "%s must be run as the user root\n", argv[0]);
    exit(1);    
  }
  /* Default values. */
  arguments.verbose = 1;
  arguments.device = "";
  arguments.type   = "";

  /* Parse our arguments; every option seen by `parse_opt' will
     be reflected in `arguments'. */
  argp_parse (&argp, argc, argv, 0, 0, &arguments);
     
  if(cluster_member_check() == 1) {
    fprintf(stderr, "%s cannot be run on a cluster member\n", argv[0]);
    exit(2);
  }

  switch_type = PWR_type();
  if (switch_type == SWT_NONE) {
    fprintf(stderr, "%s cannot operate on power switch type %s.\n", argv[0],
	PWR_TYPE_NONE);
    exit(2);
  }
  
/*   if(arguments.verbose) { */
/*     printf ("VERBOSE = %s\nCOMMAND = %s\n", */
/*             arguments.verbose ? "yes" : "no", */
/*             arguments.args[0]); */
/*   } */
  PWR_configure(NULL);

  if( ! STREQ(arguments.type, "")) {
    CFG_Set("power%type",arguments.type);
  }
  if( ! STREQ(arguments.device, "")) {
    CFG_Set("power%device",arguments.device);
  }
  if(PWR_init(&rc)) {
    if( !strcmp("status", arguments.args[0]) ) {
      rc = PWR_status();
      if(arguments.verbose) {
        print_status(rc);
      }
      if(rc &PWR_TIMEOUT || rc & PWR_ERROR) {
        return 1;
      }
    }
    else if( !strcmp("open", arguments.args[0]) ) {
      rc = PWR_off();
      if(arguments.verbose) {
        print_status(rc);
      }
      if(rc &PWR_TIMEOUT || rc & PWR_ERROR) {
        return 1;
      }
    }
    else if( !strcmp("close", arguments.args[0]) ) {
      rc = PWR_on();
      if(arguments.verbose) {
        print_status(rc);
      }
      if(rc &PWR_TIMEOUT || rc & PWR_ERROR) {
        return 1;
      }
    }
    else if( !strcmp("reboot", arguments.args[0]) ) {
      rc = PWR_reboot();
      if(arguments.verbose) {
        print_status(rc);
      }
      if(rc &PWR_TIMEOUT || rc & PWR_ERROR) {
        return 1;
      }
    }
    else {
      fprintf(stderr,"unknown command: %s\n", arguments.args[0]);
      return(1);
    }
    PWR_release();
    return 0;
  }
  else {
    if(arguments.verbose) {
      fprintf(stderr, "Failed to initialize\n");
    }
    print_status(rc);
    PWR_release();
    exit(1);
  }
}

