#include <string.h>
#include <sys/stat.h>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <glib.h>

#include "kpsettings.h"
#include "kputil.h"

static xmlDocPtr kp_settings_to_xml (void);
static void setting_to_xml (const gchar *setting, const gchar *value,
                            xmlNodePtr doc);

static GHashTable *settings;
static GString *config_file = NULL;
static gboolean parse_config (void);

/**
 * kp_settinges_init:
 * @filename: Name of the config file. If it's NULL, default value will be used.
 * 
 * Init the settings system. Must be called before
 * any settings can be used.
 *
 * Returns: TRUE if everything is OK and FALSE otherwise. 
 */
gboolean
kp_settings_init (const gchar *filename)
{
  gchar *conf;
  g_return_val_if_fail (config_file == NULL, FALSE);

  config_file = g_string_new (NULL);
  
  if (filename == NULL)
    conf = g_build_filename (G_DIR_SEPARATOR_S,
                             g_get_home_dir (), 
                            ".kipina",
                             G_DIR_SEPARATOR_S,
                            "config",
                             NULL);
  else
    conf = g_strdup (filename);

  g_string_assign (config_file, conf);
  
  if (!g_file_test (conf, G_FILE_TEST_EXISTS)) {
    g_free (conf);
    conf = g_strdup_printf ("%s/%s", KIPINA_DATA_DIR, "default_config.xml");
    g_string_assign (config_file, conf);
    g_free (conf);
  } else
    g_free (conf);

  /* FIXME: NULL => g_free ?? */
  settings = g_hash_table_new_full (g_str_hash,
                                    g_str_equal,
                                    g_free,
                                    NULL);
  if (!parse_config ())
    return FALSE;

  return TRUE;
}


void
kp_settings_deinit (void)
{
  g_string_free (config_file, TRUE);
  g_hash_table_destroy (settings);
}


/**
 * kp_settings_save:
 *
 * Save the settings to the disk.
 */
void
kp_settings_save (void)
{
  xmlDocPtr doc;
  gchar *file;
  gchar *dir;
  
  g_return_if_fail (settings != NULL);
  g_return_if_fail (config_file != NULL);

  file = g_build_filename (G_DIR_SEPARATOR_S,
                           g_get_home_dir (), 
                          ".kipina",
                           G_DIR_SEPARATOR_S,
                          "config",
                           NULL);
  g_string_assign (config_file, file);
  g_free (file);
  
  doc = kp_settings_to_xml ();

  dir = g_build_filename (G_DIR_SEPARATOR_S,
                          g_get_home_dir (),
                          G_DIR_SEPARATOR_S,
                          ".kipina",
                          NULL);
        
  if (!g_file_test (dir, G_FILE_TEST_IS_DIR)) 
    mkdir (dir, 0777);

  g_free (dir);
  
  if (xmlSaveFormatFile (config_file->str, doc, 1) <= 0)
    g_warning ("Settings could not be saved! (file: %s)", config_file->str);

  xmlFreeDoc (doc);
}


static xmlDocPtr
kp_settings_to_xml (void)
{
  xmlDocPtr doc;
  xmlNodePtr node;

  doc = xmlNewDoc ("1.0");
  node = xmlNewChild ((xmlNodePtr) doc, NULL, "config", NULL);
  
  g_hash_table_foreach (settings, (GHFunc) setting_to_xml, node);

  return doc;
}


static void
setting_to_xml (const gchar *setting, const gchar *value, xmlNodePtr node)
{
  xmlNodePtr child;
  gchar **str;
  guint i;
  
  str = g_strsplit (value, ";", -1);

  for (i=0; str[i] != NULL; i++) {
    if (strlen (str[i]) > 0) {
      child = xmlNewChild (node, NULL, "option", NULL);
      xmlSetProp (child, "name", setting);
      xmlSetProp (child, "value", str[i]);
    }
  }
  g_strfreev (str);
}

/**
 * parse_config:
 * 
 * Parses config-file and returns FALSE if something goes wrong
 * and TRUE otherwise. Settings will be stored into static
 * variable 'settings', which is GHashTable.
 *
 * Returns: TRUE if config was parsed successfully and
 * FALSE otherwise.
 */
static gboolean
parse_config (void)
{
  xmlDocPtr doc;
  xmlNodePtr cur;
  xmlChar *value_str;
  xmlChar *value;
  xmlChar *name;
  xmlChar *tmp;

  kp_debug ("Parsing config: %s\n", config_file->str);
  
  doc = xmlParseFile (config_file->str);
  if (doc == NULL)
    return FALSE;

  cur = xmlDocGetRootElement (doc);
  g_return_val_if_fail (cur != NULL, FALSE);
  g_return_val_if_fail (cur->name != NULL, FALSE);
  g_return_val_if_fail (xmlStrEqual (cur->name, "config"), FALSE);

  cur = cur->xmlChildrenNode;

  while (cur != NULL) {
    if (xmlStrEqual (cur->name, "option")) {

      name = xmlGetProp (cur, "name");
      value = xmlGetProp (cur, "value");

      if (value && name) {
        /* If the key is already in the table,
         * we don't want to replace it. */
        
        tmp = g_hash_table_lookup (settings, name);
        
        if (tmp != NULL && strlen (tmp) > 0) {
          g_hash_table_remove (settings, name);
          value_str = g_strdup_printf("%s;%s", tmp, value);
          g_hash_table_insert (settings, name, value_str);
          g_free (value);
        } else
          g_hash_table_insert (settings, name, value);
      }
    }
    cur = cur->next;
  }
  xmlFreeDoc (doc);

  return TRUE;
}

/**
 * kp_settings_add_to_list:
 * @setting: Name of the setting
 * @value: Value to add to the list
 *
 * Adds the @value to list of values or creates a new setting
 * if there is no such setting in the hashtable.
 */
void
kp_settings_add_to_list (const gchar *setting, const gchar *value)
{
  GString *s;
  gchar *str;
  gchar *tmp;
  
  g_return_if_fail (setting != NULL);
  g_return_if_fail (value != NULL);
  str = NULL; /* Prevent a compiler warning */
  
  /* g_hash_table_remove () will probably free the value of tmp */
  tmp = g_strdup (g_hash_table_lookup (settings, setting));
  
  if (tmp && strlen (tmp) > 0) {
    g_hash_table_remove (settings, setting);

    s = g_string_new (tmp);
    g_free (tmp);

    if (s->str[s->len-1] != ';')
      g_string_append_c (s, ';');

    g_string_append (s, value);
    g_hash_table_insert (settings, g_strdup (setting), s->str);
    g_string_free (s, FALSE);
  } else 
    g_hash_table_insert (settings, g_strdup (setting), g_strdup (value));
}

/**
 * kp_settings_remove_from_list:
 * @setting: Name of the setting
 * @value: Value to remove from the list
 *
 * Removes the @value from list of values or removes the setting from the
 * hashtable.
 */
void
kp_settings_remove_from_list (const gchar *setting, const gchar *value)
{
  GString *s;
  gchar *str;
  gchar *tmp;
  gchar *loc;
  guint len;
  
  g_return_if_fail (setting != NULL);
  g_return_if_fail (value != NULL);
  str = NULL; /* Prevent a compiler warning */

  tmp = g_hash_table_lookup (settings, setting);
  if (!tmp)
    return;
 
  if ((loc = strstr (tmp, value)) == NULL) 
    return;
    
  s = g_string_new (tmp);
  len = strlen (value);

  if (loc[len] == ';') 
    len++;
  else if (loc[len] != '\0') {
    g_string_free (s, TRUE);
    return;
  }
  
  g_string_erase (s, loc-tmp, len);
  g_hash_table_insert (settings, g_strdup (setting), s->str);

  g_string_free (s, FALSE);
}

/**
 * kp_settings_list_find:
 * @setting: The setting
 * @value: Value to find from the list
 *
 * Searches the list of settings found by the key named
 * @setting for @value.
 * 
 * Returns: TRUE if found and FALSE otherwise.
 */
gboolean
kp_settings_list_find (const gchar *setting, const gchar *value)
{
  gchar *tmp;
  gchar *loc;
  guint len;

  g_return_val_if_fail (setting != NULL, FALSE);
  g_return_val_if_fail (value != NULL, FALSE);

  tmp = g_hash_table_lookup (settings, setting);

  loc = strstr (tmp, value);
  if (!loc)
    return FALSE;

  len = strlen (value);

  if (tmp[loc - tmp + len] == ';' || tmp[loc - tmp + len] == '\0')
    return TRUE;

  return FALSE;
}


/**
 * kp_settings_get_str:
 * @setting: The name of the setting
 *
 * Get the value of the setting as string.
 *
 * Returns: String that must not be freed, it is not copied.
 */
gchar *
kp_settings_get_str (const gchar *setting)
{
  g_return_val_if_fail (settings != NULL, NULL);
  g_return_val_if_fail (setting != NULL, NULL);

  return g_hash_table_lookup (settings, setting);
}

/**
 * kp_settings_set_bool:
 * @setting: The name of the setting
 * @value: TRUE or FALSE
 *
 * Just set the value of some setting to TRUE or FALSE.
 */
void
kp_settings_set_bool (const gchar *setting, gboolean value)
{
  g_return_if_fail (setting != NULL);

  if (value)
    kp_settings_set_str (setting, "true", TRUE);
  else
    kp_settings_set_str (setting, "false", TRUE);
}

/**
 * kp_settings_set_int:
 * @setting: The name of the setting
 * @value: Any integer value
 *
 * Set the value.
 */
void
kp_settings_set_int (const gchar *setting, gint value)
{
  gchar *val;
  
  g_return_if_fail (setting != NULL);
  
  val = g_strdup_printf ("%d", value);
  kp_settings_set_str (setting, val, TRUE);
}


/**
 * kp_settings_set_str:
 * @setting: The name of the setting
 * @val: Value of the setting, can be almost anything
 * @replace: if TRUE and the setting is set, it will be replaced and otherwise not.
 * 
 * Sets @setting to be val. If @replace is TRUE and setting
 * with name @setting already exists, that setting will be
 * replaced with the new one. If @replace is FALSE and there
 * is already a setting with that name, they will go both to
 * the hashtable and key will be like: "setting" and value
 * will be like: "value1;value2;value3". 
 */
void
kp_settings_set_str (const gchar *setting, const gchar *val,
                     gboolean replace)
{
  GString *string;
  gchar *tmp;

  g_return_if_fail (settings != NULL);
  g_return_if_fail (setting != NULL);
  g_return_if_fail (val != NULL);

  kp_debug ("Setting '%s' to value '%s'", setting, val);

  tmp = g_strdup (g_hash_table_lookup (settings, setting));
  if (!tmp || replace) {
    /* If key doesn't exist already */
    g_hash_table_insert (settings, g_strdup (setting), g_strdup (val));
    return;
  }
  string = g_string_new (tmp);
  g_free (tmp);
 
  /* Append new stuff to an old value */
  g_string_append_c (string, ';');
  g_string_append (string, val);
   
  /* Insert new value to the table */
  g_hash_table_insert (settings, g_strdup (setting), string->str);
  g_string_free (string, FALSE);
}


void
kp_settings_remove_setting (const gchar *setting)
{
  g_return_if_fail (settings != NULL);
  g_return_if_fail (setting != NULL);

  if (g_hash_table_lookup (settings, setting))
    g_hash_table_remove (settings, setting);
}


gint
kp_settings_get_int (const gchar *setting)
{
  gchar *val = kp_settings_get_str (setting);

  if (!val)
    return -1;

  return (int) strtod (val, NULL);
}


GSList *
kp_settings_get_list (const gchar *setting)
{
  GSList *list;
  gchar **str;
  gchar *val;
  guint i;

  g_return_val_if_fail (setting != NULL, NULL);
  
  val = kp_settings_get_str (setting);

  if (val == NULL)
    return NULL;
  
  str = g_strsplit (val, ";", -1);

  for (i=0, list=NULL; str[i] != NULL; i++)
    list = g_slist_prepend (list, g_strdup (str[i]));
  
  g_strfreev (str);
  
  return list;
}


void
kp_settings_list_free (GSList *list)
{
  GSList *node;

  node = list;
  while (node) {
    g_free (node->data);

    node = node->next;
  }
  g_slist_free (list);
}

/**
 * kp_settings_get_bool:
 * @setting: The name of the setting
 *
 * Get the value of this setting as boolean.
 * 
 * Returns: TRUE if setting's value is true or if there is no setting @setting.
 */
gboolean
kp_settings_get_bool (const gchar *setting)
{
  gchar *val = kp_settings_get_str (setting);

  if (!val)
    return FALSE;

  if (strcmp (val, "true") == 0)
    return TRUE;

  return FALSE;
}

