#include <string.h>
#include <glib.h>

#include "kpmainwindow.h"
#include "kpguiutils.h"
#include "kpentries.h"
#include "kpentrypropertiesdialog.h"
#include "kpresultsview.h"
#include "kpsplitsview.h"
#include "kpviewpopupmodel.h"
#include "kpworkouteditor.h"
#include "kpnewcommentdialog.h"
#include "kpresultseditor.h"
#include "kpcontexts.h"
#include "kpnewsplitworkoutdialog.h"


#include "../kpworkoutmodel.h"
#include "../kpsplitworkout.h"
#include "../kpcalendarentry.h"
#include "../kipina-i18n.h"
#include "../kputil.h"
#include "../kpcomment.h"

GSList *types_list = NULL;

/* List of KPEntryData * items */
GSList *entries_list = NULL;

  
void
kp_entries_init (void)
{
  kp_entries_add_entry (KP_TYPE_SPLIT_WORKOUT, 
                     _("Split Workout"), 
                       "spworkout.png",
       (KPEntryNewFunc) kp_new_split_workout_dialog_new,
      (KPEntryPropFunc) kp_entries_show_prop_split_workout_dialog);

  kp_entries_add_entry (kp_workout_get_type (),
                     _("Workout"),
                       "runner.xpm",
       (KPEntryNewFunc) kp_workout_editor_new,
      (KPEntryPropFunc) kp_entries_show_prop_workout_dialog);

  kp_entries_add_entry (kp_comment_get_type (),
                     _("Comment"),
                       "comment.png",
       (KPEntryNewFunc) kp_new_comment_dialog_new,
      (KPEntryPropFunc) kp_entries_show_prop_comment_dialog);

  kp_entries_add_entry (kp_results_get_type (),
                     _("Results"),
                       "runner.xpm",
                        kp_results_editor_new,
      (KPEntryPropFunc) kp_entries_show_prop_results_dialog);
}


KPEntryData *
kp_entries_get_entry (const gchar *human_name)
{
  GSList *node;
  KPEntryData *data;

  for (node = entries_list; node; node = node->next) {
    data = KP_ENTRY_DATA (node->data);

    if (strcmp (human_name, data->name) == 0)
      return data;
  }
  return NULL;
}


void
kp_entries_add_entry (GType type, const gchar *human_name, const gchar *icon_name,
                      KPEntryNewFunc func, KPEntryPropFunc prop_func)
{
  KPEntryData *data;

  data = g_new0 (KPEntryData, 1);
  data->type = type;
  data->name = g_strdup (_(human_name));
  data->icon_name = g_strdup (icon_name);
  data->new_dialog = func;
  data->prop_dialog = prop_func; 
  
  entries_list = g_slist_prepend (entries_list, data);
}


GSList *
kp_entries_get_list_as_menu_items (GCallback cb, gpointer data)
{
  GtkWidget *image;
  GtkWidget *item;
  GSList *node;
  GSList *list = NULL;
  KPEntryData *d;
  
  for (node = entries_list; node; node = node->next) {
    d = KP_ENTRY_DATA (node->data);
    item = gtk_image_menu_item_new_with_label (d->name);
    
    if (d->icon_name) {
      image = GTK_WIDGET (kp_get_icon_as_image (d->icon_name));
      gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), image);
    } 
    
    g_signal_connect (G_OBJECT (item), "activate", cb, data);
    gtk_widget_show (item);
    
    list = g_slist_prepend (list, item);
  }

  return list;
}


static void
popup_prop_activate (GtkMenuItem *item, KPViewPopupModel *model)
{
  KPCalendarEntry *entry;

  if ((entry = kp_view_popup_model_get_active_entry (model)))
    kp_entries_show_prop_dialog (entry);
}


static void
popup_delete_activate (GtkMenuItem *item, KPViewPopupModel *model)
{
  KPCalendarEntry *entry;
  KPTrainingLog *log;
  
  entry = kp_view_popup_model_get_active_entry (model);
  g_return_if_fail (KP_IS_CALENDAR_ENTRY (entry));

  log = kp_main_window_get_log ();
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
 
  kp_training_log_remove (log, entry);
}

  
static void
popup_edit_activate (GtkMenuItem *item, KPViewPopupModel *model)
{
  KPCalendarEntry *entry;
  GtkWidget *dialog;
  KPEntryData *data;
  KPDate date;
  GSList *node;
  
  g_return_if_fail (KP_IS_VIEW_MODEL (model));
  kp_view_model_get_dmy (KP_VIEW_MODEL (model), &date.d, &date.m, &date.y);
  
  entry = kp_view_popup_model_get_active_entry (model);
  if (!entry)
    return;
  
  for (data = NULL, node = entries_list; node; node = node->next)
    if (KP_ENTRY_DATA (node->data)->type == G_OBJECT_TYPE (entry)) {
      data = KP_ENTRY_DATA (node->data);
      break;
    }
  if (!data || !data)
    return;
  
  dialog = data->new_dialog (&date, entry);
  gtk_widget_show (dialog);
}

  
static void
new_activate (GtkMenuItem *item, KPViewPopupModel *model)
{
  GtkWidget *dialog;
  GtkWidget *label;
  KPEntryData *data;
  KPDate date;

  g_return_if_fail (KP_IS_VIEW_MODEL (model));
  kp_view_model_get_dmy (KP_VIEW_MODEL (model), &date.d, &date.m, &date.y);

  label = gtk_bin_get_child (GTK_BIN (item));
  data = kp_entries_get_entry (gtk_label_get_text (GTK_LABEL (label)));
  g_return_if_fail (data != NULL);
  
  dialog = data->new_dialog (&date, NULL);
  gtk_widget_show (dialog);
}


KPEntryPopup *
kp_entries_get_popup (KPViewPopupModel *model)
{
  KPEntryPopup *data;
  GtkWidget *item;
  GSList *node;
  
  data = g_new0 (KPEntryPopup, 1);
  data->menu = gtk_menu_new ();
  gtk_widget_show (data->menu);

  item = gtk_menu_item_new_with_label ("No Date.");
  data->title_label = gtk_bin_get_child (GTK_BIN (item));
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), item);
 
  item = gtk_separator_menu_item_new ();
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), item);

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_ADD, NULL);
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), item);
  
  data->add_menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), data->add_menu);

  node = kp_entries_get_list_as_menu_items (G_CALLBACK (new_activate), model);
  for (; node; node = node->next) {
    gtk_menu_shell_append (GTK_MENU_SHELL (data->add_menu), GTK_WIDGET (node->data));
  }
  
#if GTK_MINOR_VERSION >= 6
  data->edit = gtk_image_menu_item_new_from_stock  (GTK_STOCK_EDIT, NULL);
#else
  data->edit = gtk_menu_item_new_with_label (_("Edit"));
#endif
  gtk_widget_show (data->edit);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), data->edit);

  item = gtk_separator_menu_item_new ();
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), item);

  data->delete = gtk_image_menu_item_new_from_stock (GTK_STOCK_DELETE, NULL);
  gtk_widget_show (data->delete);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), data->delete);
 
  item = gtk_separator_menu_item_new ();
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), item);

  item = gtk_menu_item_new_with_label ("Actions");
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), item);

  data->actions_menu = gtk_menu_new ();
  gtk_widget_show (data->actions_menu);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), data->actions_menu);
 
  item = gtk_separator_menu_item_new ();
  gtk_widget_show (item);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), item);

  data->prop = gtk_image_menu_item_new_from_stock (GTK_STOCK_PROPERTIES, NULL);
  gtk_widget_show (data->prop);
  gtk_menu_shell_append (GTK_MENU_SHELL (data->menu), data->prop);
 
  g_signal_connect (G_OBJECT (data->prop), "activate",
                    G_CALLBACK (popup_prop_activate), model);
  g_signal_connect (G_OBJECT (data->edit), "activate",
                    G_CALLBACK (popup_edit_activate), model);
  g_signal_connect (G_OBJECT (data->delete), "activate",
                    G_CALLBACK (popup_delete_activate), model);
  
  return data;
}


/**
 * kp_entries_popup_prepare_dynamic:
 * @data: A #KPEntryPopup
 * @object_id: Type id of the object that is active or 0
 *
 * Do all the things we need to do every time before showing the menu.
 *  - Clean actions menu
 *  - Build new actions menu
 *  - Set the date
 *  - Set items active/inactive
 */
void
kp_entries_popup_prepare_dynamic (KPEntryPopup *data, KPViewPopupModel *model)
{ 
  KPCalendarEntry *entry;
  guint flags = 0;
  GList *list, *old;
  GSList *slist, *sold;
  GType object_id;
  gchar date_str[32];
  guint d, m, y;

  entry = kp_view_popup_model_get_active_entry (model);
  kp_entries_set_popup_mode_entry (data, KP_IS_CALENDAR_ENTRY (entry));
 
  /* Set the date */
  kp_view_model_get_dmy (KP_VIEW_MODEL (model), &d, &m, &y);
  if (kp_date_dmy_to_locale_string (d, m, y, date_str, sizeof (date_str)))
    gtk_label_set_text (GTK_LABEL (data->title_label), date_str);
  
  if (entry == NULL)
    return;

  /* Context type */
  object_id = G_OBJECT_TYPE (G_OBJECT (entry));
 
  if (object_id == KP_TYPE_WORKOUT) 
    flags |= KP_CONTEXT_WORKOUT_FLAG;
  if (object_id == KP_TYPE_SPLIT_WORKOUT)
    flags |= KP_CONTEXT_SPLIT_WORKOUT_FLAG;
  if (object_id == KP_TYPE_COMMENT)
    flags |= KP_CONTEXT_COMMENT_FLAG;
  if (object_id == KP_TYPE_RESULTS)
    flags |= KP_CONTEXT_RESULTS_FLAG;
  
  /* Remove old entries */
  list = gtk_container_get_children (GTK_CONTAINER (data->actions_menu));
  while (list) {
    if (GTK_IS_MENU_ITEM (list->data))
      gtk_container_remove (GTK_CONTAINER (data->actions_menu), GTK_WIDGET (list->data));
  
    old = list;
    list = list->next;
    g_list_free_1 (old);
  }

  /* Add new entries */
  slist = kp_context_get_menu_items (flags, entry);
  for (; slist;) {
    gtk_menu_shell_append (GTK_MENU_SHELL (data->actions_menu), 
                           GTK_WIDGET (slist->data));
    sold = slist;
    slist = slist->next;
    g_slist_free_1 (sold);
  }
  
  gtk_widget_show_all (data->actions_menu);
}



void
kp_entries_set_popup_mode_entry (KPEntryPopup *data, gboolean there_is_entry)
{
  gtk_widget_set_sensitive (data->delete, there_is_entry);  
  gtk_widget_set_sensitive (data->edit, there_is_entry);  
  gtk_widget_set_sensitive (data->prop, there_is_entry);  
}


void
kp_entries_show_edit_dialog (KPCalendarEntry *entry)
{
  GtkWidget *dialog;
  KPDate date;
  GSList *node;
  GType type;

  type = G_OBJECT_TYPE (G_OBJECT (entry));
  
  for (node = entries_list; node; node = node->next) 
    if (KP_ENTRY_DATA (node->data)->type == type) {
      kp_calendar_entry_get_date (KP_CALENDAR_ENTRY (entry), &date);
      
      dialog = KP_ENTRY_DATA (node->data)->new_dialog (&date, entry);
      gtk_widget_show (GTK_WIDGET (dialog));
      
      return;
    }
  
  g_warning ("Edit func not implemented for type %s!", G_OBJECT_TYPE_NAME (entry));
}

void
kp_entries_show_prop_dialog (KPCalendarEntry *entry)
{
  GSList *node;
  GType type;

  type = G_OBJECT_TYPE (G_OBJECT (entry));
  
  for (node = entries_list; node; node = node->next) 
    if (KP_ENTRY_DATA (node->data)->type == type) {
      KP_ENTRY_DATA (node->data)->prop_dialog (entry);
      return;
    }
  
  g_warning ("Properties func not implemented for type %s!", G_OBJECT_TYPE_NAME (entry));
}


void
kp_entries_show_prop_workout_dialog (KPCalendarEntry *entry)
{
  KPParamList *list;
  GtkWidget *dialog;
  gchar *comment;
  gchar *duration;
  gchar *distance;
  const gchar *sport;
  guint pace;
  gchar *pace_str;
  gchar *speed_str;
  
  dialog = kp_entry_properties_dialog_new (entry);
  list = KP_WORKOUT (entry)->param_list;
  
  distance = g_strdup_printf ("%.3f km", kp_param_list_get_double (list, "distance"));
  duration = kp_param_list_get_as_string (list, "duration");
  comment = kp_param_list_get_as_string (list, "comment");
  sport = kp_workout_get_sport (KP_WORKOUT (entry));
  
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), 
                                      "Sport:", gtk_label_new (sport), FALSE);
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), 
                                      "Distance:", gtk_label_new (distance), 
                                       FALSE);
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), 
                                      "Duration:", gtk_label_new (duration),
                                       FALSE);
  if (distance && duration) {
    /* Pace */
    pace = kp_workout_model_get_pace (KP_WORKOUT_MODEL (entry));
    pace_str = kp_date_mseconds_to_std_string (pace);
    /* Speed */
    speed_str = g_strdup_printf ("%.1f km/h", 
                                  kp_workout_model_get_speed (KP_WORKOUT_MODEL (entry)));
    
    kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog),
                                        "Pace per km:", gtk_label_new (pace_str),
                                         FALSE);
    kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog),
                                        "Speed:", gtk_label_new (speed_str),
                                         FALSE);
     
    g_free (pace_str); 
  
  }
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), 
                                      "Comment:", gtk_label_new (comment),
                                       FALSE);
  
  g_free (duration);
  g_free (distance);
  g_free (comment);
  
  gtk_widget_show (dialog);
}


void
kp_entries_show_prop_split_workout_dialog (KPCalendarEntry *entry)
{
  GtkWidget *splits_view;
  GtkWidget *dialog;
  GtkWidget *swin;
  gchar *comment;
  gchar *duration;
  gchar *du_eff, *du_all;
  gchar *distance;
  const gchar *sport;
  gchar *pace_str;
  gchar *speed;
  guint pace1, pace2;
  gchar *str1, *str2;
  guint eduration;
  gdouble edistance;
  
  dialog = kp_entry_properties_dialog_new (entry);

  /* Distance & Duration */
  kp_split_workout_get_effective_values (KP_SPLIT_WORKOUT (entry), &eduration,
                                        &edistance);
 
  du_all = kp_date_mseconds_to_std_string (
              kp_workout_model_get_duration (KP_WORKOUT_MODEL (entry)));
  du_eff = kp_date_mseconds_to_std_string (eduration);
 
  duration = g_strdup_printf ("%s   (%s)", du_eff, du_all);
 
  g_free (du_all);
  g_free (du_eff);

  distance = g_strdup_printf ("%.3f km   (%.3f km)",
              edistance,
              kp_workout_model_get_distance (KP_WORKOUT_MODEL (entry)));
  
  /* Comment */
  comment = kp_param_list_get_as_string (KP_WORKOUT (entry)->param_list,
                                         "comment");
  /* Sport */
  sport = kp_workout_get_sport (KP_WORKOUT (entry));
  
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog),
                                      "Sport:", 
                                       gtk_label_new (sport),
                                       FALSE);
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), 
                                      "Distance:", 
                                       gtk_label_new (distance), 
                                       FALSE);
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), 
                                      "Duration:", 
                                       gtk_label_new (duration),
                                       FALSE);
  if (distance && duration) {
    /* Pace */
    pace1 = kp_workout_model_get_pace (KP_WORKOUT_MODEL (entry));
    pace2 = kp_split_workout_get_whole_pace (KP_SPLIT_WORKOUT (entry));
    
    str1 = kp_date_mseconds_to_std_string (pace1);
    str2 = kp_date_mseconds_to_std_string (pace2);

    pace_str = g_strdup_printf ("%s   (%s)", str1, str2);
    
    /* Speed */
    speed = g_strdup_printf ("%.1f km/h   (%.1f km/h)", 
                kp_workout_model_get_speed (KP_WORKOUT_MODEL (entry)),
                kp_split_workout_get_whole_speed (KP_SPLIT_WORKOUT (entry)));

    kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog),
                                        "Pace per km:", 
                                         gtk_label_new (pace_str),
                                         FALSE);
    kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog),
                                        "Speed:", 
                                         gtk_label_new (speed),
                                         FALSE);
    g_free (str1);
    g_free (str2);
  
    g_free (speed);
    g_free (pace_str);
  }
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), 
                                      "Comment:", gtk_label_new (comment),
                                       FALSE);
  splits_view = kp_splits_view_new (KP_SPLIT_WORKOUT (entry)->splits);
  gtk_widget_show (splits_view);
  
  swin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (swin),
                                         splits_view);
 
  
  
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog),
                                      "Splits:", swin, TRUE);
 
  g_free (duration);
  g_free (distance);
  g_free (comment);
  
  gtk_widget_show (dialog);
}


void
kp_entries_show_prop_comment_dialog (KPCalendarEntry *entry)
{
  GtkWidget *dialog;
  gchar *comment;
  gchar *title;

  dialog = kp_entry_properties_dialog_new (entry);

  /* Title */
  title = kp_comment_get_title (KP_COMMENT (entry));
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog),
                                      "Title:", gtk_label_new (title), FALSE);
  /* Comment */
  comment = kp_comment_get_text (KP_COMMENT (entry));
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), 
                                      "Comment:", gtk_label_new (comment), FALSE);
  g_free (comment);
  g_free (title);
  
  gtk_widget_show (dialog);
}


void
kp_entries_show_prop_results_dialog (KPCalendarEntry *entry)
{
  GtkWidget *dialog;
  GtkWidget *view;
  GtkWidget *swin;
  const gchar *location;
  const gchar *title;

  dialog = kp_entry_properties_dialog_new (entry);

  /* Title */
  title = kp_results_get_title (KP_RESULTS (entry));
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), "Title:", 
                                       gtk_label_new (title), FALSE);

  /* Location */
  location = kp_results_get_location (KP_RESULTS (entry));
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), "Location:", 
                                       gtk_label_new (location), FALSE);
  
  /* Results View */
  view = kp_results_view_new (KP_RESULTS (entry));
  gtk_widget_show (view);
  
  swin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (swin),
                                         view);
  
  kp_entry_properties_dialog_add_pair (KP_ENTRY_PROPERTIES_DIALOG (dialog), 
                                      "List:", swin, TRUE);

  gtk_widget_show (dialog);
}

