#include <gtk/gtk.h>
#include <glade/glade.h>
#include <string.h>
#include <time.h>

#include "kpintensityentry.h"
#include "kpmainwindow.h"
#include "kpworkouteditor.h"
#include "kpguiutils.h"
#include "kpdateentry.h"

#include "../kptraininglog.h"
#include "../kppresetdata.h"
#include "../kipina-i18n.h"
#include "../kpparamlist.h"
#include "../kpsettings.h"
#include "../kputil.h"
#include "../kpworkout.h"

#define SPORT_MENU_TEXT    "Choose a Sport"
#define SPORT_MENU_OTHER   "Other"

typedef enum {
  KP_WORKOUT_EDITOR_MODE_EDIT,
  KP_WORKOUT_EDITOR_MODE_NEW    
} KPWorkoutEditorMode;

static void     kp_workout_editor_class_init     (KPWorkoutEditorClass *klass);
static void     kp_workout_editor_init           (KPWorkoutEditor *dialog);
static void     kp_workout_editor_finalize       (GObject *object);
static void     fill_data                        (KPWorkoutEditor *editor);
static gchar   *get_field_sport                  (KPWorkoutEditor *dialog);
static void     set_detail_tree_view             (KPWorkoutEditor *dialog);
static void     build_sport_menu                  (KPWorkoutEditor *dialog);
static void     build_detail_menu                (KPWorkoutEditor *dialog);
static void     check_values                     (GtkButton *button,
                                                  KPWorkoutEditor *dialog);
/* Callbacks */
static void     on_sport_entry_changed           (GtkEntry *entry,
                                                  gpointer data);
static void     on_detail_add_button_clicked     (GtkButton *button,
                                                  KPWorkoutEditor *editor);
static void     on_detail_delete_button_clicked  (GtkButton *button,
                                                  gpointer data);
static void     on_sport_menu_activate           (GtkComboBox *box,
                                                  gpointer data);
static void     detail_list_row_selected         (GtkTreeSelection *selection,
                                                  gpointer data);
static void     kp_workout_editor_response_cb    (GtkDialog *dialog,
                                                  int response_id,
                                                  KPWorkoutEditor *editor);
static void     comment_changed_cb               (GtkTextBuffer *buffer,
                                                  KPWorkoutEditor *editor);
static void     duration_changed_cb              (GtkSpinButton *button, 
                                                  KPWorkoutEditor *editor);
static void     distance_changed_cb              (GtkSpinButton *button, 
                                                  KPWorkoutEditor *editor);
static void     date_selected_cb                 (KPDateEntry *entry,
                                                  KPDate *date,
                                                  KPWorkoutEditor *editor);
static void     time_selected_cb                 (KPDateEntry *entry,
                                                  KPTime *time,
                                                  KPWorkoutEditor *editor);
static void     intensity_entry_changed_cb       (KPIntensityEntry *entry,
                                                  KPWorkoutEditor *editor);
static void     intensity_entry_user_text_changed(GtkEntry *entry, 
                                                  KPWorkoutEditor *editor);

static GuiModuleSignalsData signals_data[] = {
{"detail_add_button"  ,   "clicked", CB (on_detail_add_button_clicked),   NULL},
{"detail_delete_button",  "clicked", CB (on_detail_delete_button_clicked),NULL},
{ NULL,                    NULL,     NULL,                                NULL},
};


typedef struct KPWorkoutEditorPrivateData_
{
  KPTrainingLog *log;
  KPWorkout *wo_backup;
  KPWorkout *wo;
  KPDateEntry *date_entry;

  GtkWidget *details_view;
  GtkWidget *detail_add_button;
  GtkWidget *detail_delete_button;
  GtkWidget *detail_preset_menu;
  GtkWidget *detail_name;
  GtkWidget *detail_value;

  GtkWidget *intensity_entry;
  GtkWidget *sport_entry;
  GtkWidget *okbutton;
  GtkWidget *cancelbutton;
  GtkWidget *distance_entry;
  GtkWidget *sport_menu;
  GtkWidget *comment;
  GtkWidget *calendar;

  GtkSpinButton *duration_h;
  GtkSpinButton *duration_m;
  GtkSpinButton *duration_s;
  GtkSpinButton *duration_t;
  
  KPWorkoutEditorMode mode;
} KPWorkoutEditorPrivateData;

#define KP_WORKOUT_EDITOR_PRIVATE_DATA(widget) \
  (((KPWorkoutEditorPrivateData*) \
  (KP_WORKOUT_EDITOR (widget)->private_data)))

static GObjectClass *parent_class = NULL;

GType
kp_workout_editor_get_type (void)
{
  static GType kp_workout_editor_type = 0;

  if (kp_workout_editor_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPWorkoutEditorClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_workout_editor_class_init,
      NULL,
      NULL,
      sizeof (KPWorkoutEditor),
      0,
      (GInstanceInitFunc) kp_workout_editor_init,
      NULL,
    };

    kp_workout_editor_type = g_type_register_static (GTK_TYPE_DIALOG,
                                                    "KPWorkoutEditor",
                                                    &our_info, 0);
  }
  return kp_workout_editor_type;
}


static void
kp_workout_editor_class_init (KPWorkoutEditorClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_workout_editor_finalize;
}


static void
kp_workout_editor_init (KPWorkoutEditor *dialog)
{
  KPWorkoutEditorPrivateData *p_data;
  GtkTextBuffer *buffer;
  GladeXML *xml;

  xml = kp_gui_load ("workout_editor", "workout_editor");
  
  dialog->private_data = g_new0 (KPWorkoutEditorPrivateData, 1);
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (dialog);

  gtk_window_set_modal (GTK_WINDOW (dialog), FALSE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Workout Editor"));
  
  g_signal_connect (G_OBJECT (dialog), "response",
                    G_CALLBACK (kp_workout_editor_response_cb), dialog);

  g_signal_connect_swapped (G_OBJECT (dialog), "response",
                            G_CALLBACK (gtk_widget_destroy), dialog);
  
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
                     KP_W (xml, "workout_editor"));

  p_data->wo_backup = NULL;
  p_data->wo = kp_workout_new ();
  
  /* Put widgets to private data */
  p_data->detail_delete_button = KP_W (xml, "detail_delete_button");  
  p_data->detail_preset_menu = KP_W (xml, "detail_preset_menu");
  p_data->detail_add_button = KP_W (xml, "detail_add_button");  
  p_data->details_view = KP_W (xml, "details_treeview");
  p_data->detail_value = KP_W (xml, "detail_value_entry");

  p_data->duration_h = GTK_SPIN_BUTTON (KP_W (xml, "duration_h"));
  p_data->duration_m = GTK_SPIN_BUTTON (KP_W (xml, "duration_m"));
  p_data->duration_s = GTK_SPIN_BUTTON (KP_W (xml, "duration_s"));
  p_data->duration_t = GTK_SPIN_BUTTON (KP_W (xml, "duration_t"));
  
  p_data->distance_entry = KP_W (xml, "distance_entry");
  p_data->sport_menu = KP_W (xml, "sport_menu");
  p_data->comment = KP_W (xml, "comment");
  p_data->sport_entry = KP_W (xml, "sport_entry");
  
  g_signal_connect (G_OBJECT (p_data->duration_h), "value-changed",
                    G_CALLBACK (duration_changed_cb), dialog);
  g_signal_connect (G_OBJECT (p_data->duration_m), "value-changed",
                    G_CALLBACK (duration_changed_cb), dialog);
  g_signal_connect (G_OBJECT (p_data->duration_s), "value-changed",
                    G_CALLBACK (duration_changed_cb), dialog);
  g_signal_connect (G_OBJECT (p_data->duration_t), "value-changed",
                    G_CALLBACK (duration_changed_cb), dialog);
  g_signal_connect (G_OBJECT (p_data->distance_entry), "value-changed",
                    G_CALLBACK (distance_changed_cb), dialog);

  /* Date Entry */
  p_data->date_entry = KP_DATE_ENTRY (kp_date_entry_new ());
  gtk_box_pack_start (GTK_BOX (KP_W (xml, "datebox")), 
                      GTK_WIDGET (p_data->date_entry), TRUE, TRUE, 0);
  gtk_widget_show (GTK_WIDGET (p_data->date_entry));  

  g_signal_connect (G_OBJECT (p_data->date_entry), "date-selected",
                    G_CALLBACK (date_selected_cb), dialog);
  g_signal_connect (G_OBJECT (p_data->date_entry), "time-selected",
                    G_CALLBACK (time_selected_cb), dialog);

  /* Intensity Entry */
  p_data->intensity_entry = kp_intensity_entry_new ();
  gtk_box_pack_start (GTK_BOX (KP_W (xml, "intensitybox")),
                      p_data->intensity_entry, TRUE, TRUE, 0);
  gtk_widget_show (GTK_WIDGET (p_data->intensity_entry));
  g_signal_connect (G_OBJECT (p_data->intensity_entry), "changed",
                    G_CALLBACK (intensity_entry_changed_cb), dialog);
  g_signal_connect (G_OBJECT (GTK_BIN (p_data->intensity_entry)->child),
        "changed", G_CALLBACK (intensity_entry_user_text_changed), dialog);
  
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (p_data->comment));
  g_signal_connect (G_OBJECT (buffer), "end-user-action",
                    G_CALLBACK (comment_changed_cb),
                    KP_WORKOUT_EDITOR (dialog));
 
  
  build_sport_menu (dialog);
  build_detail_menu (dialog);

  kp_gui_module_signals_connect_data (xml, signals_data, dialog);
  g_object_unref (G_OBJECT (xml));
}


static void
kp_workout_editor_finalize (GObject *object)
{
  KPWorkoutEditor *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_WORKOUT_EDITOR (object));

  dialog = KP_WORKOUT_EDITOR (object);

  g_return_if_fail (dialog->private_data != NULL);
  g_free (dialog->private_data);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


/**
 * kp_workout_editor_new:
 * @date: A #KPDate
 * @wo: A #KPWorkout
 *
 * Create an instance of #KPWorkoutEditor.
 *
 * Returns: A #KPWorkoutEditor
 */
GtkWidget *
kp_workout_editor_new (KPDate *date, KPWorkout *wo)
{
  KPWorkoutEditorPrivateData *p_data;
  GtkWidget *dialog;

  dialog = g_object_new (KP_TYPE_WORKOUT_EDITOR, NULL);
  
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (dialog);
  p_data->log = kp_main_window_get_log ();
 
  g_return_val_if_fail (KP_IS_TRAINING_LOG (p_data->log), NULL);
  
  set_detail_tree_view (KP_WORKOUT_EDITOR (dialog));

  if (KP_IS_WORKOUT (wo)) {
    g_object_unref (p_data->wo);
    p_data->wo_backup = kp_workout_copy (wo);
    p_data->wo = wo;

    fill_data (KP_WORKOUT_EDITOR (dialog));
    p_data->mode = KP_WORKOUT_EDITOR_MODE_EDIT;
  } else {
    p_data->mode = KP_WORKOUT_EDITOR_MODE_NEW;

    KP_CALENDAR_ENTRY (p_data->wo)->datetime = kp_calendar_time_new ();
    kp_calendar_time_set_dmy (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                              date->d, date->m, date->y);
  }

  kp_date_entry_set_date (p_data->date_entry, date);
  
  p_data->cancelbutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                                GTK_STOCK_CANCEL,
                                                GTK_RESPONSE_CANCEL);

  if (p_data->mode == KP_WORKOUT_EDITOR_MODE_EDIT) {
    p_data->okbutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                              GTK_STOCK_CLOSE,
                                              GTK_RESPONSE_OK);
  }  else {
    p_data->okbutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                              GTK_STOCK_ADD,
                                              GTK_RESPONSE_OK);
  }
  gtk_widget_set_sensitive (GTK_WIDGET (p_data->okbutton), 
                           (p_data->mode == KP_WORKOUT_EDITOR_MODE_EDIT));

  
  return dialog;
}


static void
kp_workout_editor_response_cb (GtkDialog *dialog, int response_id,
                               KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  KPPresetDataItem *item;
  gchar *sport;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);
 
  if (response_id != GTK_RESPONSE_CANCEL
   && response_id != GTK_RESPONSE_OK) {
    return;
  }
  
  if (p_data->mode == KP_WORKOUT_EDITOR_MODE_EDIT) {
    if (response_id == GTK_RESPONSE_CANCEL) {
      kp_debug ("Removing entry with id: %u\n", 
                 kp_calendar_entry_get_id (KP_CALENDAR_ENTRY (p_data->wo)));
      kp_training_log_remove (p_data->log,
                              KP_CALENDAR_ENTRY (p_data->wo));

      kp_debug ("Adding entry with id: %u\n",
                 kp_calendar_entry_get_id (KP_CALENDAR_ENTRY (p_data->wo_backup)));
      
      kp_training_log_add (p_data->log,
                           KP_CALENDAR_ENTRY (p_data->wo_backup));
    }
    return;
  }

  if (response_id == GTK_RESPONSE_CANCEL)
    return;
  
  g_return_if_fail (kp_workout_is_valid (p_data->wo));

  sport = get_field_sport (editor);
  
  /* Add sport item */
  item = kp_preset_data_item_new ();
  item->name = sport;
  kp_preset_data_add_item (KP_PRESET_DATA_SPORT, item);
 
  /* Add intensity item */
  item = kp_preset_data_item_new ();
  item->name = kp_param_list_get_as_string (p_data->wo->param_list, 
                                           "intensity");
  kp_preset_data_add_item (KP_PRESET_DATA_EXERCISE_TYPE, item);
  
  /* Add entry to the log */
  kp_training_log_add (p_data->log, KP_CALENDAR_ENTRY (p_data->wo));
}

/*
 * button is not used, so it can be NULL and this can be called
 * from other functions and not only as callback for buttons.
 */
static void
check_values (GtkButton *button, KPWorkoutEditor *dialog)
{
  KPWorkoutEditorPrivateData *p_data;
  gboolean valid = TRUE;
  gchar *sport;
  
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (dialog);
  
  kp_debug ("Checking values of the workout.\n");
 
  /* TODO: check time, duration and distance! */ 
  
  if (!(sport = get_field_sport (dialog)))
    valid = FALSE;
  else
    g_free (sport);

  if (GTK_IS_WIDGET (p_data->okbutton))
  gtk_widget_set_sensitive (GTK_WIDGET (p_data->okbutton), valid);
}


static void
add_detail_to_view (const gchar *key, const gchar *val, KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);

  /* Don't show these here, because they are shown in their own fields */
  if (strcmp (key, "distance") == 0
   || strcmp (key, "duration") == 0
   || strcmp (key, "comment") == 0)
    return;
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->details_view));
  
  g_return_if_fail (GTK_IS_LIST_STORE (model));

  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, key, 1, val, -1);
}


static void
fill_data (KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  KPParamListCategory *cat; 
  KPParamList *list;
  KPParam *param;
  GtkTextBuffer *buffer;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GList *node;
  const gchar *sport;
  gchar *str;
  guint raw_value, val;
  
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);

  /* Sport menu */
  model = gtk_combo_box_get_model (GTK_COMBO_BOX (p_data->sport_menu));

  if (gtk_tree_model_get_iter_first (model, &iter)) {
    sport = kp_workout_get_sport (p_data->wo);
    do {
      gtk_tree_model_get (model, &iter, 0, &str, -1);
      
      if (strcmp (str, sport) == 0) {
        gtk_combo_box_set_active_iter (GTK_COMBO_BOX (p_data->sport_menu),
                                       &iter);
        break;
      }
    } while (gtk_tree_model_iter_next (model, &iter));
  }
  
  /* Comment */
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (p_data->comment));
  str = kp_workout_get_comment (p_data->wo);
  if (str) { 
    gtk_text_buffer_set_text (buffer, str, -1);
    g_free (str);
  }
    
  /* Time entry && button */
  str = kp_calendar_time_to_time_string (KP_CALENDAR_ENTRY (p_data->wo)->datetime);
  g_return_if_fail (str != NULL);

  g_free (str);

  list = p_data->wo->param_list;
  
  /* Duration */
  raw_value = kp_param_list_get_uint (list, "duration");
  
  val = raw_value / (60 * 60 * 1000);
  raw_value -= (val * 60 * 60 * 1000);
  gtk_spin_button_set_value (p_data->duration_h, (gdouble) val);
  
  val = raw_value / (60 * 1000);
  raw_value -= (val * 60 * 1000);
  gtk_spin_button_set_value (p_data->duration_m, (gdouble) val);

  val = raw_value / 1000;
  raw_value -= val * 1000;
  gtk_spin_button_set_value (p_data->duration_s, (gdouble) val);
  gtk_spin_button_set_value (p_data->duration_t, (gdouble) raw_value);

  /* Distance */
  str = kp_param_list_get_as_string (list, "distance");
  if (str) {
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (p_data->distance_entry), 
                             strtod (str, NULL));
    g_free (str);
  }
  
  cat = kp_param_list_get_category (list, "detail");
  
  /* Details */
  for (node = cat->list; node; node = node->next) {
    param = KP_PARAM (node->data);
    str = kp_param_get_as_string (param);
    add_detail_to_view (kp_param_get_name (param), str, editor);

    g_free (str);
  }

  /* Intensity Entry */
  str = kp_param_list_get_as_string (list, "intensity");
  if (str) {
    kp_intensity_entry_set_text (KP_INTENSITY_ENTRY (p_data->intensity_entry),
                                 str);
    g_free (str);
  }
}


/* 
 * This function returns the selected sport or NULL
 * if nothing valid is selected.
 */
static gchar *
get_field_sport (KPWorkoutEditor *dialog)
{
  KPWorkoutEditorPrivateData *p_data;
  const gchar *sport_str;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *tmp;
  gchar buf[255];

  g_return_val_if_fail (KP_IS_WORKOUT_EDITOR (dialog), NULL);
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (dialog);

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (p_data->sport_menu));
  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (p_data->sport_menu), &iter)
      == FALSE)
    return NULL;

  gtk_tree_model_get (model, &iter, 0, &tmp, -1);
  if (!tmp)
    return NULL;
  
  strncpy (buf, tmp, sizeof (buf));
  g_free (tmp);
  
  /* Sport is not what it was initially set, so OK */
  if (strcmp (SPORT_MENU_TEXT, buf) != 0) {
    /* User has selected "Other" from menu and maybe written the sport */
    if (strcmp (SPORT_MENU_OTHER, buf) == 0) {
      sport_str = gtk_entry_get_text (GTK_ENTRY (p_data->sport_entry));
      
      if (!sport_str || strlen (sport_str) == 0)
        return NULL;

      return g_strdup (sport_str);
    }
      
    /* User has selected a sport from the menu */
    return g_strdup (buf);
  }
  /* Sport is not selected by user, so this workout can't be added yet */
  else 
    return NULL;
}


static void 
build_sport_menu (KPWorkoutEditor *dialog)
{
  KPWorkoutEditorPrivateData *p_data;
  GtkCellRenderer *cell;
  KPPresetDataItem *sport;
  GtkListStore *store;
  GtkComboBox *box;
  GSList *list;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (dialog);

  box = GTK_COMBO_BOX (p_data->sport_menu);

  store = gtk_list_store_new (1, G_TYPE_STRING);
  gtk_combo_box_set_model (box, GTK_TREE_MODEL (store));
  g_object_unref (store);

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (box), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (box), cell, "text", 0, NULL);
 
  list = kp_preset_data_get_items (KP_PRESET_DATA_SPORT); 
  for (;list; list = list->next) {
    sport = KP_PRESET_DATA_ITEM (list->data);
    gtk_combo_box_append_text (box, g_strdup (sport->name));
  }

  gtk_combo_box_prepend_text (box, _(SPORT_MENU_TEXT));
  gtk_combo_box_append_text (box, _(SPORT_MENU_OTHER));

  g_signal_connect (G_OBJECT (box), "changed",
                    G_CALLBACK (on_sport_menu_activate), dialog);

  g_signal_connect (G_OBJECT (p_data->sport_entry), "changed",
                    G_CALLBACK (on_sport_entry_changed), dialog);
  
  gtk_combo_box_set_active (box, 0);
}


static void
set_detail_tree_view (KPWorkoutEditor *dialog)
{
  KPWorkoutEditorPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *column1;
  GtkTreeViewColumn *column2;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->details_view);
  
  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_text_new ();

  column1 = gtk_tree_view_column_new_with_attributes (_("Field"), renderer,
                                                     "text", 0, NULL);
  column2 = gtk_tree_view_column_new_with_attributes (_("Value"), renderer,
                                                     "text", 1, NULL);
  gtk_tree_view_append_column (treeview, column1);
  gtk_tree_view_append_column (treeview, column2);

  selection = gtk_tree_view_get_selection (treeview);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (detail_list_row_selected), dialog);
}

static void
build_detail_menu (KPWorkoutEditor *dialog)
{
  KPWorkoutEditorPrivateData *p_data;
  KPPresetDataItem *detail;
  GtkWidget *item;
  GtkWidget *menu;
  GSList *list;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (dialog);
  
  menu = gtk_menu_new ();
  item = gtk_menu_item_new_with_label (_("Choose preset detail"));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  
  list = kp_preset_data_get_items (KP_PRESET_DATA_DETAIL);
  for (;list; list = list->next) {
    detail = KP_PRESET_DATA_ITEM (list->data);
    
    item = gtk_menu_item_new_with_label (detail->name);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  }

  gtk_option_menu_set_menu (GTK_OPTION_MENU (p_data->detail_preset_menu), menu);
  gtk_widget_show_all (menu);
}

static void
detail_list_row_selected (GtkTreeSelection *selection, gpointer data)
{
  KPWorkoutEditorPrivateData *p_data;
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (KP_WORKOUT_EDITOR (data));
  gtk_widget_set_sensitive (p_data->detail_delete_button, TRUE);
}


static void
on_detail_delete_button_clicked (GtkButton *button, gpointer data)
{
  KPWorkoutEditorPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *text;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (KP_WORKOUT_EDITOR (data));
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->details_view));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

    gtk_tree_model_get (model, &iter, 0, &text, -1);
    kp_param_list_remove (p_data->wo->param_list, text);
    g_free (text);
    
    (void) gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
      
    if (gtk_tree_model_iter_n_children (model, NULL) == 0)
      gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);

  }
}


static void
on_detail_add_button_clicked (GtkButton *button, KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  GtkWidget *treeview;
  GtkTreeModel *model;
  GtkTreeIter iter;
  KPParam *param;

  gchar *value_str;
  gchar *name;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);
  
  treeview = p_data->details_view;

  value_str = g_strdup (gtk_entry_get_text (GTK_ENTRY (p_data->detail_value)));

  name = kp_gui_get_option_menu_active (
      GTK_OPTION_MENU (p_data->detail_preset_menu));
  g_return_if_fail (name != NULL);
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, name, 1, value_str, -1);
  param = kp_param_new (name);
  kp_param_set_automatic_as_string (param, value_str);
  kp_param_list_insert (p_data->wo->param_list, "detail", param);
  
  g_free (value_str);
  g_free (name);
  
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_value), "");
}



static void
on_sport_menu_activate (GtkComboBox *box, gpointer data)
{
  KPWorkoutEditorPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *text;
  
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (data);
  
  if (!gtk_combo_box_get_active_iter (box, &iter))
    g_return_if_reached ();
  
  model = gtk_combo_box_get_model (box);
  gtk_tree_model_get (model, &iter, 0, &text, -1);
 
  if (strcmp (text, _(SPORT_MENU_TEXT)) == 0) {
    if (GTK_IS_WIDGET (p_data->okbutton))
      gtk_widget_set_sensitive (p_data->okbutton, FALSE);
    g_free (text);
    goto check;
  }

  if (strcmp (text, _(SPORT_MENU_OTHER))  == 0) {
    gtk_widget_show (p_data->sport_entry);
    kp_workout_set_sport (p_data->wo, NULL);
    g_free (text);
    goto check;
  }
 
  gtk_widget_set_sensitive (p_data->okbutton, TRUE);
  kp_workout_set_sport (p_data->wo, g_strstrip (text));
  g_free (text);

  if (GTK_WIDGET_VISIBLE (p_data->sport_entry))
    gtk_widget_hide (p_data->sport_entry);

check:
  check_values (NULL, KP_WORKOUT_EDITOR (data));
}


static void
duration_changed_cb (GtkSpinButton *button, KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  guint h, m, s, t;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);

  h = (guint) gtk_spin_button_get_value (p_data->duration_h);
  m = (guint) gtk_spin_button_get_value (p_data->duration_m);
  s = (guint) gtk_spin_button_get_value (p_data->duration_s);
  t = (guint) gtk_spin_button_get_value (p_data->duration_t);

  kp_param_list_set_time (p_data->wo->param_list, "default", "duration", 
                          h * 60 * 60 * 1000 + m * 60 * 1000 + s * 1000 + t);
}

static void
distance_changed_cb (GtkSpinButton *button, KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  gdouble d;
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);

  d = gtk_spin_button_get_value (GTK_SPIN_BUTTON (p_data->distance_entry));

  kp_param_list_set_double (p_data->wo->param_list, "default", "distance", d);
}


static void
time_selected_cb (KPDateEntry *entry, KPTime *time, KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);

  kp_calendar_time_set_hmst (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                             time->h, time->m, time->s, 0);
}

static void
date_selected_cb (KPDateEntry *entry, KPDate *date, KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);

  kp_calendar_time_set_dmy (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                            date->d, date->m, date->y);
}


static void
on_sport_entry_changed (GtkEntry *entry, gpointer data)
{
  KPWorkoutEditorPrivateData *p_data;
  const gchar *text;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (KP_WORKOUT_EDITOR (data));
 
  if ((text = gtk_entry_get_text (entry)))
    kp_workout_set_sport (p_data->wo, text);
 
  check_values (NULL, KP_WORKOUT_EDITOR (data));
}

static void
comment_changed_cb (GtkTextBuffer *buffer, KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  GtkTextIter s, e;
  gchar *text;

  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);

  gtk_text_buffer_get_start_iter (buffer, &s);
  gtk_text_buffer_get_end_iter (buffer, &e);

  text = gtk_text_buffer_get_text (buffer, &s, &e, -1);
  kp_workout_set_comment (p_data->wo, text);
  g_free (text);
}


static void
intensity_entry_changed_cb (KPIntensityEntry *entry, KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  gchar *text;
  
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);

  text = gtk_combo_box_get_active_text (GTK_COMBO_BOX (entry));
  if (text) {
    kp_param_list_set_string (p_data->wo->param_list, "default", 
                             "intensity", text);
    g_print ("Got real text: %s\n", text);
  }
}

static void
intensity_entry_user_text_changed (GtkEntry *entry, KPWorkoutEditor *editor)
{
  KPWorkoutEditorPrivateData *p_data;
  const gchar *text;
  
  p_data = KP_WORKOUT_EDITOR_PRIVATE_DATA (editor);

  text = gtk_entry_get_text (GTK_ENTRY (entry)); 
  if (text)
    kp_param_list_set_string (p_data->wo->param_list, "default", 
                             "intensity", text);
}


