#include <string.h>

#include "kpworkoutmodel.h"
#include "kplogstatitem.h"
#include "kpworkout.h"


static void
kp_log_stat_item_init (KPLogStatItem *item)
{
  item->period = 0;
  item->sport = NULL;
  item->intensity = NULL;
  item->days_in_period = 0;
  item->n_entries = 0;
  item->n_workouts = 0;
  
  item->tot_du = 0;
  item->avg_du = 0;
  item->max_du = 0;
  item->min_du = 0;

  item->tot_di = 0.0;
  item->avg_di = 0.0;
  item->max_di = 0.0;
  item->min_di = 0.0;
}


KPLogStatItem *
kp_log_stat_item_new (KPLogStatPeriod period, const gchar *sport, 
                      const gchar *intensity)
{
  KPLogStatItem *item;

  item = g_new (KPLogStatItem, 1);
  
  kp_log_stat_item_init (item);
  item->sport = (sport) ? g_strdup (sport) : NULL;
  item->intensity = (intensity) ? g_strdup (intensity) : NULL;
  item->period = period;
    
  return item;
}


void
kp_log_stat_item_free (KPLogStatItem *item)
{
  if (item->sport)
    g_free (item->sport);
  if (item->intensity)
    g_free (item->intensity);

  g_free (item);
}


void
kp_log_stat_item_add_data (KPLogStatItem *item, guint du, gdouble di)
{
  item->tot_du += du;
  item->tot_di += di;
 
  item->avg_du = (guint) ((gdouble) item->tot_du / (gdouble) item->n_workouts);
  
  if (item->n_workouts != 0.0)
    item->avg_di = item->tot_di / (gdouble) item->n_workouts;
  else
    item->avg_di = 0.0;

  if (di > item->max_di)
    item->max_di = di;
  if (du > item->max_du)
    item->max_du = du;
  
  if (item->min_di == 0 || di < item->min_di)
    item->min_di = di;
  if (item->min_du == 0 || du < item->min_du)
    item->min_du = du;
}


void
kp_log_stat_item_add_item (KPLogStatItem *dest, KPLogStatItem *src)
{
  dest->tot_du += src->tot_du;
  dest->tot_di += src->tot_di;

  dest->n_workouts += src->n_workouts;
  dest->n_entries += src->n_entries;

  g_return_if_fail (dest->n_workouts != 0.0);
  
  dest->avg_du = (guint) ((gdouble) dest->tot_du / (gdouble) dest->n_workouts);
  dest->avg_di = dest->tot_di / (gdouble) dest->n_workouts;

  dest->max_di = MAX (dest->max_di, src->max_di);
  dest->max_du = MAX (dest->max_du, src->max_du);
  
  if (dest->min_di == 0 || (src->min_di != 0 && src->min_di < dest->min_di))
    dest->min_di = src->min_di;
  if (dest->min_du == 0 || (src->min_du != 0 && src->min_du < dest->min_du))
    dest->min_du = src->min_du;
}

  
void
kp_log_stat_item_add_split (KPLogStatItem *item, KPSplit *split)
{
  g_return_if_fail (item != NULL);
  g_return_if_fail (split != NULL);
    
  kp_log_stat_item_add_data (item, split->duration, split->distance); 
}


gboolean
split_ok (KPSplit *split, const gchar *sport, const gchar *intensity)
{
  if (split == NULL)
    return FALSE;

  if (sport != NULL && strcasecmp (sport, split->sport) != 0)
    return FALSE;

  if (intensity != NULL && strcasecmp (intensity, split->type) != 0) {
    g_print ("Split type: %s\n", split->type);
    return FALSE;
  }
  
  return TRUE;
}


gboolean
workout_ok (KPWorkout *wo, const gchar *sport, const gchar *intensity)
{
  const gchar *sport_str;
  gchar *int_str;

  if (wo == NULL)
    return FALSE;

  
  if (sport != NULL) {
    sport_str = kp_workout_get_sport (wo);
    if (sport_str == NULL)
      return FALSE;
    
    if (strcasecmp (sport, kp_workout_get_sport (wo)) != 0) {
    return FALSE;
    }
  }

  if (intensity != NULL) {
    int_str = kp_param_list_get_as_string (KP_WORKOUT (wo)->param_list, 
                                          "intensity"); 
    if (int_str == NULL)
      return FALSE; 

    if (strcasecmp (intensity, int_str) != 0) {
      g_free (int_str);
      return FALSE;
    }
    g_free (int_str);
  }
  return TRUE;
}


void
kp_log_stat_item_add_workout (KPLogStatItem *item, KPWorkout *wo)
{
  /* Split workout */
  KPSplit *split;
  GSList *node;
  /* Workout */
  gdouble di;
  guint du;
  gboolean split_added = FALSE;
  
  g_return_if_fail (item != NULL);
  g_return_if_fail (KP_IS_WORKOUT (wo));
  
  if (KP_IS_SPLIT_WORKOUT (wo)) {
    /* N workouts must include this current item when counting below.
     * So, if it isn't valid, we'll decrease this */
    item->n_workouts++;
 
    for (node = KP_SPLIT_WORKOUT (wo)->splits; node; node = node->next) {
      split = node->data;
     
      if (split_ok (split, item->sport, item->intensity)) {
        kp_log_stat_item_add_split (item, split);
        split_added = TRUE;
      }
    }
    /* No splits were added, so no workout wasn't added either */
    if (!split_added)
      item->n_workouts--;
  } 
  else {
   if (workout_ok (wo, item->sport, item->intensity)) {
      item->n_workouts++;
      du = kp_workout_model_get_duration (KP_WORKOUT_MODEL (wo));
      di = kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo));
      
      kp_log_stat_item_add_data (item, du, di);
    }
  }
}



