/*
 * The Kissme default system class loader
 */

package java.lang;

import java.util.StringTokenizer;
import java.io.IOException;
import java.io.File;
import java.io.FileInputStream;
import java.net.URL;
import java.net.MalformedURLException;

import gnu.java.lang.SystemClassLoader;
import kissme.vm.loaders.JarClassLoader;
import kissme.vm.loaders.ZipClassLoader;
import kissme.vm.loaders.FileClassLoader;

public class DefaultClassLoader extends SystemClassLoader
{
  private String[] classPathComponents;
  private String fileSeparator;

  /**
   * Creates the class loader, using a classpath determined by the
   * system properties. Note that the parent may be null, when this is
   * created as the system class loader by ClassLoader.getSystemClassLoader.
   *
   * @param parent the parent class loader
   */
  public DefaultClassLoader(ClassLoader parent)
  {
    super(parent);
    String classPath = System.getProperty("java.class.path", ".");
    String pathSeparator = System.getProperty("path.separator");
    StringTokenizer st = new StringTokenizer(classPath, pathSeparator, false);
    int nosComponents = st.countTokens();

    st = new StringTokenizer(classPath, pathSeparator, false);
    classPathComponents = new String[nosComponents];
    for (int i = 0; i < nosComponents; i++) {
      classPathComponents[i] = st.nextToken();
    }
    fileSeparator = System.getProperty("file.separator");
  }

  public Class loadClass(String name) throws ClassNotFoundException
  {
    try {
      Class res = super.loadClass(name);
      return res;
    }
    catch (ClassNotFoundException ex) {
      throw ex;
    }
  }

  protected Class findClass(String name) throws ClassNotFoundException 
  { 
    char[] chars = name.toCharArray();
    for (int i = 0; i < chars.length; i++) {
      if (chars[i] == '.') {
	chars[i] = fileSeparator.charAt(0);
      }
    }

    String classFileName = new String(chars);

    for (int i = 0; i < classPathComponents.length; i++) {
      String component = classPathComponents[i];
      byte[] classData;
      if (component.endsWith(".jar")) {
        classData = JarClassLoader.getClassData(classFileName, component);
      }
      else if (component.endsWith(".zip")) {
        classData = ZipClassLoader.getClassData(classFileName, component);
      }
      else {
	classData = FileClassLoader.getClassData(classFileName, component);
      }
      if (classData != null) {
	return defineClass(name, classData, 0, classData.length);
      }
    }
    throw new ClassNotFoundException(name);
  }


  protected URL findResource(String name)
  { 
    String urlString = null;
    
    for (int i = 0; i < classPathComponents.length; i++) {
      String component = classPathComponents[i];
      if (component.endsWith(".jar")) {
        urlString = JarClassLoader.getResourceUrl(name, component);
      }
      else if (component.endsWith(".zip")) {
        urlString = ZipClassLoader.getResourceUrl(name, component);
      }
      else {
	urlString = FileClassLoader.getResourceUrl(name, component);
      }
      if (urlString != null) {
	break;
      }
    }
    if (urlString == null) {
      return null;
    }

    try {
      return new URL(urlString);
    }
    catch (MalformedURLException ex) {
      InternalError ex2 = 
	new InternalError("Loader returned a bad resource URL: " + urlString);
      ex2.initCause(ex);
      throw ex2;
    }
  }
}
