package hitme;

import java.io.*;

/**
 * The main entry point class for the HitMe test harness.  Running
 * hitme.HitMe with no arguments runs the tests listed in 'hitme.tests'.
 * Arguments (if supplied) are individual tests to be run.  Refer to
 * HitMeOpts for details of options.
 **/
public class HitMe
{
  private HitMeOpts m_opts;
  private int m_successCount;  
  private int m_failCount;
  private int m_abortCount;
  private int m_skipCount;
  private int m_missingCount;
  
  private PrintStream m_err;
  private PrintStream m_log;
    

  private HitMe(HitMeOpts opts) {
    m_opts = opts;
    m_err = System.err;
    m_log = System.out;
  }

  /**
   * Load and run a test class.
   * @param className the fully qualified class name.
   * @return true if the test succeeded
   */
  public boolean runTest(String className) {
    HitMeTest t;
    try { 
      Class c = Class.forName(className);
      t = (HitMeTest) (c.newInstance());
    }
    catch (Exception ex) {
      m_err.println("Test class '" + className + 
		    "' cannot be instantiated: " +
		    ex.getMessage());
      m_missingCount++;
      return false;
    }
    String spaces = "                                        ";
    String pad = ((className.length() > spaces.length()) ? "" :
		  spaces.substring(className.length()));
    m_log.println(className + pad + " " + t.describe());
    try {
      if (t.runTest(new HitMeFullLogger(m_err, m_log))) {
	m_successCount++;
	return true;
      }
      else {
	m_log.println(className + " - FAILED");
	m_failCount++;
      }
    }
    catch (Throwable ex) {
      m_log.println(className + " - ABORTED");
      ex.printStackTrace(m_err);
      m_abortCount++;
    }
    return false;
  }

  /**
   * Run the tests listed in the test case list
   */
  public void runTests() {
    try {
      BufferedReader br = m_opts.getCaseListReader();
      String line;      
      while ((line = br.readLine()) != null) {
	line = line.trim();
	if (line.length() > 0) {
	  if (line.charAt(0) == '#') {
	    m_skipCount++;
	  }
	  else {
	    runTest(line);
	  }
	}
      }
    }
    catch (IOException ex) {
      m_err.println("IO Error while reading the case list: " + ex);
    }
  }

  /**
   * Output statistics on tests passed, failed, etcetera
   */
  public void statistics() {
    m_err.println("***** HitMe Test Summary *****");
    m_err.println("Succeeded: " + m_successCount + 
		  "  Failed: " + m_failCount + 
		  "  Aborted: " + m_abortCount +
		  "  Skipped: " + m_skipCount +
		  "  Missing: " + m_missingCount);
  }

  /**
   * @return the return code for the main program.
   */
  public int getRC() {
    return (m_failCount + m_abortCount + m_missingCount > 0) ? 1 : 0;
  }

  /**
   * Test harness entry point.
   * @param args command line arguments.
   */
  public static void main (String[] args) {
    HitMeOpts opts = new HitMeOpts(args);
    if (!opts.isOk()) {
      System.err.println("usage: hitme " + opts.getShortHelp() +
			 " [ <some.test.classname> ... ]");
      System.exit(1);
    }
    else if (opts.isHelp()) {
      System.err.println("usage: hitme " + opts.getShortHelp() +
			 " [ <some.test.classname> ... ]");
      System.err.println(opts.getLongHelp());
    }
    else {
      int argPos = opts.getParameterPos();
      HitMe h = new HitMe(opts);
      if (argPos < args.length) {
	boolean success = true;
	for (int i = argPos; i < args.length; i++) {
	  success &= h.runTest(args[i]);
	}
	if (args.length - argPos > 1) {
	  h.statistics();
	}
	else {
	  if (success) {
	    h.m_err.println("Test succeeded");
	  }
	}
      }
      else {
	h.runTests();
	h.statistics();
      }
      System.exit(h.getRC());
    }
  }
}
