/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part 
 * of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
/*
 * native stuff for java.lang.Runtime
 *
 */

#ifdef KISSME_LINUX_USER
#include <stdlib.h>
#include <sys/utsname.h>
#endif
#include <string.h>
#include <unistd.h>
#include <stdio.h>

#include "vm/jni.h"
#include "vm/jni_data.h"
#include "vm/kni.h"
#include "vm/kni_methods.h"

#include "vm/newobject.h"
#include "vm/interp.h"
#include "vm/interp_methods.h"
#include "vm/classfile_methods.h"
#include "vm/garbage.h"

#ifdef DLOPEN
#include <dlfcn.h>
#endif

#ifdef PERSIST
#include "vm/rpot.h"
#endif

#include "lib/indigenous/java.lang/java_lang_Runtime.h"
#include "lib/indigenous/java.lang/VMSystem.h"


extern tOBREF OOMExceptionObject;
extern int respectGCCalls;

/*
 * private synchronized native String initializeLinkerInternal();
 * private native String buildLibName(String pathname, String filename);
 * 
 * // Helper for load and loadLibrary 
 * private native boolean loadFileInternal(String filename);
 *
 * // Runs the garbage collector.
 * public native void gc();
 *
 * public native void runFinalization();
 */

void Java_java_lang_Runtime_runFinalizersOnExitInternal(JNIEnv* env,
							jobject obj, 
							jboolean val)
{
  return;
}


void Java_java_lang_Runtime_runFinalization(JNIEnv* env, jobject obj)
{
  return;
}


/*
 * Fork and run the process 
 *
 * Return a UnixProcess object which has the streams set up to the 
 * child's input/output/error.
 */
jobject Java_java_lang_Runtime_execInternal(JNIEnv* env, jobject obj,
					    jarray arglist, jarray environment,
					    // XXX - directory arg ignored
					    jobject directory)
{
  jobject process;
  jclass processClazz;
  jmethodID processCons;
  
  int num_args = 0;
  int num_env = 0;
  int i = 0;
  int file_descriptors[6];
  
  int failure;
  
  if (arglist) {
    num_args = (*env)->GetArrayLength(env, arglist);
  }
    
  if (environment) {
    num_env = (*env)->GetArrayLength(env, environment);
  }
  
  processClazz = (*env)->FindClass(env, "java/lang/UnixProcess");
  assert(processClazz);
  processCons = (*env)->GetMethodID(env, processClazz, "<init>", "()V");
  assert(processCons);
  process = (*env)->NewObject(env, processClazz, processCons, &failure);
  if (failure) {
    (*env)->Throw(env, process);
    return NULL;
  }
  
  assert(process);
  {
    // We construct an array of char* which are the args
    char** args = (char**) sys_malloc(sizeof(char*) * (num_args + 1));

    assert(args);

    //Now each arg comes from the array
    //The first arg is the path
    for (i = 0; i < (num_args); i++) {
      char* tmp;
      jstring thestring = (*env)->GetObjectArrayElement(env, arglist, i);
      
      args[i] = (char*) sys_malloc((*env)->GetStringLength(env, thestring));
      assert(args[i]);
      
      tmp = INTERP_AscizFromString(env, thestring);
      if (tmp) {
	strcpy(args[i], tmp);
	//sys_free(tmp); XXX
      }
      else {
	strcpy(args[i], "");
      }
    }
    args[num_args] = NULL;

    {
      int nfd = 0;
      int err;
      pid_t newpid;
#define IN_IN           0
#define IN_OUT          1
#define OUT_IN          2
#define OUT_OUT         3
#define ERR_IN          4
#define ERR_OUT         5  

      /* Create the pipes to communicate with the child */
      /* Make sure fds get closed if we can't create all pipes */
      for (nfd = 0; nfd < 6; nfd += 2) {
	err = pipe(file_descriptors + nfd);
	if (err == -1) {
	  close(file_descriptors[0]);
	  close(file_descriptors[1]);
	  close(file_descriptors[2]);
	  close(file_descriptors[3]);
	  close(file_descriptors[4]);
	  close(file_descriptors[5]);
	  
	  
	  //Raise Exception
	  eprintf("Error making pipes for child\n");
	  return NULL;
	}
      }       
      
      newpid = fork();
      if (newpid == 0) {
	// We are the child ...
	/* set stdin, stdout, and stderr up from the pipes */
	dup2(file_descriptors[IN_IN], 0);
	dup2(file_descriptors[OUT_OUT], 1);
	dup2(file_descriptors[ERR_OUT], 2); 
	
	if (execvp(args[0], args) == -1) {
	  exit(1);
	}
      }
      else if (newpid == -1) {
	/* Error */
	/* Close all pipe fds */
	close(file_descriptors[0]);
	close(file_descriptors[1]);
	close(file_descriptors[2]);
	close(file_descriptors[3]);
	close(file_descriptors[4]);
	close(file_descriptors[5]);
      }
      else {
	int failure;
	/* Parent */
	/* Construct the FileDescriptors for those we do need */
	
	jobject fd1 = INTERP_NewObjectFromName(env, "java/io/FileDescriptor", 
					       &failure);
	jobject fd2 = INTERP_NewObjectFromName(env, "java/io/FileDescriptor", 
					       &failure);
	jobject fd3 = INTERP_NewObjectFromName(env, "java/io/FileDescriptor", 
					       &failure);
	
	jclass fdClass = (*env)->GetObjectClass(env, fd1);
	jfieldID fid = (*env)->GetFieldID(env, fdClass, "native_fd", "I");
	
	assert(fid);
	
	(*env)->SetIntField(env, fd1, fid, file_descriptors[IN_OUT]);
	(*env)->SetIntField(env, fd2, fid, file_descriptors[OUT_IN]);
	(*env)->SetIntField(env, fd3, fid, file_descriptors[ERR_IN]);
	
	/* Now create the streams and set the pid*/
	{
	  jclass unixPClass = (*env)->GetObjectClass(env, process);
	  jmethodID mid = (*env)->GetMethodID(env, unixPClass, "makeStreams",
					      "(Ljava/io/FileDescriptor;Ljava/io/FileDescriptor;Ljava/io/FileDescriptor;)V");
	  jfieldID fid = (*env)->GetFieldID(env, unixPClass, "pid", "I");
	  
	  assert(mid);
	  assert(fid);
	  
	  (*env)->CallVoidMethod(env, process, mid, fd1, fd2, fd3);
	  (*env)->SetIntField(env, process, fid, newpid);
	}
	/* close the fds we won't need */
	close(file_descriptors[IN_IN]);
	close(file_descriptors[OUT_OUT]);
	close(file_descriptors[ERR_OUT]);
      }
    }
  }    
  //    eprintf("Returning %x\n", process);
  return process;
}


// static native String getLibraryPath();            
jstring Java_java_lang_Runtime_getLibraryPath(JNIEnv* env,
					      jclass class)
{
#ifdef DEBUG
  eprintf("java_lang_Runtime_getLibraryPath ...\n");
#endif
  return INTERP_NewStringFromAsciz(env, ".");
}


void Java_java_lang_Runtime_gc(JNIEnv* env,
			       jobject obj)
{
  // The Java spec says that a call to java.lang.Runtime.gc() is a
  // hint.  We pay attention to it ONLY if kissme is started with
  // the -gc option.
  if (respectGCCalls) {
    GARBAGE_lock_and_gc(env, JNI_getCurrentHeap());
  }
  return;
}

jobject Java_java_lang_Runtime_initializeLinkerInternal(JNIEnv* env,
							jobject obj
							)
{
#ifdef DEBUG
  eprintf("java_lang_Runtime_initializeLinkerInternal ...\n");
#endif
  
  return INTERP_NewStringFromAsciz(env, "");
}


// native String nativeGetLibname(String pathname, String libname); 

jobject Java_java_lang_Runtime_nativeGetLibname(JNIEnv* env, jclass cls,
						jobject pathname, 
						jobject filename)
{
  char* pathstr = INTERP_AscizFromString(env, pathname);
  char* filestr = INTERP_AscizFromString(env, filename);
  // XXX - this is unsafe
  char newName[512];
 
  traceLoading("java_lang_Runtime_nativeGetLibname(%s, %s)", 
	       pathstr, filestr);
  
  strcpy(newName, "lib");
  strcat(newName, filestr);
  strcat(newName, ".so");
  sys_free(pathstr);
  sys_free(filestr);
  return INTERP_NewStringFromAsciz(env, newName);
}


/*
 * This implementation returns 0 on failure, 1 if we loaded the library
 * and 2 if it was already loaded.  NullPointerException is thrown if
 * the filename argument is null.
 */
jint Java_java_lang_Runtime_nativeLoad(JNIEnv* env, jobject obj,
				       jobject filename)
{
#ifdef DLOPEN
  char * str;
  void* temp;
  int i;
  
  if (filename == NULL) {
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "NULL filename"));
    return 0;
  }
  str = INTERP_AscizFromString(env, filename);
  
  traceLoading("java_lang_Runtime_nativeLoad(%s)", str);
  
  assert(JNI_NumberLoadedLibraries < 20);
  
  temp = dlopen(str, RTLD_LAZY);
  
  if (temp == NULL) {
    traceLoading("Failed to load %s: %s", str, dlerror());
    sys_free(str);
    return 0;
  }
  sys_free(str);
  
  //check if we've loaded this one before
  for (i = 0; i < JNI_NumberLoadedLibraries; i++) {
    if (JNI_DynLoadedLibraries[i] == temp) {
      dlclose(temp);
      return 2;
    }
  }

  JNI_DynLoadedLibraries[JNI_NumberLoadedLibraries++] = temp;
  traceLoading0("java_lang_Runtime_nativeLoad succeeded");
  //    CLASSFILE_FindOrLoad(env, "java/io/BufferedInputStream", NULL);
  //    traceLoading0("blatt");
  return 1;
#else
  traceLoading0("java.lang.nativeLoad() is disabled");
  return 0;
#endif
}


//{"java/lang/Runtime", "traceInstructions", "(Z)V", (void*) Java_java_lang_Runtime_traceInstructions }, 
void Java_java_lang_Runtime_traceInstructions(JNIEnv* env, jclass clazz, 
					      jboolean flag)
{
  eprintf("Runtime.traceInstructions(%i)\n", flag);
  DIAG_SetChannelState(OPCODE_DIAG_CHANNEL, flag);  
  DIAG_SetChannelState(OPSTACK_DIAG_CHANNEL, flag);
}


//{"java/lang/Runtime", "traceMethodCalls", "(Z)V", (void*) Java_java_lang_Runtime_traceMethodCalls }, 
void Java_java_lang_Runtime_traceMethodCalls(JNIEnv* env, jclass clazz,
					     jboolean flag)
{
  eprintf("Runtime.traceMethodCalls(%i)\n", flag);
  DIAG_SetChannelState(CALL_DIAG_CHANNEL, flag);
}


void Java_java_lang_Runtime_exitInternal(JNIEnv* env, jobject ob, jint value)
{
  exit(shutdown_machine(NULL, 1, value));
}


jlong Java_java_lang_Runtime_freeMemory(JNIEnv* env, jobject object)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  return GARBAGE_getNumFreeWords(env, kniPtr->GetHeap(env)) * 4;
}


jlong Java_java_lang_Runtime_totalMemory(JNIEnv* env, jobject object)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  return GARBAGE_getNumTotalWords(env, kniPtr->GetHeap(env)) * 4;
}


// Call the 'put' method on the properties object to set the propName
// to value propValue.
static void putProp(JNIEnv* env, jobject props, 
		    char *propName, char* propValue)
{
  int32 args[4];
  char* className = sys_malloc(80);
  tMethod* method;
  strcpy(className, "java.util.Hashtable");
  
  method = (*env)->GetMethodID(env, 
			       (*env)->FindClass(env, className),
			       "put", 
			       "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
  
  if (method == NULL) {
    // XXX wrong exception ...
    (*env)->Throw(env, OOMExceptionObject);
  }
  else {
    args[0] = (int32) props;
    args[1] = (int32) INTERP_NewStringFromAsciz(env, propName);
    args[2] = (int32) INTERP_NewStringFromAsciz(env, propValue);
    INTERP_RunVirtualMethodFromPtr(env, method, args);
  }
  sys_free(className);
}

/**
 * Get the system properties. For JDK 1.4 compatibility, these MUST 
 * include the following:
 * <dl>
 * <dt>java.version         <dd>Java version number
 * <dt>java.vendor          <dd>Java vendor specific string
 * <dt>java.vendor.url      <dd>Java vendor URL
 * <dt>java.home            <dd>Java installation directory
 * <dt>java.vm.specification.version <dd>VM Spec version
 * <dt>java.vm.specification.vendor  <dd>VM Spec vendor
 * <dt>java.vm.specification.name    <dd>VM Spec name
 * <dt>java.vm.version      <dd>VM implementation version
 * <dt>java.vm.vendor       <dd>VM implementation vendor
 * <dt>java.vm.name         <dd>VM implementation name
 * <dt>java.specification.version    <dd>Java Runtime Environment version
 * <dt>java.specification.vendor     <dd>Java Runtime Environment vendor
 * <dt>java.specification.name       <dd>Java Runtime Environment name
 * <dt>java.class.version   <dd>Java class version number
 * <dt>java.class.path      <dd>Java classpath
 * <dt>java.library.path    <dd>Path for finding Java libraries
 * <dt>java.io.tmpdir       <dd>Default temp file path
 * <dt>java.compiler        <dd>Name of JIT to use
 * <dt>java.ext.dirs        <dd>Java extension path
 * <dt>os.name              <dd>Operating System Name
 * <dt>os.arch              <dd>Operating System Architecture
 * <dt>os.version           <dd>Operating System Version
 * <dt>file.separator       <dd>File separator ("/" on Unix)
 * <dt>path.separator       <dd>Path separator (":" on Unix)
 * <dt>line.separator       <dd>Line separator ("\n" on Unix)
 * <dt>user.name            <dd>User account name
 * <dt>user.home            <dd>User home directory
 * <dt>user.dir             <dd>User's current working directory
 * </dl>
 *
 * @param p the Properties object to insert the system properties into
 */
void Java_java_lang_Runtime_insertSystemProperties(JNIEnv* env,
						   jclass class,
						   jobject props)
{
  char* valueBuffer = (char*) sys_malloc(4096);
  char* var = NULL;
#ifdef KISSME_LINUX_USER
  struct utsname sysInfo;
#endif
  int i;
  
  if (valueBuffer == NULL) {
    (*env)->Throw(env, OOMExceptionObject);
    return;
  }
  
  // Set some version properties
  putProp(env, props, "java.version", "1.4");
  putProp(env, props, "java.vendor", JAVA_VENDOR);
  putProp(env, props, "java.vendor.url", JAVA_VENDOR_URL);

  // Set the Kissme installation directory
#ifdef KISSME_LINUX_USER
  var = getenv("KISSME_HOME");
#endif
  putProp(env, props, "java.home", (var == NULL) ? DEFAULT_KISSME_HOME : var);

  // Set more version properties
  putProp(env, props, "java.vm.specification.version", "1.0");
  putProp(env, props, "java.vm.specification.vendor", 
	  "Sun Microsystems Inc."); 
  putProp(env, props, "java.vm.specification.name", 
	  "Java Virtual Machine Specification");
  putProp(env, props, "java.vm.version", KISSME_VERSION);
  putProp(env, props, "java.vm.vendor", KISSME_VENDOR); 
  putProp(env, props, "java.vm.vendor.url", KISSME_VENDOR_URL); 
  putProp(env, props, "java.vm.name", "Kissme");
  putProp(env, props, "java.specification.version", "1.4");
  putProp(env, props, "java.specification.vendor", 
	  "Sun Microsystems Inc."); 
  putProp(env, props, "java.specification.name", 
	  "Java Platform API Specification");

  // I think that this should be the Classpath version number, but
  // how do we get it?
  putProp(env, props, "java.class.version", "?");
  
  // Set the effective classpaths.
  strcpy(valueBuffer, "");
  for (i = 0; i < CLASSFILE_UserClassPath.u16NumClassPaths; i++) {
    strcat(valueBuffer, CLASSFILE_UserClassPath.ppszClassPaths[i]);
    if (i != (CLASSFILE_UserClassPath.u16NumClassPaths - 1)) {
      strcat(valueBuffer, ":");
    }
  }
  putProp(env, props, "java.class.path", valueBuffer);

  strcpy(valueBuffer, "");
  for (i = 0; i < CLASSFILE_BootClassPath.u16NumClassPaths; i++) {
    strcat(valueBuffer, CLASSFILE_BootClassPath.ppszClassPaths[i]);
    if (i != (CLASSFILE_BootClassPath.u16NumClassPaths - 1)) {
      strcat(valueBuffer, ":");
    }
  }
  putProp(env, props, "java.boot.class.path", valueBuffer);

  // Set library path used for loading JNI libraries
#ifdef KISSME_LINUX_USER
  var = getenv("LD_LIBRARY_PATH");
  if (var != NULL && var[0] != 0) {
    strcpy(valueBuffer, var);
    strcat(valueBuffer, ":/usr/lib:/lib");
  }
  else {
    strcpy(valueBuffer, "/usr/lib:/lib");
  }
  putProp(env, props, "java.library.path", valueBuffer);
#else
  putProp(env, props, "java.library.path", "?");
#endif

  // Set the temporary directory as used by the IO subsystem
  putProp(env, props, "java.io.tmpdir", "/tmp");

  // Set the default JIT compiler
  putProp(env, props, "java.compiler", DEFAULT_JIT_COMPILER);

  // Kissme does not support an "ext" directory yet.
  putProp(env, props, "java.ext.dirs", "");


  // Set the hardware & OS properties
#ifdef KISSME_LINUX_USER
  uname(&sysInfo);
  putProp(env, props, "os.name", sysInfo.sysname);
  // This copies the behaviour of JDK 1.4.  For earlier JDK versions
  // on Linux, Intel x86 platforms had "os.arch" == "i386".
  if (sysInfo.machine[0] == 'i' && 
      (sysInfo.machine[1] >= '3' && sysInfo.machine[1] <= '6') &&
      sysInfo.machine[2] == '8' && sysInfo.machine[3] == '6' &&
      sysInfo.machine[4] == 0) {
    putProp(env, props, "os.arch", "x86");
  }
  else {
    putProp(env, props, "os.arch", sysInfo.machine);
  }
  putProp(env, props, "os.version", sysInfo.release);
#else
  putProp(env, props, "os.name", "?");
  putProp(env, props, "os.arch", "?");
  putProp(env, props, "os.version", "?");
#endif

  // Set the separator strings
  putProp(env, props, "file.separator", "/");
  putProp(env, props, "path.separator", ":");
  putProp(env, props, "line.separator", "\n");

  // Set the user name
#ifdef KISSME_LINUX_USER
  var = getenv("USER");
#endif
  putProp(env, props, "user.name", (var == NULL) ? "?" : var);
  
  // Set the user home directory
#ifdef KISSME_LINUX_USER
  var = getenv("HOME");
#endif
  putProp(env, props, "user.home", (var == NULL) ? "?" : var);

  // Set the current directory
#ifdef KISSME_LINUX_USER
  getcwd(valueBuffer, 4096);
#else
  strcpy(valueBuffer, "/usr/share/teaseme");
#endif
  putProp(env, props, "user.dir", valueBuffer);

  // Set the GNU Classpath io encoding scheme
  putProp(env, props, 
	  "gnu.java.io.encoding_scheme_alias.ISO-8859-1", "8859_1");

  // Finally, insert any properties passed on the command line with
  // a -D option.
  for (i = 0; i < command_line_num_args; i++) {
    if (strncmp(command_line_args[i], "-D", 2) == 0) {
      char* equalIndex = strstr(command_line_args[i], "=");
      if (equalIndex) {
	*equalIndex = 0;
	putProp(env, props, command_line_args[i] + 2, equalIndex + 1);
      }
    }
  }
  sys_free(valueBuffer);
  return;
}
