/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h> 
#include  "vm/wrappers.h"
 
/* Hacked version of CPLIST. This stores pairs of <class, classloader>
 *
 */
 
#include <stdlib.h> 
#include <string.h> 
#include <stdio.h> 

#include "vm/interp.h" 
#include "vm/classfil.h"
#include "vm/cptuplelist.h" 
 
#include "vm/global.h" 
  
static tCPTupleListElement* pstTreeTop = NULL; 
static tCPTupleListElement* pstCurrPointer = NULL; 
static int iChanged; 
 
/* 
 * @doc FUNC 
 * @func 
 * Inserts a class into the tree in alphabetical order. 
 * 
 */ 

void CPTUPLELIST_Insert(tClassLoaderTuple* pstClass
			/* @parm The class to add */ 
			) 
{ 
  tCPTupleListElement* pstNew; 
  tCPTupleListElement* pstTemp; 
  int iDone = 0; 
  int iCompare; 
  
  iChanged = 1; 
  
  assert(CPTUPLELIST_Find(pstClass->uidName, pstClass->classLoader) == NULL);
  
  pstNew = (tCPTupleListElement*) sys_malloc(sizeof(tCPTupleListElement)); 
  
  assert(pstNew); 

  pstNew->pstClass = pstClass; 
  pstNew->pstLeft = NULL; 
  pstNew->pstRight = NULL; 
 
  /* if list is empty, simple case */ 
  if (pstTreeTop == NULL) { 
    pstTreeTop = pstNew; 
    pstNew->pstParent = NULL; 
  } 
  else { 
    /* otherwise search for place to put it */ 
    pstTemp = pstTreeTop; 
    while (!iDone) { 
      iCompare = strcmp(pstClass->uidName, pstTemp->pstClass->uidName); 
      
      /* class is already here */ 
      if ((iCompare == 0) &&
	  (pstClass->classLoader == pstTemp->pstClass->classLoader)) { 
        iDone = 1; 
      } 
      /* need to go left */ 
      else if ((iCompare < 0) || 
	       ((iCompare == 0) && 
		(pstClass->classLoader < pstTemp->pstClass->classLoader))) { 
	if (pstTemp->pstLeft) { 
          pstTemp = pstTemp->pstLeft; 
        } 
        else { 
          pstNew->pstParent = pstTemp; 
          pstTemp->pstLeft = pstNew; 
          iDone = 1; 
        } 
      } 
      /* need to go right */ 
      else if ((iCompare > 0) || 
	       ((iCompare == 0) && 
		(pstClass->classLoader > pstTemp->pstClass->classLoader))) { 
        if (pstTemp->pstRight) { 
          pstTemp = pstTemp->pstRight; 
        } 
        else { 
          pstNew->pstParent = pstTemp; 
          pstTemp->pstRight = pstNew; 
          iDone = 1; 
        } 
      }
      else {
	panic0("CPTUPLELIST_Insert: shouldn't get here");
      }
    } 
  } 
} 


/* 
 * @doc FUNC 
 * @func 
 * Searches for an element in the tree, given the name of the class. 
 * 
 * @rdesc Returns a pointer to the class if it's found. NULL otherwise. 
 * 
 */ 
 
tClassLoaderTuple* CPTUPLELIST_Find(char* pszName, tOBREF loader) 
{ 
  Uid uidName = UID_GetUid(pszName); 
  
#ifdef UID_WARNINGS 
  if (uidName == pszName) { 
    printf("Alert - Uid passed to CPLIST_Find - %s\n", pszName); 
  } 
#endif /*UID_WARNINGS*/ 
  
  return CPTUPLELIST_UidFind(uidName, loader); 
} 


/* 
 * @doc FUNC 
 * @func 
 * Searches for an element in the tree, given the name of the class. The string 
 * pointer that is passed must be a Uid. 
 * 
 * @rdesc Returns a pointer to the class if it's found. NULL otherwise. 
 * 
 */ 
 
tClassLoaderTuple* CPTUPLELIST_UidFind(Uid uidName, tOBREF loader) 
{ 
  tCPTupleListElement* pstTemp; 
  int iCompare; 
  
  if (pstTreeTop == NULL) { 
    return NULL; 
  } 
  
  pstTemp = pstTreeTop; 
  while (1) { 
    iCompare = (strcmp(uidName, pstTemp->pstClass->uidName)); 
 
    /* found it */ 
    if ((iCompare == 0) && (pstTemp->pstClass->classLoader == loader)) { 
      return pstTemp->pstClass; 
    } 
    /* else go left */ 
    else if ((iCompare < 0) || 
	     ((iCompare == 0) && (loader < pstTemp->pstClass->classLoader))) { 
      if (pstTemp->pstLeft) { 
	pstTemp = pstTemp->pstLeft; 
      } 
      else { 
        return NULL; 
      } 
    } 
    /* else go right */ 
    else if ((iCompare > 0) || 
	     ((iCompare == 0) && (loader > pstTemp->pstClass->classLoader))) { 
      if (pstTemp->pstRight) { 
	pstTemp = pstTemp->pstRight; 
      } 
      else { 
        return NULL; 
      } 
    } 
  } 
} 

 
/* 
 * @doc FUNC 
 * @func 
 * Checks whether or not the list is empty. 
 * 
 * @rdesc Returns one of the following: 
 * 
 * @flag 0 | Not empty 
 * @flag 1 | Empty 
 * 
 */ 
 
int CPTUPLELIST_IsEmpty(void) 
{ 
  return (pstTreeTop == NULL); 
} 
 

/* 
 * @doc FUNC 
 * @func 
 * Sets the current element to the first in the list. 
 * 
 */ 
 
void CPTUPLELIST_GoTop(void) 
{ 
  /* find leftmost element in tree */ 
  pstCurrPointer = pstTreeTop; 
  if (pstCurrPointer) {
    while (pstCurrPointer->pstLeft) { 
      pstCurrPointer = pstCurrPointer->pstLeft; 
    } 
  }
} 
 

/* 
 * @doc FUNC 
 * @func 
 * Sets the current element to the alphabetically next element in the list. 
 * 
 */ 
 
void CPTUPLELIST_Skip(void) 
{ 
  if (pstCurrPointer) { 
    /* if we can go right, we go right and then as left as possible */ 
    if (pstCurrPointer->pstRight) { 
      pstCurrPointer = pstCurrPointer->pstRight; 
      while (pstCurrPointer->pstLeft) { 
        pstCurrPointer = pstCurrPointer->pstLeft; 
      } 
    } 
    /* else we have to move back up */ 
    else { 
      /* if parent is NULL, we have finished */ 
      if (pstCurrPointer->pstParent == NULL) { 
        pstCurrPointer = NULL; 
        return; 
      } 
      
      /* if we are the left child, we just go to the parent */ 
      if (pstCurrPointer == pstCurrPointer->pstParent->pstLeft) { 
        pstCurrPointer = pstCurrPointer->pstParent; 
      } 
      /* if we are the right child, we go to the parent and then continue 
         going up until we go up a "left branch" */ 
      else { 
        /* keep going while we are still a right child */ 
        while (pstCurrPointer == pstCurrPointer->pstParent->pstRight) { 
          pstCurrPointer = pstCurrPointer->pstParent; 
          /* check if we've got to the top */ 
          if (pstCurrPointer->pstParent == NULL) { 
            pstCurrPointer = NULL; 
            return; 
          } 
        } 
        /* go up the left branch - if parent is NULL then so be it */ 
        pstCurrPointer = pstCurrPointer->pstParent; 
      } 
    } 
  } 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Checks if the previous call to CPTUPLELIST_Skip reached the end of the list. 
 * 
 * @rdesc Returns one of the following: 
 * 
 * @flag 0 | Reached end 
 * @flag 1 | Not at end 
 * 
 */ 
 
int CPTUPLELIST_IsMore(void) 
{ 
  return (pstCurrPointer != NULL); 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Returns the class pointer of the current element. 
 * 
 * @rdesc Returns the class pointer of the current element. 
 * 
 */ 
 
tClassLoaderTuple* CPTUPLELIST_GetCurrentClass(void) 
{ 
  if (pstCurrPointer != NULL) { 
    return pstCurrPointer->pstClass; 
  } 
  else { 
    return NULL; 
  } 
} 

 
/* 
 * @doc FUNC 
 * @func 
 * Returns the class name of the current element. 
 * 
 * @rdesc Returns the class name of the current element. 
 * 
 */ 
 
char* CPTUPLELIST_GetCurrentName(void) 
{ 
  if (pstCurrPointer != NULL){ 
    return pstCurrPointer->pstClass->uidName; 
  } 
  else{ 
    return NULL; 
  } 
} 
 

/* 
 * @doc FUNC 
 * @func 
 * Clears the "changed" flag. 
 * 
 */ 
void CPTUPLELIST_ClearChanged(void) 
{ 
  iChanged = 0; 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * This checks if the "changed" flag is set. This flag is set whenever a new 
 * element is added to the list. It can be reset with CPTUPLELIST_ClearChanged(). 
 * 
 * @rdesc Returns 0 if it's not set, non-0 if it is set. 
 * 
 */ 
 
int CPTUPLELIST_HasChanged (void) 
{ 
  return (iChanged == 1); 
} 
 

static void TreeDelete(tCPTupleListElement* pstRoot) 
{ 
  if (pstRoot->pstLeft) { 
    TreeDelete(pstRoot->pstLeft); 
  } 
  if (pstRoot->pstRight) { 
    TreeDelete(pstRoot->pstRight); 
  } 
  sys_free(pstRoot); 
} 
 
void CPTUPLELIST_Empty(void) 
{ 
  if (pstTreeTop) {
    TreeDelete(pstTreeTop); 
  } 
}
 
 
 
#ifdef DEBUG_CPLIST 

/* Do some testing */

void CPTUPLELIST_Test()
{
  tClassLoaderTuple* tmp;

  tmp = sys_malloc(sizeof(tClassLoaderTuple));
  tmp->uidName = UID_GetUid("java/lang/Object");
  tmp->classLoader = NULL;

  CPTUPLELIST_Insert(tmp);

  tmp = sys_malloc(sizeof(tClassLoaderTuple));
  tmp->uidName = UID_GetUid("java/lang/Object");
  tmp->classLoader = 0x80;

  CPTUPLELIST_Insert(tmp);

  tmp = sys_malloc(sizeof(tClassLoaderTuple));
  tmp->uidName = UID_GetUid("java/lang/Object");
  tmp->classLoader = 0x100;

  CPTUPLELIST_Insert(tmp);

  tmp = sys_malloc(sizeof(tClassLoaderTuple));
  tmp->uidName = UID_GetUid("java/lang/reflect/Constructor");
  tmp->classLoader = 0x10;

  CPTUPLELIST_Insert(tmp);

  tmp = sys_malloc(sizeof(tClassLoaderTuple));
  tmp->uidName = UID_GetUid("java/lang/Object");
  tmp->classLoader = -0x10;

  CPTUPLELIST_Insert(tmp);


  eprintf("Finding Object %p\n", CPTUPLELIST_Find("java/lang/Object", NULL));
  eprintf("Finding Object %p\n", CPTUPLELIST_Find("java/lang/Object", 0x80));
  eprintf("Finding Object %p\n", CPTUPLELIST_Find("java/lang/Object", 0x100));

}


void CPTUPLELIST_Dump() 
{ 
  eprintf("CPLIST tree\n"); 
  CPTUPLELIST_DumpNode(pstTreeTop, 0); 
  eprintf("end CPLIST tree\n"); 
} 
 
void CPTUPLELIST_DumpNode(tCPTupleListElement* root, int indent)  
{ 
  if (root != NULL) { 
    int i; 
    for (i = 0; i < indent; i++) {
      eprintf(" "); 
    }
    
    if (root->pstParent) { 
      if (root->pstParent->pstLeft == root) {
	eprintf("L"); 
      }
      else {
	eprintf("R"); 
      } 
    }
    eprintf("* %s (%p,%p)\n", root->pstClass->uidName, root->pstClass, 
	    root->pstClass->classLoader); 
    
    if (root->pstLeft) {
      CPTUPLELIST_DumpNode(root->pstLeft, indent + 1); 
    }
    if (root->pstRight) {
      CPTUPLELIST_DumpNode(root->pstRight, indent + 1); 
    }      
  } 
} 
#endif
/* end of cplist.c */ 
 
 
