//fns for ETA

#ifndef __ETA__
#define __ETA__

/* Eta is the module that implements the distributed shared memory. We expect one or more copies of eta to run on different computers. Eta communicates with the other instances of itself on remote machines, asking them for pages it wants to access, and sending them pages that they request. I have separated Eta into two source files, eta.c handles mostly the 'client' stuff, it has the code that requests pages from other Eta nodes. The eta_server.c file contains code to service requests.

Eta does not depend on any other modules. 

Below we have the data structure that represents a page (in OS parlance). Pages exist in memory only as far as ETA is concerned, but Alpha reuses the structure for the ALPHA_write_non_blocking function.

*/
/* This structure is the data stored for each page. Each page holds 4k of data. 
   The flushed parameter indicates whether this page has been written to ALPHA (0 means flushed)
 */

#define BLOCK_SIZE 4096 //The size of each page

struct page_entry {

  long long address;	//Which address this page is for
  int lock;		//ETA_lock_page and unlock_page use this variable to control access to pages
  int counter;		//Unused at present
  int access;		//Can be either 0,1 or 2. 0 means we have no access at all to this page. 1 means we have read-only access. 2 means we can write to the page.
  int copy_set;		//This is a list of hosts that have read copies of the page. The format is a just a bit indicating the host number. In my network I have machines 1.0.0.1 and 1.0.0.2, so the copy set for these two would have bit 1 and bit 2 set.

  int prob_owner_host;	//Indicates the ip address of the host we think is the owner of this page
  int prob_owner_port;	//Indicates the port of the host we think is the owner of this page

  int flushed;		//0 means this page has been flushed to ALPHA
  void* page_data;	//The actual 4k of data held by this page
  struct page_entry* next; 
};

/* These are the methods in ETA. The ones that are used publicly by GAMMA are:

get_page_for_address
create_page_for_address
lock_page_entry
unlock_page_entry

I'll discuss the methods in eta.c and eta_server.c
*/

struct page_entry* ETA_get_page_for_address(long long address, int readwrite);
int ETA_create_page_for_address(long long address);

int send_page(struct page_entry* mpage, int requesting_host, int requesting_port, int requesting_sequence);
int service_request(long long address, int requesting_host, int requesting_port, int readwrite, int requesting_sequence);

int ETA_request_read_access(long long address, int prob_owner_host, int prob_owner_port, int requesting_host, int requesting_port, int sequence);

int ETA_request_write_access(long long address, int prob_owner_host, int prob_owner_port, int requesting_host, int requesting_port, int sequence);

int ETA_lock_page_entry(struct page_entry* entry);
int ETA_unlock_page_entry(struct page_entry* entry);

void print_host(int h, int p); //Internal print function

/* Constants for communication between ETA nodes */

#define ETA_REQUEST_SIZE 4 * 6	       //The header size for a request between nodes
				       //4 ints, 1 long long 
#define ETA_READ_REQUEST 0xb0b0f00f    //Indicates a read request
#define ETA_WRITE_REQUEST 0xc0c0f00f   //Indicates a write request
#define ETA_IDENTIFY 0xabdabdef	       //This is used to verify our own ip address
#define ETA_PAGE_FOR_READ 0x0badd00b   //Delivering a page for read
#define ETA_PAGE_FOR_WRITE 0x0badbabe  //Delivering a page for writing
#define ETA_ACK 0x0000000f	       //Acknowledgment of ?
#define ETA_NACK 0xf000000f	       //Acknowledgment of ?
#define ETA_INVALIDATE 0xb000000f      //Invalidation of a page

#define ETA_GET_PAGE_FOR_READ		99	//Just constants for indicating read/write requests
#define ETA_GET_PAGE_FOR_WRITE		100


#endif
