/* 
 * This module implements entry into the GC.  The garbage collector itself
 * runs as a dedicated thread; i.e. GC_Thread.  Normally it is inactive,
 * waiting on the heap's gc_point_cond condition.  Before the GC thread
 * can do a garbage collection, it must wait for all other threads to
 * enter GC points.  
 *
 * When a worker thread requires the GC to run, it calls GC_NeedGC.  This
 * wakes the GC thread and enters the current thread into a GC point.  The
 * GC thread then waits until all other threads have entered a GC point,
 * then proceeds with the GC.
 *
 * When a thread is at a GC point, it tests to see if GC is pending.
 * If not, it simply returns.  Otherwise, it marks its stack top,
 * tells the GC thread it is ready, and then (typically) blocks until
 * the GC signals that it has completed.
 *
 * There are two kinds of GC point.  A simple GC point is entered by
 * calling GARBAGE_EnterGCPoint().  After marking its stack, the thread
 * waits on a condition variable until woken by the GC thread.  
 *
 * A GC region is entered by first calling GARBAGE_EnterGCRegion().
 * This marks the stack top and sets a flag to tell the GC thread not
 * to wait for this thread.  The thread can then do other work,
 * provided it doesn't involve the Java heap. It must not create heap
 * objects, read or write heap data, or change the values of any root
 * reference slots; e.g. on the Java stack or a JNI call's Localrefs.
 * The thread exits the GC region by calling GARBAGE_ExitGCRegion().
 * This checks to see if the GC is currently running before exitting
 * the region.
 * 
 * Note: this scheme only works if every worker thread periodically
 * enters a GC point.  The interpretter tries to ensure this by
 * calling GARBAGE_EnterGCPoint() in each method call, and each branch
 * that jumps backwards (i.e. loop iteration).  Any system calls that
 * might block a thread for a long time should made within a GC
 * region.
 *
 * Unfortunately, a thread cannot perform a system call in a GC region
 * if the syscall reads or writes to the Java heap.  (If it does, there
 * is a small risk that the GC might reallocate the source/target for
 * the syscall ... with dire consequences.)  You must either take steps
 * to avoid the syscall blocking, or accept the possibility that the
 * entire VM might freeze up because the GC con't run.  Alternatively,
 * use non-heap resident buffer.
 */


#include "config.h"

#ifdef GARBAGE2

extern int instructions_since_gc;
#ifdef KISSME_LINUX_USER
#include <unistd.h>
#include <stdio.h>
#define __USE_GNU
#include <pthread.h>
#include <errno.h>
#endif

#include "vm/garbage.h"
#include "vm/garbage2.h"
#include "vm/threads_native.h"
#include "vm/classfil.h"
#include "vm/classfile_methods.h"
#include "vm/interp_methods.h"


#define GC_THREAD_ACTIVE        0
#define GC_END_THREAD           1
#define GC_THREAD_ENDED         2

#define GC_IDLE                 0
#define GC_PENDING              1
#define GC_RUNNING              2

#ifdef KISSME_LINUX_KERNEL
static int FinalizationThread(void* arg);
static int GarbageThread(void* arg);
#else
static void* FinalizationThread(void* arg);
static void* GarbageThread(void* arg);
#endif

static tOBREF ConstructGCThreadObject(JNIEnv* env);


extern volatile int   totalThreadsInSystem;

/* 
 * There may be two threads that call this method, and this is quite
 * likely since both will be out of memory if they share the same heap
 */
int GARBAGE_NeedGC(tAllocatorHeap* pstHeap, int numWords)
{
  int waited = 0;
  tThreadNode* thisNode;
  sys_mutex_lock(pstHeap->gc_resume_mutex);
  if (pstHeap->garbageThreadControl != GC_THREAD_ACTIVE) {
    sys_mutex_unlock(pstHeap->gc_resume_mutex);
    return 1;
  }
  
  thisNode = THREAD_FindThread();
  thisNode->stackTop = (void*) &waited;

  // First check whether a GC isn't pending already
  if (pstHeap->wantGC == GC_IDLE) {
    pstHeap->wantGC = GC_PENDING;
    // Tell the GC thread that we need more memory
    traceGCThread("Waking up GC thread (tid %i)", sys_current_thread());
    sys_condition_signal(pstHeap->gc_point_cond);
  }
  sys_mutex_unlock(pstHeap->gc_resume_mutex);	  
  
  // Wait until the GC thread is ready 
  // It will lock gc_resume_mutex while changing wantGC    
  while (1) {
    waited++;
    sys_mutex_lock(pstHeap->gc_resume_mutex);
    if (pstHeap->wantGC == GC_RUNNING) {
      sys_mutex_unlock(pstHeap->gc_resume_mutex);	  
      break;
    }
    
    if (waited == 50) {
      sys_condition_signal(pstHeap->gc_point_cond);
      waited = 0;
    }
    
    sys_mutex_unlock(pstHeap->gc_resume_mutex);	  
#ifdef KISSME_LINUX_USER
    usleep(100);
#endif
  }
  
  traceGCThread("GC originating thread (tid %i) is entering GC point",
		sys_current_thread());
  GARBAGE_EnterGCPoint(pstHeap);
  return 0;
}


//This thread is the 'garbage collector thread'

#ifdef KISSME_LINUX_KERNEL
static int GarbageThread(void* args)
#else
static void* GarbageThread(void* args)
#endif
{
  int spincount = 0;
  tAllocatorHeap* pstHeap = (tAllocatorHeap*) args;
  JNIEnv* env = JNI_getJNIEnvPtr();

  //Create a pstJNIData for ourselves
  THREAD_createJNIData(NULL, NULL);

  while (pstHeap->garbageThreadControl < GC_END_THREAD) {
    traceGCThread("Garbage thread running (tid %i)", sys_current_thread());
    //We wait for a signal to start the gc
    sys_mutex_lock(pstHeap->gc_wait_mutex);	 
    pstHeap->garbageThreadReady = 1;
    
    traceGCThread("Garbage thread is sleeping (tid %i) on cond %p",
		  sys_current_thread(), &pstHeap->gc_point_cond);
    
#ifdef KISSME_RJK_KERNEL
    traceGCThread0("Garbage thread returning for RJK");
    return NULL;
#endif
    sys_condition_wait_nogc(pstHeap->gc_point_cond, pstHeap->gc_wait_mutex);
    sys_mutex_unlock(pstHeap->gc_wait_mutex);	
    
    traceGCThread("Garbage thread is woken (tid %i)", sys_current_thread());
    
    if (pstHeap->garbageThreadControl >= GC_END_THREAD) {
      break;
    }
    
    sys_mutex_lock(pstHeap->gc_resume_mutex);	
    pstHeap->wantGC = GC_RUNNING;
    
    pstHeap->threadsRequired = THREAD_GetNumberThreads(pstHeap); //hmm
    sys_mutex_unlock(pstHeap->gc_resume_mutex);	

    // XXX - what happens if a Java thread gets instantiated or destroyed
    // while we're waiting here?  Won't pstHeap->threadsPresent be out
    // of sync with THREAD_GetNumberThreads(pstHeap)?
    
    //Once we get that signal we wait for the other threads to reach
    //the GC point
    while (pstHeap->threadsPresent < pstHeap->threadsRequired) {
      // XXX - it is a bad idea to busy wait.  We should wait on the 
      // cond that regular threads signal when they enter the GC point.
#ifdef KISSME_LINUX_USER
      usleep(50); //Sleep for 50 microseconds
#endif
      if (spincount++ == 2000) {
	traceGCThread("Threads present %i and req %i",
		      pstHeap->threadsPresent, pstHeap->threadsRequired);
#ifdef DEBUG
        THREAD_dumpThreads(pstHeap);
#endif
	spincount = 0;
      }
#ifdef KISSME_LINUX_KERNEL
      schedule();
#endif
    }
    
    //Now that everyone is here, we do the GC
    traceGCThread("Garbage thread is doing gc (tid %i)", sys_current_thread());
    GARBAGE_lock_and_gc(env, pstHeap);
        
    //Indicate that we've done the GC
    sys_mutex_lock(pstHeap->gc_resume_mutex);
    pstHeap->wantGC = GC_IDLE;
    sys_mutex_unlock(pstHeap->gc_resume_mutex);
    
    //And let everyone carry on with their business
    traceGCThread("Garbage thread is signalling other threads (tid %i)",
		  sys_current_thread());
    sys_condition_broadcast(pstHeap->gc_resume_cond);
  }
  
  //The GC thread has realised it must exit
  pstHeap->garbageThreadControl = GC_THREAD_ENDED; 

  totalThreadsInSystem--;
  traceGCThread("garbage thread %i exited (number threads %i)", 
		sys_current_thread(), totalThreadsInSystem);
#ifdef KISSME_LINUX_KERNEL
  return 0;
#else
  return NULL;
#endif
}


int GARBAGE2_Init(tAllocatorHeap* pstHeap)
{
#ifdef KISSME_LINUX_USER
  int success;
  int tid = 0;
#endif

  pstHeap->garbageThreadControl = GC_THREAD_ACTIVE;
  pstHeap->finalizationThreadControl = GC_THREAD_ACTIVE;
  
  pstHeap->gc_point_cond = sys_condition_create();
  pstHeap->gc_resume_cond = sys_condition_create();
  pstHeap->gc_finalization_cond = sys_condition_create();

  pstHeap->gc_wait_mutex = sys_mutex_create(0);
  pstHeap->gc_resume_mutex = sys_mutex_create(0);  
  pstHeap->gc_finalization_mutex = sys_mutex_create(0);

  pstHeap->threadsPresent = 0;
  pstHeap->threadsRequired = 0;

  pstHeap->wantGC = GC_IDLE;

  //Create the garbage and finalization threads
#ifdef KISSME_LINUX_USER
  success = pthread_create(((pthread_t*) &tid), NULL,  
			   (void* (*)(void*)) &GarbageThread, 
			   (void*) pstHeap);
  assert(success == 0);
  traceGCThread("Created garbage thread -> tid %i", tid);

  success = pthread_create(((pthread_t*) &tid), NULL,  
			   (void* (*)(void*)) &FinalizationThread, 
			   (void*) pstHeap);
  assert(success == 0);
  traceGCThread("Created finalizer thread -> tid %i", tid);
#else
#ifdef KISSME_LINUX_KERNEL
  kernel_thread(&GarbageThread, (void*) pstHeap, 0);  
  kernel_thread(&FinalizationThread, (void*) pstHeap, 0);
#else
  kinterface->kthread_create(&GarbageThread, (void*) pstHeap, 32, 0);
  kinterface->kthread_create(&FinalizationThread, (void*) pstHeap, 32, 0);
#endif
#endif

  totalThreadsInSystem += 2;
  return 0;
}


int GARBAGE2_Finish(tAllocatorHeap* pstHeap)
{
  while (pstHeap->garbageThreadReady != 1) {
    traceGCThread("Waiting for garbage collection to end");
#ifdef KISSME_LINUX_USER
    pthread_yield();
#endif
#ifdef KISSME_LINUX_KERNEL
    schedule();
#endif
  }

  traceGCThread("Stopping finalization thread");
  pstHeap->finalizationThreadControl = GC_END_THREAD;
  while (pstHeap->finalizationThreadControl != GC_THREAD_ENDED) {
    sys_condition_signal(pstHeap->gc_finalization_cond);
  }

  traceGCThread("Stopping gc thread");
  pstHeap->garbageThreadControl = GC_END_THREAD;
  while (pstHeap->garbageThreadControl != GC_THREAD_ENDED) {
    sys_condition_signal(pstHeap->gc_point_cond);    
  }
  return 0;
}


/*
 * Calling this routine defines a "GC Point"; i.e. a point at which
 * the current thread can stop to allow Garbage Collection to
 * happen, if required.  If GC has been requested, the thread is
 * blocked here until the GC has run. 
 */
int GARBAGE_EnterGCPoint(tAllocatorHeap* pstHeap)
{
  if (pstHeap->wantGC == GC_RUNNING) {
    // 'someVariable' is used to record the top of the stack so
    // that we can mark the C stack accurately
    int someVariable;
    int rc;
    tTimeSpec timeout;
    tThreadNode* node = THREAD_FindThread();
    assert(node);
    
    traceGCThread("thread (tid %i) is in GC Point", sys_current_thread());
    
    // This is the address of the variable 'someVariable' on the C stack
    node->stackTop = &someVariable;
    // make sure compiler doesn't optimize 'someVariable' away 
    someVariable = 3; 
    
    traceGCThread("thread (tid %i) stackBottom is %p, stackTop is %p", 
		  sys_current_thread(), node->stackBottom, node->stackTop);
    traceGCThread("thread (tid %i) waiting for GC end broadcast",
		  sys_current_thread());
    
    // We're ready for the GC to run.
    sys_mutex_lock(pstHeap->gc_resume_mutex);
    pstHeap->threadsPresent++;

    // Wait until the GC has finished
#ifdef DEBUG
    // ... using a watchdog timer to guard against GC lockout cause by bugs
    // or long syscalls.  The latter is a problem we need to address fix.
    gettimeofday(&timeout, NULL);
    timeout.tv_sec += 30;
    rc = sys_condition_timedwait_nogc(pstHeap->gc_resume_cond, 
				      pstHeap->gc_resume_mutex, &timeout);
    if (rc == ETIMEDOUT) {
      // Unfortunately, the pthread_cond_timedwait call sometimes seems to
      // return prematurely.  (Any clues why??)  So a panic is a bad idea.
      nonfatalError("thread (tid %i) GC entry is stalling ...",
		    sys_current_thread());
      // Continue waiting, but without the timer.
      rc = sys_condition_wait_nogc(pstHeap->gc_resume_cond, 
				  pstHeap->gc_resume_mutex);
    }
#else
    // ... without using a timeout.  If something goes wrong, we may
    // hang forever.
    rc = sys_condition_wait_nogc(pstHeap->gc_resume_cond, 
				 pstHeap->gc_resume_mutex);
#endif

    if (rc) {
      panic("pthread_cond_timedwait failed: rc = %i", rc);
    }
    traceGCThread("thread (tid %i) got GC end broadcast", 
		  sys_current_thread());
    
#ifdef DEBUG_CLASS_FOR_GC
    instructions_since_gc = 0;
#endif
    pstHeap->threadsPresent--;
    sys_mutex_unlock(pstHeap->gc_resume_mutex);
    return 1;
  }
  
  return 0;
}


/*
 * Calling this routine defines the start of GC region; i.e. a region
 * in which Garbage Collection can proceed in parallel with the thread,
 * if required.  At the start of the region, we mark the stack and INC
 * pstHeap->threadsPresent so that the GC thread won't be blocked on
 * our account.
 */
int GARBAGE_EnterGCRegion(tAllocatorHeap* pstHeap)
{
  int someVariable;  
  tThreadNode* node = THREAD_FindThread();
  assert(node);
  
  traceGCThread("thread (tid %i) entering GC region", sys_current_thread());

  // This is the address of the variable 'someVariable' on the C stack
  node->stackTop = &someVariable;

  // make sure compiler doesn't optimize 'someVariable' away 
  someVariable = 3; 

  // this says we're happy for a GC to occur.
  sys_mutex_lock(pstHeap->gc_resume_mutex);
  pstHeap->threadsPresent++;
  sys_mutex_unlock(pstHeap->gc_resume_mutex);
  return 0;
}


/*
 * Calling this routine defines the end of a GC region.  At the end,
 * we look to see if GC is currently pending or running.  If it is, we
 * wait for it to end.  Otherwise, we just DEC pstHeap->threadsPresent.
 */
int GARBAGE_ExitGCRegion(tAllocatorHeap* pstHeap)
{
  sys_mutex_lock(pstHeap->gc_resume_mutex);
  if (pstHeap->wantGC != GC_IDLE) {
    traceGCThread("thread (tid %i) waiting to exit GC region", 
		  sys_current_thread());
    sys_condition_wait_nogc(pstHeap->gc_resume_cond, pstHeap->gc_resume_mutex);
  }
  else {
    assert(pstHeap->threadsPresent);
    pstHeap->threadsPresent--;
  }
  sys_mutex_unlock(pstHeap->gc_resume_mutex);
  traceGCThread("thread (tid %i) exited GC region", sys_current_thread());
  return 0;
}


/*
 * This method is the main line of the finalization thread
 */
#ifdef KISSME_LINUX_KERNEL
static int FinalizationThread(void* arg) 
#else
static void* FinalizationThread(void* arg) 
#endif
{
  tAllocatorHeap* pstHeap = (tAllocatorHeap*) arg;
  tStackFrame* pstFrame;
  tClassLoaderTuple* tuple;
  tMethod* pstFinalize;
  tThreadNode* node;
  int i, j;
  JNIEnv* env = JNI_getJNIEnvPtr();

  int32 i32GCStackPlace;
  tOBREF obj, pstExOb;
  tOBREF hThread = NULL;

  traceGC("Starting FinalizationThread (tid %i)", sys_current_thread());

  i32GCStackPlace = 21; //stop this being optimised away
  
  /* Create a pstJNIData for ourselves */
  THREAD_createJNIData(NULL, NULL);
  JNI_getJNIData(sys_current_thread())->pstHeap = pstHeap;

  /* Create the thread node, but don't complete the Java thread yet.
     We need to create the node before we call sys_condition_wait,
     but we can't allocate Java objects yet. */
  node = THREADINFO_CreateNewNodeForThread(sys_current_thread(), 
					   &i32GCStackPlace);
  assert(node);
  THREADINFO_addThreadNodeToList(node);
  THREADINFO_IncrementThreadCount();

  sys_mutex_lock(pstHeap->gc_finalization_mutex);

  /* Wait here until finalization is first needed ... or the thread
     needs killing. */
  traceGC("FinalizationThread: waiting for first wakeup");
  sys_condition_wait(pstHeap->gc_finalization_cond, 
		     pstHeap->gc_finalization_mutex);

  while (pstHeap->finalizationThreadControl == GC_THREAD_ACTIVE) {
    /* Complete the java Thread for running finalizers. */
    if (hThread == NULL) {
      /* Release mutex to avoid deadlock if allocating the thread
	 triggers a GC */
      sys_mutex_unlock(pstHeap->gc_finalization_mutex);
      traceGC("FinalizationThread: creating thread object");
      hThread = ConstructGCThreadObject(env);
      assert(hThread);

      /* Reacquire mutex */
      sys_mutex_lock(pstHeap->gc_finalization_mutex);
    }
    else {
      /* Get the next object to finalize */
      obj = ALLOCATOR_GetFinalizationCandidate(pstHeap); 
      if (obj == NULL) {
	/* There is no work at the moment ... wait */
	traceGC("FinalizationThread: waiting for more work");
	sys_condition_wait(pstHeap->gc_finalization_cond, 
			   pstHeap->gc_finalization_mutex);
	continue;
      }
      /* Release mutex while we run the finalize method to avoid
         deadlock if finalization triggers a GC */
      sys_mutex_unlock(pstHeap->gc_finalization_mutex);
      
      /* Find the object's finalize method */
      tuple = DEREF(obj)->pstType;
      pstFinalize = CLASSFILE_FindMethodInVT(env, tuple, "finalize", "()V");
      if (pstFinalize == NULL) {
	panic("Cannot find %s.finalize method", tuple->uidName);
      }
      
      /* Run the finalizer */
      pstFrame = InitialStackFrame(pstFinalize, obj, (int32 *) &obj, pstHeap);
      THREADINFO_SetNodeInfo(node, hThread, &i32GCStackPlace);
      THREADINFO_addInitialStackFrameToNode(node, pstFrame);
      
      traceGC("calling interpreter to finalize a %s object", tuple->uidName);
      pstExOb = Interpret(env, pstFrame, 1);
      
      traceGC("interpreter has returned from finalize");
      THREADINFO_removeInitialStackFrameFromNode(node, pstFrame);
       
      /* Reacquire mutex */
      sys_mutex_lock(pstHeap->gc_finalization_mutex);
      
     /* Deregister object from finalizer list */
      ALLOCATOR_DeregisterFinalizable(obj, pstHeap); 
    }
  }

  i32GCStackPlace = 22; //stop this being optimised away

  pstHeap->finalizationThreadControl = GC_THREAD_ENDED;
  sys_mutex_unlock(pstHeap->gc_finalization_mutex);
}


/* 
 * Construct a Java Thread object with the name "gc" 
 */
static tOBREF ConstructGCThreadObject(JNIEnv* env)
{
  tOBREF hThread;
  jmethodID methodID;
  jclass thread_cls = (*env)->FindClass(env, "java/lang/Thread");  

  assert(thread_cls);

  methodID = (*env)->GetMethodID(env, thread_cls, "<init>", 
				 "(Ljava/lang/String;)");
  assert(methodID);

  traceGC0("Creating the 'gc' Thread object");
  hThread = (*env)->NewObject(env, thread_cls, methodID, 
			      INTERP_NewStringFromAsciz(env, "gc"));
  traceGC("GC Thread object is %p", hThread);
  if ((*env)->ExceptionOccurred(env) == NULL) {
    return hThread;
  }
  else {
    traceGC("Exception %s occurred while creating 'gc' Thread",
	    DEREF((*env)->ExceptionOccurred(env))->pstType->uidName);
    return NULL;
  }
}


#endif














