 /* Routines to manage the old heap */

#include "config.h"

#ifdef GARBAGE2

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <string.h>
#endif

#include "vm/stdtypes.h"


#include "vm/garbage/public_methods.h"
#include "vm/garbage/bitfield.h"
#include "vm/garbage/oldheap.h"
#include "vm/garbage.h"
#include "vm/garbage2.h"
#include "vm/rpot.h"

#include "vm/garbage_defs.h"
#include "vm/interp.h" //for tObject*

#include "vm/jni_data.h"

#include "lib/indigenous/java.lang/Class.h"

#ifdef DEBUG
int highest_block;
#endif

int OLDHEAP_Init(tAllocatorHeap* heap)
{
  int i;

#ifdef DEBUG
  highest_block = 0;
#endif

 // Clean the bitfield for our words
  memset(heap->pi32Block + heap->i32YoungHeapBFSize, 0, 
	 heap->i32OldHeapBFSize * sizeof(uint32));

  // We simply indicate that the whole heap is free
  heap->pi32OldHeap[0] = heap->i32OldHeapSize;
  heap->pi32OldHeap[1] = (int32) NULL; //next pointer is NULL
  heap->pi32OldFirstFree = heap->pi32OldHeap;				 
  return 0;
}

/* Allocates space for an object with a handle
 */

tOBREF OLDHEAP_Alloc(tAllocatorHeap* heap, int32 i32NumWords)
{
  tOBREF hHandle;
  tObject* obj;
  
  hHandle = HANDLES_getFreeHandle(heap);
  
  if (hHandle == NULL) {
    traceGC0("No more free handles, gc'ing");
    GARBAGE_UNLOCK(heap);
    GARBAGE_NeedGC(heap, i32NumWords);
    GARBAGE_LOCK(heap);
    
    hHandle = HANDLES_getFreeHandle(heap);
    
    if (hHandle == NULL) { 
      traceGC0("Still no more free handles, giving up");
      return NULL;
    }
  }
  
  // XXXX - what happens if OLDHEAP_AllocObject causes a garbage
  // collection.  What stops the collector freeing hHandle?
  obj = OLDHEAP_AllocObject(heap, i32NumWords);
  if (obj == NULL) {
    return NULL;
  }
  else {
    *hHandle = obj;
    obj->hHandle = hHandle;
    
    assert(ALLOCATOR_ISHANDLE(heap,  hHandle) == 1); //just for debugging
    return hHandle;
  }
}

/* Allocates space for an object. Allocates i32NumWords of space 
 *
 * Whenever we allocate an object, we mark the first word of that object 
 * as used in the bitfield (the word before it is the size)
 */
tObject* OLDHEAP_AllocObject(tAllocatorHeap* heap, int32 i32NumWords)
{
  int32* pi32TempFS;

  assert(i32NumWords >= 1);

  /* garbage collect if free space is not big enough */
  if (heap->pi32OldFirstFree[0] < i32NumWords) {
    traceGC("Run out of space, must GC (our tid %x %i)", 
	    (int) sys_current_thread(), (int) sys_current_thread());
    
    GARBAGE_UNLOCK(heap);
    GARBAGE_NeedGC(heap, i32NumWords);
    GARBAGE_LOCK(heap);
    
    if (heap->pi32OldFirstFree[0] < i32NumWords) {
      traceGC0("Still not enough space");
      return NULL;
    }
  }
  
  /* store current beginning of free space (will be beginning of new object) */
  pi32TempFS = heap->pi32OldFirstFree;
  
  /* move beginning of free space to after object */
  heap->pi32OldFirstFree += i32NumWords;
  
  /* store size of smaller free space */
  heap->pi32OldFirstFree[0] = pi32TempFS[0] - i32NumWords;
  
  /* store size of new allocated bit */
  pi32TempFS[0] = i32NumWords;

  /* mark this space in the bit field */
  OLD_SET_BIT(((pi32TempFS + 1) - heap->pi32OldHeap));

#ifdef DEBUG
  highest_block = (((pi32TempFS + 1) - heap->pi32OldHeap)) / 32;
#endif
  return (tObject*) (pi32TempFS + 1);
}


int OLDHEAP_ISHANDLE(tAllocatorHeap* heap, tOBREF h)
{
  if (HANDLES_InHandleBlock(heap, h)) {
    if (((int32*) (*h) >= heap->pi32OldHeap) && 
	(((int32*) (*h) < (heap->pi32OldHeap + heap->i32OldHeapSize)))) {
      return 1;
    }
    else {
      return 0;
    }
  }
  else {
    return 0;
  }
}


int OLDHEAP_TestBitfield(tAllocatorHeap* heap) 
{
  int i = 0;
  int word;

  for (i = 0; i < 33; i++) {
    word = 3008;
    OLD_SET_BIT(word + i);
    eprintf("Word is at %i (%x) %i\n", 
	    (int) word + i,
	    (int) heap->pi32Block[(word + i) / 32 + heap->i32YoungHeapBFSize],
	    (int) ((word + i) / 32 + heap->i32YoungHeapBFSize));
  }
  word = 0;
  OLD_SET_BIT(word);
  if (IS_OLD_BIT_SET(word)) {
    eprintf("Test 1 passed\n");
  }
  OLD_CLEAR_BIT(word);
  if (IS_OLD_BIT_SET(word) == 0) {
    eprintf("Test 2 passed\n");
  }

  eprintf("Word is at %i (%x) %i\n", 
	  word, 
	  (int) (heap->pi32Block[word / 32 + heap->i32YoungHeapBFSize]), 
	  (int) ((word) / 32 + heap->i32YoungHeapBFSize));
  OLD_SET_BIT(word);
  eprintf("Word is at %i (%x) %i\n", 
	  (int) word, 
	  (int) (heap->pi32Block[word / 32 + heap->i32YoungHeapBFSize]), 
	  (int) ((word) / 32 + heap->i32YoungHeapBFSize));
  
  return 0;
}


int OLDHEAP_SweepAndCompact(tAllocatorHeap* heap)
{
  int32       i, j;
  int32*      pi32NewTop;
  tObject*    pstOldLoc;     /* object's old location */
  tObject*    pstNewLoc;     /* object's new location */
  int x;
  int iCopySize, iSpaceSize;
#ifdef DEBUG
  int32 nosFreed = 0;
  int32 nosKept = 0;
#endif

  traceGC0("OLDHEAP_SweepAndCompact: started");
#ifdef GARBAGE_VERBOSE
  traceGC("Before sweep: %d regular handles are free", 
	  HANDLES_nosFreeHandles(heap));
#endif


  /* set "new top" of compacted heap to base of heap */
  pi32NewTop = heap->pi32OldHeap;

  /* go through bitfield of this heap block */
  for (i = 0; i < (heap->i32OldHeapBFSize + 1); i++) {
    /* check if there are any bits set in this int32 */
    if (IS_ANY_OLD_BIT_SET(i * 32)) {
      /* find the set bit(s) */
      for (j = 0; j < 32; j++) {
	if (IS_OLD_BIT_SET(i * 32 + j)) {
#ifdef DEBUG
	  assert(i >= 0 && j >= 0 &&
		 pi32NewTop >= heap->pi32OldHeap &&
		 pi32NewTop <= heap->pi32OldHeap + heap->i32OldHeapSize);
#endif
	  pstOldLoc = (tObject*) (&heap->pi32OldHeap[32 * i + j]);

#ifdef DEBUG_OBJECT_FOR_GC
	  //check the magic flags
	  INTERP_CheckObject(pstOldLoc);
#endif
	  /* check if it's been marked as reachable  - if so, let it live
	     and unset the reachable flag */
	  if (pstOldLoc->i32Flags & GARBAGE_REACHABILITY_MASK) {
	    /* unflag */
	    pstOldLoc->i32Flags &= ~GARBAGE_REACHABILITY_MASK;
	    
	    /* check if we need to move this object - we move it if there
	       is space between it and the "new top" of the compacted
	       objects (which should be all cases after the first space
	       has been made) */
#ifdef DEBUG
	    assert((((int32*) pstOldLoc) >= heap->pi32OldHeap));
	    assert((((int32*) pstOldLoc) <= (heap->pi32OldHeap + 
					     heap->i32OldHeapSize)));
	    nosKept++;
#endif
	    if (((int32*) pstOldLoc) - 1 != pi32NewTop) {
	      /* clear bit for old position in bitfield */
	      OLD_CLEAR_BIT(i * 32 + j);
	      /* move to new top */
	      /* quick check to see if they overlap - ie if size of 
		 object > size of free space */
	      iCopySize = (((int32*) pstOldLoc) - 1)[0];
#ifdef DEBUG
	      assert(iCopySize >= 0 && iCopySize < heap->i32OldHeapSize);
#endif
	      iSpaceSize = ((((int32*) pstOldLoc) - 1) - pi32NewTop);
#ifdef DEBUG      
	      assert(iSpaceSize >= 0 && iSpaceSize < heap->i32OldHeapSize);
#endif
	      if (iCopySize < iSpaceSize) {
		/* no overlap... can use memcpy */
		memcpy(pi32NewTop, ((int32*) pstOldLoc) - 1, 
		       iCopySize * sizeof(int32));
		
		//test
		//zero the old image
		memset(((int32*) pstOldLoc) - 1, 0, 
		       iCopySize * sizeof(int32));
	      }
	      else {
		/* otherwise do it manually */
		for (x = 0; x < iCopySize; x++) {
		  pi32NewTop[x] = (((int32*) pstOldLoc) - 1)[x];
		  //zero old image
		  (((int32*) pstOldLoc) - 1)[x] = 0;
		}
	      }
	      /* set bit for new position in bitfield (+1 coz object starts one
		 slot after newtop coz the size gets stored there) */
	      OLD_SET_BIT(((pi32NewTop + 1) - heap->pi32OldHeap)) ;
	      
	      /* store new location for convenience */
	      pstNewLoc = (tObject*) (pi32NewTop + 1);
#ifdef DEBUG
	      assert((((int32*) pstNewLoc) >= heap->pi32OldHeap));
	      assert((((int32*) pstNewLoc) <= (heap->pi32OldHeap + 
					       heap->i32OldHeapSize)));
#endif

	      /* change handle */
#ifndef PERSIST
	      *(pstNewLoc->hHandle) = pstNewLoc;
#else
	      if (pstNewLoc->pid == 0) {
		/* only change handle here if it's not persistent */
		*(pstNewLoc->hHandle) = pstNewLoc;
	      } 
	      else {
		/* Update a persistent object's RPOT entry to match
                   the new object location, and replace the object's
                   handle pointer ('cos the RPOT entry may change).
                   Also free the old handle (if any) left over from
                   the object's promotion. */
		RPOT_ChangeLA2(heap, pstNewLoc->hHandle, pstNewLoc);
	      }
#endif	      

	      /* change variable / elements pointer */
	      if ((pstNewLoc->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT) {
		pstNewLoc->pi32Vars = 
		  (int32*) (((byte*) pstNewLoc) + ROUND32BIT(sizeof(tObject)));
	      }
	      else {
		((tArray*) pstNewLoc)->pvElements = 
		  (void*) (((byte*) pstNewLoc) + ROUND32BIT(sizeof(tArray)));
	      }

#ifdef IN_PLACE_CONDITION
	      /* change the In place) condVariable pointer */
	      pstNewLoc->condVariable = &(pstNewLoc->actualCondVariable);
#endif      

#ifdef DEBUG_OBJECT_FOR_GC
	      INTERP_CheckObject(pstNewLoc);
#endif
	    } //ends check for moving
            else {
#ifdef DEBUG_OBJECT_FOR_GC
	      INTERP_CheckObject(pstOldLoc);
#endif
	    }
	    
	  
	    /* change new top by adding size of object that is after it... this
	       might be an object that has just been moved there or it might be
	       an object that was always there (if no objects have been thrown
	       away yet) */
#ifdef DEBUG
	    assert(pi32NewTop[0] > 0);
#endif
	    pi32NewTop += pi32NewTop[0];
	  }
	  else {
	    /* it's unreachable so we throw it away */
	    /* clear bit in bitfield */
	    OLD_CLEAR_BIT(i * 32 + j);
	    
	    /* XXXX - should we run finalize method if there is one ? */
	    pstOldLoc = (tObject*) &(heap->pi32OldHeap[32 * i + j]);
#ifdef DEBUG
	    nosFreed++;
	    assert((((int32*) pstOldLoc) >= heap->pi32OldHeap) && 
		   (((int32*) pstOldLoc) <= (heap->pi32OldHeap + 
					     heap->i32OldHeapSize)));
	    assert(pstOldLoc->pstType != NULL);
#endif
	    if (strcmp(pstOldLoc->pstType->uidName, "java/lang/Class") == 0) {
	      tJNIData* pstJNIData;
	      tClassLoaderTuple* tuple;
	      pstJNIData = JNI_getJNIData(sys_current_thread());
	      tuple = CLASS_GetClassStruct(&pstJNIData->functions, 
					   pstOldLoc->hHandle);
	      
	      traceGC("GC'ing class object %p for class %s (%p)", 
		      pstOldLoc->hHandle, tuple->uidName, tuple->classLoader);
	    }
	    /* free the condition variable */
#ifdef IN_PLACE_CONDITION
#else
	    sys_condition_destroy(pstOldLoc->condVariable);
#endif
	    pstOldLoc->condVariable = NULL;
#ifdef DEBUG_OBJECT_FOR_GC
	    INTERP_CheckObject(pstOldLoc);
#endif
	    
	    //eprintf("Garbage collecting object %p (%s)\n", 
            //        pstOldLoc->hHandle, pstOldLoc->pstType->uidName);
	    //GARBAGE_VerifyObject( pstTempHB, pstOldLoc );
	    
#ifdef PERSIST
	    /* We cannot stabilise a dirty object here because that
	       would put an incomplete snapshot into the store.  [If
	       we must garbage collect a dirty persistent object, it
	       must be written to a shadow store.]  Actually, any
	       dirty persistent object must be in the RPOT, and 
	       should be marked as reachable. -- Steve C 2002-05-12 */
	    if (pstOldLoc->pid) {
	      assert((heap->pi32OldHeap[32 * i + j] & PERSIST_DirtyMask) == 0);
	      /*
		 This is wrong ...
		 RPOT_Stabilise(pstOldLoc); */
	      /* It should be safe to remove a clean persistent object
		 that is not currently reachable via the RPOT.  Just make
		 sure that we remove the RPOT entry.
                 -- Steve C 2002-05-12 */
	      RPOT_Remove(pstOldLoc->pid);
	    }
#endif
	    
	    /* Free the object's handle for reuse.  Only remove handle
	       if it has no pid, because if it has a pid then it was
	       using the LA field of its RPOT entry as a handle and
	       did not have handle in the handle table */
#ifdef PERSIST
	    if (pstOldLoc->pid == 0) {
	      HANDLES_ReleaseHandle(heap, pstOldLoc->hHandle);
	    }
#else
	    HANDLES_ReleaseHandle(heap, pstOldLoc->hHandle);
#endif
	  }
	}
      }
    }
  }
  
  heap->pi32OldFirstFree = pi32NewTop;
  pi32NewTop[0] = &(heap->pi32OldHeap[heap->i32OldHeapSize - 1]) - pi32NewTop;
  pi32NewTop[1] = (int) NULL;
  
  //try and manually zero all the data above NewTop
  {
    // XXX - use memset
    int32* pointer = pi32NewTop + 1; 
    while (pointer < (heap->pi32OldHeap + heap->i32OldHeapSize)) {
      *pointer++ = 0;
    }
  }
#ifdef DEBUG
  assert(pi32NewTop[0] >= 0);
#endif

  traceGC0("OLDHEAP_SweepAndCompact: finished");
#ifdef GARBAGE_VERBOSE
  traceGC("After sweep: %d regular handles are now free", 
	  HANDLES_nosFreeHandles(heap));
  traceGC("%d objects freed, %d objects retained", nosFreed, nosKept);
#endif
  return 0;
}



#endif
