/*
 * @doc MODULE
 * @module hash.h |
 *
 * Declarations for hash tables - same as Tcl
 *
 */

#ifndef __HASH_H__
#define __HASH_H__

#ifndef BUFSIZ
#include <stdio.h>
#endif


/*
 * Definitions that allow this header file to be used either with or
 * without ANSI C features like function prototypes.
 */

#ifdef __cplusplus
#define EXTERN extern "C"
#else
#define EXTERN extern
#endif

#ifndef _CLIENTDATA
#ifdef __STDC__
typedef void *ClientData;
#else
typedef int32 *ClientData;
#endif /* __STDC__ */
#define _CLIENTDATA
#endif

/*
 * Forward declaration of HASH_tstHashTable.  Needed by some C++ compilers
 * to prevent errors when the forward reference to HASH_tstHashTable is
 * encountered in the HASH_tstHashEntry structure.
 */

#ifdef __cplusplus
struct HASH_tsthashtable;
#endif

/*
 * @doc TYPE
 * @struct HASH_tstHashEntry |
 * Structure definition for an entry in a hash table.  No-one outside
 * Tcl should access any of these fields directly;  use the macros
 * defined below.
 */

typedef struct HASH_tstHashEntry {
  struct HASH_tstHashEntry *pstNext;     /* @field Pointer to next entry in 
					    this hash bucket, or NULL for end 
					    of chain. */
  struct HASH_tstHashTable *pstTable;	 /* @field Pointer to table containing
					    entry. */
  struct HASH_tstHashEntry **ppstBucket; /* @field Pointer to bucket that
					    points to first entry in this 
					    entry's chain: used for deleting
					    the entry. */
  ClientData pvClientData;		 /* @field Application stores something
					    here with HASH_SetHashValue. */
  union {				 /* Key has one of these forms: */
    char *pchOneWordValue;		 /* @field One-word value for key. */
    int32 ai32Words[1];		         /* @field Multiple integer words for
					    key.  The actual size will be as 
					    large as necessary for this table's
					    keys. */
    char achString[4];		         /* @field String for key.  The actual
					    size will be as large as needed to
					    hold the key. */
  } uniKey;				 /* MUST BE LAST FIELD IN RECORD!! */
} HASH_tstHashEntry;


/*
 * @doc TYPE
 * @struct HASH_tstHashTable |
 * Structure definition for a hash table.  Must be in tcl.h so clients
 * can allocate space for these structures, but clients should never
 * access any fields in this structure.
 */

#define HASH_SMALL_HASH_TABLE 4
typedef struct HASH_tstHashTable {
  HASH_tstHashEntry **papstBuckets;	 /* @field Pointer to bucket array.
					    Each element points to first entry
					    in bucket's hash chain, or NULL. */
  HASH_tstHashEntry *apstStaticBuckets[HASH_SMALL_HASH_TABLE]; 
                                         /* @field Bucket array used for small
					    tables (to avoid mallocs/frees). */
  int32 i32NumBuckets;			 /* @field Total number of buckets 
					    allocated at **papstBuckets. */
  int32 i32NumEntries;			 /* @field Total number of entries 
					    present in table. */
  int32 i32RebuildSize;                  /* @field Enlarge table when 
					    i32NumEntries gets to be this 
					    large. */
  int32 i32DownShift;			 /* @field Shift count used in hashing
					    function.  Designed to use high-
					    order bits of randomized keys. */
  int32 i32Mask;			 /* @field Mask value used in hashing
					    function. */
  int32 i32KeyType;			 /* @field Type of keys used in this 
					    table.  It's HASH_STRING_KEYS,
					    HASH_ONE_WORD_KEYS, or an integer
					    giving the nos of ints in a key. */
  HASH_tstHashEntry *(*pfFindProc)(struct HASH_tstHashTable *pstHashTable,
				   char *uniKey);
                                         /* @field pfFindProc | fptr | ... */
  HASH_tstHashEntry *(*pfCreateProc)(struct HASH_tstHashTable *pstHashTable,
				     char *key, int32 *newPtr); 
                                         /* @field pfFindProc | fptr | ... */
} HASH_tstHashTable;


/*
* @doc TYPE
 * @struct HASH_tstHashSearch |
 * Structure definition for information used to keep track of searches
 * through hash tables:
 */

typedef struct HASH_tsthashsearch {
  HASH_tstHashTable *pstTable;     /* @field Table being searched. */
  int32 i32NextIndex;	           /* @field Index of next bucket to be
				      enumerated after present one. */
  HASH_tstHashEntry *pstNextEntry; /* @field Next entry to be enumerated in the
				      the current bucket. */
} HASH_tstHashSearch;

/*
 * Acceptable key types for hash tables:
 */

/*
 * Use the string passed as the string entry.
 */
#define HASH_STRINGPTR_KEYS 0xFFFFFFFF
/*
 * Make a copy the string in the hash entry.
 */
#define HASH_STRING_KEYS		0
#define HASH_ONE_WORD_KEYS	1

/*
 * Macros for clients to use to access fields of hash entries:
 */

#define HASH_GetHashValue(h) ((h)->pvClientData)
#define HASH_SetHashValue(h, value) ((h)->pvClientData = (ClientData) (value))
#define HASH_GetHashKey(pstHashTable, h) \
  ((char *) (((pstHashTable)->i32KeyType == HASH_ONE_WORD_KEYS) ? \
   (h)->uniKey.pchOneWordValue : (h)->uniKey.achString))

/*
 * Macros to use for clients to use to invoke find and create procedures
 * for hash tables:
 */

#define HASH_FindHashEntry(pstHashTable, uniKey) \
  (*((pstHashTable)->pfFindProc))(pstHashTable, uniKey)
#define HASH_CreateHashEntry(pstHashTable, key, newPtr) \
  (*((pstHashTable)->pfCreateProc))(pstHashTable, key, newPtr)

/*
 * Exported Hash procedures:
 */

EXTERN void HASH_DeleteHashEntry(HASH_tstHashEntry *pstEntry);
EXTERN void HASH_DeleteHashTable(HASH_tstHashTable *pstHashTable);
EXTERN HASH_tstHashEntry* HASH_FirstHashEntry(HASH_tstHashTable *pstHashTable,
					      HASH_tstHashSearch *pstSearch);
EXTERN char* HASH_HashStats(HASH_tstHashTable *pstHashTable);
EXTERN void HASH_InitHashTable(HASH_tstHashTable *pstHashTable,	
			       int32 i32KeyType);
EXTERN HASH_tstHashEntry* HASH_NextHashEntry(HASH_tstHashSearch *pstSearch);
EXTERN void HASH_IntegrityCheck(HASH_tstHashTable *pstTable);


#endif /* __HASH_H__ */




