/*
 * RPOT.h
 *
 * Header for resident persistent object table (RPOT).  The RPOT is a
 * hash table that maps PIDs to in-memory object addresses.  Associated
 * methods manage the table, and perform various tasks such as promoting
 * non-persistent object to persistent, stabilising persistent objects, 
 * and faulting them into memory.
 */

#ifndef __RPOT_H__
#define __RPOT_H__

#include "vm/store.h"
#include "vm/interp.h"

#define RPOT_TABLESIZE 0x10000
#define RPOT_OVERFLOWSIZE RPOT_TABLESIZE
#ifdef EVICTIONS
#define TPOT_TABLESIZE    RPOT_TABLESIZE
#endif

#define HASHFUNC(x) (((-(x)) & (0xfff << 2)) >> 2)

#ifdef REVRPOT
#define REVHASHFUNC(x) (((int32)(x)) & 0xfff)
#endif


typedef struct trpotelement
{
  /* the 'la' field must be the first field */
  tObject* la;                 /* The local address (la) of an in-memory 
				  persistent object. The slot also serves as 
				  the in-memory object's primary handle. */
  tOBREF oldHandle;            /* The old handle for the object that has been
				  promoted. Its contents should be kept up
				  to date until the garbage collector runs */
  PID pid;                     /* A persistent object's PID */
  struct trpotelement* next;   /* The next tRPOTElement in the hash chain. */
#ifdef REVRPOT
  struct trpotelement* revnext;
#endif
} tRPOTElement;

#ifdef REVRPOT
extern tRPOTElement** RPOT_table;
extern tRPOTElement** RPOT_revtable;
#else
extern tRPOTElement* RPOT_table;
#endif

#ifdef EVICTIONS
extern tRPOTElement* TPOT_table;
#endif

/* for initialising and closing down the table */
void RPOT_Init(void);
void RPOT_Kill(void);
#ifdef EVICTIONS
void TPOT_Init(void);
void TPOT_Kill(void);
#endif

/* for using the table */
tObject* RPOT_GetLA(PID pid);
#ifdef REVRPOT
PID RPOT_GetPID(void* la);
#endif
tRPOTElement* RPOT_Add(PID pid, void* la);
void RPOT_Remove(PID pid);
void RPOT_PrintTable(void);
#ifdef EVICTIONS
tOBREF RPOT_GetObref(PID pid);
tObject* TPOT_GetLA(PID pid);
tRPOTElement* TPOT_Add(PID pid, void* la);
void TPOT_Remove(PID pid);
void TPOT_PrintTable(void);
#endif

/* for faulting, promoting and stabilising */
tOBREF RPOT_FaultObject(PID pid);
tOBREF RPOT_ChangeLA(PID pid, void* newLA);
tOBREF RPOT_ChangeLA2(tAllocatorHeap* heap, tOBREF h, void* newLA);
#ifdef EVICTIONS
tOBREF TPOT_ChangeLA(PID pid, void* newLA);
tOBREF RPOT_SwapIn(PID pid);
#endif
PID RPOT_PromoteObject(tObject* la);
void RPOT_UpdateObject(PID pid, tObject* la);
void RPOT_StabiliseAll(void);
void RPOT_Stabilise(tObject* o);

#ifdef RPOTREFS
int RPOT_InRPOT(void*);
#endif

/* macro for checking to see if a reference is a PID */
#ifdef BEOS
#define ISPID(x) (0)
#else
#define ISPID(x) ((((PID)(x)) < 0) ? 1 : 0)
#endif

#endif

