/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <stdio.h>
#include <pthread.h>
#include <string.h>

#include "vm/stdtypes.h"

#include "vm/classfil.h"
#include "vm/loading_errors.h"

#include "vm/jni.h"
#include "vm/classfile_methods.h"
#include "vm/interp_methods.h"
#include "vm/newobject.h"

#include "vm/wrappers.h"

#include "vm/lib/minizip/unzip.h"
#include <stdio.h>

#include "lib/zip/zipfile.h"

#include "vm/stdtypes.h"
#include "vm/hash.h"

static HASH_tstHashTable zipFileCache;
static BOOL cacheInitialised = FALSE; 

#define UNZ_OK (0)


extern int showLoading;  // declared in kissme_main.c
static int iHaveBootstrapped = 0;


typedef enum tjavaloadertype {
  JAR_LOADER_TYPE = 1,
  ZIP_LOADER_TYPE,
  HTTP_LOADER_TYPE,
  FS_LOADER_TYPE
} JavaLoaderType;


static tClassLoaderTuple* getFromByteArray(JNIEnv* env, byte* pbData, 
					   char* pszFileName, 
					   uint32* errorCode, 
					   pthread_mutex_t* classfilMutex);
static tClassLoaderTuple* getFromFileSystem(JNIEnv* env, char* pszFileName, 
					    char* pszClassPath,
					    uint16 u16ClassPathLen, 
					    uint32* errorCode,
					    pthread_mutex_t* classfilMutex);
static tClassLoaderTuple* getFromJavaLoader(JNIEnv* env, JavaLoaderType type, 
					    char* pszFileName,
					    char* pszClassPath, 
					    uint16 u16ClassPathLen,
					    uint32* errorCode, 
					    pthread_mutex_t* classfilMutex);
static tClassLoaderTuple* getFromZipLoader(JNIEnv* env, JavaLoaderType type,
					   char* pszFileName,
					   char* pszClassPath, 
					   uint16 u16ClassPathLen,
					   uint32* errorCode,
					   pthread_mutex_t* classfilMutex);


void setBootstrapped()
{
  iHaveBootstrapped = 1;
}


byte* getDataFromZipLoader(JNIEnv* env, JavaLoaderType type, 
			   char* pszFileName, char* pszClassPath, 
			   uint16 u16ClassPathLen, uint32* errorCode);
byte* getDataFromFileSystem(JNIEnv* env, JavaLoaderType type, 
			    char* pszFileName, char* pszClassPath, 
			    uint16 u16ClassPathLen, uint32* errorCode);

byte* getClassData(JNIEnv* env, char* pszFileName, uint32* errorCode)
{
  int u16PathLen;
  int i = 0;
  byte* classdata = NULL;

  if (CLASSFILE_BootClassPath.u16NumClassPaths == 0) {
    *errorCode = LOADING_ERROR_NO_FILE;
    return NULL;
  }
  for (i = 0; i < CLASSFILE_BootClassPath.u16NumClassPaths; i++) {
    int try_file_system = 1;
    
    u16PathLen = strlen(CLASSFILE_BootClassPath.ppszClassPaths[i]);
    /* decide whether to use the local filesystem or a jar/zip file or
       http loader */
    if (u16PathLen > 4) { //must be greater than 4 for a .jar or .zip file 
      if (strcmp(CLASSFILE_BootClassPath.ppszClassPaths[i] + (u16PathLen - 4),
		 ".jar") == 0) {
	/* it is a jar file */
	try_file_system = 0;
	classdata =
	  getDataFromZipLoader(env, ZIP_LOADER_TYPE, pszFileName,
			       CLASSFILE_BootClassPath.ppszClassPaths[i], 
			       u16PathLen, errorCode);
	if (classdata != NULL) {
	  *errorCode = 0;
	  return classdata;
	}
      }
      else if (strcmp(CLASSFILE_BootClassPath.ppszClassPaths[i] + 
		      (u16PathLen - 4), 
		      ".zip") == 0) {
	/* it is a zip file */
	try_file_system = 0;
	
	classdata = 
	  getDataFromZipLoader(env, ZIP_LOADER_TYPE, pszFileName, 
			       CLASSFILE_BootClassPath.ppszClassPaths[i], 
			       u16PathLen, errorCode);
	if (classdata != NULL) {
	  *errorCode = 0;
	  return classdata;
	}
      }
    }
    if (try_file_system != 0) {
      classdata = 
	getDataFromFileSystem(env, FS_LOADER_TYPE, pszFileName, 
			      CLASSFILE_BootClassPath.ppszClassPaths[i],
			      u16PathLen, errorCode);
      if (classdata != NULL) { 
	*errorCode = 0;
	return classdata;
      }
    }
  }
  
  if (*errorCode == 0) {
    *errorCode = LOADING_ERROR_NO_FILE;
  }
  return NULL;
}


static tClassLoaderTuple* getFromZipLoader(JNIEnv* env, JavaLoaderType type,
					   char* pszFileName,
					   char* pszClassPath,
					   uint16 u16ClassPathLen,
					   uint32* errorCode,
					   pthread_mutex_t* classfilMutex)
{
  const char *zipfilename = pszClassPath; 
  char extractFileName[256];
  int success = 0;
  ZipFile* zipf;
  char filename_inzip[256];
  byte* pbData = NULL;
  tClassLoaderTuple* pstClass = NULL;
  
  unz_file_info file_info;
  int uncompressed_size = 0;
  
  HASH_tstHashEntry* lookup;
  int32 i32Dummy; 

  traceLoading("Trying native ZIP loader to load %s from %s",
	       pszFileName, pszClassPath);

  if (cacheInitialised == FALSE) {
    HASH_InitHashTable(&zipFileCache, HASH_STRING_KEYS); 
    cacheInitialised = TRUE;
  }
  
  lookup = HASH_FindHashEntry(&zipFileCache, (void*) zipfilename);
  if (lookup == NULL) {
    /* Open up the main zipfile */
    zipf = opendir_in_zip(zipfilename, 0);
    if (zipf == NULL) {
      traceLoading("Could not open zip file %s", zipfilename);
      *errorCode = LOADING_ERROR_NOT_ZIPFILE;
      HASH_SetHashValue(HASH_CreateHashEntry(&zipFileCache, 
					     (void*) zipfilename, &i32Dummy),
			NULL);
      return NULL; //failure
    }
    else {
      HASH_SetHashValue(HASH_CreateHashEntry(&zipFileCache, 
					     (void*) zipfilename, &i32Dummy), 
			zipf);
      
      /* Build the directory cache */
      {
	ZipDirectory* zipd;
	int i;
	
	HASH_InitHashTable(&zipf->zipDirectoryCache, HASH_STRING_KEYS); 
	zipd = (ZipDirectory*) zipf->central_directory;
	
	for (i = 0; i < zipf->count; i++, zipd = ZIPDIR_NEXT (zipd)) {
	  char savebyte = 
	    (((char*) zipd) + zipd->filename_offset)[zipd->filename_length];
	  (((char*) zipd) + zipd->filename_offset)[zipd->filename_length] = 
	    '\0';
	  HASH_SetHashValue(HASH_CreateHashEntry(&zipf->zipDirectoryCache, 
						 (void*) ZIPDIR_FILENAME(zipd),
						 &i32Dummy), 
			    zipd);
	  // eprintf("Adding %s\n", ZIPDIR_FILENAME(zipd));
	  (((char*) zipd) + zipd->filename_offset)[zipd->filename_length] = 
	    savebyte;
	  
	  // eprintf("%d: size:%d, name(#%d)%s, offset:%d\n",
	  //         i, zipd->size, zipd->filename_length,
	  //         ZIPDIR_FILENAME (zipd),
	  //         zipd->filestart);
	}
      }
    }
  }
  else {
    zipf = HASH_GetHashValue(lookup);
    if (zipf == NULL) {
      traceLoading("Could not open zip file %s (cached)", zipfilename);
      *errorCode = LOADING_ERROR_NOT_ZIPFILE;
      return NULL; //failure
    }
  }
  
  /* Locate this class file inside that zip file */
  
  strcpy(extractFileName, ""); 
  strcat(extractFileName, pszFileName);
  strcat(extractFileName, ".class");

  {
    HASH_tstHashEntry* lookupFile;
    byte* pbData;
    byte* pbDataEnd;
    ZipDirectory* zipd;

    pthread_mutex_lock( classfilMutex );
    lookupFile = HASH_FindHashEntry(&zipf->zipDirectoryCache, 
				    (void*) extractFileName);
    if (lookupFile == NULL) {
      traceLoading("file %s not found in ZIP file %s",
		   extractFileName, zipfilename);
      *errorCode = LOADING_ERROR_NO_FILE;
      pthread_mutex_unlock( classfilMutex );
      return NULL;
    }
      
    zipd = HASH_GetHashValue(lookupFile);
    if (zipd->compression_method == Z_NO_COMPRESSION) {
      pbData = sys_malloc(zipd->size);
      if (pbData == NULL) {
	nonfatalError("malloc(%i) failed in zip_classloader",
		      uncompressed_size);
	*errorCode = LOADING_ERROR_OUT_OF_MEMORY;
	pthread_mutex_unlock( classfilMutex );
	return NULL; //failure
      }
      if (lseek (zipf->fd, zipd->filestart, 0) < 0 || 
	  read (zipf->fd, pbData, zipd->size) != (long) zipd->size) {
	*errorCode = LOADING_ERROR_ZIP_ERROR;
	sys_free(pbData);
	pthread_mutex_unlock( classfilMutex );
	return NULL;
      }
    }
    else {
      byte* compressedBuffer;
      z_stream d_stream; /* decompression stream */
      d_stream.zalloc = (alloc_func) 0;
      d_stream.zfree = (free_func) 0;
      d_stream.opaque = (voidpf) 0;
      
      pbData = sys_malloc(zipd->uncompressed_size);
      if (pbData == NULL) {
	nonfatalError("malloc(%i) failed in zip_classloader", 
		      uncompressed_size);
	*errorCode = LOADING_ERROR_OUT_OF_MEMORY;
	pthread_mutex_unlock( classfilMutex );
	return NULL; //failure
      }
      
      d_stream.next_out = pbData;
      d_stream.avail_out = zipd->uncompressed_size;
      compressedBuffer = sys_malloc(zipd->size);
      if (compressedBuffer == NULL) {
	nonfatalError("malloc(%i) failed in zip_classloader", 
		      uncompressed_size);
	*errorCode = LOADING_ERROR_OUT_OF_MEMORY;
	sys_free(pbData);
	pthread_mutex_unlock( classfilMutex );
	return NULL; //failure
      }
      d_stream.next_in = compressedBuffer;
      d_stream.avail_in = zipd->size;
      if (lseek (zipf->fd, zipd->filestart, 0) < 0 || 
	  read (zipf->fd, compressedBuffer, zipd->size) != (long) zipd->size)
	{
	  *errorCode = LOADING_ERROR_ZIP_ERROR;
	  sys_free(compressedBuffer);
	  sys_free(pbData);
	  pthread_mutex_unlock( classfilMutex );
	  return NULL;
	}
      /* Handle NO_HEADER using undocumented zlib feature.
	 This is a very common hack.  */
      inflateInit2 (&d_stream, -MAX_WBITS);
      inflate (&d_stream, Z_NO_FLUSH);
      inflateEnd (&d_stream);
      sys_free(compressedBuffer);
      //XXX check error codes?
    }
    
    /* Get the Java class */
    pthread_mutex_lock( classfilMutex);
    //XXX this lock is held!!!
    
    // eprintf("Doing CLASS_Load for %s, data at %p\n", pszFileName, pbData);
    pstClass = CLASSFILE_ClassCreate(env, pbData, pszFileName, 
				     NULL, errorCode); //boot class loader
    if (pstClass == NULL) {
      // XXX - should throw an exception
    }
    else {
      CLASSFILE_CompleteClass(env, pstClass);
    }

    //lock a second time
    pthread_mutex_unlock(classfilMutex);
  
    if (pstClass != NULL) {
      if (showLoading) {
	eprintf("[Loaded %s from %s]\n", pszFileName, pszClassPath);
      }
    }

    sys_free(pbData);    
    pthread_mutex_unlock(classfilMutex);
    return pstClass;

  }
}


byte* getDataFromFileSystem(JNIEnv* env, JavaLoaderType type,
					   char* pszFileName,
					   char* pszClassPath,
					   uint16 u16ClassPathLen,
				  uint32* errorCode)
{
  char* pszFullPath;
  FILE* fp;
  int iFileSize;
  byte* pbData;
  int read;

  traceLoading("Trying native FileSystem loader to load %s from %s",
	       pszFileName, pszClassPath);
  
  pszFullPath = sys_malloc(u16ClassPathLen + strlen(pszFileName) + 10);
  if (pszFullPath == NULL) {
    *errorCode = LOADING_ERROR_OUT_OF_MEMORY;
    return NULL;
  }
  strcpy(pszFullPath, pszClassPath);
  if (pszFullPath[strlen(pszFullPath) - 1] != '/') {
    strcat(pszFullPath, "/");
  }
  strcat(pszFullPath, pszFileName);
  strcat(pszFullPath, ".class");
  
  fp = fopen(pszFullPath, "r");
  
  if (fp != NULL) {
    if (showLoading) {
      eprintf("[Loading %s]\n", pszFullPath);
    }
    sys_free(pszFullPath);
    
    /* We have found the binary data for the class, read it */
    fseek(fp, 0, SEEK_END);
    iFileSize = ftell(fp);
    fseek(fp, 0, SEEK_SET);
    pbData = (byte*) sys_malloc(iFileSize);
    assert(pbData);
    read = fread(pbData, iFileSize, 1, fp);
    assert(read == 1);
    fclose(fp);
    
    return pbData;
  }
  else {
    traceLoading("couldn't open %s", pszFullPath);
    sys_free(pszFullPath);
    return NULL;
  }
}


byte* getDataFromZipLoader(JNIEnv* env, JavaLoaderType type,
			   char* pszFileName,
			   char* pszClassPath,
			   uint16 u16ClassPathLen,
			   uint32* errorCode)
{
  const char *zipfilename = pszClassPath; 
  char extractFileName[256];
  int success = 0;
  ZipFile* zipf;
  char filename_inzip[256];
  byte* pbData = NULL;
  
  unz_file_info file_info;
  int uncompressed_size = 0;
  
  HASH_tstHashEntry* lookup;
  int32 i32Dummy; 

  traceLoading("Trying native ZIP loader to load %s from %s",
	       pszFileName, pszClassPath);

  if (cacheInitialised == FALSE) {
    traceLoading0("Initializing hash table for ZIP file cache");
    HASH_InitHashTable(&zipFileCache, HASH_STRING_KEYS); 
    cacheInitialised = TRUE;
  }
  
  lookup = HASH_FindHashEntry(&zipFileCache, (void*) zipfilename);
  if (lookup == NULL) {
    /* Open up the main zipfile */
    zipf = opendir_in_zip(zipfilename, 0);
    if (zipf == NULL) {
      traceLoading("Could not open ZIP file %s", zipfilename);
      *errorCode = LOADING_ERROR_NOT_ZIPFILE;
      HASH_SetHashValue(HASH_CreateHashEntry(&zipFileCache, 
					     (void*) zipfilename, &i32Dummy), 
			NULL);
      return NULL; //failure
    }
    else {
      traceLoading("Caching directory info for ZIP file %s", zipfilename);
      HASH_SetHashValue(HASH_CreateHashEntry(&zipFileCache, 
					     (void*) zipfilename, &i32Dummy), 
			zipf);
      
      /* Build the directory cache */
      {
	ZipDirectory* zipd;
	int i;
	
	HASH_InitHashTable(&zipf->zipDirectoryCache, HASH_STRING_KEYS); 
	zipd = (ZipDirectory*) zipf->central_directory;
	
	for (i = 0; i < zipf->count; i++, zipd = ZIPDIR_NEXT (zipd)) {
	  char savebyte = 
	    (((char*) zipd) + zipd->filename_offset)[zipd->filename_length];
	  (((char*) zipd) + zipd->filename_offset)[zipd->filename_length] =
	    '\0';
	  HASH_SetHashValue(HASH_CreateHashEntry(&zipf->zipDirectoryCache, 
						 (void*) ZIPDIR_FILENAME(zipd),
						 &i32Dummy), 
			    zipd);
	  // eprintf("Adding %s\n", ZIPDIR_FILENAME(zipd));
	  (((char*) zipd) + zipd->filename_offset)[zipd->filename_length] = 
	    savebyte;
	  
	  // eprintf ("%d: size:%d, name(#%d)%s, offset:%d\n",
	  //          i, zipd->size, zipd->filename_length,
	  //          ZIPDIR_FILENAME (zipd),
	  // zipd->filestart);
	}
      }
    }
  }
  else {
    zipf = HASH_GetHashValue(lookup);
    if (zipf == NULL) {
      traceLoading("Could not open zip file %s (cached)", zipfilename);
      *errorCode = LOADING_ERROR_NOT_ZIPFILE;
      return NULL; //failure
    }
  }
  
  /* Locate this class file inside that zip file */
  
  strcpy(extractFileName, ""); 
  strcat(extractFileName, pszFileName);
  strcat(extractFileName, ".class");

  {
    HASH_tstHashEntry* lookupFile;
    byte* pbData;
    byte* pbDataEnd;
    ZipDirectory* zipd;

    lookupFile = HASH_FindHashEntry(&zipf->zipDirectoryCache, 
				    (void*) extractFileName);
    if (lookupFile == NULL) {
      traceLoading("file %s not found in ZIP file %s",
		   extractFileName, zipfilename);
      *errorCode = LOADING_ERROR_NO_FILE;
      //	pthread_mutex_unlock( classfilMutex );
      return NULL;
    }
    
    zipd = HASH_GetHashValue(lookupFile);
    if (zipd->compression_method == Z_NO_COMPRESSION) {
      pbData = sys_malloc(zipd->size);
      if (pbData == NULL) {
	nonfatalError("malloc(%i) failed in zip_classloader",
		      uncompressed_size);
	*errorCode = LOADING_ERROR_OUT_OF_MEMORY;
	//	  pthread_mutex_unlock( classfilMutex );
	return NULL; //failure
      }
      if (lseek(zipf->fd, zipd->filestart, 0) < 0 || 
	  read(zipf->fd, pbData, zipd->size) != (long) zipd->size) {
	*errorCode = LOADING_ERROR_ZIP_ERROR;
	sys_free(pbData);
	//	    pthread_mutex_unlock( classfilMutex );
	return NULL;
      }
    }
    else {
      byte* compressedBuffer;
      z_stream d_stream; /* decompression stream */
      d_stream.zalloc = (alloc_func) 0;
      d_stream.zfree = (free_func) 0;
      d_stream.opaque = (voidpf) 0;
      
      pbData = sys_malloc(zipd->uncompressed_size);
      if (pbData == NULL) {
	nonfatalError("malloc(%i) failed in zip_classloader", 
		      uncompressed_size);
	*errorCode = LOADING_ERROR_OUT_OF_MEMORY;
	//	  pthread_mutex_unlock( classfilMutex );
	return NULL; //failure
      }
      
      d_stream.next_out = pbData;
      d_stream.avail_out = zipd->uncompressed_size;
      compressedBuffer = sys_malloc(zipd->size);
      if (compressedBuffer == NULL) {
	nonfatalError("malloc(%i) failed in zip_classloader", 
		      uncompressed_size);
	*errorCode = LOADING_ERROR_OUT_OF_MEMORY;
	sys_free(pbData);
	//	  pthread_mutex_unlock( classfilMutex );
	return NULL; //failure
      }
      d_stream.next_in = compressedBuffer;
      d_stream.avail_in = zipd->size;
      if (lseek(zipf->fd, zipd->filestart, 0) < 0 || 
	  read(zipf->fd, compressedBuffer, zipd->size) != (long) zipd->size) {
	  *errorCode = LOADING_ERROR_ZIP_ERROR;
	  sys_free(compressedBuffer);
	  sys_free(pbData);
	    //	    pthread_mutex_unlock( classfilMutex );
	    return NULL;
	  }

	/* Handle NO_HEADER using undocumented zlib feature.
	   This is a very common hack.  */
	inflateInit2(&d_stream, -MAX_WBITS);
	inflate(&d_stream, Z_NO_FLUSH);
	inflateEnd(&d_stream);
	sys_free(compressedBuffer);
	//XXX check error codes?
      }

    return pbData;
  }
}


