/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include "vm/jni.h"
#include "vm/kni.h"
#include "vm/kni_methods.h"

#include "vm/interp.h"
#include <pthread.h>

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <stdlib.h>
#endif

#include "vm/garbage.h"

#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>

#include "vm/threads_native.h"
#include "vm/threadinfo.h"
#include "vm/interp_methods.h"
#include "vm/classfile_methods.h"
#include "vm/interp/methodstacks.h"


extern tAllocatorHeap* system_heap;

// this can be globally used to get the type for class
extern tClassLoaderTuple* pstClassType; 

/* Stuff to manipulate the thread objects */

tThreadNode* pstThreadList = NULL; //List of threads
static pthread_mutex_t threadListMutex;

volatile uint32 iLiveThreads = 0; //Number of live threads left 
static pthread_cond_t threadDeathCond = PTHREAD_COND_INITIALIZER;


/* Prototypes */
int pthread_mutexattr_setkind_np(pthread_mutexattr_t* kind, int tkind);

static void THREAD_signalHandler(int signal);


pthread_mutex_t* THREAD_getThreadListMutex()
{
  return &threadListMutex;
}


void THREAD_Init()
{
  pthread_mutexattr_t kind;
  struct sigaction action;
  sigset_t signals;

  traceThreads("THREAD_Init: tid is %i", pthread_self());

  // create the threadListMutex and make it recursive
  pthread_mutexattr_init (&kind);
  pthread_mutexattr_setkind_np(&kind, PTHREAD_MUTEX_RECURSIVE_NP);
  pthread_mutex_init(&threadListMutex, &kind);

  // register a signal handler for the SIGUSR1 signal
  sigemptyset(&signals);
  sigaddset(&signals, SIGUSR1);
  action.sa_handler = THREAD_signalHandler;
  action.sa_mask = signals;
  action.sa_flags = 0;
  if (sigaction(SIGUSR1, &action, NULL) == -1) {
    panic("THREAD_Init: sigaction failed: %s", sys_errlist[errno]);
  }
  if (siginterrupt(SIGUSR1, 1) == -1) {
    panic("THREAD_Init: siginterrupt failed: %s", sys_errlist[errno]);
  }
  if (pthread_sigmask(SIG_UNBLOCK, &signals, NULL) == -1) {
    panic("THREAD_Init: pthread_sigmask failed: %s", sys_errlist[errno]);
  }

  THREAD_initThreadIds();
}


static void THREAD_signalHandler(int signal)
{
  traceThreads("THREAD_signalHandler(%d): tid %i", signal, pthread_self());
}


tThreadNode* THREAD_getThreadList()
{
  return pstThreadList;
}


void THREAD_setThreadList(tThreadNode* nlist)
{
  pstThreadList = nlist;
  return;
}


tThreadNode* THREAD_lockAndGetThreadList()
{
  pthread_mutex_lock(&threadListMutex);
  return pstThreadList;
}


int THREAD_unlockThreadList()
{
  pthread_mutex_unlock(&threadListMutex);
  return 0;
}


int THREAD_GetNumberThreads(tAllocatorHeap* pstHeap)
{
  return iLiveThreads;
}


#ifdef DEBUG
void THREAD_dumpThreads()
{
  eprintf("=====Thread Dump====\n");
  tThreadNode* iter = THREAD_lockAndGetThreadList();

  while (iter != NULL) {
    eprintf("  thread: id %i, pthread %i, object %x, state %i\n",
	    iter->threadId, iter->pthread, iter->object, iter->state);
    eprintf("    interrupted %i, daemon %i, contention %i, waitObj %p\n", 
	    iter->isInterrupted, iter->isDaemon, iter->i32ContentionBit,
	    iter->waitObject);
    iter = iter->next;
  }
  THREAD_unlockThreadList();
  eprintf("====================\n");
}
#endif


/*
 * @doc TYPE
 * @struct
 * This structure contains offsets of all of the relevant Java Thread object's
 * fields. It is used so that the offsets only have to be lookup up once and
 * can be used often.
 */
struct tthreadoffsets
{
  int iGroupOffset;	   /* @field The thread group to which this thread
			      belongs */
  int iTargetOffset;	   /* @field The Runnable target of this thread, if
			      there is one; null otherwise */
  int iNameOffset;	   /* @field The name of the thread. */
};


/*
 * @doc TYPE
 * @type tThreadOffsets | typdef of tthreadoffsets struct
 */
typedef struct tthreadoffsets tThreadOffsets;


/*
 * Structure containing all of the offsets we need.
 */
static tThreadOffsets stThreadOffsets;

tJNIData* THREAD_GetCurrentJNIData(void)
{
  return JNI_getJNIData(pthread_self());
}

/*
 * This is a linked list of locks that have been obtained by various threads
 * threadListMutex protects this structure
 */
struct native_thread_lock* THREAD_locks = NULL;


/*
 * Routine to dump the locks 
 * Assumes that threadListMutex is locked
 */
void dumpLocks()
{
  tThreadLock* iter = THREAD_locks;
  eprintf("Dumping locks from thread %x --------------> \n", 
	  (int) pthread_self());
  while (iter != NULL) {
    eprintf("Object %p was locked by pthread %x, mutex is %x\n", 
	    iter->object, (int) (iter->pthread), (int) &(iter->mutex));
    iter = iter->next;
  }
  eprintf("<--------------- End Dumping locks \n");
}


/* Stuff called from INTERP */

tStackFrame* THREAD_GetNewStackFrame(tStackFrame* pstOldFrame 
				     /* @parm Environment frame of old 
					thread */
				     )
{
  panic0("THREAD_GetNewStackFrame: not implemented");
  return NULL;
}


/*
 * @doc FUNC
 * @func
 * Destroy a thread. This is invoked when a thread has run to completion.
 */
void THREAD_Destroy(tOBREF objThread  /* @parm Thread to destroy */)
{
  tThreadNode* iter, *prev;
  
  if (objThread == NULL) {
    return;
  }
  pthread_mutex_lock(&threadListMutex);
  
  prev = iter = pstThreadList;
  
  assert(pstThreadList);
  while (iter != NULL) {
    if (iter->object == objThread) {
      prev->next = iter->next;
      pthread_mutex_unlock(&threadListMutex);    
      return;
    }
    prev = iter;
    iter = iter->next;
  }
  pthread_mutex_unlock(&threadListMutex);    
  panic("FAILED to DESTROY thread object %p %p", objThread, *objThread);
}


jobject java_lang_Thread_currentThread(JNIEnv* env, jclass thisclass)
{
  tThreadNode* iter;
  pthread_t id;

  pthread_mutex_lock(&threadListMutex);
  iter = pstThreadList;

  // XXX ... this may be wrong: see vm/interp/method_return.h:31
  if ((env == NULL) && (thisclass == NULL)) {
    pthread_mutex_unlock(&threadListMutex);
    return NULL;
  }

  id = (pthread_t) pthread_self();

  assert(pstThreadList);
  while (iter != NULL) {
    if (iter->pthread == id) {
      pthread_mutex_unlock(&threadListMutex);    
      return iter->object;
    }
    iter = iter->next;
  }
  pthread_mutex_unlock(&threadListMutex);

  // This happens when the constructor for the first thread tries to
  // find the (non-existent) current thread to determine its thread
  // group and priority.
  return NULL;
}


/* The java.lang.Thread native methods */ 
void java_lang_Thread__initC(JNIEnv* env, jclass clazz)
{

  tField* fieldoffset;
  tClassLoaderTuple *pstTempClass;

  struct
  {
    char *pszField;
    int *pui32Offset;
  }

  astFields[] =
  {
    {"group", &(stThreadOffsets.iGroupOffset)},
    {"toRun", &(stThreadOffsets.iTargetOffset)},
    {"name",  &(stThreadOffsets.iNameOffset)},
  };
  int iField;

  traceThreads("lang_Thread__initC(%p)", clazz);  
  
  pstTempClass = CLASSFILE_FindOrLoad(env, "java/lang/Thread", pstClassType); //XXX THREAD?
#ifdef DEBUG
  assert(pstTempClass);
#endif
  
  /* find the offsets of the instance variables */
  for (iField = 0; 
       iField < sizeof(astFields) / sizeof(astFields[0]);
       iField++) {
    fieldoffset = CLASSFILE_InstFieldOffset(env, pstTempClass, 
					    astFields[iField].pszField);
#ifdef DEBUG
    assert(fieldoffset);
#endif
    *(astFields[iField].pui32Offset) = fieldoffset->u16Offset;
  }
  
  /* initialise random number generator... used by notify */
  srand((unsigned) time(NULL));
}


void java_lang_Thread_yield(JNIEnv* env, jobject obj)
{
  traceThreads0("lang_Thread_yield()");
  pthread_yield();
}


void java_lang_Thread_nativeResume(JNIEnv* env, jobject obj)
{
  traceThreads("lang_Thread_nativeResume(%p)", obj);
  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", 
							      "Thread.nativeResume not supported in kissme yet"));
}


void java_lang_Thread_sleep(JNIEnv* env, jclass clazz, 
			    jlong millis, jint nanos)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  struct timespec tv;
  tThreadNode* thread;
  int rc;

  traceThreads("lang_Thread_sleep(%p, %ld, %d)", clazz, millis, nanos);

  if (millis < 0) {
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", 
							      "'millis' parameter is -ve"));
    return;
  }
  if (nanos < 0 || nanos > 999999) {
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", 
							      "'nanos' parameter is out of range"));
    return;
  }

  if (millis == 0 && nanos == 0) {
    // The 'sleep forever' case
    tv.tv_sec = 0x7fffffff;
    tv.tv_nsec = 0;
  }
  else {
    tv.tv_sec = millis / 1000;
    tv.tv_nsec = (millis % 1000) * 1000000 + nanos;
  }
  
  thread = THREAD_FindThread();
  assert(thread);

  // XXX - this should be done in a critical region to avoid a potential
  // race with Thread_Interrupt()
  if (thread->isInterrupted) {
    thread->isInterrupted = 0;
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InterruptedException", ""));
    return;
  }
  thread->state = THREAD_STATE_SLEEPING;
  kniPtr->EnterGCRegion(env);
  rc = nanosleep(&tv, NULL);
  kniPtr->ExitGCRegion(env);
  thread->state = THREAD_STATE_RUNNABLE;

  if (rc == -1) {
    if (errno == EINTR) {
      traceThreads("sleep interrupted: waking up");
      thread->isInterrupted = 0;
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InterruptedException", ""));
    }
    else {
      panic("lang_Thread_sleep: nanosleep failed: %s", sys_errlist[errno]);
    }
  }
  else {
    traceThreads("sleep completed: waking up");
  }
}


jboolean java_lang_Thread_isAlive(JNIEnv* env, jobject obj)
{
  tThreadNode* thread;

  traceThreads("lang_Thread_isAlive(%p)", obj);
  thread = THREAD_FindThreadByObject(obj);
  return ((thread != NULL && thread->state != THREAD_STATE_TERMINATED) ? 
	  JNI_TRUE : JNI_FALSE);
}

/*
 * Note: the 'size' parameter is ignored.  Kissme is capable of growing
 * a thread's Java stack.
 */
void java_lang_Thread_nativeInit(JNIEnv* env, jobject obj, jlong size)
{
  traceThreads("lang_Thread_nativeInit(%p, %ld)", obj, size);
  /* Nothing to do here.  Thread state and the stack are allocated and
     initialised when the thread is started. */
}


void java_lang_Thread_nativeSuspend(JNIEnv* env, jobject obj)
{
  traceThreads("lang_Thread_nativeSuspend(%p)", obj);
  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", 
							      "Thread.nativeSuspend not supported in kissme yet"));
}


void java_lang_Thread_nativeStop(JNIEnv* env, jobject obj, jobject throwable)
{
  traceThreads("lang_Thread_nativeStop(%p, %p)", obj, throwable);
  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", 
							      "Thread.nativeStop not supported in kissme yet"));
}


void java_lang_Thread_nativeSetPriority(JNIEnv* env, jobject obj, jint priority)
{
  traceThreads("lang_Thread_nativeSetPriority(%p, %d)", obj, priority);
  // We have to ignore this because Linux only supports one priority
  // value for non-realtime threads.
}


jint java_lang_Thread_countStackFrames(JNIEnv* env, jobject obj)
{
  traceThreads("lang_Thread_countStackFrames(%p)", obj); 
  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", 
							      "Thread.countStackFrames not supported in kissme yet"));
}


jboolean java_lang_Thread_holdsLock(JNIEnv* env, jobject obj, jobject target)
{
  traceThreads("lang_Thread_holdsLock(%p, %p)", obj, target);  
  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", 
							      "Thread.holdsLock not supported in kissme yet"));
}


jboolean java_lang_Thread_interrupted(JNIEnv* env, jobject obj)
{
  jboolean res;
  tThreadNode* thread;
 
  traceThreads0("lang_Thread_interrupted()");

  thread = THREAD_FindThread();
  if (thread == NULL) {
    return JNI_FALSE;
  }

  pthread_mutex_lock(&threadListMutex);

  res = (thread->isInterrupted) ? JNI_TRUE : JNI_FALSE;
  thread->isInterrupted = 0;

  pthread_mutex_unlock(&threadListMutex);

  return res;
}


jboolean java_lang_Thread_isInterrupted(JNIEnv* env, jobject obj)
{
  tThreadNode* thread;

  traceThreads("lang_Thread_isInterrupted(%p)", obj);  
  
  thread = THREAD_FindThreadByObject(obj);
  return (thread != NULL && thread->isInterrupted) ? JNI_TRUE : JNI_FALSE;
}


void java_lang_Thread_nativeInterrupt(JNIEnv* env, jobject obj)
{
  tThreadNode* thread;

  traceThreads("lang_Thread_nativeInterrupt(%p)", obj);

  thread = THREAD_FindThreadByObject(obj);
  if (thread == NULL) {
    // XXXX We can't set 'isInterrupted' on a thread that hasn't started.
    traceThreads("Ignoring interrupt for a thread that has not started");
  }
  else {
    THREAD_Interrupt(env, thread);
  }
}


/* 
 * Just a little helper method 
 * It is the C method which runs a new Java thread
 */
void Interpret_MultiThread(void* arg)
{
  int i ; 
  pthread_t newThread;
  tThreadArgs* targs;
  tThreadNode* nnode;
  JNIEnv* env;
  tJNIData* pstJNIData;
  jfieldID groupFieldID;
  jclass threadCls, groupCls;
  jobject group;

  THREADINFO_IncrementThreadCount();
  newThread = pthread_self();
  
  targs = (tThreadArgs*) arg;
  env = targs->env;
  threadCls = (*env)->FindClass(env, "java/lang/Thread");
  assert(threadCls);
  groupCls = (*env)->FindClass(env, "java/lang/ThreadGroup");
  assert(groupCls);

  assert(targs->threadObject);
  pstJNIData = THREAD_createJNIData(NULL, targs->threadObject); 
  
  /* We store the bottom (although high mem) of the stack for this thread
     Normally each stack is 2 megabytes, which we search for the GC
  */
  
  targs->returnStackBottom = (void*) &i;
  
  /* Now put this new thread in the global list of threads */
  
  nnode = THREADINFO_CreateNewNodeForThread(sys_current_thread(), &i);
  nnode->stackTop = nnode->stackBottom;
  THREADINFO_addThreadNodeToList(nnode);
  THREADINFO_SetNodeInfoForCurrentThread(targs->threadObject, &i);
  
  traceThreads("Interpret_MultiThread is starting (tid %i)", 
	       (int) pthread_self());
  nnode->state = THREAD_STATE_RUNNABLE;
  
  THREADINFO_addInitialStackFrameToNode(THREAD_FindThread(), targs->newFrame);
  INTERP_Interpret(env, targs->newFrame, 1);
  THREADINFO_removeInitialStackFrameFromNode(THREAD_FindThread(), 
					     targs->newFrame);
  
  //let's free any locks on our waiting list 
  THREAD_FreeLocks();

  //find the thread's ThreadGroup (the 'group' field of the Thread object)
  groupFieldID = (*env)->GetFieldID(env, threadCls, "group",
				    "Ljava/lang/ThreadGroup;");
  assert(groupFieldID);
  group = (*env)->GetObjectField(env, targs->threadObject, groupFieldID);

  //if the Thread's group is non-null, call group.removeThread(thread)
  if (group) {
    jmethodID removeMethodID = 
      (*env)->GetMethodID(env, groupCls, "removeThread", 
			  "()Ljava/lang/ThreadGroup;");
    assert(removeMethodID);
    (*env)->CallObjectMethod(env, group, removeMethodID, targs->threadObject);
  }

  //indicate that the GC shouldn't scan this thread stack any longer
  nnode->state = THREAD_STATE_TERMINATED;
  
  traceThreads("Interpret_MultiThread has finished (tid %i)", 
	       (int) pthread_self());

  // Tell the world we are dead
  THREADINFO_DecrementThreadCount();
  pthread_cond_broadcast(&threadDeathCond);
  
  sys_free(targs);
  THREAD_destroyJNIData(pstJNIData);

  i = 77; //prevent optimising away
  pthread_exit(NULL);
}


void java_lang_Thread_start(JNIEnv* env, jobject obj)
{
  tStackFrame* pstNewFrame;
  tOBREF pstObject;
  tClassLoaderTuple* pstThreadClass;
  tClassLoaderTuple* toRunClass;
  tMethod* pstMethod;

  tThreadArgs* targs;

  jboolean isThread = JNI_FALSE;

  pthread_t newThread = 0;
  int result;

  traceThreads("lang_Thread_start(%p)", obj);  
  assert(obj);
  
  /* get pointer to thread-inherited object */
  pstObject = obj;
  
  /* get pointer to thread class */
  pstThreadClass = CLASSFILE_FindOrLoad(env, "java/lang/Thread",
					pstClassType); //XXX?
  
  /* get pointer to the object's class */
  toRunClass = ODEREF(pstObject)->pstType;
  
  /* Make sure this object is an instance of java.lang.Thread */
  if (INTERP_CheckCast(env, pstThreadClass, toRunClass) != 1) {
    panic("%s is not an instance of java/lang/Thread", toRunClass->uidName);
  }
  
  /* get pointer to run method */
  pstMethod = CLASSFILE_FindMethodInVT(env, toRunClass, "run", "()V");
  
  if (pstMethod == NULL) {
    panic("Couldn't find method run() in class %s", toRunClass->uidName);
  }
  
  /* make a stack frame */
  pstNewFrame = InitialStackFrame(pstMethod, pstObject, 
				  (int32*) &pstObject, system_heap);
  
  targs = sys_malloc(sizeof(tThreadArgs));
  
  assert(targs); 
  
  targs->newFrame = pstNewFrame;
  targs->threadObject = obj;
  targs->returnStackBottom = (void*) 0;
  targs->env = env;
  
  result = pthread_create(&newThread, NULL, 
			  (void* (*)(void*)) &Interpret_MultiThread, 
			  (void*) targs);
  
  assert(result == 0);
  
  /* Spin until we know the new thread has been added */
  while (THREAD_FindThreadByObject(obj) == 0) {
    // XXX - If thread creation causes a GC, this may block the GC :-( 
    usleep(100);
  }
}


/*
 * This is called when the 'main' Thread returns normally. It "joins" 
 * with all other non-daemon Threads.
 */
int THREAD_WaitForNativeThreads(JNIEnv* env)
{
  jclass threadCls = (*env)->FindClass(env, "java/lang/Thread");
  jfieldID daemonFieldID;

  assert(threadCls);
  daemonFieldID = (*env)->GetFieldID(env, threadCls, "daemon", "Z");
  assert(daemonFieldID);

  pthread_mutex_lock(THREAD_getThreadListMutex());
  while (1) {
    tThreadNode* iter;
    int allDaemon = 1;
  
    traceThreads("WaitForNativeThreads: iLiveThreads is %i",
		 (int) iLiveThreads);
    // If there are threads at all, we're done
    if (iLiveThreads <= 0) {
      break;
    }

    // This is the only reliable to see if we have any non daemon threads, 
    // now that Thread.setDaemon doesn't make a native call.
    iter = pstThreadList; 
    while (iter != NULL && allDaemon) {
      traceThreads("WaitForNativeThreads: checking thread %i", 
		   iter->threadId);
      if (iter->isDaemon) {
	// if this Thread was daemon last time, it is now
      }
      else {
	// be careful ... the finalization thread may not have Thread
	// object yet.
	if (iter->object) {
	  // if this Thread wasn't a daemon last time, it might be now.
          // look at the value of its 'daemon' field to find out. 
	  iter->isDaemon = (*env)->GetBooleanField(env, iter->object, 
						   daemonFieldID);
	  if (!iter->isDaemon) {
	    traceThreads("WaitForNativeThreads: found non-daemon thread %i", 
			 iter->threadId);
	    allDaemon = 0;
	  }
	}
      }
      iter = iter->next;
    }

    // If all remaining threads are daemon threads, they are expendable
    if (allDaemon) {
      break;
    }
    
    // Wait for a thread death to be notified
    pthread_cond_wait(&threadDeathCond, THREAD_getThreadListMutex());
  }
  pthread_mutex_unlock(&threadListMutex);
  return 0;
}


tJNIData* THREAD_createJNIData(JNIEnv* env, jobject obj)
{
  tJNIData* pstJNIData;
  pthread_t pthread = sys_current_thread();
  threadId_t threadId = THREAD_newThreadId(pthread);

  /* create this thread's JNIData structure */
  pstJNIData = (tJNIData*) sys_malloc(sizeof(tJNIData));
  assert(pstJNIData);

  pstJNIData->functions = JNI_pstFunctionTable;
  pstJNIData->pstException = NULL;
  pstJNIData->ppstLocalRefs = NULL;
  pstJNIData->u16NumLocalRefs = 0;
  pstJNIData->u16NumLocalRefsUsed = 0;
  pstJNIData->pstTopFrame = NULL;
#ifndef TEASEME
  pstJNIData->pstHeap = system_heap;
#endif
  pstJNIData->pthread = pthread;
  pstJNIData->threadId = threadId;
  JNI_setJNIData(pstJNIData, pthread);
  return pstJNIData;
}


void THREAD_destroyJNIData(tJNIData* pstJNIData) 
{
  assert(pstJNIData);

  THREAD_freeThreadId(pstJNIData->threadId);
  JNI_setJNIData(NULL, pstJNIData->pthread);
  sys_free(pstJNIData);
}


tThreadNode* THREAD_FindThreadByID(threadId_t threadId)
{
  tThreadNode* iter;
  pthread_mutex_lock(&threadListMutex);

  iter = pstThreadList; 
  while (iter != NULL) {
    if (iter->threadId == threadId) {
      pthread_mutex_unlock(&threadListMutex);
      return iter;
    }
    iter = iter->next;
  }
  pthread_mutex_unlock(&threadListMutex);
  traceThreads("Couldn't find thread by ID (%x)", threadId);
  return NULL;
}


tThreadNode* THREAD_FindThread()
{
  tThreadNode* iter;
  pthread_t comp = pthread_self();

  pthread_mutex_lock(&threadListMutex);

  iter = pstThreadList; 
  while (iter != NULL) {
    if (iter->pthread == comp) {
      pthread_mutex_unlock(&threadListMutex);
      return iter;
    }
    iter = iter->next;
  }
  pthread_mutex_unlock(&threadListMutex);
  traceThreads("Couldn't find current thread");
  return NULL;
}


tThreadNode* THREAD_FindThreadByObject(tOBREF pstObject)
{
  tThreadNode* iter;

  pthread_mutex_lock(&threadListMutex);

  iter = pstThreadList; 
  while (iter != NULL) {
    if (iter->object == pstObject) {
      pthread_mutex_unlock(&threadListMutex);
      return iter;
    }
    iter = iter->next;
  }
  pthread_mutex_unlock(&threadListMutex);
  traceThreads("Couldn't find thread by object (%p)", pstObject);
  return NULL;
}









