# Copyright (c) 2008 Hideki Ikemoto
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

require 'cgi'
require 'uri'
require 'nkf'
require 'jcode'

module Util
  TYPE_2CH = 0
  TYPE_MACHIBBS = 1
  TYPE_JBBS = 2
  TYPE_UNKNOWN = -1

  def self.board_type(url)
    url_2ch = %r|http://.*\.2ch\.net/.*|
    url_bbspink = %r|http://.*\.bbspink\.com/.*|
    url_machibbs = %r|http://.*\.machi\.to/.*|
    url_jbbs = %r|http://jbbs\.livedoor\.jp/.*|

    return TYPE_UNKNOWN if url.empty?
    return TYPE_2CH if url_2ch.match(url)
    return TYPE_2CH if url_bbspink.match(url)
    return TYPE_MACHIBBS if url_machibbs.match(url)
    return TYPE_JBBS if url_jbbs.match(url)
    return TYPE_UNKNOWN
  end

  def self.board_url?(url)
    url_2ch = %r|http://.*\.2ch\.net/.*|
    url_bbspink = %r|http://.*\.bbspink\.com/.*|
    url_www_2ch = %r|http://www\.2ch\.net/.*|
    url_machibbs = %r|http://.*\.machi\.to/.*|

    return false if url.empty?

    return false if url_www_2ch.match(url)

    return false if URI.parse(url).path == nil
    return false if URI.parse(url).path.split("/").size != 2

    return true  if url_2ch.match(url) || url_bbspink.match(url) ||
                    url_machibbs.match(url)

    return false
  end

  def self.thread_url?(url)
    if URI.parse(url).path.split("/").size == 5 then
      true
    else
      false
    end
  end

  def self.board_url_to_board_id(url)
    case self.board_type(url)
    when TYPE_2CH, TYPE_MACHIBBS, TYPE_UNKNOWN
      URI.parse(url).path.split("/")[1]
    when TYPE_JBBS
      URI.parse(url).path.split("/")[1] + "/" + URI.parse(url).path.split("/")[2]
    end
  end

  def self.build_post_str(name, mail, body, board_id, thread_id, server_time, session_id)
    str = ""

    str += "submit=%8F%91%82%ab%8d%9e%82%de"
    str += "&FROM=" + CGI.escape(NKF.nkf("-Wsx", name)) # UTF-8 -> SJIS with JISX0201
    str += "&mail=" + CGI.escape(NKF.nkf("-Wsx", mail)) # UTF-8 -> SJIS with JISX0201
    str += "&MESSAGE=" + CGI.escape(NKF.nkf("-Wsx", body)) # UTF-8 -> SJIS with JISX0201
    str += "&bbs=" + board_id
    str += "&hana=mogera"
    str += "&key=" + thread_id
    str += "&time=" + server_time.to_i.to_s
    str += "&sid=" + session_id if session_id

    str
  end

  def self.dat_url_to_dat_file_name(url)
    case self.board_type(url)
    when TYPE_2CH, TYPE_MACHIBBS, TYPE_UNKNOWN
      URI.parse(url).path.split("/")[3]
    when TYPE_JBBS
      URI.parse(url).path.split("/")[5]
    end
  end

  def self.dat_url_to_thread_id(url)
    dat_file_name = self.dat_url_to_dat_file_name(url)
    dat_file_name.split("\.")[0]
  end

  def self.dat_url_to_board_id(url)
    case self.board_type(url)
    when TYPE_2CH, TYPE_MACHIBBS, TYPE_UNKNOWN
      URI.parse(url).path.split("/")[1]
    when TYPE_JBBS
      URI.parse(url).path.split("/")[3] + "/" + URI.parse(url).path.split("/")[4]
    end
  end

  def self.dat_url_to_write_url(url)
    "http://#{URI.parse(url).host}/test/bbs.cgi"
  end

  def self.dat_url_to_board_url(url)
    board_id = self.dat_url_to_board_id(url)
    "http://#{URI.parse(url).host}/#{board_id}/"
  end

  def self.dat_url_to_thread_url(url)
    board_id = self.dat_url_to_board_id(url)
    thread_id = self.dat_url_to_thread_id(url)
    "http://#{URI.parse(url).host}/test/read.cgi/#{board_id}/#{thread_id}/"
  end

  def self.dat_url_to_index_path(dat_url)
    dir = self.dat_url_to_dat_dir_path(dat_url)
    dat_file_name = self.dat_url_to_dat_file_name(dat_url)

    dir + "/" + dat_file_name + ".idx"
  end

  def self.dat_url_to_dat_dir_path(dat_url)
    board_id = self.dat_url_to_board_id(dat_url)
    case self.board_type(dat_url)
    when TYPE_2CH, TYPE_MACHIBBS, TYPE_UNKNOWN
      ENV['HOME'] + "/.kita/cache/2ch/" + board_id
    when TYPE_JBBS
      ENV['HOME'] + "/.kita/cache/#{URI.parse(dat_url).host}/" + board_id.tr('/', '_')
    end
  end

  def self.thread_url_to_board_id(url)
    URI.parse(url).path.split("/")[3]
  end

  def self.thread_url_to_thread_id(url)
    URI.parse(url).path.split("/")[4]
  end

  def self.thread_url_to_dat_url(url)
    board_id = self.thread_url_to_board_id(url)
    thread_id = self.thread_url_to_thread_id(url)
    "http://#{URI.parse(url).host}/#{board_id}/dat/#{thread_id}.dat"
  end

  def self.make_dat_url(board_url, dat_file_name)
    case self.board_type(board_url)
    when TYPE_2CH, TYPE_MACHIBBS, TYPE_UNKNOWN
      board_url + "dat/" + dat_file_name
    when TYPE_JBBS
      board_id = self.board_url_to_board_id(board_url)
      thread_id = dat_file_name.split("\.")[0]
      "http://jbbs.livedoor.jp/bbs/rawmode.cgi/#{board_id}/#{thread_id}/"
    end
  end

  def self.truncate_str(str, num)
    count = 0
    ret = ""
    str.each_char {|char|
      break if count >= num
      ret += char
      count += 1
    }
    ret += "…"
    ret
  end

  def self.to_utf8(raw_str, type)
    case type
    when TYPE_2CH, TYPE_MACHIBBS, TYPE_UNKNOWN
      NKF.nkf("-Swx", raw_str) # SJIS -> UTF-8 with JISX0201
    when TYPE_JBBS
      NKF.nkf("-Ewx", raw_str) # EUC-JP -> UTF-8 with JISX0201
    end
  end
end
