/*
 * Set of functions to deal with internationalization (translation).
 */
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "support.h"
#include "interface.h"
#include "main.h"
#include "keyboard.h"
#include "velocity.h"
#include "fluidness.h"
#include "translation.h"

/*
 * Variables
 */
extern GtkWidget *window_main_;
extern gboolean callbacks_shield;
extern int _nl_msg_cat_cntr;

const gchar *languages_set = LANG_SET;
gchar lang_name[N_LANGS][50];
gchar lang_code[N_LANGS][6];
gchar language_env[32] = "";

/**********************************************************************
 * Get the code of the current language set by the environmental variables
 * LANGUAGE, LC_ALL, LC_MESSAGES or LANG.
 */
void
trans_get_lang_env (gchar * tmp_code)
{
  gchar *tmp_str = NULL;

  tmp_str = (gchar *) g_getenv ("LANGUAGE");

  if (tmp_str == NULL)
    tmp_str = (gchar *) g_getenv ("LC_ALL");
  else if (strlen (tmp_str) < 1)
    tmp_str = (gchar *) g_getenv ("LC_ALL");

  if (tmp_str == NULL)
    tmp_str = (gchar *) g_getenv ("LC_MESSAGES");
  else if (strlen (tmp_str) < 1)
    tmp_str = (gchar *) g_getenv ("LC_MESSAGES");

  if (tmp_str == NULL)
    tmp_str = (gchar *) g_getenv ("LANG");
  else if (strlen (tmp_str) < 1)
    tmp_str = (gchar *) g_getenv ("LANG");

  if (tmp_str == NULL)
    strcpy (tmp_code, "C");
  else if (strlen (tmp_str) < 1)
    strcpy (tmp_code, "C");
  else
    strcpy (tmp_code, tmp_str);
}

/**********************************************************************
 * Set the code of the current language defined by the environmental variable LANGUAGE
 */
void
trans_set_language_env (gchar * code)
{
  if (g_setenv ("LANGUAGE", code, 1) == FALSE)
    g_warning ("the environment variable LANGUAGE couldn't be set.");
}

/**********************************************************************
 * Initialize 'lang' and 'code' accordingly to 'language_set',
 * which was defined in "languages.h".
 */
void
trans_set_lang_name_code ()
{
  gint i;
  gchar delimiter[2];
  gchar tmp_str[50];
  gchar **temp_lang = NULL;

  strcpy (delimiter, "\n");
  temp_lang = g_strsplit (languages_set, delimiter, N_LANGS);
  strcpy (delimiter, "(");
  for (i = 0; i < N_LANGS; i++)
    {
      /*
       * Initialize 'lang'
       */
      strncpy (tmp_str, temp_lang[i], 50);
      tmp_str[50] = '\0';
      g_strdelimit (tmp_str, delimiter, '\0');
      strcpy (lang_name[i], g_strstrip (tmp_str));

      /*
       * Initialize 'code'
       */
      strcpy (tmp_str, strchr (temp_lang[i], '('));
      *(strchr (tmp_str, ')')) = '\0';
      strcpy (lang_code[i], tmp_str + 1);
    }
  g_strfreev (temp_lang);
}
gboolean
trans_estas_Esperanto ()
{
  return ((((unsigned) time (NULL)) % 8) == 1 ? TRUE : FALSE);
}

/**********************************************************************
 * Get the current locale and change it if necessary
 */
void
trans_init_language_env ()
{
  gchar *aux_str;
  gchar tmp_code[33];
  gchar aux_code_2[3];
  gint i;

  /* 
   * Read the current locale and cut eventual colons
   */
#ifdef G_OS_UNIX
  trans_get_lang_env (tmp_code);
#endif

#ifdef G_OS_WIN32
  gchar *tmp_win_code;
  tmp_win_code = g_win32_getlocale ();
  strcpy (tmp_code, tmp_win_code);
  g_free (tmp_win_code);
#endif

  /*
   * If the language is already set in preferences, then override the current locale
   */
  if (g_key_file_has_key (main_preferences_get (), "interface", "language", NULL))
  {
    aux_str = g_key_file_get_string (main_preferences_get (), "interface", "language", NULL);
    strncpy (tmp_code, aux_str, 6);
    g_free (aux_str);
  }

  if (strcmp (tmp_code, "C") != 0)
  {
	  tmp_code[0] = g_ascii_tolower (tmp_code[0]);
	  tmp_code[1] = g_ascii_tolower (tmp_code[1]);
	  if (tmp_code[2] == '_')
	    tmp_code[5] = '\0';
	  else
	    tmp_code[2] = '\0';
  }

  /*
   * Test if the current locale is available
   */
  trans_set_lang_name_code ();
  for (i = 0; i < N_LANGS; i++)
    if (strcmp (lang_code[i], tmp_code) == 0)
      break;

  /*
   * If not available, test for a similar one
   */
  if (i == N_LANGS)
    {
      for (i = 0; i < N_LANGS; i++)
      {
	if (strstr (lang_code[i], tmp_code) || strstr (tmp_code, lang_code[i]))
	  {
	    strcpy (tmp_code, lang_code[i]);
	    break;
	  }
      }
    }

  g_key_file_set_string (main_preferences_get (), "interface", "language", tmp_code);

  /*
   * If even a similar is not available, then change it
   */
  if (i == N_LANGS)
    {
      g_message ("as the locale \"%s\" isn't available,", tmp_code);

      strcpy (aux_code_2, tmp_code);
      aux_code_2[2] = '\0';

      if (strcmp (aux_code_2, "pt") == 0)
	strcpy (tmp_code, "pt_BR");

      /* this second is just a template */
      else if (strcmp (tmp_code, "pt_PT") == 0)
	strcpy (tmp_code, "pt_BR");

      else if (strcmp (aux_code_2, "es") == 0)
	strcpy (tmp_code, "pt_BR");

      else if (trans_estas_Esperanto () == TRUE)
	strcpy (tmp_code, "eo_EO");

      else
	strcpy (tmp_code, "C");

      g_message ("we are using this another one: \"%s\"", tmp_code);
      g_message ("if you don't agree, try to change it at the main menu...");

      g_key_file_remove_key (main_preferences_get (), "interface", "language", NULL);
    }
      
  trans_set_language_env (tmp_code);
  
  /* Old way:
  ++_nl_msg_cat_cntr;
   */
  
  /* New way:
   */
  gtk_set_locale ();
}

/**********************************************************************
 * Inserts the list of available languages in the 'combo_language'.
 */
void
trans_set_combo_language ()
{
  gint i;
  gint i_env;
  gchar tmp_code[12];
  GList *languages = NULL;
  GtkWidget *wg;

  trans_get_lang_env (tmp_code);

  for (i = 0, i_env = -1; i < N_LANGS; i++)
    {
      if (strcmp (lang_code[i], tmp_code) != 0)
	languages = g_list_insert_sorted (languages, lang_name[i], &comp_str);
      else
	i_env = i;
    }

  if (i_env == -1)
    g_error ("set_combo_language() ==> the locale \"%s\" is not available!",
	     tmp_code);
  else
    languages = g_list_insert (languages, lang_name[i_env], 0);

  wg = lookup_widget (window_main_, "combo_language");
  gtk_combo_set_popdown_strings (GTK_COMBO (wg), languages);

  g_list_free (languages);
}

/**********************************************************************
 * Update the current language used accordingly to that selected in the
 * 'combo_language'
 */
void
trans_change_language ()
{
  gint i;
  gchar *selection = NULL;
  GtkWidget *wg;

  wg = lookup_widget (window_main_, "entry_language");
  selection = gtk_editable_get_chars (GTK_EDITABLE (wg), 0, -1);

  for (i = 0; i < N_LANGS; i++)
    if (strcmp (lang_name[i], selection) == 0)
      break;

  if (i == N_LANGS)
    {
      g_warning ("change_language() --> couldn't find the language: %s",
		 selection);
      g_free (selection);
      return;
    }
  g_free (selection);

  trans_set_language_env (lang_code[i]);
  /*++_nl_msg_cat_cntr; */
  gtk_set_locale ();

  callbacks_shield = TRUE;
  gtk_widget_destroy (window_main_);
  window_main_ = create_window_main ();
  wg = lookup_widget (window_main_, "entry_keyboard");
  gtk_entry_set_text (GTK_ENTRY (wg), keyb_get_name ());
  trans_set_combo_language ();
  gtk_widget_show (window_main_);
  callbacks_shield = FALSE;

  velo_reset_dict ();
  fluid_reset_paragraph ();

  g_key_file_set_string (main_preferences_get (), "interface", "language", lang_code[i]);
  g_message ("selected language code --> %s", lang_code[i]);
}

/**********************************************************************
 * Reads the text of a file to be presented at some dialog,
 * accordingly to the environment language.
 * 'text': buffer where the text is copied into.
 * 'max_text_len': the size of the buffer.
 * 'file_end': how the file name ends (isn't needed the leading
 * underscore, only the termination, ex.: 'basic_end.txt').
 */
void
trans_read_text (gchar * text, gint max_text_len, const gchar * file_end)
{
  gchar *tmp_name = NULL;
  gchar *tmp_path = NULL;
  gchar tmp_code[16];
  FILE *fh;

  trans_get_lang_env (tmp_code);
  if (strlen (tmp_code) > 5)
    tmp_code[5] = '\0';
  tmp_name = g_strconcat (tmp_code, "_", file_end, NULL);

  /*
   * Try at home
   */
  tmp_path = g_strconcat (main_get_user_dir (), tmp_name, NULL);
  fh = (FILE*) g_fopen (tmp_path, "r");

  /*
   * Try at PACKAGE
   */
  if (fh == NULL)
    {
      g_free (tmp_path);
      tmp_path = g_strconcat (main_get_data_path (), tmp_name, NULL);
      fh = (FILE*) g_fopen (tmp_path, "r");
    }

  /*
   * Default to C
   */
  if (fh == NULL && strcmp (tmp_code, "C") != 0)
    {
      g_message ("trans_read_text() --> couldn't open the data file: %s\n"
		 " So, we have to apply the default one: C_%s", tmp_name,
		 file_end);
      g_free (tmp_name);
      g_free (tmp_path);
      trans_set_language_env ("C");
      trans_read_text (text, max_text_len, file_end);
      trans_set_language_env (tmp_code);
      return;
    }

  if (fh == NULL)
    g_error ("trans_read_text() --> couldn't open the data file:\n %s",
	     tmp_name);
  else
    {
      gint i, j, tmpc = 0;

      max_text_len -= 2;
      i = 0;
      do
	{
	  for (j = 0; j < 60; j++)
	    {
	      tmpc = fgetc (fh);
	      text[i] = (tmpc != EOF) ? (gchar) tmpc : '\0';
	      if (i <= max_text_len)
		i++;
	      if (tmpc == '\n')
		break;
	    }
	  if (tmpc == '\n')
	    continue;
	  for (j = 0; j < 20; j++)
	    {
	      tmpc = fgetc (fh);
	      if (tmpc == ' ')
		{
		  //text[i] = '\n';
		  text[i] = ' ';
		  break;
		}
	      else if (tmpc == '\n')
		{
		  text[i] = '\n';
		  break;
		}
	      else if (tmpc == EOF)
		{
		  text[i] = '\0';
		  break;
		}
	      text[i] = (gchar) tmpc;
	      if (i < max_text_len)
		i++;
	    }
	  i++;
	}
      while (i < max_text_len && tmpc != EOF);
      fclose (fh);
      text[max_text_len - 1] = '\0';
    }

  g_free (tmp_name);
  g_free (tmp_path);
}
