/*
 * Velocity exercise
 */
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "support.h"
#include "main.h"
#include "translation.h"
#include "tutor.h"
#include "velocity.h"

/*
 * Global variables
 */
extern GtkWidget *window_tutor_;

GList *word_list;

struct
{
  GList *list;
  gint len;
  gchar name[21];
} dict;

/*******************************************************************************
 * Interface functions
 */
gchar *
velo_get_dict_name ()
{
  return (dict.name);
}

void
velo_reset_dict ()
{
  g_list_free (dict.list);
  dict.list = NULL;
  dict.len = 0;
  dict.name[0] = '\0';
}

/**********************************************************************
 * Initialize the velo exercise window.
 */
void
velo_init ()
{
  gchar *tmp_name;
  gchar *wl_name;
  FILE *fh;

  if (dict.len == 0)
    {
      if (g_key_file_has_key (main_preferences_get (), "tutor", "word_list", NULL))
	{
          wl_name = g_key_file_get_string (main_preferences_get (), "tutor", "word_list", NULL);
	  tmp_name =
	    g_strconcat (main_get_user_dir (), wl_name, ".words", NULL);
	  if ((fh = (FILE*) g_fopen (tmp_name, "r")))
	    {
	      fclose (fh);
	      velo_init_dict (wl_name);
	    }
	  g_free (wl_name);
          g_free (tmp_name);
	}
    }
  if (dict.len == 0)
    velo_init_dict (NULL);
}

/**********************************************************************
 * Retrieves words from the dictionary.
 */
void
velo_init_dict (gchar * list_name)
{
  static gchar *word_str = NULL;	/* no doubts: this is initialized only
					   at the first call */
  gchar *tmp_name;
  gchar tmp_code[16];
  gchar str_32[33];
  FILE *fh;

  if (list_name && strcmp (list_name, _("--> Default")))
    {
      g_key_file_set_string (main_preferences_get (), "tutor", "word_list", list_name);
      tmp_name = g_strconcat (main_get_user_dir (),
			      list_name, ".words", NULL);
      g_message ("loading dictionary: %s.words", list_name);
      strncpy (dict.name, list_name, 20);
      dict.name[20] = '\0';
    }
  else
    {
      g_key_file_remove_key (main_preferences_get (), "tutor", "word_list", NULL);
      trans_get_lang_env (tmp_code);
      tmp_name =
	g_strconcat (main_get_data_path (), tmp_code, ".words", NULL);
      g_message ("loading dictionary: %s.words", tmp_code);
      strcpy (dict.name, "Default");
    }

  if ((fh = (FILE*) g_fopen (tmp_name, "r")))
    {
      g_free (word_str);
      g_list_free (dict.list);
      dict.list = NULL;
      word_str = g_strdup ("");
      while (fgets (str_32, 33, fh))
	{
	  str_32[strlen (str_32) - 1] = '\0';
	  word_str = g_strconcat (word_str, " ", str_32, NULL);
	  dict.list = g_list_append (dict.list, strrchr (word_str, ' ') + 1);
	}
      fclose (fh);
      dict.len = g_list_length (dict.list);
      g_message ("dictionary loaded!\n");
    }
  else
    {
      g_message ("could not open the file: %s", tmp_name);
      g_free (tmp_name);
      trans_get_lang_env (tmp_code);
      if (strcmp (tmp_code, "C") == 0)
	g_error ("so, we must quit!");
      trans_set_language_env ("C");
      velo_init_dict (list_name);
      trans_set_language_env (tmp_code);
      return;
    }

  g_free (tmp_name);
}

/**********************************************************************
 * Draw random words selected from a 'discretionary'
 */
void
velo_draw_random_words ()
{
  gchar *text;
  gchar str_32[33];
  gint i, j;

  for (i = 0; i < 4; i++)
    {				/* 4 paragraphs per exercise */
      text = NULL;
      for (j = 0; j < 20; j++)
	{			/* 20 words per paragraph */
	  velo_get_random_word (str_32);
	  if (!j)
	    {
	      str_32[0] = g_ascii_toupper (str_32[0]);
	      text = g_strconcat (str_32, NULL);
	    }
	  else
	    text = g_strconcat (text, " ", str_32, NULL);
	}
      text = g_strconcat (text, ".\n", NULL);
      tutor_draw_paragraph (text);
      g_free (text);
    }
}

/**********************************************************************
 * Selects one word from a 'discretionary'
 */
void
velo_get_random_word (gchar str_32[33])
{
  strncpy (str_32, g_list_nth_data (dict.list, rand () % dict.len), 33);
  str_32[32] = '\0';
}

/**********************************************************************
 * Reads the text file "file_name" and write to the dictionary.
 * If overwrite is TRUE, then besides overwriting any .words file,
 * it loads a lesson with the new dictionary.
 */
#define MAX_WORD_LEN 30
void
velo_create_dict (gchar * file_name, gboolean overwrite)
{
  gint i;
  gint nlin;
  gint nw;
  gchar *dict_path;
  gchar *dictio_name;
  gchar tmp_str[1000];
  gchar *tmp_utf8;
  gchar *tmp_utf8_word;
  gunichar tmp_unichar;
  gunichar tmp_word[MAX_WORD_LEN];
  FILE *fh_source;
  FILE *fh_destiny;

  if (file_name == NULL)
    {
      gdk_beep ();
      g_warning ("velo_create_dict(): null file name as first argument.");
      return;
    }

  if (!(fh_source = (FILE*) g_fopen (file_name, "r")))
    {
      gdk_beep ();
      g_warning ("couldn't open the file:\n <%s>", file_name);
      return;
    }

  dictio_name = g_strdup (strrchr (file_name, DIRSEP) + 1);
  dict_path = g_strconcat (main_get_user_dir (), dictio_name, ".words", NULL);

  assert_user_dir ();
  if (!(fh_destiny = (FILE*) g_fopen (dict_path, "w")))
    {
      gdk_beep ();
      g_warning ("couldn't create the file:\n <%s>", dict_path);
      if (overwrite == FALSE)
	{
	  fclose (fh_source);
	  g_free (dict_path);
	  g_free (dictio_name);
	  return;
	}
    }
  g_free (dict_path);

  /********************
   * Process the file
   */
  i = 0;
  nlin = 0;
  nw = 0;
  while (fgets (tmp_str, 2000, fh_source) != NULL)
  {
    nlin++;
    if (g_utf8_validate (tmp_str, -1, NULL) == FALSE)
    {
	    g_message ("Not valid UTF-8 text in paragraph %i", nlin);
	    continue;
    }

    tmp_utf8 = tmp_str;
    do
    {
      tmp_unichar = g_utf8_get_char (tmp_utf8);
      tmp_utf8 = g_utf8_next_char (tmp_utf8);

      if (g_unichar_isalpha (tmp_unichar) == FALSE)
	{
          tmp_word[i] = '\0';
	  if (i > 1)
	    {
	      nw++;
	      tmp_utf8_word = g_ucs4_to_utf8 (tmp_word, i, NULL, NULL, NULL);
	      fputs (tmp_utf8_word, fh_destiny);
	      fputc ('\n', fh_destiny);
	      g_free (tmp_utf8_word);
	    }
	  i = 0;
	}
      else if (i < (MAX_WORD_LEN - 1))
	  tmp_word[i++] = tmp_unichar;

    } while (tmp_unichar != L'\n' && tmp_unichar != L'\0'); 
  }
  if (nw == 0)
    fputs ("01234\n56789\n43210\n98765\n:-)\n", fh_destiny);

  fclose (fh_source);
  fclose (fh_destiny);

  if (overwrite == TRUE)
    {
      velo_init_dict (dictio_name);
      tutor_set_query (QUERY_INTRO);
      tutor_process_touch ('\0');
    }
  g_free (dictio_name);
}

/**********************************************************************
 * Put on the screen the final comments
 */
void
velo_comment (gdouble accuracy, gdouble velocity)
{
  gchar str[1000];
  GtkWidget *wg;
  GtkTextBuffer *buf;

  /*
   * Comments
   */
  if (accuracy < 97)
    g_sprintf (str,
	       _(" Your accuracy rate is too low.\n"
		 " I only can rely on you if it is greater than 97 %%.\n"
		 " Please do it again.\n"));
  else if (velocity < 10)
    g_sprintf (str,
	       _(" You must be kidding. You are so slow!\n"
		 " Please do it again and again.\n"));
  else if (velocity < 20)
    g_sprintf (str,
	       _(" Still away from the highway. You can make better...\n"
		 " Try at least 20 WPM.\n"));
  else if (velocity < 30)
    g_sprintf (str,
	       _(" You are doing well, but need to go faster.\n"
		 " And don't forget the accuracy. Try to get 30 WPM.\n"));
  else if (velocity < 40)
    g_sprintf (str,
	       _(" Fine. Now you need to start running.\n"
		 " Can you reach 40 WPM?\n"));
  else if (velocity < 50)
    g_sprintf (str, _(" Very good. You are almost there.\n"
		      " Can you finally reach 50 WPM?\n"));
  else if (velocity < 60)
    g_sprintf (str, _(" Excellent. For this course, that is enough.\n"
		      " Try now the fluidness exercises, OK?\n"));
  else if (velocity < 70)
    g_sprintf (str, _(" Fast! Are you training for a competition?\n"
		      " So, try to get 70 WPM!\n"));
  else if (velocity < 80)
    g_sprintf (str,
	       _(" Top of \"qwerty\"."
		 " Now it's time to change to the Dvorak mode.\n"
		 " Are you afraid of reaching 80 WPM?\n"));
  else if (velocity < 90)
    g_sprintf (str,
	       _(" Dvorak mode dominated!\n" " Can you fly at 90 WPM?\n"));
  else
    g_sprintf (str,
	       _(" Dvorak master!\n"
		 " I have no words to express my admiration!\n"));

  wg = lookup_widget (window_tutor_, "text_tutor");
  buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));
  gtk_text_buffer_insert_at_cursor (buf, str, strlen (str));
}
