/**************************************************************************/
/*  Klavaro - a flexible touch typing tutor                               */
/*  Copyright (C) 2005 - 2008  Felipe Castro                              */
/*                                                                        */
/*  This program is free software, licensed under the terms of the GNU    */
/*  General Public License as published by the Free Software Foundation,  */
/*  which is also included in this package, in a file named "COPYING".    */
/**************************************************************************/

/*
 * Contest for fluidness performance
 */

#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <time.h>
#include <math.h>
#include <sys/stat.h>
#include <curl/curl.h>
#include <curl/easy.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include "interface.h"
#include "support.h"

#include "main.h"
#include "translation.h"
#include "top10.h"

/**************************************************
 * Variables
 */
GtkWidget *window_top10_;
Statistics top10_local[10];
Statistics top10_global[10];
GKeyFile *keyfile = NULL;

/**************************************************
 * Functions
 */

static void
top10_window_init ()
{
	gchar *tmp;
	GtkWidget *wg;

	wg = lookup_widget (window_top10_, "checkbutton_top10_default");
	if (!main_preferences_exist ("game", "default"))
		main_preferences_set_boolean ("game", "default", TRUE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg),
				      main_preferences_get_boolean ("game", "default"));

	wg = lookup_widget (window_top10_, "checkbutton_top10_personal");
	if (!main_preferences_exist ("game", "personal"))
		main_preferences_set_boolean ("game", "personal", FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg),
				      main_preferences_get_boolean ("game", "personal"));

	wg = lookup_widget (window_top10_, "checkbutton_top10_manager");
	if (!main_preferences_exist ("game", "manager"))
		main_preferences_set_boolean ("game", "manager", FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg),
				      main_preferences_get_boolean ("game", "manager"));

	if (main_preferences_exist ("game", "uphost"))
	{
		wg = lookup_widget (window_top10_, "entry_top10_uphost");
		tmp = main_preferences_get_string ("game", "uphost");
		gtk_entry_set_text (GTK_ENTRY (wg), tmp);
		g_free (tmp);
	}

	if (main_preferences_exist ("game", "upuser"))
	{
		wg = lookup_widget (window_top10_, "entry_top10_upuser");
		tmp = main_preferences_get_string ("game", "upuser");
		gtk_entry_set_text (GTK_ENTRY (wg), tmp);
		g_free (tmp);
	}

	if (main_preferences_exist ("game", "uppass"))
	{
		wg = lookup_widget (window_top10_, "entry_top10_uppass");
		tmp = main_preferences_get_string ("game", "uppass");
		gtk_entry_set_text (GTK_ENTRY (wg), tmp);
		g_free (tmp);
	}

	if (main_preferences_exist ("game", "downhost"))
	{
		wg = lookup_widget (window_top10_, "entry_top10_downhost");
		tmp = main_preferences_get_string ("game", "downhost");
		gtk_entry_set_text (GTK_ENTRY (wg), tmp);
		g_free (tmp);
	}

	if (main_preferences_exist ("game", "downuser"))
	{
		wg = lookup_widget (window_top10_, "entry_top10_downuser");
		tmp = main_preferences_get_string ("game", "downuser");
		gtk_entry_set_text (GTK_ENTRY (wg), tmp);
		g_free (tmp);
	}

	if (main_preferences_exist ("game", "downpass"))
	{
		wg = lookup_widget (window_top10_, "entry_top10_downpass");
		tmp = main_preferences_get_string ("game", "downpass");
		gtk_entry_set_text (GTK_ENTRY (wg), tmp);
		g_free (tmp);
	}
}

static void
top10_print (gchar * msg)
{
	GtkWidget *wg;
	GtkTextBuffer *buf;

	if (window_top10_ == NULL)
	{
		g_print ("%s\n", msg);
		return;
	}

	wg = lookup_widget (window_top10_, "notebook_top10");
	gtk_notebook_set_current_page (GTK_NOTEBOOK (wg), 1);
	wg = lookup_widget (window_top10_, "textview_top10_external");
	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));
	gtk_text_buffer_set_text (buf, msg, -1);
}

static void
top10_check_writable (gchar *scorefile)
{
	gchar *path;

	path = g_build_filename ("/var/games/klavaro", scorefile, NULL);
	if (!g_file_test (path, G_FILE_TEST_IS_REGULAR))
	{
		g_free (path);
		return;
	}
#ifdef	G_OS_UNIX
	g_chmod (path, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH);
#endif
	g_free (path);
}

#define NOBODY "xxx"
void
top10_clean_stat (gint i, gboolean locally)
{
	Statistics *top10;

	top10 = locally ? top10_local : top10_global;

	if (i > 9)
		return;

	top10[i].lang[0] = 'x';
	top10[i].lang[1] = 'x';
	top10[i].genv = 'x';
	top10[i].when = 0;
	top10[i].nchars = MIN_CHARS_TO_LOG;
	top10[i].accur = 0.0;
	top10[i].velo = 0.0;
	top10[i].fluid = 0.0;
	top10[i].score = 0.0;
	top10[i].name_len = strlen (NOBODY);
	strcpy (top10[i].name, NOBODY);
}

void
top10_init_stats (gboolean locally)
{
	gint i;
	Statistics *top10;

	top10 = locally ? top10_local : top10_global;

	for (i = 0; i < 10; i++)
		top10_clean_stat (i, locally);
}

void
top10_insert_stat (Statistics * stat, gint i, gboolean locally)
{
	gint j;
	Statistics *top10;

	top10 = locally ? top10_local : top10_global;

	if (i > 9)
		return;

	for (j = 8; j >= i; j--)
		memmove (&top10[j + 1], &top10[j], sizeof (Statistics));

	memmove (&top10[i], stat, sizeof (Statistics));
}

gboolean
top10_compare_insert_stat (Statistics * stat, gboolean locally)
{
	gint i, j;
	Statistics *top10;

	top10 = locally ? top10_local : top10_global;

	for (i = 0; i < 10; i++)
	{
		if (stat->score > top10[i].score)
		{
			for (j = i - 1; j >= 0; j--)
			{
				if (g_str_equal (stat->name, top10[j].name))
					return (FALSE);
			}

			for (j = i; j < 10; j++)
			{
				if (g_str_equal (stat->name, top10[j].name))
					top10_delete_stat (j, locally);
			}

			top10_insert_stat (stat, i, locally);
			return (TRUE);
		}
	}
	return (FALSE);
}

void
top10_delete_stat (gint i, gboolean locally)
{
	gint j;
	Statistics *top10;

	top10 = locally ? top10_local : top10_global;

	if (i > 9)
		return;

	for (j = i; j < 9; j++)
		memmove (&top10[j], &top10[j + 1], sizeof (Statistics));

	top10_clean_stat (9, locally);
}

gfloat
top10_calc_score (Statistics * stat)
{
	gfloat score;
	// Don't touch this anymore!
	score = (6 * stat->velo + 3 * stat->accur + stat->fluid) / 100;
	return (score);
}

gboolean
top10_validate_stat (Statistics * stat)
{

	if (stat->score > 10.0 || stat->score < 0.0)
	{
		g_message ("Invalid score (%g): it must be between 0 and 10", stat->score);
		return (FALSE);
	}

	if ((stat->score - top10_calc_score (stat) > 1.0e-4))
	{
		g_message ("Invalid score (%g): did you try to edit the scores file?", stat->score);
		return (FALSE);
	}

	if (stat->fluid > 95)
	{
		g_message
			("Invalid score (fluidness = %g): robots shouldn't compete with humans...",
			 stat->fluid);
		return (FALSE);
	}

	return (TRUE);
}

gchar *
top10_get_score_file (gboolean locally, gint lang)
{
	gchar *tmp = NULL;
	gchar *ksc;

	if (lang < 0)
		tmp = main_preferences_get_string ("interface", "language");
	else
		tmp = g_strdup (trans_get_code (lang));
	if (tmp == NULL)
		tmp = g_strdup ("en");

	if (tmp[0] == 'C')
		ksc = g_strdup (locally ? "local_en.ksc" : "global_en.ksc");
	else
		ksc = g_strdup_printf ("%s_%c%c.ksc", locally ? "local" : "global", tmp[0], tmp[1]);
	g_free (tmp);

	return (ksc);
}

gboolean
top10_read_stats_from_file (gboolean locally, gchar * file)
{
	gint i, j, n;
	gboolean success;
	FILE *fh;
	Statistics *top10;

	top10 = locally ? top10_local : top10_global;

	fh = g_fopen (file, "r");
	if (fh == NULL)
		return FALSE;

	for (i = 0; i < 10; i++)
	{
		success = FALSE;

		/* lang[0] */
		top10[i].lang[0] = fgetc (fh);
		if (!g_ascii_isalpha (top10[i].lang[0]))
		{
			g_message ("Problem: lang[0] = %c", top10[i].lang[0]);
			break;
		}

		/* lang[1] */
		top10[i].lang[1] = fgetc (fh);
		if (!g_ascii_isalpha (top10[i].lang[1]))
		{
			top10[i].lang[0] = 'e';
			top10[i].lang[1] = 'o';
		}

		/* genv */
		top10[i].genv = fgetc (fh);
		if (top10[i].genv != 'x' && top10[i].genv != 'w')
		{
			g_message ("Problem: genv = %c", top10[i].genv);
			break;
		}

		/* when */
		n = fread (&top10[i].when, sizeof (time_t), 1, fh);
		if (n == 0)
		{
			g_message ("Problem: when = %lu", top10[i].when);
			break;
		}

		/* nchars */
		n = fread (&top10[i].nchars, sizeof (gint), 1, fh);
		if (n == 0 || top10[i].nchars < MIN_CHARS_TO_LOG)
		{
			g_message ("Problem: nchars = %i, < %i", top10[i].nchars, MIN_CHARS_TO_LOG);
			break;
		}

		/* accur */
		n = fread (&top10[i].accur, sizeof (gfloat), 1, fh);
		if (n == 0 || top10[i].accur < 0 || top10[i].accur > 100)
		{
			g_message ("Problem: accur = %f <> [0, 100]", top10[i].accur);
			break;
		}

		/* velo */
		n = fread (&top10[i].velo, sizeof (gfloat), 1, fh);
		if (n == 0 || top10[i].velo < 0 || top10[i].velo > 300)
		{
			g_message ("Problem: velo = %f <> [0, 300]", top10[i].velo);
			break;
		}

		/* fluid */
		n = fread (&top10[i].fluid, sizeof (gfloat), 1, fh);
		if (n == 0 || top10[i].fluid < 0 || top10[i].fluid > 100)
		{
			g_message ("Problem: fluid = %f <> [0, 100]", top10[i].fluid);
			break;
		}

		/* score */
		n = fread (&top10[i].score, sizeof (gfloat), 1, fh);
		if (n == 0 || top10[i].score < 0 || top10[i].score > 20)
		{
			g_message ("Problem: score = %f <> [0, 20]", top10[i].score);
			break;
		}

		/* name_len */
		n = fread (&top10[i].name_len, sizeof (gint), 1, fh);
		if (n == 0 || top10[i].name_len < 0 || top10[i].name_len > MAX_NAME_LEN)
		{
			g_message ("Problem: name_len = %i <> [0, MAX_NAME_LEN]", top10[i].name_len);
			break;
		}

		/* name */
		n = fread (&top10[i].name, sizeof (gchar), top10[i].name_len, fh);
		top10[i].name[top10[i].name_len] = '\0';
		if (!g_utf8_validate (top10[i].name, -1, NULL))
		{
			for (j = 0; j < top10[i].name_len; j++)
				if (!g_ascii_isalpha (top10[i].name[j]))
					top10[i].name[j] = '?';
		}
		success = TRUE;
	}
	fclose (fh);

	return (success);
}

void
top10_read_stats (gboolean locally, gint lang)
{
	gchar *local_ksc;
	gchar *tmp;
	Statistics *top10;

	top10 = locally ? top10_local : top10_global;

	top10_init_stats (locally);

	local_ksc = top10_get_score_file (locally, lang);

	tmp = g_build_filename (main_get_score_path (), local_ksc, NULL);
	g_free (local_ksc);

	if (!top10_read_stats_from_file (locally, tmp))
	{
		g_message ("Could not read the scores file '%s'.\n Creating a blank one.", tmp);
		top10_init_stats (locally);
		top10_write_stats (locally, lang);
	}
	g_free (tmp);
}

void
top10_write_stats (gboolean locally, gint lang)
{
	gint i;
	gchar *path;
	gchar *filename;
	gchar *lsfile;
	gchar *command;
	FILE *fh;
	gboolean success;
	GError *error = NULL;
	Statistics *top10;

	top10 = locally ? top10_local : top10_global;

	if (UNIX_OK)
		path = g_strdup ("/tmp/klavaro");
	else
		path = g_strdup (main_get_score_path ());

	if (!g_file_test (path, G_FILE_TEST_IS_DIR))
	{
#ifdef G_OS_UNIX
		g_mkdir (path, S_IRWXU | S_IRWXG | S_IRWXO | S_IWGRP | S_IWOTH);
#else
		g_mkdir (path, 0xFFFF);
#endif
	}

	filename = top10_get_score_file (locally, lang);

	lsfile = g_build_filename (path, filename, NULL);

	fh = g_fopen (lsfile, "w");
	if (fh == NULL)
	{
		g_warning ("Could not write the scores file in %s", path);
		g_free (path);
		g_free (filename);
		g_free (lsfile);
		return;
	}

	for (i = 0; i < 10; i++)
	{
		fputc (top10[i].lang[0], fh);
		fputc (top10[i].lang[1], fh);
		fputc (top10[i].genv, fh);
		fwrite (&top10[i].when, sizeof (time_t), 1, fh);
		fwrite (&top10[i].nchars, sizeof (gint), 1, fh);
		fwrite (&top10[i].accur, sizeof (gfloat), 1, fh);
		fwrite (&top10[i].velo, sizeof (gfloat), 1, fh);
		fwrite (&top10[i].fluid, sizeof (gfloat), 1, fh);
		fwrite (&top10[i].score, sizeof (gfloat), 1, fh);
		fwrite (&top10[i].name_len, sizeof (gint), 1, fh);
		if (top10[i].name && top10[i].name_len > 0)
			fputs (top10[i].name, fh);
	}
	fputs ("KLAVARO!", fh);
	fclose (fh);

	if (UNIX_OK)
	{
		top10_check_writable (filename);
		command = g_strconcat ("klavaro_helper ", filename, NULL);
		success = g_spawn_command_line_sync (command, NULL, NULL, NULL, &error);
		if (!success)
		{
			g_free (command);
			command = g_strconcat ("./klavaro_helper ", filename, NULL);
			success = g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
		}
		g_free (command);
		if (success)
		{
			g_unlink (lsfile);
			g_rmdir (path);
			top10_check_writable (filename);
		}
		else
			g_message ("Failed to copy scorings file.\n %s", error->message);
		if (error != NULL)
			g_error_free (error);
	}

	g_free (path);
	g_free (filename);
	g_free (lsfile);
}

/* Test function to show every field of a scoring record, at the terminal
 */
void
top10_show_stat (Statistics * stat)
{
	g_print ("Language: %c%c\n", stat->lang[0], stat->lang[1]);
	g_print ("Graphical environment: %s\n", stat->genv == 'x' ? "Linux" : "Windous");
	g_print ("When: %li\n", stat->when);
	g_print ("# of characters: %i\n", stat->nchars);
	g_print ("Accuracy: %2.1f\n", stat->accur);
	g_print ("Velocity: %2.1f\n", stat->velo);
	g_print ("Fluidness: %2.1f\n", stat->fluid);
	g_print ("Score: %05.1f\n", stat->score);
	g_print ("Name length: %i\n", stat->name_len);
	g_print ("Name: %s\n", stat->name);
}

void
top10_show_stats (gboolean locally)
{
	gint i;
	gchar *tmp;
	gchar *date;
	struct tm *ltime;
	GtkWidget *wg;
	GtkTextBuffer *buf;
	Statistics *top10;

	top10 = locally ? top10_local : top10_global;

	top10_read_stats (locally, -1);

	if (window_top10_ == NULL)
	{
		window_top10_ = create_window_top10 ();
		if (!UNIX_OK)
		{
			wg = lookup_widget (window_top10_, "notebook_top10");
			gtk_notebook_set_show_tabs (GTK_NOTEBOOK (wg), FALSE);
		}
		else
			top10_window_init ();
	}

	if (locally)
		wg = lookup_widget (window_top10_, "textview_top10_local");
	else
		wg = lookup_widget (window_top10_, "textview_top10_external");
	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));
	gtk_text_buffer_set_text (buf, "", -1);
	for (i = 0; i < 10; i++)
	{
		/* First line: main info
		 */
		tmp = g_strdup_printf ("%02i) %3.4f - %s\n", i + 1, top10[i].score, top10[i].name);
		gtk_text_buffer_insert_at_cursor (buf, tmp, -1);
		g_free (tmp);

		/* Second line: further info
		 */
		if (locally)
			wg = lookup_widget (window_top10_, "togglebutton_top10_local");
		else
			wg = lookup_widget (window_top10_, "togglebutton_top10_external");
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wg)))
		{
			ltime = localtime (&top10[i].when);
			date = g_strdup_printf ("%i-%2.2i-%2.2i %02i:%02i", (ltime->tm_year) + 1900,
						(ltime->tm_mon) + 1, (ltime->tm_mday),
						(ltime->tm_hour), (ltime->tm_min));
			tmp = g_strdup_printf
				("\t%s: %s\t%s: %2.1f\t%s: %2.1f\t%s: %2.1f\t%s: %i %s: %c%c\n\n",
				 _("When"), date, _("Velocity"), top10[i].velo, _("Accuracy"),
				 top10[i].accur, _("Fluidness"), top10[i].fluid, _("Chars"),
				 top10[i].nchars, _("Language"), top10[i].lang[0],
				 top10[i].lang[1] != '\0' ?top10[i].lang[1]:' ');
			gtk_text_buffer_insert_at_cursor (buf, tmp, -1);
			g_free (tmp);
			g_free (date);
		}
	}
	gtk_widget_show (window_top10_);
}

gboolean
top10_global_update (gpointer data)
{
	gboolean success;
	gboolean fail;
	gchar *tmp;
	gchar *ksc;
	gchar *host;
	gchar *hostsub;
	gchar *subdir;
	gchar *command;
	GError *error = NULL;
	GtkImage *img;
	gpointer gp;
	CURL *curl;
	FILE *fh;
	
	gp = data;

	img = GTK_IMAGE (lookup_widget (window_top10_, "image_top10_update"));

	if (!main_get_curl_ok ())
	{
		tmp = g_strconcat (_("Not able to download files"), ": 'libcurl' ", _("not found"), ". ",
		       _("Are you sure you have it installed in your system?"), NULL);
		top10_print (tmp);
		g_free (tmp);
		gtk_image_set_from_stock (img, "gtk-goto-bottom", GTK_ICON_SIZE_BUTTON);
		return FALSE;
	}

	hostsub = top10_get_host ("downhost");
	if (hostsub == NULL)
	{
		g_message ("No valid host name: nothing updated");
		gtk_image_set_from_stock (img, "gtk-goto-bottom", GTK_ICON_SIZE_BUTTON);
		return FALSE;
	}
	subdir = strchr (hostsub, '/');
	if (!subdir)
		host = g_strconcat (hostsub, "/klavaro", NULL);
	else
		host = g_strdup (hostsub);
	g_free (hostsub);

	/**************************************************
	 * Download from downhost
	 */
	ksc = top10_get_score_file (GLOBAL, -1);

	if (! (curl = curl_easy_init ()) )
	{
		g_message ("Not able to initialize 'curl'");
		gtk_image_set_from_stock (img, "gtk-goto-bottom", GTK_ICON_SIZE_BUTTON);
		g_free (host);
		return FALSE;
	}
	tmp = g_strdup ("/tmp/klavaro");
	if (!g_file_test (tmp, G_FILE_TEST_IS_DIR))
#ifdef G_OS_UNIX
		g_mkdir (tmp, S_IRWXU | S_IRWXG | S_IRWXO | S_IWGRP | S_IWOTH);
#else
		g_mkdir (tmp, 0xFFFF);
#endif
	g_free (tmp);
	tmp = g_build_filename ("/tmp/klavaro", ksc, NULL);
	fail = TRUE;
	command = g_strdup_printf ("http://www.%s/%s", host, ksc);
	if ( (fh = g_fopen (tmp, "wb")) )
	{
		//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
		curl_easy_setopt (curl, CURLOPT_URL, command);
		curl_easy_setopt (curl, CURLOPT_WRITEDATA, fh);
		fail = curl_easy_perform (curl);
		fclose (fh);
	}
	curl_easy_cleanup (curl);
	g_free (command);
	g_free (host);

	if (fail)
	{
		top10_print (_("Could not download file from the host server."));
		gtk_image_set_from_stock (img, "gtk-goto-bottom", GTK_ICON_SIZE_BUTTON);
		g_free (ksc);
		g_free (tmp);
		g_rmdir ("/tmp/klavaro");
		return FALSE;
	}

	success = g_file_test (tmp, G_FILE_TEST_IS_REGULAR);
	if (success)
	{
		top10_check_writable (ksc);
		command = g_strconcat ("klavaro_helper ", ksc, NULL);
		success = g_spawn_command_line_sync (command, NULL, NULL, NULL, &error);
		if (!success)
		{
			g_free (command);
			command = g_strconcat ("./klavaro_helper ", ksc, NULL);
			success = g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
		}
		g_free (command);
		g_unlink (tmp);
		if (success)
		{
			top10_show_stats (GLOBAL);
			top10_check_writable (ksc);
		}
		else
		{
			g_free (tmp);
			tmp = g_strdup_printf ("%s\n %s", _("Failed to copy scoring file."),
					       error->message);
			top10_print (tmp);
		}
		if (error != NULL)
			g_error_free (error);
	}
	else
		top10_print (_("No file to download in the host server."));

	gtk_image_set_from_stock (img, "gtk-goto-bottom", GTK_ICON_SIZE_BUTTON);
	g_free (tmp);
	g_free (ksc);
	g_rmdir ("/tmp/klavaro");

	return FALSE;
}

gboolean
top10_global_publish (gpointer data)
{
	gboolean success;
	gchar *tmp;
	gchar *ksc;
	gchar *hostsub;
	gchar *host;
	gchar *subdir;
	gchar *user;
	gchar *pass;
	gchar *usrpas;
	gchar *path;
	gchar *username;
	gchar *command;
	GtkImage *img;
	gpointer gp;
	FILE *fh;
	struct stat fs;
	CURL *curl;
	struct curl_slist *comms = NULL;
	
	gp = data;
	fs.st_size = 0;
	img = GTK_IMAGE (lookup_widget (window_top10_, "image_top10_publish"));

	if (!main_get_curl_ok ())
	{
		tmp = g_strconcat (_("Not able to upload files"), ": 'libcurl' ", _("not found"), ". ",
		       _("Are you sure you have it installed in your system?"), NULL);
		top10_print (tmp);
		g_free (tmp);
		gtk_image_set_from_stock (img, "gtk-goto-top", GTK_ICON_SIZE_BUTTON);
		return FALSE;
	}

	if (! (curl = curl_easy_init ()))
	{
		g_message ("Not able to initialize curl session");
		gtk_image_set_from_stock (img, "gtk-goto-top", GTK_ICON_SIZE_BUTTON);
		return FALSE;
	}

	hostsub = top10_get_host ("uphost");
	if (hostsub == NULL)
	{
		g_message ("No valid host name: nothing uploaded");
		gtk_image_set_from_stock (img, "gtk-goto-top", GTK_ICON_SIZE_BUTTON);
		return FALSE;
	}
	subdir = strchr (hostsub, '/');
	if (!subdir)
		host = g_strdup (hostsub);
	else
		host = g_strndup (hostsub, subdir - hostsub);
	g_free (hostsub);

	user = top10_get_user ("upuser");
	if (user == NULL)
	{
		g_message ("No user name: nothing uploaded");
		gtk_image_set_from_stock (img, "gtk-goto-top", GTK_ICON_SIZE_BUTTON);
		return FALSE;
	}

	pass = top10_get_pass ("uppass");
	if (pass == NULL)
	{
		g_message ("No password: nothing uploaded");
		gtk_image_set_from_stock (img, "gtk-goto-top", GTK_ICON_SIZE_BUTTON);
		return FALSE;
	}
	usrpas = g_strconcat (user, ":", pass, NULL);
	g_free (user); g_free (pass);

	/**************************************************
	 * Upload to uphost
	 */
	tmp = main_preferences_get_string ("interface", "language");
	if (tmp[0] == 'C')
	{
		g_free (tmp);
		tmp = g_strdup ("en");
	}
	ksc = g_strdup_printf ("local_%c%c.ksc", tmp[0], tmp[1]);
	path = g_build_filename ("/var/games/klavaro/", ksc, NULL);
	g_free (ksc);
	g_stat (path, &fs);

	username = g_strdup (g_get_real_name ());
	username = g_strdelimit (username, " ", '_');
	ksc = g_strdup_printf ("%s_%s_%c%c.ksc", username, g_get_host_name (), tmp[0], tmp[1]);
	g_free (username);
	g_free (tmp);

	command = g_strdup_printf ("ftp://%s/%s", host, ksc);
	g_free (host);
	g_free (ksc);

	//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
	curl_easy_setopt (curl, CURLOPT_UPLOAD, TRUE);
	curl_easy_setopt (curl, CURLOPT_USERPWD, usrpas);
	curl_easy_setopt (curl, CURLOPT_INFILESIZE, (long) fs.st_size);
	curl_easy_setopt (curl, CURLOPT_URL, command);
	success = FALSE;
	if ( (fh = g_fopen (path, "rb")) )
	{
		curl_easy_setopt (curl, CURLOPT_READDATA, fh);
		if (curl_easy_perform (curl))
			g_message ("FTP upload failed!");
		else
			success = TRUE;
		fclose (fh);
	}
	curl_easy_cleanup (curl);
	g_free (path);
	g_free (usrpas);
	g_free (command);

	if (!success)
		top10_print (_("Could not upload local scores."));
	else
		top10_print (_("Thanks, it seems that your local scores were uploaded correctly. "
				"Every day a new ranking should be managed by the maintainer of the host server. "
				"You are supposed to wait some time before getting your achievement up there..."));

	gtk_image_set_from_stock (img, "gtk-goto-top", GTK_ICON_SIZE_BUTTON);
	return FALSE;
}

#define MAX_KSC_FILES 200
#define MAX_LANGS 79
void
top10_global_rank ()
{
	glong i, j, k;
	gsize size;
	gchar *file[MAX_KSC_FILES];
	glong file_len;
	gboolean success;
	gboolean fail;
	gchar *msg;
	gchar *tmp;
	gchar *ksc;
	gchar *hostsub;
	gchar *host;
	gchar *subdir;
	gchar *user;
	gchar *pass;
	gchar *usrpas;
	gchar *command;
	gchar **filelist;
	gchar str[6][400];
	gint dummy[2];
	gboolean lang_update[MAX_LANGS];
	FILE *fh;
	struct stat fs;
	CURL *curl;
	struct curl_slist *comms = NULL;

	if (!main_get_curl_ok ())
	{
		g_warning ("Not able to upload files: 'libcurl' not installed.");
		return;
	}

	host = top10_get_host ("uphost");
	if (host == NULL)
	{
		g_message ("Nothing updated");
		return;
	}

	user = top10_get_user ("upuser");
	if (user == NULL)
	{
		g_message ("Nothing updated");
		return;
	}

	pass = top10_get_pass ("uppass");
	if (pass == NULL)
	{
		g_message ("Nothing updated");
		return;
	}

	/* Get listing of files in the upload host
	 */
	fail = TRUE;
	command = g_strdup_printf ("ftp://%s/", host);
	curl = curl_easy_init ();
	if (curl)
	{
		tmp = g_strconcat (user, ":", pass, NULL);
		if ( (fh = g_fopen ("/tmp/.listing", "wb")) )
		{
			//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
			curl_easy_setopt (curl, CURLOPT_FTPLISTONLY, 1);
			curl_easy_setopt (curl, CURLOPT_WRITEDATA, fh);
			curl_easy_setopt (curl, CURLOPT_USERPWD, tmp);
			curl_easy_setopt (curl, CURLOPT_URL, command);
			curl_easy_perform (curl);
			if (curl_easy_perform (curl))
				g_message ("Retrieving .listing file failed!");
			else
				fail = FALSE;
			fclose (fh);
		}
		curl_easy_cleanup (curl);
		g_free (tmp);
	}

	g_free (command);
	if (fail)
	{
		g_free (host);
		g_free (user);
		g_free (pass);
		g_warning ("Not able to connect to the uploading host");
		return;
	}

	if (!g_file_get_contents ("/tmp/.listing", &tmp, NULL, NULL))
	{
		g_free (host);
		g_free (user);
		g_free (pass);
		g_warning ("Not able to get a file list from the uploading host");
		return;
	}
	unlink ("/tmp/.listing");

	filelist = g_strsplit_set (tmp, "\n\r", 0);
	g_free (tmp);

	/* Select which files to download
	 */
	fail = FALSE;
	i = j = k = 0;
	while (filelist[i])
	{
		if (! g_str_has_suffix (filelist[i], ".ksc"))
		{
			i++;
			continue;
		}
		for (k = 0; k < j; k++)
			if (g_str_equal (filelist[i], file[k]))
			{
				fail = TRUE;
				break;
			}
		if (fail)
			break;
		file[j++] = g_strdup (filelist[i]);
		if (j == MAX_KSC_FILES)
		{
			g_warning ("Maximum number of files to process reached: %i", j);
			break;
		}
		i++;
	}
	file_len = j;
	g_strfreev (filelist);

	/* Download the new files from upload host
	 */
	fail = TRUE;
	msg = g_strconcat (user, ":", pass, NULL);
	g_free (user); g_free (pass);
	if (! (curl = curl_easy_init ()) )
	{
		g_free (host);
		g_free (msg);
		return;
	}
	//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
	curl_easy_setopt (curl, CURLOPT_USERPWD, msg);
	g_mkdir ("/tmp/klavaro", S_IRWXU | S_IRWXG | S_IRWXO | S_IWGRP | S_IWOTH);
	g_printf ("Downloaded:\n");
	for (i = 0; i < file_len; i++)
	{
		command = g_strdup_printf ("ftp://%s/%s", host, file[i]);
		tmp = g_build_filename ("/tmp/klavaro", file[i], NULL);
		if ( (fh = g_fopen (tmp, "wb")) )
		{
			curl_easy_setopt (curl, CURLOPT_URL, command);
			curl_easy_setopt (curl, CURLOPT_WRITEDATA, fh);
			curl_easy_perform (curl);
			fclose (fh);
			g_printf ("\t%s\n", file[i]);
		}
		g_free (tmp);
		g_free (command);
	}
	curl_easy_cleanup (curl);
	g_free (msg);
	g_free (host);

	/* Rank local files into global files
	 */
	for (i = 0; trans_get_code (i) != NULL && i < MAX_LANGS; i++)
	{
		lang_update[i] = FALSE;
		if (i > 0)
			if (g_str_equal (trans_get_code (i), trans_get_code (i - 1)))
				continue;

		top10_read_stats (GLOBAL, i);
		tmp = g_strconcat ("_", trans_get_code (i), ".ksc", NULL);
		success = 0;
		for (j = 0; j < file_len; j++)
		{
			if (g_str_has_suffix (file[j], tmp))
			{
				ksc = g_build_filename ("/tmp/klavaro", file[j], NULL);
				if (!top10_read_stats_from_file (LOCAL, ksc))
					g_message ("Invalid file: %s", ksc);
				else
				{
					for (k = 0; k < 10; k++)
					{
						if (!top10_validate_stat (&top10_local[k]))
							continue;
						if (top10_compare_insert_stat (&top10_local[k], GLOBAL))
						{
							lang_update[i] = TRUE;
							success++;
						}
					}
				}
				msg = g_build_filename ("/tmp", file[j], NULL);
				if (g_rename (ksc, msg))
					g_unlink (ksc);
				g_free (ksc);
				g_free (msg);
			}
		}
		g_free (tmp);
		if (success)
			top10_write_stats (GLOBAL, i);
	}
	g_rmdir ("/tmp/klavaro");

	/* Upload new rankings to download host
	 */
	hostsub = top10_get_host ("downhost");
	if (hostsub == NULL)
	{
		g_message ("Nothing updated");
		return;
	}
	subdir = strchr (hostsub, '/');
	if (!subdir)
		host = g_strdup (hostsub);
	else
		host = g_strndup (hostsub, subdir - hostsub);
	g_free (hostsub);

	user = top10_get_user ("downuser");
	if (user == NULL)
	{
		g_message ("Nothing updated");
		return;
	}

	pass = top10_get_pass ("downpass");
	if (pass == NULL)
	{
		g_message ("Nothing updated");
		return;
	}

	curl = curl_easy_init ();
	if (! curl)
	{
		g_message ("Nothing updated");
		g_free (host);
		g_free (user);
		g_free (pass);
		return;
	}
	curl_easy_setopt (curl, CURLOPT_UPLOAD, TRUE);
	usrpas = g_strdup_printf ("%s:%s", user, pass);
	curl_easy_setopt (curl, CURLOPT_USERPWD, usrpas);
	//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
	g_printf ("Uploaded:\n");
	for (i = 0; trans_get_code (i) != NULL && i < MAX_LANGS; i++)
	{
		if (i > 0)
			if (g_str_equal (trans_get_code (i), trans_get_code (i - 1)))
				continue;
		if (lang_update[i] == FALSE)
			continue;

		msg = g_strconcat ("global_", trans_get_code (i), ".ksc", NULL);
		ksc = g_strconcat ("/var/games/klavaro/", msg, NULL);
		command = g_strdup_printf ("ftp://%s/%s", host, msg);
		success = FALSE;
		if ( (fh = g_fopen (ksc, "rb")))
		{
			fs.st_size = 0;
			g_stat (ksc, &fs);
			curl_easy_setopt (curl, CURLOPT_INFILESIZE, (long) fs.st_size);
			curl_easy_setopt (curl, CURLOPT_URL, command);
			curl_easy_setopt (curl, CURLOPT_READDATA, fh);
			if (curl_easy_perform (curl))
				g_message ("FTP upload failed!");
			else
				success = TRUE;
			fclose (fh);
			g_printf ("\t%s\n", msg);
		}
		g_free (msg);
		g_free (ksc);
		g_free (command);
	}
	curl_easy_cleanup (curl);

	g_free (host);
	g_free (user);
	g_free (pass);
	g_free (usrpas);

	/* Remove downloaded files in uphost
	 */
	host = top10_get_host ("uphost");
	if (host == NULL)
	{
		g_message ("Nothing updated");
		return;
	}

	user = top10_get_user ("upuser");
	if (user == NULL)
	{
		g_message ("Nothing updated");
		return;
	}

	pass = top10_get_pass ("uppass");
	if (pass == NULL)
	{
		g_message ("Nothing updated");
		return;
	}

	curl = curl_easy_init ();
	if (! curl)
	{
		g_message ("Nothing updated");
		g_free (host);
		g_free (user);
		g_free (pass);
		return;
	}

	//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
	for (i = 0; i < file_len; i++)
	{
		tmp = g_strdup_printf ("DELE %s", file[i]);
		comms = curl_slist_append (comms, tmp);
		g_free (tmp);
	}
	usrpas = g_strdup_printf ("%s:%s", user, pass);
	command = g_strdup_printf ("ftp://%s", host);

	g_printf ("---\nNot deleted:\n");
	curl_easy_setopt (curl, CURLOPT_QUOTE, comms);
	curl_easy_setopt (curl, CURLOPT_FTPLISTONLY, 1);
	curl_easy_setopt (curl, CURLOPT_USERPWD, usrpas);
	curl_easy_setopt (curl, CURLOPT_URL, command);
	if (curl_easy_perform (curl))
		g_message ("Remote deletion failed!");
	curl_easy_cleanup (curl);
	
	g_free (host);
	g_free (user);
	g_free (pass);
	g_free (usrpas);
	g_free (command);
	curl_slist_free_all (comms);
}

gchar *
top10_get_host (gchar * updown)
{
	gchar *tmp;
	gchar *host = NULL;

	if (keyfile == NULL)
		keyfile = g_key_file_new ();

	if (main_preferences_get_boolean ("game", "default") == TRUE)
	{
		top10_keyfile_proverbaro ();
		host = g_key_file_get_string (keyfile, "proverbo", "kie", NULL);
	}
	else if (main_preferences_get_boolean ("game", "personal") == FALSE)
	{
		tmp = g_strconcat (_("No host is configured"), ": ", _("nothing done"), ".", NULL);
		top10_print (tmp);
		g_free (tmp);
		return (NULL);
	}
	else
		host = main_preferences_get_string ("game", updown);

	tmp = g_strconcat (_("No valid host"), ": ", _("nothing done"), ".", NULL);
	if (host == NULL)
		top10_print (tmp);
	else if (strlen (host) == 0)
	{
		top10_print (tmp);
		g_free (host);
		host = NULL;
	}
	g_free (tmp);
	return (host);
}

gchar *
top10_get_user (gchar * updown)
{
	gchar *tmp;
	gchar *user = NULL;
	gchar *host = NULL;

	if (keyfile == NULL)
		keyfile = g_key_file_new ();

	if (main_preferences_get_boolean ("game", "default") == TRUE)
	{
		if (top10_keyfile_proverbaro () == FALSE)
			return (NULL);
		host = g_key_file_get_string (keyfile, "proverbo", "kie", NULL);
		if (g_str_equal (updown, "downuser"))
			user = g_strconcat ("klavaro@", host, NULL);
		else
		{
			tmp = g_key_file_get_string (keyfile, "proverbo", "kiu", NULL);
			user = g_strconcat (tmp, "@", host, NULL);
			g_free (tmp);
		}
		g_free (host);
	}
	else if (main_preferences_get_boolean ("game", "personal") == FALSE)
	{
		tmp = g_strconcat (_("No host is configured"), ": ", _("nothing done"), ".", NULL);
		top10_print (tmp);
		g_free (tmp);
		return (NULL);
	}
	else
		user = main_preferences_get_string ("game", updown);

	if (user == NULL)
	{
		tmp = g_strconcat (_("No valid user is defined"), ": ", _("nothing done"), ".", NULL);
		top10_print (tmp);
		g_free (tmp);
		return (NULL);
	}
	else if (strlen (user) == 0)
	{
		tmp = g_strconcat (_("No valid user is defined"), ": ", _("nothing done"), ".", NULL);
		top10_print (tmp);
		g_free (tmp);
		g_free (user);
		return (NULL);
	}
	return (user);
}

gchar *
top10_get_pass (gchar * updown)
{
	gsize i, j;
	gsize max;
	gchar *tmp;
	gchar *pass = NULL;
	time_t when;
	struct tm *ltime;

	when = time (NULL);
	ltime = localtime (&when);

	if (keyfile == NULL)
		keyfile = g_key_file_new ();

	if (main_preferences_get_boolean ("game", "default") == TRUE)
	{
		if (g_str_equal (updown, "downpass"))
		{
			tmp = g_strconcat (main_get_user_dir (), "adm.txt", NULL);
			if (! g_file_get_contents (tmp, &pass, NULL, NULL))
			{
				g_free (tmp);
				g_free (pass);
				g_message ("You are not the administrator of Klavaro's default host. Nothing done");
				return (NULL);
			}
			g_free (tmp);
		}
		else
		{
			if (top10_keyfile_proverbaro () == FALSE)
				return (NULL);
			tmp = g_key_file_get_string (keyfile, "proverbo", "kiel", NULL);
			pass = g_strdup (tmp);
			max = strlen (pass) - 1;
			for (i = 0, j = 0; i < max; i++)
			{
				tmp[j] = pass[i];
				if (pass[i] == ' ')
					pass[i + 1] = g_ascii_toupper (pass[i + 1]);
				else
					j++;
			}
			tmp[j] = pass[i];
			tmp[j + 1] = '\0';
			g_free (pass);
			pass = g_strdup_printf ("%s%i", tmp, ltime->tm_year + 1900);
			g_free (tmp);
		}
	}
	else if (main_preferences_get_boolean ("game", "personal") == FALSE)
	{
		tmp = g_strconcat (_("No host is configured"), ": ", _("nothing done"), ".", NULL);
		top10_print (tmp);
		g_free (tmp);
		return (NULL);
	}
	else
		pass = main_preferences_get_string ("game", updown);

	if (pass == NULL)
	{
		tmp = g_strconcat (_("No valid password is defined"), ": ", _("nothing done"), ".", NULL);
		top10_print (tmp);
		g_free (tmp);
		return (NULL);
	}
	if (strlen (pass) == 0)
	{
		tmp = g_strconcat (_("No valid password is defined"), ": ", _("nothing done"), ".", NULL);
		top10_print (tmp);
		g_free (tmp);
		g_free (pass);
		return (NULL);
	}
	return (pass);
}

gboolean
top10_keyfile_proverbaro ()
{
	gboolean success;
	gchar *tmp;
	gchar *command;
	FILE *fh;
	CURL *curl;

	if (keyfile == NULL)
		keyfile = g_key_file_new ();

	tmp = g_strconcat (main_get_user_dir (), "proverbaro.desktop", NULL);
	if (!g_file_test (tmp, G_FILE_TEST_IS_REGULAR))
	{
		fh = g_fopen (tmp, "wb");
		if (!fh)
		{
			g_message ("Not able to write to: %s", tmp);
			g_free (tmp);
			return FALSE;
		}
		if (! (curl = curl_easy_init ()) )
		{
			g_message ("Not able to initialize 'curl'");
			fclose (fh);
			g_free (tmp);
			return FALSE;
		}
		//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
		curl_easy_setopt (curl, CURLOPT_WRITEDATA, fh);
		curl_easy_setopt (curl, CURLOPT_URL, "http://klavaro.sourceforge.net/proverbaro.desktop");
		if (curl_easy_perform (curl))
		{
			top10_print (_("Not able to reach default host."));
			curl_easy_cleanup (curl);
			fclose (fh);
			g_free (tmp);
			return FALSE;
		}
		curl_easy_cleanup (curl);
		fclose (fh);
	}
	g_key_file_load_from_file (keyfile, tmp, G_KEY_FILE_NONE, NULL);
	g_free (tmp);
	return TRUE;
}

gboolean
top10_host_test (gpointer data)
{
	guint i, j;
	guint max;
	static gint step = 0;
	static gint step_default = 0;
	gboolean success;
	gchar *msg;
	gchar *hostsub;
	static gchar *host = NULL;
	gchar *subdir;
	static gchar *user = NULL;
	static gchar *pass = NULL;
	gchar *output = NULL;
	gchar *outerr = NULL;
	gchar *command;
	gchar *usrpas;
	time_t when;
	static struct tm *ltime;
	static gchar *upfile = NULL;
	static GtkTextBuffer *buf = NULL;
	static GtkTextView *tv;
	GtkWidget *wg;
	FILE *fh;
	CURL *curl;

	when = time (NULL);
	ltime = localtime (&when);

	if (keyfile == NULL)
		keyfile = g_key_file_new ();

	if (upfile == NULL)
		upfile = g_strconcat (main_get_data_path (), "klavaro_test.txt", NULL);
	if (!g_file_test (upfile, G_FILE_TEST_IS_REGULAR))
		g_error (upfile);

	switch (step)
	{
	case 0:
		/* Prepare to show messages
		 */
		tv = GTK_TEXT_VIEW (lookup_widget (window_top10_, "textview_top10_external"));
		buf = gtk_text_view_get_buffer (tv);
		gtk_text_buffer_set_text (buf, "", -1);

		/* Check libcurl
		 */
		gtk_text_buffer_insert_at_cursor (buf, _("Testing 'libcurl': "), -1);
		if (success = main_get_curl_ok ())
			gtk_text_buffer_insert_at_cursor (buf, _("YES! :-)"), -1);
		else
			gtk_text_buffer_insert_at_cursor (buf, _("NO... :-("), -1);
		gtk_text_buffer_insert_at_cursor (buf, "\n====================\n\n", -1);

		if (!success)
		{
			step = 0;
			return FALSE;
		}

		if (main_preferences_get_boolean ("game", "default") == TRUE)
		{
			wg = lookup_widget (window_top10_, "image_top10_update");
			gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-yes", GTK_ICON_SIZE_BUTTON);

			msg = g_strconcat (_("Default host"), " - ", _("querying"),
					   "\n", NULL);
			gtk_text_buffer_insert_at_cursor (buf, msg, -1);
			g_free (msg);
			gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));
		}

		step++;
		return TRUE;
		break;

	case 1:
		/**************************************************
		 * Test default host
		 */
		if (main_preferences_get_boolean ("game", "default") == FALSE)
		{
			step++;
			step_default = 0;
			return TRUE;
		}
		break;

	case 2:
		/**************************************************
		 * Test no host
		 */
		if (main_preferences_get_boolean ("game", "personal") == FALSE)
		{
			msg = g_strconcat (_("No host is configured"), ": ", _("nothing done"), ".",
					   NULL);
			gtk_text_buffer_set_text (buf, msg, -1);
			g_free (msg);
			step = 0;
			return FALSE;
		}
		wg = lookup_widget (window_top10_, "image_top10_publish");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-yes", GTK_ICON_SIZE_BUTTON);

		msg = g_strconcat (_("HOST 1"), " - ", _("uploading"), "\n",
				   NULL);
		gtk_text_buffer_insert_at_cursor (buf, msg, -1);
		g_free (msg);
		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		step++;
		return TRUE;
		break;

	case 3:
		/**************************************************
		 * Test uploading to uphost
		 */
		hostsub = main_preferences_get_string ("game", "uphost");
		subdir = strchr (hostsub, '/');
		if (!subdir)
			host = g_strdup (hostsub);
		else
			host = g_strndup (hostsub, subdir - hostsub);
		g_free (hostsub);

		user = main_preferences_get_string ("game", "upuser");
		pass = main_preferences_get_string ("game", "uppass");
		usrpas = g_strdup_printf ("%s:%s", user, pass);
		g_free (pass);
		g_free (user);
		
		/* Do the upload
		 */
		command = g_strdup_printf ("ftp://%s/klavaro_test.txt", host);
		g_free (host);
		success = FALSE;
		curl = curl_easy_init ();
		if (curl)
		{
			if ( (fh = g_fopen (upfile, "rb")) )
			{
				//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
				curl_easy_setopt (curl, CURLOPT_UPLOAD, TRUE);
				curl_easy_setopt (curl, CURLOPT_READDATA, fh);
				curl_easy_setopt (curl, CURLOPT_INFILESIZE, 0);
				curl_easy_setopt (curl, CURLOPT_USERPWD, usrpas);
				curl_easy_setopt (curl, CURLOPT_URL, command);
				if (curl_easy_perform (curl))
					g_message ("FTP upload failed!");
				else
					success = TRUE;
				fclose (fh);
			}
			curl_easy_cleanup (curl);
		}
		g_free (usrpas);
		g_free (command);

		/* Put the results
		 */
		gtk_text_buffer_insert_at_cursor (buf, "=====> ", -1);
		if (!success)
			gtk_text_buffer_insert_at_cursor (buf, _("FAIL\n\n"), -1);
		else
			gtk_text_buffer_insert_at_cursor (buf, _("OK!\n\n"), -1);

		wg = lookup_widget (window_top10_, "image_top10_publish");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-goto-top", GTK_ICON_SIZE_BUTTON);

		wg = lookup_widget (window_top10_, "image_top10_update");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-yes", GTK_ICON_SIZE_BUTTON);

		msg = g_strconcat (_("HOST 2"), " - ", _("downloading"), "\n",
				   NULL);
		gtk_text_buffer_insert_at_cursor (buf, msg, -1);
		g_free (msg);
		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		step++;
		return TRUE;
		break;

	case 4:
		/**************************************************
		 * Test downloading from downhost
		 */
		host = main_preferences_get_string ("game", "downhost");
		command = g_strdup_printf ("http://www.%s/klavaro_test.txt", host);
		g_free (host);
		success = FALSE;
		curl = curl_easy_init ();
		if (curl)
		{
			if ( (fh = g_fopen ("/tmp/klavaro_test.txt", "wb")) )
			{
				//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
				curl_easy_setopt (curl, CURLOPT_WRITEDATA, fh);
				curl_easy_setopt (curl, CURLOPT_URL, command);
				curl_easy_perform (curl);
				if (curl_easy_perform (curl))
					g_message ("Downloading file from downhost failed!");
				else
					success = TRUE;
				fclose (fh);
			}
			curl_easy_cleanup (curl);
		}
		g_free (command);

		gtk_text_buffer_insert_at_cursor (buf, "=====> ", -1);
		if (success)
		{
			gtk_text_buffer_insert_at_cursor (buf, _("OK!\n\n"), -1);
			g_unlink ("/tmp/klavaro_test.txt");
		}
		else
			gtk_text_buffer_insert_at_cursor (buf, _("FAIL\n\n"), -1);

		wg = lookup_widget (window_top10_, "image_top10_update");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-goto-bottom", GTK_ICON_SIZE_BUTTON);

		if (main_preferences_get_boolean ("game", "manager") == FALSE)
		{
			gtk_text_buffer_insert_at_cursor (buf,
							  _("No manager is configured: skipping upload to HOST 2."),
							  -1);
			gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));
			step = 0;
			return FALSE;
		}

		wg = lookup_widget (window_top10_, "image_top10_publish");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-yes", GTK_ICON_SIZE_BUTTON);

		msg = g_strconcat (_("HOST 2"), " - ", _("uploading"), "\n",
				   NULL);
		gtk_text_buffer_insert_at_cursor (buf, msg, -1);
		g_free (msg);
		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		step++;
		return TRUE;
		break;

	case 5:
		/**************************************************
		 * Test upload to downhost
		 */
		hostsub = main_preferences_get_string ("game", "downhost");
		subdir = strchr (hostsub, '/');
		if (!subdir)
			host = g_strdup (hostsub);
		else
			host = g_strndup (hostsub, subdir - hostsub);
		g_free (hostsub);

		user = main_preferences_get_string ("game", "downuser");
		pass = main_preferences_get_string ("game", "downpass");
		usrpas = g_strdup_printf ("%s:%s", user, pass);
		g_free (pass);
		g_free (user);

		/* Do the upload
		 */
		command = g_strdup_printf ("ftp://%s/klavaro_test.txt", host);
		g_free (host);
		success = FALSE;
		curl = curl_easy_init ();
		if (curl)
		{
			if ( (fh = g_fopen (upfile, "rb")) )
			{
				//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
				curl_easy_setopt (curl, CURLOPT_UPLOAD, TRUE);
				curl_easy_setopt (curl, CURLOPT_READDATA, fh);
				curl_easy_setopt (curl, CURLOPT_INFILESIZE, 0);
				curl_easy_setopt (curl, CURLOPT_USERPWD, usrpas);
				curl_easy_setopt (curl, CURLOPT_URL, command);
				if (curl_easy_perform (curl))
					g_message ("FTP upload failed!");
				else
					success = TRUE;
				fclose (fh);
			}
			curl_easy_cleanup (curl);
		}
		g_free (usrpas);
		g_free (command);

		/* Put the results
		 */
		gtk_text_buffer_insert_at_cursor (buf, "=====> ", -1);
		if (!success)
			gtk_text_buffer_insert_at_cursor (buf, _("FAIL\n\n"), -1);
		else
			gtk_text_buffer_insert_at_cursor (buf, _("OK!\n\n"), -1);

		wg = lookup_widget (window_top10_, "image_top10_publish");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-goto-top", GTK_ICON_SIZE_BUTTON);

		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		step = 0;
		return FALSE;
		break;
	}

	/**************************************************
	 * Step in the default hosting tests
	 */
	switch (step_default)
	{
	case 0:
		/* Get the config file of the default host
		 */
		success = FALSE;
		curl = curl_easy_init ();
		if (curl)
		{
			msg = g_strconcat (main_get_user_dir (), "proverbaro.desktop", NULL);
			if ( (fh = g_fopen (msg, "wb")) )
			{
				//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
				curl_easy_setopt (curl, CURLOPT_WRITEDATA, fh);
				curl_easy_setopt (curl, CURLOPT_URL, "http://klavaro.sourceforge.net/proverbaro.desktop");
				curl_easy_perform (curl);
				if (curl_easy_perform (curl))
					g_message ("Config file download failed!");
				else
					success = TRUE;
				fclose (fh);
			}
			curl_easy_cleanup (curl);
			g_free (msg);
		}

		/* Set inactive state
		 */
		wg = lookup_widget (window_top10_, "image_top10_update");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-goto-bottom", GTK_ICON_SIZE_BUTTON);

		/* Show the result
		 */
		gtk_text_buffer_insert_at_cursor (buf, "=====> ", -1);
		if (!success)
		{
			gtk_text_buffer_insert_at_cursor (buf, _("FAIL\n\n"), -1);
			gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));
			step_default = 0;
			step = 0;
			return FALSE;
		}

		gtk_text_buffer_insert_at_cursor (buf, _("OK!\n\n"), -1);
		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		/* Prepare next step for uploading
		 */
		wg = lookup_widget (window_top10_, "image_top10_publish");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-yes", GTK_ICON_SIZE_BUTTON);

		msg = g_strconcat (_("HOST 1"), " - ", _("uploading"), "\n",
				   NULL);
		gtk_text_buffer_insert_at_cursor (buf, msg, -1);
		g_free (msg);
		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		msg = g_strconcat (main_get_user_dir (), "proverbaro.desktop", NULL);
		g_key_file_load_from_file (keyfile, msg, G_KEY_FILE_NONE, NULL);
		g_free (msg);

		host = g_key_file_get_string (keyfile, "proverbo", "kie", NULL);
		msg = g_key_file_get_string (keyfile, "proverbo", "kiu", NULL);
		pass = g_key_file_get_string (keyfile, "proverbo", "kiel", NULL);
		user = g_strconcat (msg, "@", host, NULL);
		g_free (msg);

		step_default++;
		return TRUE;
		break;

	case 1:
		/* Test uploading to default uphost
		 */
		wg = lookup_widget (window_top10_, "image_top10_publish");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-goto-top", GTK_ICON_SIZE_BUTTON);

		msg = g_strdup (pass);
		max = strlen (pass) - 1;
		for (i = 0, j = 0; i < max; i++)
		{
			pass[j] = msg[i];
			if (msg[i] == ' ')
				msg[i + 1] = g_ascii_toupper (msg[i + 1]);
			else
				j++;
		}
		pass[j] = msg[i];
		pass[j + 1] = '\0';
		g_free (msg);
		msg = g_strdup_printf ("%s%i", pass, ltime->tm_year + 1900);
		g_free (pass);

		/* Do the upload
		 */
		command = g_strdup_printf ("ftp://%s/klavaro_test.txt", host);
		usrpas = g_strdup_printf ("%s:%s", user, msg);
		success = FALSE;
		curl = curl_easy_init ();
		if (curl)
		{
			if ( (fh = g_fopen (upfile, "rb")) )
			{
				//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
				curl_easy_setopt (curl, CURLOPT_UPLOAD, TRUE);
				curl_easy_setopt (curl, CURLOPT_READDATA, fh);
				curl_easy_setopt (curl, CURLOPT_INFILESIZE, 0);
				curl_easy_setopt (curl, CURLOPT_USERPWD, usrpas);
				curl_easy_setopt (curl, CURLOPT_URL, command);
				if (curl_easy_perform (curl))
					g_message ("FTP upload failed!");
				else
					success = TRUE;
				fclose (fh);
			}
			curl_easy_cleanup (curl);
		}
		g_free (usrpas);

		/* Put the results
		 */
		if (!success)
		{
			gtk_text_buffer_insert_at_cursor (buf, outerr, -1);
			gtk_text_buffer_insert_at_cursor (buf, "=====> ", -1);
			gtk_text_buffer_insert_at_cursor (buf, _("FAIL\n\n"), -1);
			success = FALSE;
		}
		else
		{
			gtk_text_buffer_insert_at_cursor (buf, "=====> ", -1);
			gtk_text_buffer_insert_at_cursor (buf, _("OK!\n\n"), -1);
			success = TRUE;
		}
		g_free (msg);
		g_free (user);
		g_free (command);

		/* Prepare for next step: downloading
		 */
		wg = lookup_widget (window_top10_, "image_top10_publish");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-goto-top", GTK_ICON_SIZE_BUTTON);

		wg = lookup_widget (window_top10_, "image_top10_update");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-yes", GTK_ICON_SIZE_BUTTON);

		msg = g_strconcat (_("HOST 2"), " - ", _("downloading"), "\n",
				   NULL);
		gtk_text_buffer_insert_at_cursor (buf, msg, -1);
		g_free (msg);
		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		step_default++;
		return TRUE;
		break;

	case 2:
		/* Test downloading from default downhost
		 */
		command = g_strdup_printf ("http://www.%s/klavaro/klavaro_test.txt", host );
		success = FALSE;
		curl = curl_easy_init ();
		if (curl)
		{
			if ( (fh = g_fopen ("/tmp/klavaro_test.txt", "wb")) )
			{
				//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
				curl_easy_setopt (curl, CURLOPT_WRITEDATA, fh);
				curl_easy_setopt (curl, CURLOPT_URL, command);
				curl_easy_perform (curl);
				if (curl_easy_perform (curl))
					g_message ("Downloading file from downhost failed!");
				else
					success = TRUE;
				fclose (fh);
			}
			curl_easy_cleanup (curl);
		}
		g_free (command);

		/* Put the results
		 */
		gtk_text_buffer_insert_at_cursor (buf, "=====> ", -1);
		if (success)
		{
			gtk_text_buffer_insert_at_cursor (buf, _("OK!\n\n"), -1);
			g_unlink ("/tmp/klavaro_test.txt");
		}
		else
			gtk_text_buffer_insert_at_cursor (buf, _("FAIL\n\n"), -1);
		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		/* Prepare for next test
		 */
		wg = lookup_widget (window_top10_, "image_top10_update");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-goto-bottom", GTK_ICON_SIZE_BUTTON);

		msg = g_strconcat (main_get_user_dir (), "adm.txt", NULL);
		if (!g_file_get_contents (msg, &pass, NULL, NULL))
		{
			step = 0;
			step_default = 0;
			g_free (host);
			g_free (msg);
			return FALSE;
		}
		g_free (msg);
		user = g_strconcat ("klavaro@", host, NULL);

		wg = lookup_widget (window_top10_, "image_top10_publish");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-yes", GTK_ICON_SIZE_BUTTON);

		msg = g_strconcat (_("HOST 2"), " - ", _("uploading"), "\n",
				   NULL);
		gtk_text_buffer_insert_at_cursor (buf, msg, -1);
		g_free (msg);
		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		step_default++;
		return TRUE;
		break;

	case 3:
		/* Test upload to default downhost
		 */
		command = g_strdup_printf ("ftp://%s/klavaro_test.txt", host);
		usrpas = g_strdup_printf ("%s:%s", user, pass);
		success = FALSE;
		curl = curl_easy_init ();
		if (curl)
		{
			if ( (fh = g_fopen (upfile, "rb")) )
			{
				//curl_easy_setopt (curl, CURLOPT_VERBOSE, 1);
				curl_easy_setopt (curl, CURLOPT_UPLOAD, TRUE);
				curl_easy_setopt (curl, CURLOPT_READDATA, fh);
				curl_easy_setopt (curl, CURLOPT_INFILESIZE, 0);
				curl_easy_setopt (curl, CURLOPT_USERPWD, usrpas);
				curl_easy_setopt (curl, CURLOPT_URL, command);
				if (curl_easy_perform (curl))
					g_message ("FTP upload to downhost failed!");
				else
					success = TRUE;
				fclose (fh);
			}
			curl_easy_cleanup (curl);
		}

		gtk_text_buffer_insert_at_cursor (buf, "=====> ", -1);
		if (success)
			gtk_text_buffer_insert_at_cursor (buf, _("OK!\n\n"), -1);
		else
			gtk_text_buffer_insert_at_cursor (buf, _("FAIL\n\n"), -1);

		g_free (host);
		g_free (user);
		g_free (pass);
		g_free (usrpas);
		g_free (command);

		/* Finish testing tasks
		 */
		wg = lookup_widget (window_top10_, "image_top10_publish");
		gtk_image_set_from_stock (GTK_IMAGE (wg), "gtk-goto-top", GTK_ICON_SIZE_BUTTON);
		gtk_text_view_scroll_mark_onscreen (tv, gtk_text_buffer_get_insert (buf));

		step = 0;
		step_default = 0;
		return FALSE;
		break;
	}
	return TRUE;
}

