/*
 * This file is a part of KleanSweep.
 *
 * Copyright (C) 2005 Pawel Stolowski <pawel.stolowski@wp.pl>
 *
 * KleanSweep is free software; you can redestribute it and/or modify it
 * under terms of GNU General Public License by Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY. See GPL for more details.
 */

#include "sweeperwizard.h"
#include "sweepscanner.h"
#include <ktabwidget.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <kpushbutton.h>
#include <qvbuttongroup.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qsizepolicy.h>
#include <qlineedit.h>
#include <qcursor.h>
#include <qtooltip.h>
#include <qfile.h>
#include <qtextstream.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <ktar.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kurlrequester.h>
#include <unistd.h>
#include <errno.h>
#include <qdatetime.h>
#include <kprocess.h>
#include <kapplication.h>
#include <kaboutapplication.h>
#include <kiconloader.h>
#include <errno.h>
#include <string.h>
#include "logwidget.h"
#include "sweepfileinfo.h"
#include "fileviewitem.h"
#include "filelistview.h" 
#include "summarylabel.h"
#include "configdlg.h"
#include "kleansweepcfg.h"
#include "scanprogressdlg.h"

#ifdef DEBUG
#include <iostream>
#endif

int SweeperWizard::CommonColumns = FileName | FileDir | ModTime | Owner | Group | Permissions;

SweeperWizard::ScanOpts SweeperWizard::scanopts[] = {
	{SweepScanner::EMPTY_FILE, i18n("Empty files"),                 CommonColumns, i18n("Search for zero-length files")},
	{SweepScanner::EMPTY_DIR, i18n("Empty directories"),            CommonColumns, QString::null},
        {SweepScanner::BROKEN_SYMLINK, i18n("Broken symlinks"),         CommonColumns | Target, i18n("Search for symbolic links pointing to non-existing files")},
        {SweepScanner::BROKEN_EXEC, i18n("Broken executables"),         CommonColumns | Target, i18n("Search for executables in PATH with missing libraries")},
        {SweepScanner::BACKUP_FILE, i18n("Backup files"),               CommonColumns | Size, i18n("Search for files with common backup extensions (.bak, .tmp and others)")},
        {SweepScanner::ORPHAN_FILE, i18n("Orphaned files"),        CommonColumns | Size, i18n("Search for files that cannot be found in installed packages database of your system")},
        {SweepScanner::DEAD_MENUENTRY, i18n("Dead menu entries"),       CommonColumns | Size | Target, i18n("Search for desktop menu entires (.desktop files) pointing to non-existing executables")},
	{SweepScanner::DEAD_THUMBNAIL, i18n("Obsolete thumbnails"),     CommonColumns | Size | Target, i18n("Search for thumbnails pointing to non-existing images")},
        {SweepScanner::DUPLICATED_FILE, i18n("Duplicated files"),       CommonColumns | Size, i18n("Search for files with the same contents")},
	//{SweepScanner::OLD_FILE, i18n("Files not accessed recently"),   CommonColumns | AccessTime | Size, i18n("Search for files not accessed for specified amount of time")},
	{SweepScanner::UNUSED_LOCALE, i18n("Unused locales"),           CommonColumns | Size, i18n("Search for .po locale files for languages not used by your system")},
	{SweepScanner::UNUSED_LOCALE, QString::null, 0, QString::null}
};

SweeperWizard::SweeperWizard(QWidget *parent): KWizard(parent, 0, false, Qt::WDestructiveClose), scanner(NULL), numchoosen(0), progress(NULL), shown_startup_warning(false)
{
	setCaption("KleanSweep");

	kleansweep_bar1 = new QPixmap(::locate("data", "kleansweep/kleansweep_bar.png"));

        setupWelcomePage();
        setupStartPage();
        setupResultsPage();
	setupArchivePage();
	setupRemovePage();
	setNextEnabled(resultspage, false);
	setNextEnabled(startpage, false);
	setBackEnabled(removepage, false);

	setHelpEnabled(welcomepage, false);
	setHelpEnabled(startpage, false);
	setHelpEnabled(resultspage, false);
	setHelpEnabled(removepage, false);
	setHelpEnabled(archivepage, false);

	KleanSweepConfig::instance().readConfig();
		
}

SweeperWizard::~SweeperWizard()
{
	KleanSweepConfig::instance().writeConfig();
	delete scanner;
	delete kleansweep_bar1;
}

void SweeperWizard::show()
{
	KWizard::show();
	if (!shown_startup_warning)
	{
		shown_startup_warning = true;
		if (getuid() == 0)
			KMessageBox::information(NULL,
					i18n("You are running KleanSweep with root privileges. Therefore\n"
						"extreme caution is recommended as improper use may render\n"
						"your system unusable. Please keep in mind that because KleanSweep\n"
						"uses simple heuristics, all files suggested for removal should be\n"
						"manually reviewed."),
					QString::null, QString::null, KMessageBox::Dangerous);
		else
			KMessageBox::information(NULL,
					i18n("Since you don't have administrative privileges, you will be able to\n"
						"delete only those files that you have write access to! If you wish to\n"
						"perform system-wide cleaning, please run KleanSweep as root.\n"),
					QString::null, "RegularUserWarning");
	}
}

void SweeperWizard::moveEvent(QMoveEvent *e)
{
	if (progress)
	{
		const QPoint p = e->pos();
		const int x = p.x() + ((width() - progress->width()) / 2);
		const int y = p.y() + ((height() - progress->height()) / 2);
		progress->move(x, y);
	}
}

void SweeperWizard::resizeEvent(QResizeEvent *e)
{
	if (progress)
	{
		const int x = this->x() + ((width() - progress->width()) / 2);
		const int y = this->y() + ((height() - progress->height()) / 2);
		progress->move(x, y);
	}
}

void SweeperWizard::setupWelcomePage()
{
        welcomepage = new QWidget(this);

	QHBoxLayout *box0 = new QHBoxLayout(welcomepage, 5, 5);
	addLeftSidePic(welcomepage, box0);

	QString msg = i18n("Welcome to <B>KleanSweep</B>! This application allows you to free some disk space "
				       "by finding unneeded files on your system. You'll be given a chance to backup and "
				       "remove selected files and reclaim your disk space!<BR><BR><B>Beware! This utility is "
				       "meant for advanced users only. It is assumed that you have some basic understanding "
				       "of what can be safely deleted</B>. The author takes no responsibility "
				       "for any damage caused by using this program.");
	QLabel *info = new QLabel(msg, welcomepage);
	info->setTextFormat(Qt::RichText);			  
	info->setAlignment(Qt::AlignLeft|Qt::AlignTop|Qt::WordBreak);
	info->setFrameStyle(QFrame::Panel|QFrame::Sunken);
	box0->addWidget(info);

        addPage(welcomepage, "Welcome!");
}

void SweeperWizard::setupStartPage()
{
	KleanSweepConfig *cfg = &KleanSweepConfig::instance();

        startpage = new QWidget(this);
	QHBoxLayout *boxh = new QHBoxLayout(startpage, 5, 5);
	addLeftSidePic(startpage, boxh);
	QVBoxLayout *box0 = new QVBoxLayout(NULL, 0, 5);
	boxh->addLayout(box0);
	
	QLabel *info = new QLabel(i18n("Please select one or more checks to perform. Please keep in mind that some checks "
	                          "(seeking for duplicated or orphaned files) may take a lot of time, "
				  "depending on CPU, memory and disk space. Click <B>Next</B> when ready.") , startpage);
	info->setTextFormat(Qt::RichText);			  
	info->setAlignment(Qt::AlignLeft|Qt::AlignTop|Qt::WordBreak);
	box0->addWidget(info);

	QVButtonGroup *b_group0 = new QVButtonGroup(i18n("Seek for"), startpage);
	box0->addWidget(b_group0);
	//
	// create checkboxes for search categories
	for (int i=0; scanopts[i].optname!=QString::null; i++)
	{
		QCheckBox *cb = cb_scan[scanopts[i].ftype] = new QCheckBox(scanopts[i].optname, b_group0);
		if (scanopts[i].info != QString::null)
			QToolTip::add(cb, scanopts[i].info);
		connect(cb, SIGNAL(toggled(bool)), this, SLOT(scanCategoryToggled(bool)));
	}
	//KPushButton *b_allscans = new KPushButton(i18n("All"), b_group0);
	//b_allscans->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));

	//
	// disable unfinished checks
	cb_scan[SweepScanner::UNUSED_LOCALE]->setDisabled(true);
	
	//
	// disable checks for which essential external tools are missing
	if (KStandardDirs::findExe("identify") == QString::null)
		cb_scan[SweepScanner::DEAD_THUMBNAIL]->setDisabled(true);
	if (cfg->distroPackageManager() == OTHER)
		cb_scan[SweepScanner::ORPHAN_FILE]->setDisabled(true);

	QVButtonGroup *b_group1 = new QVButtonGroup(startpage);
	box0->addWidget(b_group1);
	QHBox *box2 = new QHBox(b_group1);
	box2->setSpacing(KDialog::spacingHint());
	KPushButton *pb_settings = new KPushButton(KGlobal::iconLoader()->loadIconSet("configure", KIcon::Small), i18n("Settings"), box2);
	pb_settings->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
	connect(pb_settings, SIGNAL(clicked()), this, SLOT(showConfigDialog()));
	cb_writable =  new QCheckBox(i18n("Only files/directories writable for me"), box2);
	QToolTip::add(cb_writable, i18n("Skips all files that cannot be written and therefore deleted by you"));
	QHBox *box1 = new QHBox(b_group1);
	box1->setSpacing(KDialog::spacingHint());
	new QLabel(i18n("Starting directory"), box1);
	url_startpath = new KURLRequester(box1);
	url_startpath->setMode(KFile::Directory | KFile::ExistingOnly | KFile::LocalOnly);
	//QToolTip::add(url_startpath, i18n("The directory to scan"));
	box0->addStretch();

	//
	// set starting directory
	url_startpath->setURL(getuid() == 0 ? "/" : QDir::homeDirPath());
	
        addPage(startpage, i18n("Select scan options"));
}

void SweeperWizard::setupResultsPage()
{
	resultspage = new QWidget(this);
	QVBoxLayout *box0 = new QVBoxLayout(resultspage, 5, 5);
	
        resultstabs = new KTabWidget(resultspage);
	box0->addWidget(resultstabs);

	QHBoxLayout *box1 = new QHBoxLayout(NULL, 0, 5, NULL);
	b_selectall = new KPushButton(i18n("Select all"), resultspage);
	b_clearall = new KPushButton(i18n("Clear all"), resultspage);
	b_saveresrep = new KPushButton(KGlobal::iconLoader()->loadIconSet("filesaveas", KIcon::Small), i18n("Save report"), resultspage);
	summary = new SummaryLabel(resultspage);
	b_stop = new KPushButton(KGlobal::iconLoader()->loadIconSet("stop", KIcon::Small), i18n("Stop"), resultspage);
	box1->addWidget(b_selectall);
	box1->addWidget(b_clearall);
	box1->addWidget(b_saveresrep);
	box1->insertSpacing(-1, 20);
	box1->addWidget(b_stop);
	box1->addStretch(1);
	box1->addWidget(summary);

	connect(b_selectall, SIGNAL(clicked()), this, SLOT(selectAllClicked()));
	connect(b_clearall, SIGNAL(clicked()), this, SLOT(clearAllClicked()));
	connect(b_saveresrep, SIGNAL(clicked()), this, SLOT(saveReport()));
	connect(b_stop, SIGNAL(clicked()), this, SLOT(stopClicked()));
	
	box0->addLayout(box1);

	for (int i=0; i<SweepScanner::NONE; i++)
		views[i] = NULL;
        addPage(resultspage, i18n("Scan results"));
}

void SweeperWizard::setupArchivePage()
{
	archivepage = new QWidget(this);	
	QVBoxLayout *box0 = new QVBoxLayout(archivepage, 5, 5);
	
	QLabel *info = new QLabel(i18n("Selected files will be removed. Please review your choices and click <B>Next</B> when ready. "
				  "It is recommended to <B>create backup archive</B> as there is no other way to restore deleted files."),
			          archivepage);
	info->setTextFormat(Qt::RichText);
	box0->addWidget(info);

	remview = new FileListView(archivepage, NULL);
	remview->setupColumns(CommonColumns | Size);
	box0->addWidget(remview);

	cb_createarch = new QCheckBox(i18n("Create backup archive"), archivepage);
	QToolTip::add(cb_createarch, i18n("Enables creating of backup archive containg selected files before they are removed"));
	cb_createarch->setChecked(true);
	box0->addWidget(cb_createarch);

	addPage(archivepage, i18n("Review your choices"));
}

void SweeperWizard::setupRemovePage()
{
	removepage = new QWidget(this);
	QVBoxLayout *box0 = new QVBoxLayout(removepage, 5, 5);
	logwidget = new LogWidget(removepage);
	box0->addWidget(logwidget);
	QHBoxLayout *box1 = new QHBoxLayout(box0);
	KPushButton *saverep = new KPushButton(KGlobal::iconLoader()->loadIconSet("filesaveas", KIcon::Small), i18n("Save report"), removepage);
	box1->addWidget(saverep);
	box1->addStretch(1);
	connect(saverep, SIGNAL(clicked()), this, SLOT(saveReport()));
	addPage(removepage, i18n("Sweeping selected files"));
}

void SweeperWizard::addLeftSidePic(QWidget *page, QBoxLayout *parent)
{
	QVBoxLayout *box1 = new QVBoxLayout(parent, 0);

	QLabel *pic = new QLabel(page, 0, Qt::AlignTop);
	pic->setPixmap(*kleansweep_bar1);
	pic->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::MinimumExpanding));
	box1->addWidget(pic, 0, Qt::AlignTop);

	QLabel *fake = new QLabel(page);
	fake->setPaletteBackgroundColor(QColor(0x7d, 0xa0, 0xe7));
	fake->setFixedWidth(kleansweep_bar1->width());
	fake->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::MinimumExpanding));
	box1->addWidget(fake);

	box1->insertSpacing(-1, 5);
	KPushButton *b_about = new KPushButton(i18n("About"), page);
	connect(b_about, SIGNAL(clicked()), this, SLOT(showAbout()));
	box1->addWidget(b_about);
}

void SweeperWizard::showPage(QWidget *w)
{
	if (w == startpage && scanner != NULL)
	{
		if (KMessageBox::warningYesNo(NULL, i18n("Searching will need to be restarted\nif you exit this page. Are you sure?")) == KMessageBox::No)
		{
			KWizard::showPage(resultspage);
			return;
		}
		delete scanner;
		scanner = NULL;
	}
	else if (w == resultspage && scanner == NULL)
	{
		QFileInfo finfo(url_startpath->url());
		//
		// test if start path exists
		if (!finfo.isDir())
		{
			KMessageBox::error(this, i18n("Starting directory must be an existing path"));
			url_startpath->clear();
			KWizard::showPage(startpage);
			return;
		}
		const QString path = finfo.absFilePath();

		scanner = new SweepScanner();
		connect(scanner, SIGNAL(newFileEntry(SweepScanner::FileType, SweepFileInfo *)), this, SLOT(addFileEntry(SweepScanner::FileType, SweepFileInfo *)));
		connect(scanner, SIGNAL(newFileEntry(SweepScanner::FileType, SweepFileInfo *, SweepFileInfo *)), this, SLOT(addFileEntry(SweepScanner::FileType, SweepFileInfo *, SweepFileInfo *)));
		connect(scanner, SIGNAL(updateTotals(int, unsigned int)), summary, SLOT(setTotals(int, unsigned int)));
		connect(scanner, SIGNAL(finished(bool)), this, SLOT(scannerFinished(bool)));

		KleanSweepConfig *cfg = &KleanSweepConfig::instance();
		FileListView *prev = remview;
		scanner->setOnlyWritable(cb_writable->isChecked());
		
		remview->clear();
		summary->reset();

		//
		// remove any existing filelist views
		for (int i=0; i<SweepScanner::NONE; i++)
			if (views[i])
			{
				resultstabs->removePage(views[i]);
				delete views[i];
				views[i] = NULL;
			}

		//
		// create views for desired scan types
		for (int i=0; scanopts[i].optname!=QString::null; i++)
		{
			const int n = scanopts[i].ftype;
			scanner->enableFileType(scanopts[i].ftype, cb_scan[n]->isChecked());
			if (cb_scan[n]->isChecked())
			{
				int columns = scanopts[i].columns;
				if (cfg->showScore())
					columns |= Score;
				views[n] = new FileListView(resultstabs, prev);
				views[n]->setupColumns(columns);
				connect(views[n], SIGNAL(openDirectory(const QString &)), this, SLOT(openDirectory(const QString &)));
				resultstabs->insertTab(views[n], scanopts[i].optname);
				prev = views[n];
				connect(prev, SIGNAL(selectionChanged(int, unsigned int)), summary, SLOT(updateSelectedInfo(int, unsigned int)));
			}
		}
		//
		// enable open/close signs on root node for duplicates view
		if (cb_scan[SweepScanner::DUPLICATED_FILE] && cb_scan[SweepScanner::DUPLICATED_FILE]->isChecked())
			views[SweepScanner::DUPLICATED_FILE]->setRootIsDecorated(true);


		if (KleanSweepConfig::instance().showProgress())
		{
			progress = new ScanProgressDialog(this);
			connect(scanner, SIGNAL(currentPath(const QString &)), progress, SLOT(setCurrentPath(const QString &)));
			progress->show();
		}

		if (scanner->start(path))
		{

			setNextEnabled(resultspage, false);
			setBackEnabled(resultspage, false);
			b_selectall->setDisabled(true);
			b_clearall->setDisabled(true);
			b_saveresrep->setDisabled(true);
			b_stop->setDisabled(false);
			setCursor(QCursor(Qt::WaitCursor));
		}
		else
		{
			KMessageBox::sorry(this, i18n("Can't execute helper process. Make sure you have kleansweep-helper in your PATH."));
			if (progress)
			{
				progress->close();
				progress = NULL;
			}
		}
	}
	else if (w == archivepage)
	{
		remview->clear();
		QPtrList<SweepFileInfo> files = scanner->getFilesForRemoval();
		for (SweepFileInfo *finfo = files.first(); finfo; finfo = files.next())
			FileViewItem *item = new FileViewItem(remview, finfo);

		setNextEnabled(archivepage, files.count() > 0);
	}
	else if (w == removepage)
	{
		QString fname = QDateTime::currentDateTime().toString("yyyyMMdd-hhmmss");
		fname = "kleansweep-" + fname;// + ".tar.gz";
		while (1) 
		{
			if (!cb_createarch->isChecked() || fname == QString::null)
			{
				int res = KMessageBox::warningYesNoCancel(NULL, i18n("Do you really want to skip archive creation?"));
				if (res == KMessageBox::Yes)
				{
					fname = QString::null;
					break;
				}
				if (res == KMessageBox::Cancel)
				{
					back();
					return;
				}
			}
			fname = KFileDialog::getSaveFileName(fname, "*.tar.gz", NULL, i18n("Select archive"));
			if (fname != QString::null)
				break;
		}
        	KWizard::showPage(w);
		bool status = true;
		QPtrList<SweepFileInfo> files = scanner->getFilesForRemoval();
		if (fname != QString::null)
		{
			logwidget->appendOperation(i18n("Creating backup archive ") + " " + fname, LogWidget::STATUS_INFO);
			KApplication::kApplication()->processEvents();

			KTar tar(fname);
			status = tar.open(IO_WriteOnly);
			if (!status)
				logwidget->setOperationStaus(LogWidget::STATUS_FAIL);
			else
			{
				for (SweepFileInfo *finfo = files.first(); finfo; finfo = files.next())
				{
					logwidget->appendOperation(i18n("Archiving") + " " + finfo->url().path());
					status = tar.addLocalFile(finfo->url().path(), finfo->url().path());
					logwidget->setOperationStaus(status ? LogWidget::STATUS_OK : LogWidget::STATUS_FAIL);
					KApplication::kApplication()->processEvents();
					if (!status)
					{
						logwidget->appendOperation(i18n("Aborting files removal due to archiving errors"), LogWidget::STATUS_INFO);
						break;
					}
					KApplication::kApplication()->processEvents();
				}
				tar.close();
			}
		}
		else
		{
			logwidget->appendOperation(i18n("Skipping archive creation"), LogWidget::STATUS_INFO);
		}

		int removed_cnt = 0;
		int notremoved_cnt = 0;
		int removed_size = 0; //total number of bytes reclaimed
		if (status) //continue if no errors occured
		{
			for (SweepFileInfo *finfo = files.first(); finfo; finfo = files.next())
			{
				QString descr(QString::null);
				int size = finfo->size();
				logwidget->appendOperation(i18n("Removing") + " " + finfo->url().path());
				KApplication::kApplication()->processEvents();

				if (finfo->isDir())
					status = (rmdir(finfo->url().path().local8Bit()) == 0);
				else
					status = (unlink(finfo->url().path().local8Bit()) == 0);
				
				#ifdef DEBUG
				std::cerr << "Removing: " << finfo->url().path() << " (" << status << ")" << std::endl;
				#endif
				
				if (!status)
					descr = strerror(errno);

				logwidget->setOperationStaus(status ? LogWidget::STATUS_OK : LogWidget::STATUS_FAIL, descr);
				KApplication::kApplication()->processEvents();
				if (status)
				{
					removed_cnt++;
					removed_size += size;
				}
				else
				{
					notremoved_cnt++;
				}
			}
			logwidget->appendOperation(i18n("Successfully removed:") + " " + QString::number(removed_cnt), LogWidget::STATUS_INFO);
			logwidget->appendOperation(i18n("Failed removing:") + " " + QString::number(notremoved_cnt), LogWidget::STATUS_INFO);
			logwidget->appendOperation(i18n("Freed disk space:") + " " + QString::number(removed_size) + " " + QString(i18n("bytes")), LogWidget::STATUS_INFO);
		}
		setFinishEnabled(removepage, true);
		return; //page is already shown, exit
	}
        KWizard::showPage(w);
}

void SweeperWizard::addFileEntry(SweepScanner::FileType t, SweepFileInfo *f)
{
	if (views[t])
		new FileViewItem(views[t], f);
}

void SweeperWizard::addFileEntry(SweepScanner::FileType t, SweepFileInfo *f, SweepFileInfo *parent)
{
	if (views[t])
	{
		if (FileViewItem *item = views[t]->find(parent))
		{
			new FileViewItem(item, f);
			views[t]->setOpen(item, true);
		}
		else
		{
			new FileViewItem(views[t], f);
		}
	}
}

void SweeperWizard::scannerFinished(bool status)
{
	if (progress)
	{
		progress->close();
		progress = NULL;
	}
	setNextEnabled(resultspage, true);
	setBackEnabled(resultspage, true);
	b_selectall->setDisabled(false);
	b_clearall->setDisabled(false);
	b_saveresrep->setDisabled(false);
	b_stop->setDisabled(true);
	setCursor(QCursor());
}

void SweeperWizard::scanCategoryToggled(bool f)
{
	if (f)
		++numchoosen;
	else
		--numchoosen;
	setNextEnabled(startpage, numchoosen>0);
}
		
void SweeperWizard::openDirectory(const QString &path)
{
        KProcess *proc = new KProcess(this);
        *proc << "kfmclient" << "openURL" << path;
        proc->start(KProcess::DontCare, KProcess::NoCommunication);
}

void SweeperWizard::selectAllClicked()
{
	if (FileListView *w = dynamic_cast<FileListView*>(resultstabs->currentPage()))
	{
		KMessageBox::information(NULL,
				i18n("KleanSweep uses simple heuristics to find files and\nit's not wise to assume everything can be safely deleted!\nPlease carefully review each item."),
				QString::null, "SelectAllWarning", KMessageBox::Dangerous);
		setCursor(QCursor(Qt::WaitCursor));
		w->selectAll();
		setCursor(QCursor());
	}
}

void SweeperWizard::clearAllClicked()
{
	if (FileListView *w = dynamic_cast<FileListView*>(resultstabs->currentPage()))
	{
		setCursor(QCursor(Qt::WaitCursor));
		w->selectNone();
		setCursor(QCursor());
	}
}

void SweeperWizard::stopClicked()
{
	if (scanner)
		scanner->stop();
}

void SweeperWizard::showConfigDialog()
{
	KleanSweepConfigDialog *d = new KleanSweepConfigDialog(this);
	d->exec();
	QCheckBox *cb = cb_scan[SweepScanner::ORPHAN_FILE];
	if (KleanSweepConfig::instance().distroPackageManager() == OTHER)
	{
		cb->setChecked(false);
		cb->setDisabled(true);
	}
	else
		cb->setDisabled(false);
	delete d;
}

void SweeperWizard::showAbout()
{
	KAboutApplication *d = new KAboutApplication;
	d->show();
}

void SweeperWizard::saveReport()
{
	QString fname = KFileDialog::getSaveFileName(QString::null, "*.txt", NULL, i18n("Select destination file"));
	if (fname != QString::null)
	{
		QFile f(fname);
		if (f.open(IO_WriteOnly))
		{
			QTextStream str(&f);
			str << "KleanSweep " VERSION " log file, created on " << QDateTime::currentDateTime().toString() << "\n\n";
			scanner->writeLog(str);
			str << "\n\n";
			logwidget->writeLog(str);
			f.close();
		}
		else
		{
			KMessageBox::sorry(this, i18n("Can't write report file"));
		}
	}
}

#include "sweeperwizard.moc"
