/*
 * This file is a part of KleanSweep.
 *
 * Copyright (C) 2005 Pawel Stolowski <pawel.stolowski@wp.pl>
 *
 * KleanSweep is free software; you can redestribute it and/or modify it
 * under terms of GNU General Public License by Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY. See GPL for more details.
 */

#include "sweepscanner.h"
#include "sweepfileinfo.h"
#include "kleansweepcfg.h"
#include <kprocess.h>
#include <kstandarddirs.h>
#include <qtextstream.h>
#include <cstdlib>
#include <cctype>

#ifdef DEBUG
#include <qvaluelist.h>
#include <iostream>
#endif

#define MAX_LINE_LEN 1024

SweepScanner::SweepScanner(): QObject(), lineptr(0), writable(false)
{
        line = new char [MAX_LINE_LEN];
	for (int i=0; i<NONE; i++)
		ftypes[i] = false;
	files.setAutoDelete(true);
        
	proc = new KProcess(this);
        connect(proc, SIGNAL(receivedStdout(KProcess *, char *, int)), this, SLOT(onProcessData(KProcess *, char *, int)));
	connect(proc, SIGNAL(processExited(KProcess *)), this, SLOT(helperExited(KProcess *)));
}

SweepScanner::~SweepScanner()
{
	stop();
        delete line;
}

void SweepScanner::enableFileType(FileType f, bool enable)
{
	if (ftypes[f] != enable)
		ftypes[f] = enable;
}

void SweepScanner::setOnlyWritable(bool f)
{
	if (writable != f)
		writable = f;
}

bool SweepScanner::getOnlyWritable() const
{
	return writable;
}

bool SweepScanner::isEnabled(FileType f)
{
	return ftypes[f];
}

void SweepScanner::clear()
{
	files.clear();
	filemap.clear();
}

bool SweepScanner::start(const QString &path)
{
	if (proc->isRunning())
		return false;

	clear();
	lineptr = totalitems = totalsize = 0;

	KleanSweepConfig *cfg = &KleanSweepConfig::instance();
	const QString bl = ::locate("data", "kleansweep/kleansweep-blacklist");
	const QString ru = ::locate("data", "kleansweep/kleansweep-rules");

	proc->clearArguments();
        *proc << "kleansweep-helper";
	if (!bl.isEmpty())
		*proc << "--blacklist" << bl; 
	if (!ru.isEmpty())
		*proc << "--scoring" << ru;
	if (cfg->showProgress())
		*proc << "--progress";
	if (writable)
		*proc << "--writable";
	if (isEnabled(EMPTY_FILE))      *proc << "-z";
	if (isEnabled(BACKUP_FILE))     *proc << "-b";
	if (isEnabled(EMPTY_DIR))       *proc << "-y";
	if (isEnabled(BROKEN_SYMLINK))  *proc << "-s";
	//if (isEnabled(OLD_FILE))        *proc << "--unused" << QString::number(cfg->oldFilesAge());
	if (isEnabled(DUPLICATED_FILE)) *proc << "-d";
	if (isEnabled(ORPHAN_FILE))
	{
		const DistroPkg pkg = cfg->distroPackageManager();
		if (pkg == RPM)
			*proc << "--rpm";
		else if (pkg == DPKG)
			*proc << "--deb";
	}
	if (isEnabled(DEAD_MENUENTRY))  *proc << "-m";
	if (isEnabled(DEAD_THUMBNAIL))  *proc << "-t";
	if (isEnabled(BROKEN_EXEC))     *proc << "-x";
	if (isEnabled(UNUSED_LOCALE))   *proc << "-l";

	QStringList paths = cfg->globalExcludePaths();
	for (QStringList::iterator it = paths.begin(); it != paths.end(); ++it)
		*proc << "--exclude" << *it;

	if (isEnabled(ORPHAN_FILE))
	{
		paths = cfg->orphansExcludePaths();
		for (QStringList::iterator it = paths.begin(); it != paths.end(); ++it)
			*proc << "--orphans-exclude" << *it;
		paths = cfg->orphansIncludePaths();
		for (QStringList::iterator it = paths.begin(); it != paths.end(); ++it)
			*proc << "--orphans-include" << *it;
	}
	
	*proc << path;
        proc->setPriority(10);

	#ifdef  DEBUG
	//
	// print helper arguments
	const QValueList<QCString> alist = proc->args();
	for (QValueList<QCString>::const_iterator it = alist.begin(); it != alist.end(); it++)
		std::cerr << *it << " ";
	std::cerr << std::endl;
	#endif
	
        if (!proc->start(KProcess::NotifyOnExit, KProcess::Stdout))
		return false;
	last_path = path;
	return true;
}

bool SweepScanner::isRunning()
{
	return proc->isRunning();
}

bool SweepScanner::stop()
{
	if (isRunning())
		proc->kill(SIGTERM);
	return true;
}

void SweepScanner::onProcessData(KProcess *proc, char *buf, int len)
{
        for (int i=0; i<len; i++)
        {
                if (buf[i] == '\n')
                {
                        line[lineptr] = 0;
                        lineptr = 0;
                        extractFileInfo();
                }
                else
                {
                        line[lineptr++] = buf[i];
                        if (lineptr == MAX_LINE_LEN-1)
                        {
                                ;
                        }
                }
        }
}

void SweepScanner::extractFileInfo()
{
	int pos = 0;
	char score = 0;
	long id = -1;
	SweepFileInfo *f = NULL;
	SweepFileInfo *dupparent = NULL;

	#ifdef DEBUG
	std::cerr << line << std::endl;
	#endif

	if (line[0] == '@')
	{
		emit currentPath(line + 2);
		return;
	}
	if (isdigit(line[pos]))
	{
		score = line[pos++] - '0';
		if (isdigit(line[pos]))
		{
			id = atol(line + pos);
			while (isdigit(line[pos])) //skip digits
				++pos;
		}
	}

	//
	// find file name
	int namepos = pos;
	while (line[namepos])
	{
		if (line[namepos] == ':')
		{
			line[namepos++] = 0;
			break;
		}
		else if (line[namepos] == '\\')
		{
			namepos += 2;
		}
		else
			++namepos;
	}
	
	KURL path;
	path.setPath(line + namepos);

	if (filemap.contains(id))
	{
		if (filemap[id]->url().path() == (line + namepos))
		{
			f = filemap[id];
		}
		else
		{
			dupparent = filemap[id];
			f = new SweepFileInfo(path, score);
			files.append(f);
			totalsize += f->size();
			emit updateTotals(++totalitems, totalsize);
		}
	}
	else
	{
		f = new SweepFileInfo(path, score);
		filemap.insert(id, f);
		files.append(f);
		totalsize += f->size();
		emit updateTotals(++totalitems, totalsize);
	}
        
        for (pos=0; line[pos];)
        { 
		FileType ft;
		switch (line[pos])
		{
			case 'y': ft = EMPTY_DIR; break;
			case 'z': ft = EMPTY_FILE; break;
			case 'l': ft = UNUSED_LOCALE; break;
			case 'b': ft = BACKUP_FILE; break;
			case 'm': ft = DEAD_MENUENTRY; break;
			case 'd': ft = DUPLICATED_FILE; break;
			case 's': ft = BROKEN_SYMLINK; break;
			case 't': ft = DEAD_THUMBNAIL; break;
			case 'x': ft = BROKEN_EXEC; break;
			//case 'U': ft = OLD_FILE; break;
			case 'O': ft = ORPHAN_FILE; break;
			default: ft = NONE; break;
		}
		++pos;
		if (ft == DUPLICATED_FILE)
		{
			if (dupparent)
			{
				emit newFileEntry(ft, f, dupparent);
			}
			else
			{
				emit newFileEntry(ft, f);
			}
			break; //?????
		}
		if (ft == DEAD_MENUENTRY || ft == DEAD_THUMBNAIL || ft == BROKEN_SYMLINK || ft == BROKEN_EXEC)
		{
			int targetpos = pos;
			//
			// parse target
			// - puts \0 in place of closing ','
			// - removes escape backslash-characters (replaces original string)
			for (int dst = pos; line[pos]; dst++)
			{
				if (line[pos] == ',')
				{
					line[dst] = 0;
					++pos;
					break;
				}
				if (line[pos] == '\\') //backslash found - take next character 'as is'
					line[dst] = line[++pos];
				else
					line[dst] = line[pos];
				++pos;
			}
			f->setTarget(line + targetpos);
		}
		if (ft != NONE && ftypes[ft])
			emit newFileEntry(ft, f);
        }
}

void SweepScanner::helperExited(KProcess *proc)
{
	filemap.clear(); //no longer needed
	emit finished(true);
}

QPtrList<SweepFileInfo> SweepScanner::getFilesForRemoval()
{
	QPtrList<SweepFileInfo> remfiles;
	remfiles.setAutoDelete(false);
	for (SweepFileInfo *finfo = files.first(); finfo; finfo = files.next())
		if (finfo->isMarkedForRemoval() && finfo->getScore()>0) //score can't be 0 if file is marked for removal -- sanity check...
			remfiles.append(finfo);
	return remfiles;
}

QString SweepScanner::lastScanPath() const
{
	return last_path;
}

void SweepScanner::writeLog(QTextStream &str)
{
	for (SweepFileInfo *finfo = files.first(); finfo; finfo = files.next())
		str << (finfo->isMarkedForRemoval() ? "[+] " : "[-] ") << finfo->getScore() << ' ' << QString(finfo->url().path()) << "\n";
}

#include "sweepscanner.moc"
