// ------------------------------------------------
// matrix device
//
// Andreas Rostin
// 07.03.2001
// ------------------------------------------------

#ifndef __DEVICE_MATRIX__
#define __DEVICE_MATRIX__


#include <math.h>


#include <qpainter.h>
#include "xdevice.h"
#include "tool.h"
#include "klogic.h"

class MatrixDevice : public XDevice
{
public:
	MatrixDevice(int x, int y, int size = -1, int = -1, int undef = -1, int clock = -1)
		: XDevice(DeviceType::fMATRIX, x, y, size, 0, undef, clock),
		colorDark(0, 89, 13),
		colorLight(0, 255, 0)
	{
		m_matrix_input = 0;
		m_column = 0;
		m_c1_id = 0;

		init();
	}

	virtual ~MatrixDevice()
	{
		if (m_matrix_input) delete [] m_matrix_input;
	}

	virtual void init()
	{
		setSize(8);

		// the clock input
		QString input_clk = "C1";
		m_c1_id = addInputName(input_clk, 0);

		// internal clock: clock-1
		QString internal_clk = "C1-1";
		m_c1_1_id = addInternalName(internal_clk);

		setEquation();
		parseEquation();

		deviceOutputInverted = false;
		deviceMaxOutput = 0;
	}

	virtual bool setSize(int new_size)
	{
		m_rows = new_size - 1;
		m_cols = (int)(m_rows / 1.5);

		// size for the calculated number of dots
		int dot_height = m_rows * 2 + 2 * XObject::IOREG;
		int dot_width = m_cols * 2 + 3 * XObject::IOREG;

		// needed size
		m_height = (new_size * GRID) + (2 * XObject::IOREG);
		if (m_height < dot_height)
			m_height = dot_height;
		m_width = (int)(new_size * GRID / 1.5) + XObject::IOREG;
		if (m_width < dot_width)
			m_width = dot_width;

		// the first input is for the clock
		bool ret = XDevice::setSize(new_size);
		setImage();
		actualizeInputConnectors();

		return ret;
	}

	virtual void * getInstance()
	{
		return (void *)this;
	}

	virtual void setEquation()
	{
		// internal clock = previous clock value
		Device::setEquation("C1", m_c1_1_id);

		// the matrix data input
		QString equation;
		QString input_name;
		for (int i = 0; i < deviceSize - 1; i++) {
			input_name.sprintf("D%d", i);
			if (i > 0)
				equation += "  ";
			equation += input_name;
		}
		Device::setEquation(equation);
	}

	virtual bool hasClock()
	{
		return false;
	}

	virtual bool hasNamedInput()
	{
		return true;
	}

	virtual bool hasNamedOutput()
	{
		return false;
	}

	virtual bool sizeChangeable()
	{
		return true;
	}

	// ------------------------------------
	// set device image
	// ------------------------------------
	virtual void setImage()
	{	QPixmap *pic = getImage();
		QPainter p;

		// resize image first, erase everything
		pic->resize(m_width, m_height);
		pic->fill();

		p.begin(pic);

		// body
		p.setPen(Qt::black);
		p.setBrush(Qt::black);
		p.drawRect(XObject::IOREG, 0, m_width - XObject::IOREG, m_height);

		// the lightning dots
		p.setPen(colorLight);
		p.setBrush(colorLight);
		for(int i = 0; i < m_cols; i++) {
			int x = (i + XObject::IOREG) * 2;
			for (int j = 0; j < m_rows; j++) {
				int y = j * 2 + XObject::IOREG;
				p.drawPoint(x, y);
			}
		}

		drawConnectionLines(&p);
		XObject::setImage();

		p.end();
	}

	// ------------------------------------
	// draw to printer
	// ------------------------------------
	virtual void printImage(QPainter *, const QRect &)
	{
	}

	// ------------------------------------
	// no outputs
	// ------------------------------------
	virtual bool outputIsActive()
	{
		return false;
	}

	// ------------------------------------
	// device image changes in dependance
	// of input value
	// ------------------------------------
	virtual bool setColor()
	{
		//if (!outputChanged()) return false;

		unsigned int val = (unsigned int)output();
		int c1 = input(m_c1_id);
		int c1_1 = output(m_c1_1_id);
		if (!c1 && c1_1) {
			m_column++;
			if (m_column >= m_cols)
				m_column = 0;
		}
		int x = (m_column + XObject::IOREG) * 2;
		printf("value=%d c1=%d c1_1=%d col=%d\n", val, c1, c1_1, m_column);

		QPixmap *pic = getImage();
		QPainter p;
		p.begin(pic);

		// set dark
		p.setPen(colorDark);
		p.setBrush(colorDark);
		for (int i = 0; i < m_rows; i++) {
			int y = i * 2 + XObject::IOREG;
			//fprintf(stdout, "val=%d col=%d row=%d shift=%d\n", val, col, col, val >> (col + j));
			if (!(val >> i & 1))
				p.drawPoint(x, y);
		}

		// set lights
		p.setPen(colorLight);
		p.setBrush(colorLight);
		for (int i = 0; i < m_rows; i++) {
			int y = i * 2 + XObject::IOREG;
			//fprintf(stdout, "val=%d col=%d row=%d shift=%d\n", val, col, col, val >> (col + j));
			if (val >> i & 1)
				p.drawPoint(x, y);
		}
		p.end();
		return true;
	}

private:
	// create connectors for matrix data input
	void actualizeInputConnectors()
	{
		int *tmp_arry = 0;
		int tmp_size = 0;

		// resize input id array; create temp array
		if (m_matrix_input_size != deviceSize - 1) {
			if (m_matrix_input_size) {
				int tmp_arry[m_matrix_input_size];
				for (int i = 0; i < m_matrix_input_size; i++)
					tmp_arry[i] = m_matrix_input[i];
				tmp_size = m_matrix_input_size;
			}
			m_matrix_input_size = deviceSize - 1;
			delete [] m_matrix_input;
			m_matrix_input = new int[m_matrix_input_size];
		} else return;	// nothing to do!

		// remove useless/add new input connectors
		if (tmp_size > m_matrix_input_size) {
			// remove inputs not needed anymore
			for (int i = m_matrix_input_size; i < tmp_size; i++)
				removeInputName(tmp_arry[i]);
		} else {
			// create new inputs
			QString input_name;
			for (int i = tmp_size; i < m_matrix_input_size; i++) {
				input_name.sprintf("D%d", i);
				m_matrix_input[i] = addInputName(input_name, i + 1);
			}
		}

		// (re)create equations for the (new) input connectors
		setEquation();
		parseEquation();
	}

	char *matrixMem;
	int m_width;	// width in pixel
	int m_height;	// height in pixel
	int m_rows;	// current number of dots per row
	int m_cols;	// current number of rows
	int m_column;	// current column
	int *m_matrix_input;
	int m_matrix_input_size;
	int m_c1_id;	// clock input id

	const QColor colorDark;
	const QColor colorLight;
};

#endif
