#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*#include <gdk/gdk.h>*/
#include "looperdata.h"
#include "interpol.h"

const float PI = 3.14159265358979323846f;
const float S2M_FACTOR = 0.7071;

typedef struct _dirty_block {
	double start;
	double end;
} dirty_block_t;

looper_data_t* looperdata_new(buffer_info_t* buf, long samplerate,int id){
	int i;
	looper_data_t *data = (looper_data_t*) malloc (sizeof(looper_data_t));

	data->buf = buf;
	data->id = id;
	data->samplerate = samplerate;
	data->speed = 1.;
	data->modespeed = 1.;
	data->vol = 1.;
	data->pan = 0.;
	data->panswing = 0.0;
	data->panpos = 0.;
	data->pandir = 1.;
	data->Lfact = 1.;
	data->Rfact = 1.;
	data->inmaxL = .0;
	data->inmaxR = .0;
	data->outmaxL = .0;
	data->outmaxR = .0;
	data->recmix = 1.;
	data->recmixnew = 1.;
	data->recmixorig = 0.;
	data->playpos = 0.;
	data->playindex = 0.;
	data->pointer = 0.;
	data->loopstart = 0;
	data->loopend = 0;
	data->recstart = 0;
	data->recend = 0;
	if (data->buf){
		data->loopend = data->buf->size;
		data->recend = data->buf->size;
	}
	data->recpos = 0.;
	data->isplaying = 0;
	data->isrecording = 0;
	data->linkrec2play = 1;
	data->keepabsolute = 0;
	data->limiter = 1;
	data->limit = 1.;
	data->limknee = .5;
	data->limconst = .25;
	data->playvuL = .0;
	data->playvuR = .0;
	data->recvuL = .0;
	data->recvuR = .0;
	data->playmode = LOOP_PLAYMODE_LOOP;
	data->cm_size = 1024;
	data->cm_min = 0.001;
	data->cm_max = 5.;
	data->customplaymode 
		= malloc (data->cm_size * sizeof(float));

	for (i=0;i < data->cm_size; i++){
		*(data->customplaymode + i) = 1.;
	}

	data->midichannel = -1; 
	data->ngrains = 0;
	data->grainmixfactor = 1.;
	data->mingrainspeed = 1.;
	data->maxgrainspeed = 1.;
	data->mingrainlength = 1024;
	data->maxgrainlength = 1024;
	data->grainpitchmode = GRAINPITCH_AUTO;
	data->grainpitchbend = 1.;
	data->graindensity = 100;
	data->grainindexes = NULL;
	data->grainoffsets = NULL;
	data->grainsizes = NULL;
	data->grainfadelengths = NULL;
	data->grainspeeds = NULL;
	data->grainsactive = NULL;
	data->grainpitchtable = NULL;

	data->dirtylist = speciallist_new();
	pthread_mutex_init (&data->looperdatamutex, NULL);
	return data;
}

void looperdata_free(looper_data_t* data){
	/* LATER free dirty block list */

	speciallist_destroy(data->dirtylist);
	free (data->customplaymode);
	free (data->grainindexes);
	free (data->grainoffsets);
	free (data->grainsizes);
	free (data->grainfadelengths);
	free (data->grainspeeds);
	free (data->grainsactive);
	free (data->grainpitchtable);

	free (data);
	data = NULL;
}

void looperdata_lock(looper_data_t* data){
	pthread_mutex_lock (&data->looperdatamutex);
}

void looperdata_unlock(looper_data_t* data){
	pthread_mutex_unlock (&data->looperdatamutex);
}

void looperdata_copy_settings (looper_data_t* dst, looper_data_t* src){
	int i;

	if (dst && src){
/* ???		dst->samplerate = src->samplerate; */
		dst->speed = src->speed;
		dst->playmode = src->playmode;
		dst->modespeed = src->modespeed;
		dst->vol = src->vol;
		dst->pan = src->pan;
		dst->panswing = src->panswing;
		dst->panpos = src->panpos;
		dst->pandir = src->pandir;
		dst->Lfact = src->Lfact;
		dst->Rfact = src->Rfact;
		dst->inmaxL = src->inmaxL;
		dst->inmaxR = src->inmaxR;
		dst->outmaxL = src->outmaxL;
		dst->outmaxR = src->outmaxR;
		dst->recmix = src->recmix;
		dst->recmixnew = src->recmixnew;
		dst->recmixorig = src->recmixorig;
		dst->linkrec2play = src->linkrec2play;
		dst->keepabsolute = src->keepabsolute;
		dst->limiter = src->limiter;
		dst->limit = src->limit;
		dst->limknee = src->limknee;
		dst->limconst = src->limconst;
		dst->midichannel = src->midichannel;
		dst->graindensity = src->graindensity;
		dst->grainpitchmode = src->grainpitchmode;
		if (dst->buf && src->buf){
			dst->isplaying = src->isplaying;
			dst->isrecording = src->isrecording;
			dst->playpos = src->playpos;
			dst->playindex = src->playindex;
			if (dst->playpos > dst->buf->size){
				dst->playpos = dst->buf->size;
			}
			if (dst->playindex > dst->buf->size){
                                dst->playindex = dst->buf->size;
                        }
			dst->recpos = src->recpos;
			dst->pointer = src->pointer;
			if (dst->recpos > dst->buf->size){
                                dst->recpos = dst->buf->size;
			}
			dst->loopstart = src->loopstart;
			if (dst->loopstart > dst->buf->size){
                                dst->loopstart = 0; 
				/* LATER find a mor intelligent solution */
                        }
			dst->loopend = src->loopend;	
			if (dst->loopend > dst->buf->size){
				dst->loopend = dst->buf->size;
			}
			dst->recstart = src->recstart;
			if (dst->recstart > dst->buf->size){
                                dst->recstart = 0;
				/* LATER find a mor intelligent solution */
                        }
			dst->recend = src->recend;
			if (dst->recend > dst->buf->size){
                                dst->recend = dst->buf->size;
                        }
		}	

		if (dst->cm_size != src->cm_size){
			dst->customplaymode = realloc(dst->customplaymode,
				src->cm_size * sizeof(float));
			dst->cm_size = src->cm_size;
		}
		for (i = 0; i < dst->cm_size; i++){
                	*(dst->customplaymode + i) = *(src->customplaymode + i);
		}
		dst->mingrainlength = src->mingrainlength;
		dst->maxgrainlength = src->maxgrainlength;
		dst->mingrainspeed = src->mingrainspeed;
		dst->maxgrainspeed = src->maxgrainspeed;
		looperdata_set_ngrains(dst,src->ngrains);
		/* LATER: should probably copy grainpitchtable */
	}
}

char* looperdata_cm2str (looper_data_t* data, char* str){
	int i;
	long s = 0;
	char *num = malloc(sizeof(char) * 32);

	if (str){
		free(str);
		str = NULL;
	}

	for (i = 0; i < data->cm_size; i++){
		sprintf (num,"%f ",*(data->customplaymode + i));
		s += strlen(num);
/*		printf ("s:%ld,num:%s-\n",(s + 1),num);*/
		str = (char*)realloc (str,sizeof(char) * (s + 1));
		memset((void*)(str + s  - strlen(num)),0,strlen(num));
		str = strcat(str, num);
        }
	free(num);
	return str;
}

void looperdata_str2cm (looper_data_t* data, char* str, int cm_size){
	char *a = str;
	char *b;
	char *num = malloc(sizeof(char) * 32);
	int i;

	data->cm_size = cm_size;	
	data->customplaymode = realloc(data->customplaymode,
                                data->cm_size * sizeof(float));

	for (i = 0;i < data->cm_size; i++){
		b = strstr((char*)(a + 1)," ");
		if (b){
			memset((void*)num,0,31);
			num = strncpy (num,(a),(b-a));
			*(data->customplaymode + i) = (float)atof (num);
			a = b;
		}
	}
}

void looperdata_write_dirtyblock(looper_data_t* data, double start, double end){
	/* return number of blocks including this one */
	dirty_block_t* block;
	int overlap = 0;

	block = speciallist_get_first(data->dirtylist);

	while (block){
		if (((block->start <= start) && (block->end >= start)) ||
			((block->start <= end) && (block->end >= end)) ){
			if (!overlap){
				/* LATER: delete if previous overlap overlaps */
				overlap = 1;
				if (block->start > start) block->start = start;
				if (block->end < end) block->end = end;
			}
		}
		block = speciallist_get_next(data->dirtylist,(void*)block);
	}

	if (!overlap){	
		block = malloc (sizeof(dirty_block_t));
        	block->start = start;
        	block->end = end;
		speciallist_append(data->dirtylist,(void*)block);
	}
}

int looperdata_read_once_dirtyblock(looper_data_t* data, double* start, double* end){
	/* return 1 if a block exists otherwise 0 */
	/* set start and  end */
	/* delete after read */
	dirty_block_t* block;
	
	block = speciallist_get_first(data->dirtylist);
	
	if (block){
		*start = block->start;
		*end = block->end;
		speciallist_delete(data->dirtylist, (void*)block);
		return 1;
	}		

	*start = 0.0;
	*end = 0.0;
	return 0;
}

void looperdata_calc_pan (looper_data_t* data){
	if (data->panswing > 0.){
		data->panpos += data->pandir 
			* data->panswing  * 2 / data->samplerate;
			
		if (data->panpos > 1.){
			data->panpos = 1.;
			data->pandir = -1.;
		}else if (data->panpos < -1.){
			data->panpos = -1.;
			data->pandir = 1.;
		}

		if (data->panpos > .0){
                	data->Lfact = 1. - data->panpos;
                	data->Rfact = 1.;
        	}else{
                	data->Lfact = 1.;
                	data->Rfact = 1. + data->panpos;
        	}

		if (data->pan > .0){
			
		}
        }else{
        	if (data->pan > .0){
                	data->Lfact = 1. - data->pan;
                	data->Rfact = 1.;
        	}else{
                	data->Lfact = 1.;
                	data->Rfact = 1. + data->pan;
        	}
	}
        data->Lfact *= data->vol;
        data->Rfact *= data->vol;
}

void looperdata_set_pan(looper_data_t* data, float pan){
	data->pan = pan;
	looperdata_calc_pan(data);
}
        
void looperdata_set_panswing(looper_data_t* data, float panswing){
        data->panswing = panswing;
	looperdata_calc_pan(data);
}     

void looperdata_set_vol(looper_data_t* data, float vol){
        if (vol < .0){vol = 0.;}
        data->vol = vol;
        /* recalculate Lfact + Rfact */
        looperdata_calc_pan(data);
}

float looperdata_get_vol(looper_data_t* data){
	return data->vol;
}

void looperdata_set_recmix(looper_data_t* data, float recmix){
/*	if (recmix < -1.){recmix = -1.;}
	if (recmix > 1.){recmix = 1.;}*/
	data->recmix = recmix;

	if (data->recmix > 1.){
		data->recmixorig = 0.;
		data->recmixnew = data->recmix;
	}else if (data->recmix < -1.){
		data->recmixorig =data->recmix;
		data->recmixnew = 0.;
	}else if (data->recmix > .0){
		data->recmixorig = 1. - data->recmix;
		data->recmixnew = 1.;
	}else{
		data->recmixorig = 1.;
		data->recmixnew = 1. + data->recmix;
	}
}	

int looperdata_get_midichannel (looper_data_t* data){
	/* muse and my keyboard send 0 when they write 1 ?!? */
	return (data->midichannel - 1);
}

void looperdata_set_midichannel (looper_data_t* data, int channel){
	if ((channel > -1) && (channel < 16))
		data->midichannel = channel;
	else	data->midichannel = -1;
}

void looperdata_set_mingrainspeed(looper_data_t* data,double speed){
	if (data){
		data->mingrainspeed = speed;
	}
}

void looperdata_set_maxgrainspeed(looper_data_t* data,double speed){
        if (data){
                data->maxgrainspeed = speed;
        }
}

void looperdata_set_mingrainlength(looper_data_t* data,long length){
/*        if ((data) && (length > 10)){*/
	if (data && (length > 1)){
                data->mingrainlength = length;
		if (data->maxgrainlength < data->mingrainlength)
                        data->maxgrainlength = data->mingrainlength;
        }
}

void looperdata_set_maxgrainlength(looper_data_t* data,long length){
        if (data && (length > 1)){
                data->maxgrainlength = length;
		if (data->maxgrainlength < data->mingrainlength)
			data->mingrainlength = data->maxgrainlength;
        }
}

void looperdata_set_graindensity(looper_data_t* data,int density){
	if (data) data->graindensity = density;
}


double looperdata_get_mingrainspeed(looper_data_t* data){
        if (data){
                return data->mingrainspeed;
        }else{
		return 0.;
	}
}

double looperdata_get_maxgrainspeed(looper_data_t* data){
        if (data){
                return data->maxgrainspeed;
        }else{
                return 0.;
        }
}

long looperdata_get_mingrainlength(looper_data_t* data){
        if (data){
                return data->mingrainlength;
        }else{
                return 0;
        }
}

long looperdata_get_maxgrainlength(looper_data_t* data){
        if (data){
                return data->maxgrainlength;
        }else{
                return 0;
        }
}

int looperdata_get_graindensity(looper_data_t* data){
	if (data)  return data->graindensity;
	else return 0;
}

int looperdata_get_cm_size(looper_data_t* data){
	return data->cm_size;
}

void looperdata_get_playmeters(looper_data_t* data,float* L,float* R){
	*L = data->outmaxL;
	*R = data->outmaxR;	
	data->outmaxL = 0.;
	data->outmaxR = 0.;
}

void looperdata_get_recmeters(looper_data_t* data,float* L,float* R){
	*L = data->inmaxL;
	*R = data->inmaxR;
	data->inmaxL = 0.;
	data->inmaxR = 0.;
}

static void looperdata_reset_grain (looper_data_t* data, int grain){
	int i,a = 0;

	*(data->grainindexes + grain) = 0;
/*
	*(data->grainoffsets + grain) = data->playpos + rand()%data->maxgrainlength;
*/

	if (data->ngrains > 1)
		*(data->grainoffsets + grain) = data->playpos + 
			rand()%(data->maxgrainlength / 15 * (data->ngrains + 1) + 1);
	else *(data->grainoffsets + grain) = data->playpos;
/*	printf ("ngrains:%d, max:%ld, distance:%ld\n",
		data->ngrains,data->maxgrainlength,(long)(*(data->grainoffsets + grain) - data->playpos));
*/

/*	printf ("size:%ld - %ld = %ld\n",data->maxgrainlength,data->mingrainlength,(data->maxgrainlength - data->mingrainlength));*/
	*(data->grainsizes  + grain) = data->mingrainlength + 
		rand()%(data->maxgrainlength - data->mingrainlength + 1);

	*(data->grainfadelengths + grain) = *(data->grainsizes + grain) / 10;
	/* is it coorect to multiply with speeds ? */
	if ((*(data->grainfadelengths + grain) * *(data->grainspeeds + grain)) > 200) *(data->grainfadelengths + grain) = 200;
	if (*(data->grainfadelengths + grain) < 5) *(data->grainfadelengths + grain) = 5;

	/* pitch according to pitchmode: */
	if (data->grainpitchmode == GRAINPITCH_CHORD){
		if (*(data->grainpitchtable + grain) != 0.){
			*(data->grainspeeds + grain) = *(data->grainpitchtable + grain);
			if (data->grainpitchbend) 
				*(data->grainspeeds + grain) *= data->grainpitchbend;
			*(data->grainsactive + grain) = 1;
		}else{
			*(data->grainspeeds  + grain) = 1;
			*(data->grainsactive + grain) = 0;	
		}
	} else {
		/* GRAINPITCH_AUTO and GRAINPITCH_RANGE are based on the predefined pitch range */
		*(data->grainspeeds  + grain) = data->mingrainspeed + 
			(double)(rand()%100001) / 100000. * (data->maxgrainspeed - data->mingrainspeed);

		if (data->grainpitchmode == GRAINPITCH_AUTO){
			if (*(data->grainpitchtable + grain) != 0.){
                        	*(data->grainspeeds + grain) *= *(data->grainpitchtable + grain);
				if (data->grainpitchbend) 
					*(data->grainspeeds + grain) *= data->grainpitchbend;
/*
                        	*(data->grainsactive + grain) = 1;
*/
				if (data->graindensity > (int)(100.0*rand()/(RAND_MAX+1.0)) )
					*(data->grainsactive + grain) = 1;
				else *(data->grainsactive + grain) = 0;
                	}else{
                        	*(data->grainsactive + grain) = 0;
                	}
		}else{
			/* this should be GRAINPITCH_RANGE */
			if (data->graindensity > (int)(100.0*rand()/(RAND_MAX+1.0)) )
                        	*(data->grainsactive + grain) = 1;
                	else *(data->grainsactive + grain) = 0;
		}
	}

	/* LATER: this might wrap wrong !?!*/
	if (*(data->grainoffsets + grain) > data->loopend){
		*(data->grainoffsets + grain) -= data->loopend - data->loopstart;
	}

	/*calculate mix factor for volume: */
	for (i = 0; i < data->ngrains;i++){
		if (*(data->grainsactive + grain)) a++;
        }
	if (a < 1) a = 1;
        data->grainmixfactor =  1.0f / sqrt((float)a);
}

void looperdata_set_grainpitchmode ( looper_data_t* data, int mode){
	int i;

	if (mode == GRAINPITCH_AUTO){
		data->grainpitchmode = GRAINPITCH_AUTO;
		for (i = 0; i <  data->ngrains; i++){
			*(data->grainpitchtable + i) = 1.;
		}
	}else if (mode == GRAINPITCH_CHORD){
		data->grainpitchmode = GRAINPITCH_CHORD;
		 for (i = 0; i <  data->ngrains; i++){
                        *(data->grainpitchtable + i) = 0.;
                }
	}else{
		data->grainpitchmode = GRAINPITCH_RANGE;
		for (i = 0; i <  data->ngrains; i++){
                        *(data->grainpitchtable + i) = 1.;
			data->grainpitchbend = 1.;
                }
	}
}

int looperdata_get_grainpitchmode (looper_data_t* data){
	return data->grainpitchmode;
}

void looperdata_set_grainpitchtablevalue ( looper_data_t* data, double value, double vol){
	int i = 0;
	int found = 0;

/*	printf ("set grainpitch\n");	*/
	if (vol == 0.){
		while (!found && (i < data->ngrains)){
			if (*(data->grainpitchtable + i) == value){
				if (data->grainpitchmode == GRAINPITCH_CHORD)
					*(data->grainpitchtable + i) = 0.; /* fall back to regular */
				else 	*(data->grainpitchtable + i) = 1.; /* mute */
				found++;
			}
			i++;
		}		
	}else{
		while (!found && (i < data->ngrains)){
			if ((*(data->grainpitchtable + i) == 1.) || (*(data->grainpitchtable + i) == 0.)){
				*(data->grainpitchtable + i) = value;
				found++;
			}
			i++;
		}
	}
}

void looperdata_set_grainpitchbend (looper_data_t* data, double bend){
/*	printf ("set grainpitchbend\n");*/
	if ((bend > -1.) && (bend < 2.))
		data->grainpitchbend = bend;
}

void looperdata_set_loopstart(looper_data_t* data, long start){
	if (data && (start > 1)){
		data->loopstart = start;
		if (data->loopstart >= data->loopend)
			data->loopend  = data->loopstart + 1;
	}
}

void looperdata_set_loopend(looper_data_t* data, long end){
	if (data && (end > 1)){
        	data->loopend = end;
		if (data->loopstart >= data->loopend)
			data->loopend  = data->loopstart + 1;
	}
}

void looperdata_set_playmode(looper_data_t* data,int mode){
	data->playmode = mode;
	if (data->playmode == LOOP_PLAYMODE_EXTERNALPOS)
		data->modespeed = 0.;
	else 	data->modespeed = 1.;
}

long looperdata_get_loopstart(looper_data_t* data){
        return data->loopstart;
}    

long looperdata_get_loopend(looper_data_t* data){
        return data->loopend;
}

void looperdata_set_recstart(looper_data_t* data, long start){
        data->recstart = start;
}

void looperdata_set_recend(looper_data_t* data, long end){
        data->recend = end;
}

void looperdata_set_playpos(looper_data_t* data, double playpos){
        data->playpos = playpos;
	data->playindex = playpos;
}

void looperdata_set_recpos(looper_data_t* data, double recpos){
        data->recpos = recpos;
}

long looperdata_get_recstart(looper_data_t* data){
        return data->recstart;
}

long looperdata_get_recend(looper_data_t* data){
        return data->recend;
}

void looperdata_set_speed(looper_data_t* data, double speed){
        data->speed = speed;
}

double looperdata_get_speed(looper_data_t* data){
        return data->speed;
}

double looperdata_get_playpos(looper_data_t* data){
        return data->playpos;
}

double looperdata_get_recpos(looper_data_t* data){
        return data->recpos;
}

static void calc_limconst (looper_data_t* data){
	data->limconst = (data->limit - data->limknee) *  data->limknee;
/*	printf ("knee:%f, limit:%f, const:%f\n",data->limknee,data->limit,data->limconst);*/
}

int looperdata_get_limiter(looper_data_t* data){
	return data->limiter;
}

void looperdata_aet_limiter(looper_data_t* data, int limiter){
	if (limiter) data->limiter = 1;
	else data->limiter = 0;
}

float looperdata_get_limknee(looper_data_t* data){
	return  data->limknee;	
}

void looperdata_set_limknee(looper_data_t* data, float limknee){
	data->limknee = limknee;
	calc_limconst(data);
}

float looperdata_get_limit(looper_data_t* data){
	return data->limit;
}

void looperdata_set_limit(looper_data_t* data, float limit){
	data->limit = limit;
	calc_limconst(data);
}


/*
void looperdata_set_grainlength (looper_data_t* data, long length){
	if (length < 256){length = 256;}
	data->grainlength = length;
}
*/

void looperdata_set_pointer (looper_data_t* data, float *pointer){
	buffer_lock(data->buf);
	data->pointer = *pointer;
	buffer_unlock(data->buf);
}

void looperdata_get_pointer (looper_data_t* data, float *pointer){
	float f = data->playindex - data->loopstart;

	if (f == 0.0) 	*pointer = 0.;
	else 	*pointer = f / (float)(data->loopend - data->loopstart);

//	if (f)
//		printf ("diff:%ld ,end:%ld pointer:%f\n",(data->loopend - data->loopstart),data->loopend,*pointer);
		
}

int looperdata_get_ngrains (looper_data_t* data){
	return data->ngrains;
}

void looperdata_set_ngrains (looper_data_t* data, int ngrains){
	int i;
/*	int a = 0;*/

	if (ngrains){
		data->grainindexes = realloc (data->grainindexes, sizeof(float) * ngrains);
		data->grainoffsets = realloc (data->grainoffsets, sizeof(long) * ngrains);
		data->grainsizes   = realloc (data->grainsizes, sizeof(long) * ngrains);
		data->grainfadelengths = realloc (data->grainfadelengths, sizeof(long) * ngrains);
		data->grainspeeds  = realloc (data->grainspeeds, sizeof(double) * ngrains);
		data->grainsactive = realloc (data->grainsactive, sizeof(int) * ngrains);
		data->grainpitchtable = realloc (data->grainpitchtable, sizeof(double) * ngrains);

		if (data->buf){
			for (i = data->ngrains; i < ngrains; i++){
				looperdata_reset_grain(data,i);
				if (data->grainpitchmode == GRAINPITCH_CHORD)
					data->grainpitchtable[i] = 0.;
				else 	data->grainpitchtable[i] = 1.;
/*				if (data->grainsactive[i]) a++;*/
			}
		}
/*		data->grainmixfactor =  1.0f / (1. + log10f(sqrt(ngrains)) * 3.333);*/
		data->grainmixfactor =  1.0f / sqrt(ngrains);
		/* change = 20 * log10(sqrt(number)) db; 6db = factor 2 */
	}
	data->ngrains = ngrains;
}

void looperdata_write_sample(looper_data_t* data,float* L, float* R){
	float *Ll;
	float *Rl;

	if (data->buf->channels > 1){
		if (data->inmaxL < fabs(*L)){data->inmaxL = fabs(*L);}
		if (data->inmaxR < fabs(*R)){data->inmaxR = fabs(*R);}

		buffer_lock(data->buf);
		*(data->buf->L + (long)data->recpos) = *L * data->recmixnew + 
			*(data->buf->L + (long)data->recpos) * data->recmixorig;
		Ll = (data->buf->L + (long)data->recpos);

		*(data->buf->R + (long)data->recpos) = *R * data->recmixnew + 
			*(data->buf->R + (long)data->recpos) * data->recmixorig;
		Rl = (data->buf->R + (long)data->recpos);
/*
                if (data->limiter){     
                        if (*Ll > .5)            *Ll = 1. - .25 / *Ll;
                        else if (*Ll < -.5)      *Ll = -1. - .25 / *Ll;
                        if (*Rl > .5)            *Rl = 1. - .25 / *Rl;
                        else if (*Rl < -.5)      *Rl = -1. - .25 / *Rl;
                }
*/
                if (data->limiter){
                        if (*L > data->limknee) *L = data->limit - data->limconst / *L;
                        else if (*L < -data->limknee) *L = -data->limit - data->limconst / *L;
                        if (*R > data->limknee) *R = data->limit - data->limconst / *R;
                        else if (*R < -data->limknee) *R = -data->limit - data->limconst / *R;
                }


		buffer_unlock(data->buf);
	}else{
		if (data->inmaxL < fabs(*L)){data->inmaxL = fabs(*L);}
		if (data->inmaxR < fabs(*R)){data->inmaxR = fabs(*R);}
		buffer_lock(data->buf);
		*(data->buf->L + (long)data->recpos) = (*L + *R) * S2M_FACTOR * data->recmixnew +
                        *(data->buf->L + (long)data->recpos) * data->recmixorig;
		Ll = (data->buf->L + (long)data->recpos);
/*
		if (data->limiter){
			if (*Ll > .5)            *Ll = 1. - .25 / *Ll;
                        else if (*Ll < -.5)      *Ll = -1. - .25 / *Ll;
		}
*/
                if (data->limiter){
                        if (*L > data->limknee) *L = data->limit - data->limconst / *L;
                        else if (*L < -data->limknee) *L = -data->limit - data->limconst / *L;
                }

		buffer_unlock(data->buf);
	}

	if (++data->recpos > data->recend){
        	data->recpos = data->recstart;
	}
	if (data->recpos < data->recstart){
        	data->recpos = data->recstart;
	}
}

static double grain_inc(looper_data_t* data, int i){
	double pos = 0.;

        *(data->grainindexes + i) += 1;

        if (*(data->grainindexes + i) >= *(data->grainsizes + i))
                looperdata_reset_grain (data,i);
        else{
		/* check if still inside the loop: */
                pos = (float)*(data->grainoffsets + i) + *(data->grainindexes + i) * *(data->grainspeeds + i);
                if (pos > (float)data->loopend){
                        *(data->grainoffsets + i) -= (data->loopend - data->loopstart)/* * *(data->grainspeeds + i)*/;
/*			printf ("loopend:%ld, recend:%ld, pos:%.2lf\n",data->loopend, data->recend,pos);*/
			pos = (float)*(data->grainoffsets + i) + *(data->grainindexes + i) * *(data->grainspeeds + i);
                } else if (pos < (float)data->loopstart){
                        *(data->grainoffsets + i) += (data->loopend - data->loopstart)/* * *(data->grainspeeds + i)*/;
			pos = (float)*(data->grainoffsets + i) + *(data->grainindexes + i) * *(data->grainspeeds + i);
                }
        }
	return pos;
}

void looperdata_calc_sample(looper_data_t* data,float* L, float* R){
	int i;
	float vol = 1.;
	double pos, ipos;

	if (data->isplaying){
		if (data->ngrains < 1){
			/* single playhead */
			*L = inter4pol (data->buf->L, data->playpos, data->buf->size);
			if (data->buf->channels > 1)
       				*R = inter4pol (data->buf->R, data->playpos, data->buf->size);
			else 	*R = *L;
		}else{
			/* granular playhead(s) */
			*L = .0;
			*R = .0;
			for (i = 0; i < data->ngrains; i++){
				pos = grain_inc(data,i);
				if (*(data->grainsactive + i)){
	                                if (*(data->grainindexes + i) < *(data->grainfadelengths + i))
                                        	vol = (float)*(data->grainindexes + i) / (float)*(data->grainfadelengths + i);
                                	else if ( (ipos = (*(data->grainsizes + i) - *(data->grainindexes + i))) < *(data->grainfadelengths + i))
                                        	vol = ipos / (float)*(data->grainfadelengths + i);
                                	else vol = 1.;
                                	vol *= data->grainmixfactor;

					*L += inter4pol (data->buf->L, pos, data->buf->size) * vol;
					if (data->buf->channels > 1)
						*R += inter4pol (data->buf->R, pos, data->buf->size) * vol;	
					else 	*R = *L; 
				}
			}	
		}
		if (data->playmode == LOOP_PLAYMODE_SINE){
			data->modespeed = 
				sin((data->playpos - data->loopstart) * PI / (data->loopend - data->loopstart)) * 1.5 + .5;
		}else if (data->playmode == LOOP_PLAYMODE_CUSTOMSPEED){
			data->modespeed = 
				inter4pol(data->customplaymode,
				(float)(data->playpos - data->loopstart) * (float)data->cm_size / 
				(float)(data->loopend - data->loopstart),data->cm_size);
		}else if (data->playmode == LOOP_PLAYMODE_EXTERNALPOS){
			data->playindex = data->loopstart +
				(int)(data->pointer * (data->loopend - data->loopstart));
		}

		data->playindex += data->speed * data->modespeed;

		if (data->playmode == LOOP_PLAYMODE_EXTERNALPOS){
			if (data->playindex > data->loopend)
				data->playindex =  data->loopstart
						+ ((long)(data->playindex - data->loopstart)
						% (long)(data->loopend - data->loopstart));
			if (data->playindex < data->loopstart)
				data->playindex = data->loopend 
						- ((long)(data->loopend - data->playindex) 
						% (long)(data->loopend - data->loopstart));
		}else if (data->playindex > data->loopend){
			if (data->playmode == LOOP_PLAYMODE_BACKFORTH){
				data->modespeed *= -1;
			}else if (data->playmode == LOOP_PLAYMODE_LOOP){
				data->modespeed = 1;
			}
                	if ((data->speed * data->modespeed) > .0){
                        	data->playindex += data->loopstart - data->loopend;
                	}else{
                        	data->playindex = data->loopend;
                	}
        	}else if (data->playindex < data->loopstart){
			if (data->playmode == LOOP_PLAYMODE_BACKFORTH){
                        	data->modespeed *= -1;
                	}else if ((data->playmode == LOOP_PLAYMODE_LOOP) && (data->modespeed < 0)){
				data->modespeed = 1;
			}
	
                	if ((data->speed  * data->modespeed) > .0){
                        	data->playindex = data->loopstart;
                	}else{
                        	data->playindex += data->loopend - data->loopstart;
                	}
        	}
		if (data->playmode == LOOP_PLAYMODE_POSITION){
			data->playpos = data->loopstart + (data->loopend - data->loopstart) / 
				(data->cm_max - data->cm_min) * (inter4pol (data->customplaymode,
				(data->playindex - data->loopstart) * data->cm_size /
				(data->loopend - data->loopstart),data->cm_size) - data->cm_min);
 		}else{
			data->playpos = data->playindex;
		}

		if (data->panswing)
			looperdata_calc_pan(data);
		*L *= data->Lfact;
		*R *= data->Rfact;

		if (data->limiter){
			if (*L > data->limknee)	*L = data->limit - data->limconst / *L;
			else if (*L < -data->limknee) *L = -data->limit - data->limconst / *L;
			if (*R > data->limknee) *R = data->limit - data->limconst / *R;
                        else if (*R < -data->limknee) *R = -data->limit - data->limconst / *R;
		}
	
		if (data->outmaxL < *L){data->outmaxL = *L;}
                if (data->outmaxR < *R){data->outmaxR = *R;}
		
	}else{
		*L = 0.;
		*R = 0.;
	}
}
