/***************************************************************************
                                 ksworkbook.h
                             -------------------                                         
    begin                :
    copyright            : (C) 2001 by Kamil Dobkowski
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef KSWORKBOOK_H
#define KSWORKBOOK_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include<qlist.h>
#include<qstring.h>
#include"widgets/qsworkbook.h"


//---------------------------------------------------------------------------------------------//

/**
  * Command
  * @author Kamil Dobkowski
  */
class KSCommand {
 public:
  /**
    *  Constructor
    */
  KSCommand( const QString& title );
  /**
    * Destructor
    */
  virtual ~KSCommand();	
  /**
    * Execute command
    */
  virtual bool execute() = 0;
  /**
    * Unexecute command
    */
  virtual void unexecute() = 0;
  /**
    * Command title
    */
  QString title() const { return m_title; }
  /**
    * Sets an error message
    */
  void setError( const QString& message );
  /**
    * Returns an error messsage
    */
  QString error() const { return m_error; }

 private:
   QString m_title;
   QString m_error;
 };

//---------------------------------------------------------------------------------------------//

/**
  * Command history
  * @author Kamil Dobkowski
  */
class KSCommandHistory : public QObject {
  Q_OBJECT

  public:
   /**
     * Constructor
     */
   KSCommandHistory( QObject *parent=NULL );
   /**
     * Destructor
     */
   virtual ~KSCommandHistory();
   /**
     * Clears command history and sheets
     */
   void clear();
   /**
     * Sets max undo levels
     */
   static void setUndoLevels( int levels );
   /**
     * Returns undo levels
     */
   static int undoLevels() { return m_undo_levels; }
   /**
     * Executes command, adds it to the undo list
     */
   bool execute( KSCommand *command );
   /**
     * Undoes the last commands
     */
   void undo();
   /**
     * Redoes the last undid command
     */
   void redo();
   /**
     * Is command for undo available.
     */
   bool isUndoPossible();
   /**
     * Is command for redo available.
     */
   bool isRedoPossible();
   /**
     * Title of the last executed command or QString::null
     */
   QString undoCommandTitle();
   /**
     * Title of the last unexecuted command or QString::null
     */
   QString redoCommandTitle();

  signals:
   /**
     * New undo command
     */
   void sigNewUndo();
   /**
     * New redo command
     */
   void sigNewRedo();
   /**
     * Error during executing a command.
     */
   void sigError( const QString& message );

  private:
   static int m_undo_levels;
   int m_last_executed_command;
   QList<KSCommand> m_command_history;
 };

//---------------------------------------------------------------------------------------------//

class KSSheetList;
/**
  * Extended workbook with command history and sheets.
  * @author Kamil Dobkowski
  */
class KSWorkbook : public QSWorkbook
 {
  Q_OBJECT
  public:
   /**
     * Constructor
     */
   KSWorkbook( QObject *parent );
   /**
     * Destructor
     */
   virtual ~KSWorkbook();
   /**
     * Clears also command history.
     */
   virtual void clear();
   /**
     * Command  history
     */
   KSCommandHistory *commandHistory() const { return m_command_history; }
   /**
     * Sheet list
     */
   KSSheetList *sheets() const { return m_sheets; }
   /**
     * Calls commandHistory()->execute()
     */
   bool execute( KSCommand *command ) { return m_command_history->execute( command ); }

  private:
   KSSheetList *m_sheets;
   KSCommandHistory *m_command_history;
 };

//---------------------------------------------------------------------------------------------//

#endif
