/***************************************************************************
                                 qslegend.cpp
                             -------------------
    begin                : 01-January-2001
    copyright            : (C) 2001 by Kamil Dobkowski
    email                : kamildobk@poczta.onet.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include"qsclegend.h"
#include"qsaxes.h"
#include"qsplot.h"

const double QSCLegend::frmSpace = 10.0;
const double QSCLegend::rowSpace = 5.0;
const double QSCLegend::colSpace = 8.0;

QSCLegend::QSCLegend( QSAxes *parentAxes, QObject *parent )
:QSCObject( parent )
 {
  setParentAxes(parentAxes);
  m_pos.set( 0.9, 0.1, 0.0 );
  m_axis.set( QSAxes::normCoord, QSAxes::normCoord, QSAxes::normCoord );
  m_align = Qt::AlignTop | Qt::AlignRight;
  m_columns = 1;
  m_shadow_pos.x = 3;
  m_shadow_pos.y = 3;
  m_shadow_fill.color = QSGColor( 0, 0, 0 );
  m_shadow_fill.style = QSGFill::Solid;
  m_fill.style = QSGFill::Solid;
  m_fill.color = QSGColor( 255, 255, 255 );
  m_frame.style = QSGLine::Solid;
 }

//-------------------------------------------------------------//

QSCLegend::~QSCLegend()
 {
 }
	
//-------------------------------------------------------------//

void  QSCLegend::setBox( const QSRectf& rect, QSDrv *drv )
 {
   // moved on canvas by d pixels ( we ingnore size )
   parametersChanging();
   QSRectf r = rect.normalize();
   QSPt2f d = r.pos - box( drv ).pos;
   QSPt3f cpos = mixedToCanvas( m_pos, m_axis.x, m_axis.y, m_axis.z, drv->dpi );
   cpos.x += d.x;
   cpos.y += d.y;
   m_pos = canvasToMixed( cpos, m_axis.x, m_axis.y, m_axis.z, drv->dpi );
   parametersChanged();
 }

//-------------------------------------------------------------//

QSRectf QSCLegend::box( QSDrv *drv )
 {
  QSRectf r;
  do_everything( drv, r, get_height(drv), false );
  return r;
 }

//-------------------------------------------------------------//

void QSCLegend::draw( QSDrv *drv, bool, bool )
 {
  QSRectf r;
  double height = get_height(drv);
  do_everything( drv, r, height, false );

  // draw frame
  double x1 = r.pos.x;
  double y1 = r.pos.y;
  double x2 = r.pos.x+r.size.x;
  double y2 = r.pos.y+r.size.y;
  double sdx = drv->toPixels(m_shadow_pos.x);
  double sdy = drv->toPixels(m_shadow_pos.y);

  if ( m_shadow_fill.style != QSGFill::Transparent ) {
	drv->setLine( QSGLine::invisibleLine );
	drv->setFill( m_shadow_fill    );
     	drv->drawRect( QSPt2f(x1+sdx,y1+sdy),
		       QSPt2f(x2+sdx,y2+sdy) );
       }
  if (  m_fill.style  != QSGFill::Transparent ||
        m_frame.style != QSGLine::Invisible    ) {
         drv->setFill( m_fill  );
         drv->setLine( m_frame );
         drv->drawRect( QSPt2f(x1,y1),
		        QSPt2f(x2,y2) );
        }

  // draw legend items
  do_everything( drv, r, height, true  );
  emit sigDrawEnds( this );
 }

//-------------------------------------------------------------//

void QSCLegend::do_everything( QSDrv *drv, QSRectf& rect, double average_height, bool draw )
  {
   int plots = parentAxes() ? parentAxes()->plotCount() : 0;
   double fspace = drv->toPixels(frmSpace);
   double rspace = drv->toPixels(rowSpace);
   double cspace = drv->toPixels(colSpace);
   drv->setFont( m_font );
   double max_col_width = 0.0;
   QSPt2f cpos;

   if ( draw ) {
   	 cpos.x = rect.pos.x;
   	 cpos.y = rect.pos.y;
   	} else {
   	 QSPt3f cpos3 = mixedToCanvas( m_pos, m_axis.x, m_axis.y, m_axis.z, drv->dpi );
   	 cpos.set( cpos3.x, cpos3.y );
   	}
   	
   QSPt2f size( 0.0, 0.0 );
   QSPt2f curr_pos( 0.0, 0.0 );
   QSPt2f marg( fspace, fspace );
   for ( int i=plots-1; i>=0; i-- ) {
	if ( parentAxes()->plot(i)->legendItemVisible() ) {
	        drv->setFont( m_font );
		QSPt2f curr_size = parentAxes()->plot(i)->legendItemSize(drv);
		if ( draw ) {
			drv->setFont( m_font );
			QSPt2f item_pos = cpos+marg+curr_pos;
			parentAxes()->plot(i)->drawLegendItem( QSPt2f(item_pos.x,item_pos.y), drv );
			}
		size.set( QMAX(size.x,curr_pos.x+curr_size.x),
			  QMAX(size.y,curr_pos.y+curr_size.y) );
		curr_pos.y += curr_size.y + rspace;
		max_col_width = QMAX( max_col_width, curr_size.x );
		if ( curr_pos.y > average_height ) {
			 curr_pos.y = 0.0;
			 curr_pos.x += max_col_width + cspace;
			 max_col_width = 0.0;
			}
		}
	}
   size = size + marg + marg;

   if ( !draw ) {	
   	rect = QSRectf(cpos.x,cpos.y,size.x,size.y);
   	if ( m_align & AlignVCenter ) rect.pos.y -= rect.size.y/2.0;
   	if ( m_align & AlignHCenter ) rect.pos.x -= rect.size.x/2.0;
   	if ( m_align & AlignBottom  ) rect.pos.y -= rect.size.y;
   	if ( m_align & AlignRight   ) rect.pos.x -= rect.size.x;
   	}
  }

//-------------------------------------------------------------//

double QSCLegend::get_height( QSDrv *drv )
 {
   double total_height = 0.0;
   int plots = parentAxes() ? parentAxes()->plotCount() : 0;
   // fixed values are given when dpi is 72
   double rspace = drv->toPixels(rowSpace);

   // sets default font
   for ( int i=0; i<plots; i++ )
     if ( parentAxes()->plot(i)->legendItemVisible() ) {
     	drv->setFont( m_font );
     	total_height += parentAxes()->plot(i)->legendItemSize(drv).y + rspace;
     	}
   return (double )total_height / m_columns;
 }

//-------------------------------------------------------------//

QString QSCLegend::name()
 {
  return tr("Legend: ") + (m_parent_axes ? m_parent_axes->title() : QString::null);
 }

//-------------------------------------------------------------//

void QSCLegend::setColumns( int columns )
 {
  SET_COBJECT_PROPERTY( m_columns, columns );
 }

//-------------------------------------------------------------//

void QSCLegend::setAlign( int align )
 {
  SET_COBJECT_PROPERTY( m_align, align );
 }

//-------------------------------------------------------------//

void QSCLegend::setPos( const QSPt3f& pos )
 {
  SET_COBJECT_PROPERTY( m_pos, pos );
 }
	
//-------------------------------------------------------------//

void QSCLegend::setPosX( double value )
 {
  SET_COBJECT_PROPERTY( m_pos.x, value );
 }

//-------------------------------------------------------------//

void QSCLegend::setPosY( double value )
 {
  SET_COBJECT_PROPERTY( m_pos.y, value );
 }

//-------------------------------------------------------------//

void QSCLegend::setPosZ( double value )
 {
  SET_COBJECT_PROPERTY( m_pos.z, value );
 }

//-------------------------------------------------------------//

void QSCLegend::setCoord( const QSPt3& coordSystem )
 {
  SET_COBJECT_PROPERTY( m_axis, coordSystem );
 }
	
//-------------------------------------------------------------//

void QSCLegend::setCoordX( int coordSystem )
 {
  SET_COBJECT_PROPERTY( m_axis.x, coordSystem );
 }

//-------------------------------------------------------------//

void QSCLegend::setCoordY( int coordSystem )	
 {
  SET_COBJECT_PROPERTY( m_axis.y, coordSystem );
 }

//-------------------------------------------------------------//
   	
void QSCLegend::setCoordZ( int coordSystem )
 {
  SET_COBJECT_PROPERTY( m_axis.z, coordSystem );
 }

//-------------------------------------------------------------//

void QSCLegend::setFont( const QSGFont& font )
 {
  SET_COBJECT_PROPERTY( m_font, font );
 }

//-------------------------------------------------------------//

void QSCLegend::setFontProperty( const QString& data )
 {
  QSGFont new_font = toQSGFont(data);
  setFont( new_font );
 }

//-------------------------------------------------------------//

void QSCLegend::setFill( const QSGFill& fill )
 {
  SET_COBJECT_PROPERTY( m_fill, fill );
 }

//-------------------------------------------------------------//

void QSCLegend::setFillProperty( const QString &data )
 {
  QSGFill new_fill = toQSGFill(data);
  setFill( new_fill );
 }

//-------------------------------------------------------------//

void QSCLegend::setFrame( const QSGLine& line )
 {
  SET_COBJECT_PROPERTY( m_frame, line );
 }

//-------------------------------------------------------------//

void QSCLegend::setFrameProperty( const QString& data )
 {
  QSGLine new_line = toQSGLine(data);
  setFrame( new_line );
 }

//-------------------------------------------------------------//

void QSCLegend::setShadowFill( const QSGFill& fill )
 {
  SET_COBJECT_PROPERTY( m_shadow_fill, fill );
 }

//-------------------------------------------------------------//

void QSCLegend::setShadowFillProperty( const QString& data )
 {
  QSGFill new_shadow_fill = toQSGFill(data);
  setShadowFill( new_shadow_fill );
 }

//-------------------------------------------------------------//

void QSCLegend::setShadowPos( const QSPt2& shift )
 {
  SET_COBJECT_PROPERTY( m_shadow_pos, shift );
 }

//-------------------------------------------------------------//

void QSCLegend::setShadowPosX( int value )
 {
  SET_COBJECT_PROPERTY( m_shadow_pos.x, value );
 }

//-------------------------------------------------------------//

void QSCLegend::setShadowPosY( int value )
 {
  SET_COBJECT_PROPERTY( m_shadow_pos.y, value );
 }

//-------------------------------------------------------------//

void QSCLegend::loadStateFromStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSCObject::loadStateFromStream( stream, factory );
 }

//-------------------------------------------------------------//

void QSCLegend::saveStateToStream( QDataStream& stream, QSObjectFactory *factory )
 {
  QSCObject::saveStateToStream( stream, factory );
 }

//-------------------------------------------------------------//



