/***************************************************************************
                                qsgattr.h                              
                             -------------------                                         
    begin                : 01-January-2000
    copyright            : (C) 2000 by Kamil Dobkowski                         
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef QSGATTRIBUTE_H
#define QSGATTRIBUTE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include<qnamespace.h>
#include<qstring.h>
#include<qdom.h>

/**
  * \brief Base class for all graphics attributes.
  * @author Kamil Dobkowski
  */
class QSGAttr {};

//-------------------------------------------------------------//

/**
  * \brief Color definition.
  * @author Kamil Dobkowski
  */
class QSGColor : public QSGAttr {
         public:
             unsigned char r;
             unsigned char g;
             unsigned char b;
             unsigned char a;
   	     /**
               * Creates a black color.
               */
             QSGColor();
             QSGColor( unsigned char red,
                       unsigned char green,
                       unsigned char blue,
                       unsigned char alpha = 255U );
             const QSGColor &set( unsigned char red,
                                  unsigned char green,
                                  unsigned char blue,
                                  unsigned char alpha = 255U );
             friend bool operator==( const QSGColor&, const QSGColor& );
             friend bool operator!=( const QSGColor&, const QSGColor& );
             friend QString  toQString( const QSGColor& a );
             friend QSGColor toQSGColor( const QString&  s );
            };

 //-------------------------------------------------------------//

/**
  * \brief Font definition.
  * @author Kamil Dobkowski
  */
class QSGFont : public QSGAttr {
         public:
              QString  family;
              int      size;     // points
              bool     bold;
              bool     italic;
              QSGColor color;
	      /**
                * Creates "helvetica" 10pts, black font.
                */
              QSGFont();
              QSGFont( const QSGFont& f );
              void operator=( const QSGFont& f );
              friend bool operator==( const QSGFont&, const QSGFont& );
              friend bool operator!=( const QSGFont&, const QSGFont& );
              friend QString toQString( const QSGFont& font );
              friend QSGFont toQSGFont( const QString& string );
             };

//-------------------------------------------------------------//

/**
  * \brief Line style definition.
  * @author Kamil Dobkowski
  */
class QSGLine : public QSGAttr {

        public:
  	   static const QSGLine invisibleLine;

            /** Line type. */
            enum Style { Invisible  = Qt::NoPen,
			 Solid      = Qt::SolidLine,
			 Dash       = Qt::DashLine,
			 DashDot    = Qt::DashDotLine,
			 Dot        = Qt::DotLine,
			 DashDotDot = Qt::DashDotDotLine };
            Style   style;
            int         width;
            QSGColor    color;

            /**
              * Creates a solid, black, 0-width line.
              */
            QSGLine();
            QSGLine( Style style );
            friend bool operator==( const QSGLine&, const QSGLine& );
            friend bool operator!=( const QSGLine&, const QSGLine& );
            friend QString toQString( const QSGLine& line );
            friend QSGLine toQSGLine( const QString& string );
           };



//-------------------------------------------------------------//

/**
  * \brief Fill definition.
  * @author Kamil Dobkowski
  */
class QSGFill : public QSGAttr {

        public:
	
	     static const QSGFill transparentFill;

             /** Fill style. */
             enum Style { Transparent = Qt::NoBrush,
                              Solid       = Qt::SolidPattern,
                              Horiz       = Qt::HorPattern,
                              Vert        = Qt::VerPattern,
                              Cross       = Qt::CrossPattern,
                              BDiag       = Qt::BDiagPattern,
                              FDiag       = Qt::FDiagPattern,
                              DiagCross   = Qt::DiagCrossPattern,
                              Half        = Qt::Dense5Pattern };

             Style style;
             QSGColor  color;

	     /**
               * Creates a solid, white fill.
               */
             QSGFill();
             QSGFill( Style style );
             friend bool operator==( const QSGFill&, const QSGFill& );
             friend bool operator!=( const QSGFill&, const QSGFill& );
            friend QString toQString( const QSGFill& fill );
            friend QSGFill toQSGFill( const QString& string );
       };



//-------------------------------------------------------------//

/**
  * \brief Point mark style definition
  * @author Kamil Dobkowski
  */
class QSGPoint : public QSGAttr {

        public:

        enum Style { Invisible = 0,
                          Circle,
                          Rect,
                          Triangle,
                          Diamond,
                          Cross,
                          Plus,
                          HLine,
                          VLine
                          };
	enum Fill {
			 Transparent = 0,
			 Opaque,
			 Filled
		       };
		
        Style  style;
        Fill   fill;
        int         size;       // points
        QSGColor    color;

	/**
          * Creates black point, with size 5 and style - Invisible.
          */
        QSGPoint();
        friend bool operator==( const QSGPoint&, const QSGPoint& );
        friend bool operator!=( const QSGPoint&, const QSGPoint& );
	friend QString toQString( const QSGPoint& point );
	friend QSGPoint toQSGPoint( const QString& string );
       };

//-------------------------------------------------------------//

/**
  * \brief Arrow dart style definition
  * @author Kamil Dobkowski
  */
class QSGArrow: public QSGAttr {

        public:

         /**
           * Arrow style. 'F' - filled, 'R' - reversed, 'N' - narrow.
           */
         enum Style { 	None      = 0,
  			   	Arrow     = 1,
				FArrow    = 2,
				NArrow    = 3,
				RArrow    = 4,
				RFArrow   = 5,
				RNArrow   = 6,
				Rect      = 7,
				Diamond   = 8,
				Circle    = 9, 	
				Line      = 10,									
				FDiagLine = 11,
				BDiagLine = 12
				};
         int style;
	 int size;

	/**
          * Creates an arrow with size 5 and style - None.
          */
         QSGArrow();
		
        friend bool operator==( const QSGArrow&, const QSGArrow& );
        friend bool operator!=( const QSGArrow&, const QSGArrow& );
	friend QString toQString( const QSGArrow& arrow );
	friend QSGArrow toQSGArrow( const QString& string );
	};

//-------------------------------------------------------------//

/**
  * \brief Gradient style definition
  *
  * Gradient is an object which returns a fill style for each parameter value between 0.0 and 1.0, see fill()
  *@author kamil
  */
class QSGGradient : public QSGAttr {

    public:
	 enum Type { GDefault, GGray, GStrange, GTwoColors, GFiveColors };

	 QSGGradient( Type t = GDefault );
	 QSGGradient( const QSGFill& f0,
			    const QSGFill& f4 );
         QSGGradient( const QSGFill& f0,
			    const QSGFill& f1,
			    const QSGFill& f2,
			    const QSGFill& f3,
			    const QSGFill& f4,
			    Type t = GFiveColors );
	virtual ~QSGGradient();
	/**
	  * Sets 'fill' argument to some fill style for level values between 0.0 and 1.0.
	  * Returns 'fill'
	  */
 	virtual QSGFill& fill( double level, QSGFill& fill ) const;
	virtual bool isEqualTo( const QSGGradient& gradient ) const;

	QSGFill fill( int nr ) const { return f[nr]; }
	Type type() const { return t; }
        friend bool operator==( const QSGGradient&, const QSGGradient& );
        friend bool operator!=( const QSGGradient&, const QSGGradient& );
	friend QString toQString( const QSGGradient& arrow );
	friend QSGGradient toQSGGradient( const QString& string );
    protected:

        Type t;
	QSGFill f[5];
	bool m_all_colors;
	void set_default_colors();
    };


//-------------------------------------------------------------//

#endif





