/***************************************************************************
                                   qgraphicaldata.h
                             -------------------
    version              : 0.2
    begin                : 01-January-2001
    copyright            : (C) 2000 by Kamil Dobkowski
    email                : kamildobk@poczta.onet.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef QSGRAPHICALDATA_H
#define QSGRAPHICALDATA_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include"qsdata.h"

#define SET_PROPERTY( property, new_value )  if ((property)!=(new_value)) { parametersChanging(); (property)=(new_value); parametersChanged(); }

/**
  * \brief Data object which holds a set of graphics attributes such as fonts, fills, line styles etc.
  *
  * See new functions :  setFill(), setFont(), setLine(), setPoint().
  * Contains also implementation of the QSData interface.
  * @author Kamil Dobkowski
  */
class QSGraphicalData : public QSData
   {
    Q_OBJECT
	Q_PROPERTY( QString title READ title WRITE setTitle )

    public:
     /**
       * Constructor
       */
     QSGraphicalData( QObject *parent=0, QSGraphicalData *parentObject=0, const char *name=0 );
      /**
	* Destructor.
	*/
     virtual ~QSGraphicalData();
     /**
       * Sets a plot title.
       */
     void setTitle( const QString& title );
     /**
       * Returns title.
       */
     QString title() const { return m_title_str; }
      /**
        * Sets a new matrix as the given channel in the plot.
        */
      virtual bool setMatrix( int channel, QSMatrix *m );
      /**
        * Returns matrix 'channel' if any, or NULL.
        * Notice that you should not operate on it during drawing ( see QSAxes::state )
        * because this affects internal iterator positions in those matrices.
        */
      virtual QSMatrix *matrix( int channel ) const;
      /**
        * Deletes the choosen matrix.
        */
      virtual void deleteMatrix( int channel );
      /**
        * Return matrix and sets data to NULL.
        */
      virtual QSMatrix *takeMatrix( int channel, QSMatrix *newMatrix );
      /**
        * Returns the total number of channels.
        */
      int channelCount() const { return m_channels_count; }
      /**
        * Returns value at the position (x, y) from the channel 'channel'.
        * You have to pass correct arguments to this function because
        * it does not perform any channel or range checking.
        */
      inline double value( int row, int col, int channel ) { return m_channels[channel]->value(row,col); }
      /**
       * Returs the height of the choosen matrix or 0 if it is not set.
       */
      int matrixRows( int channel ) const { return channel<m_channels_count?(m_channels[channel]?m_channels[channel]->rows():0):0; }
      /**
       * Returs the width of the choosen matrix or 0 if it is not set.
       */
      int matrixCols( int channel ) const { return channel<m_channels_count?(m_channels[channel]?m_channels[channel]->cols():0):0; }
      /**
        * Returns 'true' if matrix at the given channel exist.
        */
      bool isChannel( int channel ) const;
      /**
        * Sets the font for the 'e' element.
        */
      void setFont( int e, const QSGFont &font );
       /**
        * Sets the fill for the 'e' element.
        */
      void setFill( int e, const QSGFill &fill );
      /**
        * Set a line style for the 'e' element.
        */
      void setLine( int e, const QSGLine &line );
      /**
        * Set a point style for the 'e' element.
        */
      void setPoint( int e, const QSGPoint &point );
       /**
        * Returns a font style of the given element
        */
      QSGFont font( int e ) const { return e<m_settings.fonts_count?m_settings.fonts[e]:QSGFont();  }
       /**
        * Returns a fill style of the given element
        */
      QSGFill fill( int e ) const { return e<m_settings.fills_count?m_settings.fills[e]:QSGFill(); }
      /**
        * Returns a point style of the given element
        */
      QSGLine line( int e ) const { return e<m_settings.lines_count?m_settings.lines[e]:QSGLine(); }
      /**
        * Returns a point style of the given element
        */
      QSGPoint point( int e ) const { return e<m_settings.points_count?m_settings.points[e]:QSGPoint(); }
      /**
        * Returns the total number of font elements.
        */
      int fontCount() const { return m_settings.fonts_count; }
     /**
       * Returns the total number of fill elements.
       */
     int fillCount() const { return m_settings.fills_count; }
     /**
       * Returns the total number of line elements.
       */
     int lineCount() const { return m_settings.lines_count; }
     /**
       * Returns the total number of points elements.
       */
     int pointCount() const { return m_settings.points_count; }
     /**
       * Restores all graphics atributes
       */
     virtual void loadStateFromStream( QDataStream& stream, QSObjectFactory *factory );
     /**
       * Saves all graphics attributes
       */
     virtual void saveStateToStream( QDataStream& stream, QSObjectFactory *factory );

   signals:                                             	
	/**
	  * Title has changed.
	  */
	void sigTitleChanged( const QString& newTitle );

   protected:
     /**
       * Reads font from matrix. Row formst is [ family(text), size, bold, italic, red, green. blue, alpha ) ]
       */
     static QSGFont fontFromData( QSMatrix *source, int row, int start_col, const QSGFont& default_value );
     /**
       * Reads fill style from matrix. Row format is [ style, red, green, blue, alpha ]
       */
     static QSGFill fillFromData( QSMatrix *source, int row, int start_col, const QSGFill& default_value );
     /**
       * Reads a line style from matrix. Row format is [ style, width, red, green, blue, alpha ]
       */
     static QSGLine lineFromData( QSMatrix *source, int row, int start_col, const QSGLine& default_value );
     /**
       * Reads a point form matrix. Row format is [ style, fill, size, red, green, blue, alpha ]
       */
     static QSGPoint pointFromData( QSMatrix *source, int row, int start_col, const QSGPoint& default_value );
     /**
       * Reads an arrow style from matrix. Row format is [ style, size ]
       */
     static QSGArrow arrowFromData( QSMatrix *source, int row, int start_col, const QSGArrow& default_value );
     /**
       * Reads a color from matrix. Row format is [ red, green, blue, alpha ]
       */
     static QSGColor colorFromData( QSMatrix *source, int row, int start_col, const QSGColor& default_value );
     /**
       * Standard column types for font. See fontFromData().
       */
     static ColumnType fontColumnFormat( int column );
     /**
       * Standard column types for font. See fillFromData().
       */
     static ColumnType fillColumnFormat( int column );
     /**
       * Standard column types for font. See lineFromData().
       */
     static ColumnType lineColumnFormat( int column );
     /**
       * Standard column types for point. See pointFromData().
       */
     static ColumnType pointColumnFormat( int column );
     /**
       * Standard column types for arrow. See arrowFromData().
       */
     static ColumnType arrowColumnFormat( int column );
     /**
       * Standard column types for font. See colorFromData().
       */
     static ColumnType colorColumnFormat( int column );

    /**
      *  Allocates memory for channels
      */
    void initChannelTable( int channels_num );
     /**
       * Allocates memory for fonts, fills, ... tables.
       */
     void initAttributeTables( int fonts_num, int fills_num,  int lines_num, int points_num );
      /**
        * This function is called before any parameter change.
        * This function can be reimplemented but unfortunately there is no
        * information, which parameter has changed.
        * It informs derived classes to stop the drawing process.
        * Default implementation calls parent->paremerersChanging()
        */
      virtual void parametersChanging();
      /**
        * This function is called after any parameter change.
        * This function can be reimplemented but unfortunately there is no
        * information, which parameter has changed.
        * It informs derived classes that they should repaint themselves.
	* Default implementation calls parent->parametersChanging();
        */
      virtual void parametersChanged();
      /**
        * Calculates a logarithm of value 'value'.
        * Fix it to remember log10(logBase);
        */
      double _log(  double value, double logBase = 10.0 ) { return log10(value)/log10(logBase); }
     /**
       *  fonts, fills and lines styles
       */
     struct settings_t {
        QSGFont  *fonts;
        QSGFill  *fills;
        QSGLine  *lines;
        QSGPoint *points;
	int fonts_count;
	int fills_count;
	int lines_count;
        int points_count;
       } m_settings;

     /**
       * list of matrices
       */
     QSMatrix **m_channels;
     int m_channels_count;

     QString m_title_str;
     QSGraphicalData *m_parent_base_object;
   };

#endif


