/***************************************************************************
                                 qsmatrix.h                               
                             -------------------                                         
    begin                : 01-January-2000
    copyright            : (C) 2000 by Kamil Dobkowski                         
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#ifndef QSMATRIX_H
#define QSMATRIX_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include<qstring.h>
#include<qobject.h>
#include"qsserializable.h"

class QSData;

/**
  * \brief  Matrix class contains data used by QSData
  *
  * If you want your data to be used by this library you will have to make your implementation of this class.
  * It is not as complicated as it looks. You will have to reimplement only : rows(), cols(), value() and optionally string().
  * There are other functions, which  can be implemented: setValue(), setString(), resize(), transpose(),
  * isReference(), isEditable(), isString(), loadStateFromStream(), saveStateToStream()
  * Those memebers are not used by this library, but you will probably want to implement them in nontrivial projects for your own use..
  *
  * When you call QSData::setMatrix(), setDataObject() is invoked. Later you can find out the parent object with dataObject() method.
  * When you are going to change the data stored in the matrix you will have to call dataChanging() before and dataChanged()
  * after the change. This notifies the parent QSData object about the change. It is important because it can be busy drawing the data
  * at the moment and changing it during the operation can cause the crash.
  *
  * see QSData::setMatrix()
  */
class QSMatrix : public QObject, public QSSerializable {
	Q_OBJECT
	public:
	  /**
            * Constructor
            */
	 QSMatrix();
	  /**
            * Destructor. Emits 'sigDeleting()'
            */
	 virtual ~QSMatrix();
	/**
	  * Returns a value at position ( row, col ) in this matrix,
	  * converted to 'double'. This method must be redefined.
	  */
	virtual double value( int row, int col ) = 0;
	/**
	  * Returns string at position ( row, col ).
	  * Default implementation calls value() and
	  * converts the result to string.
	  */
	virtual QString string( int row, int col ) { return QString::number(value(row,col),'g',9); }
	/**
	  * Returns the number of rows in this matrix.
	  */
	virtual int rows() const = 0;
	/**
	  * Return the number of columns in this matrix.
	  */
	virtual int cols() const = 0;
	 /**
           * Resizes matrix ( preserves old data ). This should be reimplemented for
	   * each matrix type.Default implementation does nothing and returns 'false' ( 'operation not implemented' )
           */
        virtual bool resize( int rows, int cols );
	/**
	  * Does transposition. This should be reimplemented. Default implementation does nothing and returns 'false'
          *( which means 'operation not implemented' ).
	  */
	virtual bool transpose() { return false; }
	/**
	  * Copy range.'dst' can be the same matrix.
	  */	
	void copyRange( int row, int col, QSMatrix *src, int srcStartRow, int srcStartCol, int srcEndRow, int srcEndCol );
	 /**
           * Sets a data object. It is called atomatically by QSData object, so
           * there is no need to call it by hand.
           */
         virtual void setDataObject( QSData *data, int channel );
         /**
           * Returns data object
           */
	 QSData *dataObject() const { return m_data_object; }
	 /**
           *  Returns channel
           */
	 int channel() const { return m_channel; }
	 /**
           *  This should be reimplemented to set a new value if editable() is TRUE.
	   *  Doesn't call dataChanging() and dataChanged.
           */
         virtual void setValue( int row, int col, double value );
	 /**
           * Set string
           */
	 virtual void setString( int row, int col, const QString& string );
         /**
	   * If true editor will be painted in blue.
           */
	 virtual bool isReference() const;
         /**
           * If is editable using 'setValue()' this function should return true.
	   * Returns false by default.
           */
	 virtual bool isEditable() const;
	 /**
           * Returns if this matrix holds elements as string raher than matrix.
           */
	 virtual bool isString() const;	 	
	 /**
           * Calls QSData::dataChanging()
           */
	 virtual void dataChanging();
	 /**
           * Calls QSData::dataChanged()
           */
	 virtual void dataChanged();
	/**
          * Loads all properties
          */
	virtual void loadStateFromStream( QDataStream& stream, QSObjectFactory *factory );
	/**
          * Saves all properties
          */
	virtual void saveStateToStream( QDataStream& stream, QSObjectFactory *factory );

	protected:	
	 QSData *m_data_object;
	 int m_channel;

      	 QSMatrix( const QSMatrix& );
      	 void operator=(const QSMatrix& );
	
	};
#endif








