/***************************************************************************
                          contactlist.h  -  description
                             -------------------
    begin                : Sun Jan 5 2003
    copyright            : (C) 2003 by Mike K. Bennett
                           (C) 2005 by Diederik van der Boor
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CONTACTLIST_H
#define CONTACTLIST_H

#include <qptrlist.h>
#include <qobject.h>
#include <qdict.h>

// Forward declarations
class Contact;
class CurrentAccount;
class Group;
class KConfig;
class KMessTest;


/**
 * @brief Data class for the contact list.
 *
 * This class stores the state of the current contact list, it does not update the server.
 * The MsnNotificationConnection class sends update requests to the server.
 * When the server acknowledges the change, it will be updated here.
 * The user interface classes repond to the signals of this class to update their view.
 *
 * The instance of this class is shared between the MsnNotificationConnection class,
 * and re-used when the user connects with a different account.
 * After the MsnNotificationConnection class is initialized, a pointer
 * to the ContactList is set in the CurrentAccount class too.
 * The CurrentAccount class is only used to provide access to the current contact list,
 * the administration of the contacts is still done through the MsnNotificationConnection class,
 *
 * @author Mike K. Bennett
 * @ingroup Contact
 */
class ContactList : public QObject
{
  Q_OBJECT

  friend class KMessTest;

  public: // Public methods
    // The constructor
                         ContactList();
    // The destructor
                        ~ContactList();
    // Add a contact to the contact list
    Contact *            addContact( QString handle, QString friendlyName, int lists, QString groupIds, QString guid );
    // Add a group to the contact list
    Group *              addGroup( QString groupId, QString groupName );
    // Change a contact's status (and update the friendly name)
    void                 changeContactStatus( QString handle, QString friendlyName, QString status,
                                              uint capabilities = 0, QString msnObject = 0, bool showBaloon = true );
    // Return the contact with the given guid
    Contact*             getContactByGuid( QString guid ) const;
    // Return the contact with the given handle
    Contact*             getContactByHandle( QString handle ) const;
    // Return the list of contacts
    const QDict<Contact>& getContactList() const;
    // Return the group with the given id
    Group*               getGroupById( QString groupId ) const;
    // Return the group with the given sort position
    Group*               getGroupBySortPosition(int sortPosition) const;
    // Return the list of groups
    const QPtrList<Group>& getGroupList() const;
    // Read in account and other properties
    void                 readProperties(KConfig *config);
    // Remove a group
    void                 removeGroup( QString groupId );
    // Change a contact's name
    void                 renameContact( QString handle, QString newName );
    // Change a group's name
    void                 renameGroup( QString groupId, QString newName );
    // Restore the orginal state of the contact list (empty it)
    void                 reset();
    // Save account and other properties
    void                 saveProperties(KConfig *config);
    // Set a contact to offline
    void                 setContactOffline( QString handle );

  private: // Private methods
    // Return whether or not the contact with the given handle exists
    bool                 contactExists( QString handle );
    // Delete all the contacts
    void                 deleteAllContacts();
    // Delete all non-special groups
    void                 deleteNonSpecialGroups();
    // Return whether or not a group with the given id exists
    bool                 groupExists( QString groupId );

  private slots: // Private slots
    // Forward from a contact that it's msn object changed
    void                 slotForwardContactChangedMsnObject(Contact *contact);
    // Forward from a contact that is has moved
    void                 slotForwardContactMoved(Contact *contact);
    // Forward from a contact that it is offline
    void                 slotForwardContactOffline(Contact *contact, bool showBaloon);
    // Forward from a contact that it is contact is online
    void                 slotForwardContactOnline(Contact *contact, bool showBaloon);


  private: // Private attributes
    // The list of contacts
    QDict<Contact>       contacts_;
    // The list of groups
    QPtrList<Group>      groups_;

  signals: // Signals
    // Signal that a contact has been added.
    void                 contactAdded(Contact *contact);
    // Signal that a contact changed it's status.
    void                 contactChangeStatus( Contact *contact, bool showBaloon );
    // Signal that a contact changed it's msnobject.
    void                 contactChangedMsnObject(Contact *contact);
    // Signal that a contact has moved to a different position in the group list
    void                 contactMoved( Contact *contact );
    // Signal that a contact is offline
    void                 contactOffline( Contact *contact, bool showBaloon );
    // Signal that a contact is online
    void                 contactOnline( Contact *contact, bool showBaloon );
    // Signal that a contact has been removed.
    void                 contactRemoved(const Contact *contact);
    // Signal that a group has been added
    void                 groupAdded(Group *group);
    // Signal that a group has been added
    void                 groupRemoved(const Group *group);
};

#endif
