/***************************************************************************
                          directconnectionbase.cpp -  description
                             -------------------
    begin                : Tue 12 27 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : vdboor --at-- codingdomain.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "directconnectionbase.h"

#include "../../kmessdebug.h"

#include <qcstring.h>

#include <kextsock.h>
#include <ksockaddr.h>
#include <errno.h>

#ifdef KMESSDEBUG_DIRECTCONNECTION
  #define KMESSDEBUG_DIRECTCONNECTION_GENERAL
#endif


int DirectConnectionBase::nextServerPort_(6891);


// The constructor
DirectConnectionBase::DirectConnectionBase( QObject *parent, const char *name)
  : QObject(parent, name)
  , authorized_(false)
  , isServer_(false)
  , lastWriteFailed_(false)
  , server_(0)
  , serverPort_(-1)
  , socket_(0)
  , timeout_(false)
  , writeHandlerCount_(0)
{
  connect( &connectionTimer_, SIGNAL(               timeout() ),
           this,              SLOT  ( slotConnectionTimeout() ));
}



// The destructor
DirectConnectionBase::~DirectConnectionBase()
{
  // Close and clean up
  if(socket_ != 0 || server_ != 0)
  {
    closeConnection();
  }

#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DESTROYED DirectConnectionBase [connectTo=" << connectingTo_ << "]" << endl;
#endif
}



// Close the connection
void DirectConnectionBase::closeConnection()
{
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase: closing sockets" << endl;
#endif

  // Reset globals
  // Also see slotSocketClosed()
  isServer_   = false;
  authorized_ = false;
  connectionTimer_.stop();

  // Check whether we should emit a signal:
  bool closed = closeClientSocket() || closeServerSocket();

#ifdef KMESS_NETWORK_WINDOW
  KMESS_NET_CLOSE(this);
#endif

  // Signal (only once) that the connection was closed
  if(closed)
  {
    emit connectionClosed();
  }
}


// Close and delete the client socket
bool DirectConnectionBase::closeClientSocket()
{
  connectionTimer_.stop();

  // Notify the socket was already closed.
  if( socket_ == 0 )
  {
    return false;
  }

#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase::closeClientSocket() - destroying client socket." << endl;
#endif

  // Close and delete the socket
  socket_->flush();
  socket_->closeNow();  // also calls cancelAsyncConnect()
  delete socket_;
  socket_ = 0;
  writeHandlerCount_ = 0;
  return true;
}


// Close and delete the server socket
bool DirectConnectionBase::closeServerSocket()
{
  connectionTimer_.stop();

  // Notify the socket was already closed.
  if(server_ == 0)
  {
    return false;
  }

#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase::closeServerSocket() - destroying server socket." << endl;
#endif

  server_->closeNow();
  delete server_;
  server_ = 0;
  return true;
}



// Register a slot which will be called each time the write buffer is empty (for autonomous/asynchonous writing).
void DirectConnectionBase::connectWriteHandler(QObject *receiver, const char *slot)
{
  if( KMESS_NULL(socket_) ) return;

  // Enable the readyWrite signal.
  // This is invoked each time the write-buffer is empty,
  // meaning the derived class can send data fast in a autonomous way.

  // When you don't need to write commands very often,
  // this will generate a lot of of CPU overhead.
  // Since our socket is buffered, writeBlock() can be called directly too.

  // Just before KMess 1.5, we've added an internal slot here.
  // If a block can't be written properly,
  // it's remaining part will be buffered here to be sent ASAP.
  // That why there is a slotSocketReadyWrite() between the receiver and the socket now.

  if( writeHandlerCount_++ <= 0 )
  {
    socket_->enableWrite( true );
    connect( socket_, SIGNAL(readyWrite()), this, SLOT(slotSocketReadyWrite()) );
  }

  connect( this, SIGNAL(writeHandlerReady()), receiver, slot );
}



// Unregister the slot which will be called each time the write buffer is empty.
void DirectConnectionBase::disconnectWriteHandler(QObject *receiver, const char *slot)
{
  if( KMESS_NULL(socket_) ) return;

  // If all listeners are unset, disable the signal again.
  if( --writeHandlerCount_ <= 0 )
  {
    socket_->enableWrite( false );
    disconnect( socket_, SIGNAL(readyWrite()), this, SLOT(slotSocketReadyWrite()) );
  }

  disconnect( this, SIGNAL(writeHandlerReady()), receiver, slot );
}



// Return the number of bytes which are already received in the local buffer
int DirectConnectionBase::getAvailableBytes() const
{
  if(KMESS_NULL(socket_)) return -2;  // same as KExtendedSocket API, means "invalid state"
  return socket_->bytesAvailable();
}



// Get the server port that will be used
int DirectConnectionBase::getLocalServerPort()
{
  // Port not chosen yet
  if( serverPort_ < 0 )
  {
    // It's possible to use a random port nowadays,
    // old clients with the "only port 6891" are no longer in use.
    //
    // KMess continues to use ports between 6891-6900 so it's easier
    // to firewall these.
    //
    // TODO: verify whether we can use the port by calling openServerPort() earlier.

    if(nextServerPort_ > 6900)
    {
      nextServerPort_ = 6891;
    }

    serverPort_ = nextServerPort_;
    nextServerPort_++;

#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
    kdDebug() << "DirectConnectionBase::getLocalServerPort: Reserved port " << serverPort_ << " for server socket." << endl;
#endif
  }

  return serverPort_;
}



// Get the remote ip the socket is connected with.
QString DirectConnectionBase::getRemoteIp() const
{
  if(socket_ == 0)
  {
    return QString::null;
  }
  else if( socket_->peerAddress() == 0 )
  {
    return QString::null;
  }
  else
  {
    return socket_->peerAddress()->nodeName();
  }
}



// Get the remote port the socket is connected with.
QString DirectConnectionBase::getRemotePort() const
{
  if(socket_ == 0)
  {
    return QString::null;
  }
  else if( socket_->peerAddress() == 0 )
  {
    return QString::null;
  }
  else
  {
    return socket_->peerAddress()->serviceName();
  }
}



// Return the error description
QString DirectConnectionBase::getSocketError() const
{
  if( timeout_ )
  {
    return "connection timeout event";
  }
  else if(socket_ == 0)
  {
    return QString::null;
  }
  else
  {
    // The return value of systemError() is the "errno" value from errno.h at the
    // time the socket system-call failed. This function hides the low-level stuff
    // like "errno" (see man:errno(3)) and perror() (see man:perror(3))
    return KExtendedSocket::strError( socket_->status(), socket_->systemError() );
  }
}



// Return true if the last write action failed.
bool DirectConnectionBase::hasLastWriteFailed() const
{
  return lastWriteFailed_;
}



// Return true when the write buffer is full
bool DirectConnectionBase::hasTemporaryWriteError() const
{
  return ! additionalWriteBuffer_.isEmpty();
}



// Find out if the connection has been inactive since 15 minutes
bool DirectConnectionBase::hasTimedOut() const
{
  // The int returned by QTime::elapsed() is in milliseconds, so it's 900 seconds or 15 min
  return ( lastActivity_.elapsed() > 900000 );
}



// Initialize the connection, return true when this was succesful
bool DirectConnectionBase::initialize()
{
  // The base class doesn't need any initialisation, MsnDirectConnection does.
  return true;
}



// Return whether the connection login was successful.
bool DirectConnectionBase::isAuthorized() const
{
  return authorized_;
}



// Return true if a connection is active
bool DirectConnectionBase::isConnected() const
{
  return (socket_ != 0 && socket_->socketStatus() == KExtendedSocket::connected);
}



// Return true if this class acts as server, false if it acts as client.
bool DirectConnectionBase::isServer() const
{
  return isServer_;
}



// Return true when a write handler is connnected.
bool DirectConnectionBase::isWriteHandlerConnected() const
{
  return ( writeHandlerCount_ > 0 );
}



// Connect to a host
bool DirectConnectionBase::openConnection(const QString &ipAddress, const int port, bool async)
{
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase::openConnection: Connecting to " << ipAddress << " port " << port << "." << endl;
#endif
#ifdef KMESSTEST
  ASSERT( socket_ == 0 );
#endif
  connectingTo_ = ipAddress + ":" + QString::number(port);

  // Configure the socket for asynchronous operation
  // Note that a buffered socket only emits the readyRead() signal when data is really received.
  // The socket is made non-blocking, so the connectionSuccess() signal tells us when the connection is made.
  // Otherwise KMess would freeze until a connection is made or fails.
  socket_ = new KExtendedSocket(ipAddress, port, KExtendedSocket::bufferedSocket );
  socket_->enableRead(true);
  socket_->enableWrite(false);
  socket_->setBlockingMode(false);  // avoid blocking until connect() returns
  socket_->setTimeout(10);
  connect( socket_, SIGNAL( readyRead()           ), this, SLOT( slotDataReceived()    ));
  connect( socket_, SIGNAL( connectionFailed(int) ), this, SLOT( slotSocketFailed()    ));
  connect( socket_, SIGNAL( connectionSuccess()   ), this, SLOT( slotSocketConnected() ));
  connect( socket_, SIGNAL( closed(int)           ), this, SLOT( slotSocketClosed(int) ));

  // Indicate we'll become a direct-connection client.
  isServer_ = false;
  timeout_  = false;

  // Start timer, as the asyncConnect doesn't seam to honor the timeout value.
  // Needs to be started before startAsyncConnect() because it could call slotSocketFailed() directly.
  connectionTimer_.start(10000, true);

  // Connect the socket
  int connectResult = 0;
  if( async )
  {
    connectResult = socket_->startAsyncConnect();
  }
  else
  {
    connectResult = socket_->connect();
  }

  // Connect the socket
  if(connectResult != 0)
  {
    kdWarning() << "DirectConnectionBase::openConnection: Couldn't connect new socket" << endl;
    delete socket_;
    socket_ = 0;
    writeHandlerCount_ = 0;
    return false;
  }

  // The code continues with slotSocketConnected() or slotSocketFailed()
  return true;
}



// Wait for an incoming connection
bool DirectConnectionBase::openServerPort()
{
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase::openServerPort - Creating server socket." << endl;
#endif
#ifdef KMESSTEST
  ASSERT( server_ == 0 );
#endif

  int  port;
  bool portSuccess;
  int  listenResult;

  // Create the server socket to the given port.
  server_ = new KExtendedSocket();
  server_->setSocketFlags( KExtendedSocket::noResolve | KExtendedSocket::passiveSocket );

  // Set the port
  port        = getLocalServerPort();
  portSuccess = server_->setPort(port);
  if(! portSuccess )
  {
    kdWarning() << "DirectConnectionBase: Could not use port " << port << "." << endl;
    serverPort_ = -1;  // reset, so getLocalServerPort() picks the next port with another openServerPort() call.
    delete server_;
    server_ = 0;
    return false;
  }

  connectingTo_ = "0.0.0.0:" + QString::number(port);

  // Connect the server to signal when it is ready to accept a connection
  connect( server_, SIGNAL( readyAccept() ), this, SLOT( slotAcceptConnection() ));

  // Start timer, but allow a longer timeout.
  // Start before listen() so event's will stop the timer agian.
  connectionTimer_.start(30000, true);

  // Indicate we'll become a direct-connection server.
  isServer_ = true;
  timeout_  = false;

  // Put the socket in listen mode.
  listenResult = server_->listen(1);  // Refuse parallel connections
  if(listenResult != 0)
  {
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
    kdDebug() << "DirectConnectionBase: Could not listen at port " << port << ", freeing for next one." << endl;
#endif
    serverPort_ = -1;  // reset, so getLocalServerPort() picks the next port with another openServerPort() call.
    delete server_;
    server_ = 0;
    return false;
  }

  // The code continues with slotAcceptConnection()
  return true;
}



// Verify how many bytes the read buffer has. Note this actually reads the data to test it.
int DirectConnectionBase::peekBlock( const uint size )
{
  // Avoid crashes
  if(KMESS_NULL(socket_)) return -1;

  // Read the data and see how much got read.
  // This is the only reliable method, but expensive in terms of performance.
  QByteArray buffer( size );
#ifdef KMESSTEST
  ASSERT( size > 0 );
  ASSERT( buffer.size() == size );
#endif
  return socket_->peekBlock( buffer.data(), size );
}



// Read data from the socket
int DirectConnectionBase::readBlock( char *buffer, uint size )
{
  // Avoid crashes
  if(KMESS_NULL(socket_)) return -1;
  if(KMESS_NULL(buffer))  return -1;

  // Fill the buffer
  Q_LONG noBytesRead = socket_->readBlock( buffer, size );
  if( noBytesRead < 0 )
  {
    if( socket_->systemError() == EWOULDBLOCK )
    {
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
      kdDebug() << "DirectConnectionBase::readBlock() - got system error EWOULDBLOCK, buffer is empty." << endl;
#endif
      return 0;
    }
    else
    {
      kdWarning() << "DirectConnectionBase::readBlock() - Error while reading " << size << " bytes from socket, return code: " << noBytesRead << ", system error: " << getSocketError() << " remote address: " << getRemoteIp() << ":" << getRemotePort() << "!" << endl;
      return -1;
    }
  }

#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  if( (uint) noBytesRead < size )
  {
    // TODO: Even if the code detects it's not connected anymore, we don't signal this yet.
    //       It opens a new can of worms to handle various event flows which are covered now.
    kdDebug() << "DirectConnectionBase::readBlock() - only " << noBytesRead
              << " of " << size << " bytes could be read (still connected=" << isConnected() << ")." << endl;
  }
#endif

  // Return the amount of bytes actually read
  return noBytesRead;
}



// Read data from the socket (uses the QByteArray size() as block size)
int DirectConnectionBase::readBlock( QByteArray &buffer, const uint maxSize, const uint bufferOffset )
{
  // Avoid crashes
  if(KMESS_NULL(socket_)) return -1;

  // Make final connected check
  if( ! isConnected() )
  {
    kdWarning() << "DirectConnectionBase::readBlock() - Attempting to read data from a disconnected socket." << endl;
    return false;
  }

  // Determine the maximum number of bytes we can write to the buffer
  int bufferSpace = buffer.size() - bufferOffset;

  // API usage check.
  // Warn if certain sizes get a unusual value
  if(buffer.size() == 0)
  {
    kdWarning() << "DirectConnectionBase::readBlock() - Buffer is not initialized!" << endl;
    return -1;
  }
  else if(bufferSpace < 0)
  {
    kdWarning() << "DirectConnectionBase::readBlock() - Offset of " << bufferOffset << " bytes"
                << " exceeds the buffer size of " << buffer.size() << " bytes!" << endl;
  }
  else if( (uint) bufferSpace < maxSize)
  {
    kdWarning() << "DirectConnectionBase::readBlock() - Maximum allowed read size"
                << " exceeds the buffer size of " << buffer.size() << " bytes!" << endl;
  }

  // Fill the buffer from the give offset,
  // size argument is limited to what the buffer can/may sustain.
  char *bufferStart = buffer.data() + bufferOffset;
  uint size = QMIN( (uint) bufferSpace, maxSize );
  return readBlock( bufferStart, size );
}



// Mark the remote host as authorized (usually after the handshake was successful)
void DirectConnectionBase::setAuthorized(bool authorized)
{
#ifdef KMESSTEST
  ASSERT( isConnected() );
#endif

  // notify when it wasn't authorized before, but will be authorized now.
  bool notify = (! authorized_ && authorized);

  authorized_ = authorized;

  if( notify )
  {
    emit connectionAuthorized();
  }
  else
  {
    kdWarning() << "DirectConnectionBase::setAuthorized() - connection is already authorized, not notifying applications again." << endl;
  }
}



// Accept incoming connections on the socket.
void DirectConnectionBase::slotAcceptConnection()
{
#ifdef KMESSTEST
  ASSERT( socket_ == 0 );
#endif
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase: Accept a socket" << endl;
#endif

  if(socket_ != 0)
  {
    // Already accepted a connection, close server
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
    kdDebug() << "DirectConnectionBase: Already got a connection, rejecting this request" << endl;
#endif

    closeServerSocket();
    return;
  }

  // Wait until a connection was established
  server_->setBlockingMode(true);

  if(server_->accept(socket_) != 0)
  {
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
    kdWarning() << "DirectConnectionBase: Failed to accept incoming connection." << endl;
#endif
    // TODO: signal failure?
    return;
  }

  // Accept success
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase: Accept success" << endl;
#endif

  // Close the server, we don't need to anymore (no connectionClosed events()).
  server_->closeNow();
  delete server_;
  server_ = 0;
  connectionTimer_.stop();

  // Configure the socket for asynchronous operation
  socket_->enableRead(true);
  socket_->enableWrite(false);
  socket_->setBlockingMode(false);
  socket_->setTimeout(10);
  connect( socket_, SIGNAL( readyRead()           ), this, SLOT( slotDataReceived()      ));
  connect( socket_, SIGNAL( connectionFailed(int) ), this, SLOT( slotConnectionFailed()  ));

  // Inform there is a connection
  emit connectionEstablished();
  slotConnectionEstablished(); // call manually, no need to connect
}



// This is called when the connection is established
void DirectConnectionBase::slotConnectionEstablished()
{
  // This method is added for derived classes to overload.
  // It pretends we're also listening to our own signals.

  // This is both called when a outgoing connection was made or
  // an incoming connection was accepted.
  // By default, it's empty,
}



// This is called when there's an error in the socket.
void DirectConnectionBase::slotConnectionFailed()
{
  // This method is added for derived classes to overload.
  // It pretends we're also listening to our own signals.

  QString sysError = getSocketError();  // avoids poking sys_errlist[] manually

  if( ! sysError.isNull() )
  {
    kdWarning() << "DirectConnectionBase: Failed to connect with " << connectingTo_ << ", system error: " << sysError << "." << endl;
  }
  else
  {
    kdWarning() << "DirectConnectionBase: Failed to connect with " << connectingTo_ << "!" << endl;
  }

  // No events here, this method is called from slotSocketFailed()
}



// A timeout occured to connect a socket
void DirectConnectionBase::slotConnectionTimeout()
{
  if( isConnected() )
  {
    // Somehow the code execution seams to get here when:
    // - a connection is established
    // - the other contact didn't send any data yet.
    // This could be happening before because the timer.start() was placed after the startAsyncConnect() calls.
    // This bug is fixed, but keep the check nevertheless.

#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
    kdDebug() << "DirectConnectionBase::slotConnectionTimeout: Timer fired but connection "
                 "with " << connectingTo_ << " is already established, ignore." << endl;
#endif
    return;
  }

#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase::slotConnectionTimeout: Timer fired before connection "
               "with " << connectingTo_ << " was established, "
               "closing sockets and reporting failure." << endl;
#endif

  // Delete sockets, no connectionClosed events.
  closeClientSocket();
  closeServerSocket();

  // Set state for getSocketError();
  timeout_ = true;
  connectionTimer_.stop();

  // Make sure the sockets are removed.
  // When the class is reused, create new sockets.
  // Avoids calls to isConnected() at a socket in bad state.
  closeClientSocket();
  closeServerSocket();

  // Inform derived class and other listeners
  // This pretends we're connected to our signals but without generating the overhead.
  slotConnectionFailed();
  emit connectionFailed();
}



// Signal that the connection was closed
void DirectConnectionBase::slotSocketClosed( int state )
{
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase::slotSocketClosed() - Socket got closed, state=" << state << "." << endl;
#endif

  // Reset the globals.
  isServer_   = false;
  authorized_ = false;
  connectionTimer_.stop();

  if( ! ( state & KBufferedIO::closedNow ) )
  {
    // Make sure the objects are cleaned up too.
    closeClientSocket();
    closeServerSocket();
  }

#ifdef KMESS_NETWORK_WINDOW
  KMESS_NET_CLOSE(this);
#endif

  // Only notify if the remote connection closed the socket.
  // Otherwise it was done by this class.
  if( state & KBufferedIO::involuntary )
  {
    emit connectionClosed();
  }
}



// Signal that the connection was established
void DirectConnectionBase::slotSocketConnected()
{
  // This is called when the socket in openConnection() was connected.
#ifdef KMESSTEST
  ASSERT( ! isServer_ );
#endif
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase::slotSocketConnected() - Connected with host " << connectingTo_ << "." << endl;
#endif

  connectionTimer_.stop();

  // Inform derived class and other listeners
  // This pretends we're connected to our signals but without generating the overhead.
  slotConnectionEstablished();
  emit connectionEstablished();
}



// Signal that the connection could not be made.
void DirectConnectionBase::slotSocketFailed()
{
  // This is called when the socket in openConnection() failed.
#ifdef KMESSTEST
  ASSERT( ! isServer_ );
#endif
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
  kdDebug() << "DirectConnectionBase::slotSocketFailed: Failed to connect to host " << connectingTo_ << "." << endl;
#endif

  connectionTimer_.stop();

  // Make sure the sockets are removed.
  // When the class is reused, create new sockets.
  // Avoids calls to isConnected() at a socket in bad state.
  closeClientSocket();
  closeServerSocket();

  // Inform derived class and other listeners
  // This pretends we're connected to our signals but without generating the overhead.
  slotConnectionFailed();
  emit connectionFailed();
}



// Slot called when the socket is ready to write data.
void DirectConnectionBase::slotSocketReadyWrite()
{
  // First see if the additional write buffer can be cleared.
  if( ! additionalWriteBuffer_.isEmpty() )
  {
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
    kdDebug() << "DirectConnectionBase::slotSocketReadyWrite() - flushing buffer with unsent data first. "
                 "There are " << additionalWriteBuffer_.size() << " bytes to send." << endl;
#endif

    // TODO: some notice in the network window that we'll be sending crap now (remaining parts of a p2p packet).

    // Simply re-use the function.
    // Not the most efficient method, but the most reliable.
    QByteArray newBuffer;
    newBuffer.duplicate( additionalWriteBuffer_ );  // TODO: for Qt4 porting, verify QByteArray sharing changes.
    additionalWriteBuffer_.truncate(0);

    // See what get's sent, and what's added to additionalWriteBuffer_ again.
    bool success = writeBlock( newBuffer );
    if( ! success )
    {
      if( additionalWriteBuffer_.isEmpty() )
      {
        kdWarning() << "DirectConnectionBase::slotSocketReadyWrite() - the remaining " << newBuffer.size() <<
                       " unsent bytes could still not be sent!" << endl;
      }
      else
      {
        // else ok, remaining is in additional buffer again.
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
        kdDebug() << "DirectConnectionBase::slotSocketReadyWrite() - buffer not completely flushed yet, "
                     "still " << additionalWriteBuffer_.size() << " bytes to send." << endl;
#endif
      }

      // Since the data is not completely sent yet, don't signal our listeners yet.
      return;
    }

#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
    kdDebug() << "DirectConnectionBase::slotSocketReadyWrite() - buffer flushed, notifying real listeners." << endl;
#endif
  }


  // Finally notify the real sender
  emit writeHandlerReady();
}



/**
 * @brief Write data to the socket
 *
 * This function writes the packet to the socket.
 * If there is a write error, false is returned.
 * If the error is considered to be temporary, isSentBuffer
 *
 * @param  block  Pointer to the data block to write.
 * @param  size   Size of the block to read.
 * @returns  Whether the has been a write error.
 */
bool DirectConnectionBase::writeBlock( const char *block, const uint size )
{
  lastWriteFailed_ = true;
  if(KMESS_NULL(socket_)) return false;

  // Make final connected check
  if( ! isConnected() )
  {
    kdWarning() << "DirectConnectionBase::writeBlock() - Attempting to write data to a disconnected socket." << endl;
    return false;
  }

  // If the additional buffer is still not written, don't write next block
  // This would send messages in the wrong order.
  if( ! additionalWriteBuffer_.isEmpty() )
  {
#ifdef KMESSTEST
    ASSERT( isWriteHandlerConnected() );
#endif

    // Warn!
    if( additionalWriteBuffer_.size() > 1000 )
    {
      // only warn with release for larger additions. avoid false-positives for size preamble field.
      kdWarning() << "DirectConnectionBase::writeBlock() - received another mesage of " << size << " bytes "
                     "while the write buffer is full!" << endl;
    }
    else
    {
      // else only warn in debug mode.
#ifdef KMESSDEBUG_DIRECTCONNECTION_GENERAL
      kdWarning() << "DirectConnectionBase::writeBlock() - received another mesage of " << size << " bytes "
                     "while the write buffer is full!" << endl;
#endif
    }

    additionalWriteBuffer_.add( block, size );

    // Protect against consuming all system memory
    // 50 kB is more then enough if you're writing messages of 1300 bytes!
    if( additionalWriteBuffer_.size() > 50000 )
    {
      kdWarning() << "DirectConnectionBase::writeBlock() - write buffers are flooding, "
                     "dropping connection immediately!" << endl;
      additionalWriteBuffer_.truncate(0);
      closeConnection();
      return false;
    }
  }

  // Write the data
  Q_LONG noBytesWritten = socket_->writeBlock( block, size );
  if( noBytesWritten < 0 )
  {
    if( socket_->systemError() == EWOULDBLOCK )
    {
      kdWarning() << "DirectConnectionBase::writeBlock() - got system error EWOULDBLOCK, buffer is full." << endl;
      additionalWriteBuffer_.add( block, size );
      return false;
    }
    else
    {
      kdWarning() << "DirectConnectionBase::writeBlock() - Error while writing " << size << " bytes to socket, return code: " << noBytesWritten << ", system error: " << getSocketError() << " remote address: " << getRemoteIp() << ":" << getRemotePort() << "!" << endl;
      return false;
    }
  }

  // Log in network window.
  // Use actual size of written block.
#ifdef KMESS_NETWORK_WINDOW
  QByteArray wrapper;
  wrapper.setRawData( block, noBytesWritten );
  KMESS_NET_SENT(this, wrapper);
  wrapper.resetRawData( block, noBytesWritten );
#endif

  // Warn if number of written bytes is off,
  // this corrupts file transfers.
  if( (uint) noBytesWritten < size )
  {
    kdWarning() << "DirectConnectionBase::writeBlock() - only " << noBytesWritten
                << " of " << size << " bytes could be written!" << endl;

    additionalWriteBuffer_.add( block + noBytesWritten, ( size - noBytesWritten ) );
  }
  else
  {
    // Finally set to positive.
    lastWriteFailed_ = false;
  }

  // Return true if the data was indeed written.
  return ( (uint) noBytesWritten == size );
}



// Write data to the socket (convenience function)
bool DirectConnectionBase::writeBlock( const QByteArray &block )
{
  return writeBlock( block.data(), block.size() );
}


#include "directconnectionbase.moc"
