/***************************************************************************
                          directconnectionpool.h -  description
                             -------------------
    begin                : Thu 1 5 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : "vdboor" --at-- "codingdomain.com"
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef DIRECTCONNECTIONPOOL_H
#define DIRECTCONNECTIONPOOL_H

#include <qobject.h>
#include <qptrlist.h>

class DirectConnectionBase;

/**
 * A pool for direct connection attempts.
 *
 * Multiple connection objects can be added, each attempting to establish a connection.
 * When one connection is succesfully established, it's returned by a signal.
 * The other connection attemps will be cancelled.
 * It's used to establish the connection for P2PApplication invitations.
 *
 * @author Diederik van der Boor
 * @ingroup NetworkExtra
 */
class DirectConnectionPool : public QObject
{
  Q_OBJECT

  public:
    // The constructor
                           DirectConnectionPool();
    // The destructor
    virtual               ~DirectConnectionPool();

    // Add a connection to the list, tells the object to connect to the given ipaddress/port.
    bool                   addConnection(DirectConnectionBase *connection, const QString &ipAddress, const int port);
    // Add a connection to the list, starts listening for incoming connections (returns zero if couldn't start listening).
    int                    addServerConnection(DirectConnectionBase *connection);
    // Remove all connections from the pending list
    void                   clearPending();
    // Return the active connection.
    DirectConnectionBase * getActiveConnection() const;
    // Indicate whether the is a active connection or not.
    bool                   hasActiveConnection() const;
    // Indicate whether there are connections pending or not.
    bool                   hasPendingConnections() const;
    // Verify whether the currently active connection is still connected
    bool                   verifyActiveConnection();


  public slots:
    // A connection was authorized
    void                   slotConnectionAuthorized();
    // A connection was closed
    void                   slotConnectionClosed();
    // A connection is established.
    void                   slotConnectionEstablished();
    // A connection could not be made.
    void                   slotConnectionFailed();

  private:  // private properties
    // The currently active connection
    DirectConnectionBase           *activeConnection_;
    // The list of pending connections
    QPtrList<DirectConnectionBase> pendingConnections_;

  signals:
    // The active connection was authorized
    void                   activeConnectionAuthorized();
    // The active connection closed
    void                   activeConnectionClosed();
    // No direct connections could be established
    void                   allConnectionsFailed();
    // A connection was established
    void                   connectionEstablished();
};

#endif
