/***************************************************************************
                          httpsoapconnection.h -  description
                             -------------------
    begin                : Sun Sep 25 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : vdboor --at-- codingdomain.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef HTTPSOAPCONNECTION_H
#define HTTPSOAPCONNECTION_H

#include "../tcpconnectionbase.h"

#include <kurl.h>

#include <qdom.h>

class SoapMessage;
class MimeMessage;


/**
 * @brief SOAP transport over a HTTP connection.
 *
 * The SOAP protocol is used to implement webservices.
 * It's commonly used by .Net applications,
 * MSN Messenger is not an exception here.
 * SOAP is used for new features like:
 * - games / activities.
 * - offline-im.
 * - passport 3.0 authentication.
 * - addressbook data (as of Windows Live Messenger).
 *
 * The class implements SOAP at a basic level.
 * - it operates much like the <code>XmlHttpRequest</code> class works in web browsers.
 * - sniffed XML strings can be sent directly with sendRequest().
 * - no need to translate XML to SOAP encodings.
 * - no need to reverse engineer a WSDL to send SOAP messages.
 * - http://wanderingbarque.com/nonintersecting/2006/11/15/the-s-stands-for-simple/
 *
 * To use this class, extend it to implement method wrappers for the SOAP calls.
 * The request can be sent with sendRequest(). The response is received as QDomElement in parseSoapResult.
 * Overwrite parseSoapResult() to implement the response handling.
 * The XmlFunctions class is useful to parse complex result structures.
 *
 * @author Diederik van der Boor
 * @ingroup NetworkSoap
 */
class HttpSoapConnection : public TcpConnectionBase
{
  Q_OBJECT

  public:  // public methods
    // The constructor
                         HttpSoapConnection( const QString &endpoint, QObject *parent = 0, const char *name = "HttpSoapConnection" );
    // The destructor
    virtual             ~HttpSoapConnection();

    // Close the connection
    virtual void         closeConnection();

    // Return the endpoint
    QString              getEndpoint() const;
    // Whether the connection is idle, not processing a SOAP request/response
    bool                 isIdle();
    // Return the received HTTP status code
    int                  getHttpStatusCode() const;
    // Return the last request ID
    const QString &      getRequestId() const;
    // Return the last SOAP action
    const QString &      getSoapAction() const;

  protected:
    // Escape a string for inclusion in XML
    QString              escapeString( QString value ) const;
    // Parse the HTTP header
    virtual bool         parseHttpHeader( const QString &preamble, const MimeMessage &header );
    // The connection received the full response
    virtual bool         parseHttpBody( const QByteArray &responseBody );
    // The connection received the full response
    virtual void         parseSoapResult( const QDomElement &resultNode, const QDomElement &headerNode ) = 0;
    // Parse the SOAP fault
    virtual void         parseSoapFault( const QDomElement &faultNode, const QDomElement &headerNode );
    // Send a SOAP request
    void                 sendRawPostRequest( const QString &content, const QString &soapAction = QString::null, const QString &requestId = QString::null );
    // Send a SOAP request
    void                 sendRequest( const QString &soapAction, const SoapMessage &message, const QString &requestId = QString::null );
    // Send a SOAP request
    void                 sendRequest( const QString &soapAction, const QString &messageBody, const QString &messageHeader = QString::null, const QString &requestId = QString::null );
    // Change the SOAP endpoint.
    void                 setEndpoint( const QString &endpoint );
    // The connection failed
    virtual void         slotConnectionFailed();

  private:
    // Connect to the endpoint
    bool                 connectToEndpoint();
    // The connection is established
    void                 slotConnectionEstablished();
    // The connection received data
    void                 slotDataReceived();

  private:  // private attributes
    /// The endpoint for the request
    KURL                 endpoint_;
    /// Whether the server requests to keep the connection alive
    bool                 keepAlive_;
    /// Whether there is a parsing error.
    bool                 parseError_;
    /// The message to be sent
    QCString             request_;
    /// Whether there is an request active.
    bool                 requestActive_;
    /// The remote ID of the request.
    QString              requestId_;
    /// The response code
    uint                 responseCode_;
    /// The length of the response
    uint                 responseLength_;
    /// The last SOAP action
    QString              soapAction_;

  signals:
    /**
     * @brief Fired when another error occured.
     * @param  client  The object sending the signal.
     * @param  reason  The message to display in the user interface.
     */
    void                 requestFailed( HttpSoapConnection *client, const QString &reason );

    /**
     * @brief Fired when the SOAP request returned a SOAP fault.
     * @param  client       The object sending the signal.
     * @param  faultCode    The SOAP fault code.
     * @param  faultString  The SOAP fault string value.
     * @param  faultNode    The XML node of the SOAP fault, to extract custom fields if needed.
     */
    void                 requestFailed( HttpSoapConnection *client, const QString &faultCode,
                                        const QString &faultString, QDomElement faultNode );
};

#endif
