/***************************************************************************
                          notificationchat.cpp -  notifies when a
                            contact communicates with the user
                             -------------------
    begin                : Tuesday April 10 2007
    copyright            : (C) 2007 by Valerio Pilo
    email                : amroth@coldshock.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "notificationchat.h"

#include <qfile.h>
#include <qpixmap.h>
#include <qstylesheet.h>

#include <kdebug.h>
#include <klocale.h>
#include <knotifyclient.h>

#include "../kmessdebug.h"
#include "../currentaccount.h"
#include "../contact/contactbase.h"


#define MAX_BALLOON_MESSAGE_SIZE      128



// Class constructor
NotificationChat::NotificationChat( NotificationManager* manager )
  : enableChatStart_(true)
  , enableChatMessages_(false)
  , manager_(manager)
{
}



// Trigger this event's action (Raise the chat window for the calling contact)
void NotificationChat::activate( QStringList /*stringList*/, QObject *object )
{
  ChatWindow *chatWindow = static_cast<ChatWindow*>( object );
  if ( chatWindow != 0 )
  {
    // Force the window to be raised to front and focused
    emit raiseChat( chatWindow, true );
  }
  else
  {
#ifdef KMESSDEBUG_NOTIFICATIONCHAT
    kdDebug() << "NotificationChat::activate() - The given chat window was not valid." << endl;
#endif
  }
}



// Notify the user about this event (a contact has contacted the user)
void NotificationChat::notify( const ChatMessage &message, ChatWindow *chatWindow )
{
  // End if the event is coming from us instead of from a contact
  if( ! message.isIncoming() )
  {
#ifdef KMESSDEBUG_NOTIFICATIONCHAT
    kdDebug() << "NotificationChat::notify() - Not displaying a popup for an outgoing message." << endl;
#endif
    return;
  }

  // Check the chat window pointer, we need it
  if( ! chatWindow )
  {
#ifdef KMESSDEBUG_NOTIFICATIONCHAT
    kdDebug() << "NotificationChat::notify() - Invalid pointer to chat window." << endl;
#endif
    return;
  }

  // First send the sound notification; even if the balloon is disabled.
  if( chatWindow->isChatFirstMessage() )
  {
#if KDE_IS_VERSION(3,2,0)
    KNotifyClient::event( chatWindow->winId(), "chat start" );
#else
    KNotifyClient::event( "chat start" );
#endif
  }
  else
  {
#if KDE_IS_VERSION(3,2,0)
    KNotifyClient::event( chatWindow->winId(), "new message" );
#else
    KNotifyClient::event( "new message" );
#endif
  }

  // If all possible notifications for this event are disabled, do nothing.
  if( ! enableChatStart_ && ! enableChatMessages_ )
  {
#ifdef KMESSDEBUG_NOTIFICATIONCHAT
    kdDebug() << "NotificationChat::notify() - Start and messages notifications are off, skipping." << endl;
#endif
    return;
  }

  int messageType = message.getType();

  // If we should only notify chat starts, skip notificating further messages
  if( ! enableChatMessages_ && enableChatStart_ &&
      messageType == ChatMessage::TYPE_INCOMING && ! chatWindow->isChatFirstMessage() )
  {
#ifdef KMESSDEBUG_NOTIFICATIONCHAT
  kdDebug() << "NotificationChat::notify() - Messages notification is off and it's not the first message, skipping." << endl;
#endif
    return;
  }


  QString text;
  QString friendlyName, status;
  QString body = QStyleSheet::escape( message.getBody() );
  const QString &handle = message.getContactHandle();
  CurrentAccount *currentAccount = CurrentAccount::instance();
  ContactBase *contact = currentAccount->getContactByHandle( handle );

#ifdef KMESSDEBUG_NOTIFICATIONCHAT
  kdDebug() << "NotificationChat::notify() - Contact " << message.getContactHandle() << endl;
#endif

  // Check for a contact handle, we need it for the popup
  if( handle.isEmpty() )
  {
    kdWarning() << "NotificationChat::notify() - Cannot notify chat event, no contact handle given!" << endl;
    return;
  }

  if( contact )
  {
    friendlyName = QStyleSheet::escape( contact->getFriendlyName() );
    status = contact->getStatus();
  }
  else
  {
    friendlyName = handle;
    status = ""; // Invalid status code, will be shown as Online
  }

  // Truncate the message so it doesn't make the balloon too big
  if( body.length() > MAX_BALLOON_MESSAGE_SIZE )
  {
    body.truncate( MAX_BALLOON_MESSAGE_SIZE );
    body += " ...";
  }


  // Set the balloon text to a generic message, which will be overridden with more useful messages, if available
  switch( messageType )
  {
    case ChatMessage::TYPE_NOTIFICATION:
      text = i18n( "In <b>%1</b>'s chat: %2" ).arg( friendlyName ).arg( body );
      break;
    case ChatMessage::TYPE_INCOMING:
      text = i18n( "<b>%1</b> says:<br>'%2'" ).arg( friendlyName ).arg( body );
      break;
    case ChatMessage::TYPE_OFFLINE_INCOMING:
      text = i18n( "<b>%1</b> has sent you an offline message:<br>'%2'" ).arg( friendlyName ).arg( body );
      break;
    case ChatMessage::TYPE_APPLICATION:
    case ChatMessage::TYPE_APPLICATION_WEBCAM:
    case ChatMessage::TYPE_APPLICATION_FILE:
    case ChatMessage::TYPE_APPLICATION_AUDIO:
      text = i18n( "%1's chat requests attention!" ).arg( friendlyName );
      break;

    case ChatMessage::TYPE_SYSTEM:
    default:
      text = i18n( "<b>%1</b>:<br>%2" ).arg( friendlyName ).arg( body );
      break;
  }

  switch( message.getContentsClass() )
  {
    case ChatMessage::CONTENT_NOTIFICATION_NUDGE:
      text = i18n( "<b>%1</b> has sent you a nudge!" ).arg( friendlyName );
      break;
    case ChatMessage::CONTENT_NOTIFICATION_WINK:
      text = i18n( "<b>%1</b> has sent you a wink!" ).arg( friendlyName );
      break;

    case ChatMessage::CONTENT_APP_INVITE:
      switch( messageType )
      {
        case ChatMessage::TYPE_APPLICATION_WEBCAM: text = i18n( "<b>%1</b> wants to use the webcam!" ).arg( friendlyName ); break;
        case ChatMessage::TYPE_APPLICATION_FILE: text = i18n( "<b>%1</b> is sending you a file!" ).arg( friendlyName ); break;
        default: text = i18n( "<b>%1</b> has sent you an invitation!" ).arg( friendlyName ); break;
      }
      break;
    case ChatMessage::CONTENT_APP_CANCELED:
      switch( messageType )
      {
        case ChatMessage::TYPE_APPLICATION_WEBCAM: text = i18n( "<b>%1</b> has canceled the webcam session!" ).arg( friendlyName ); break;
        case ChatMessage::TYPE_APPLICATION_FILE: text = i18n( "<b>%1</b> has canceled the file transfer!" ).arg( friendlyName ); break;
        default: text = i18n( "<b>%1</b>'s activity has been canceled!" ).arg( friendlyName ); break;
      }
      break;
    case ChatMessage::CONTENT_APP_STARTED:
      switch( messageType )
      {
        case ChatMessage::TYPE_APPLICATION_WEBCAM: text = i18n( "<b>%1</b> has accepted to use the webcam!" ).arg( friendlyName ); break;
        case ChatMessage::TYPE_APPLICATION_FILE: text = i18n( "<b>%1</b> has accepted the file transfer!" ).arg( friendlyName ); break;
        default: text = i18n( "<b>%1</b> has accepted your invitation!" ).arg( friendlyName ); break;
      }
      break;
    case ChatMessage::CONTENT_APP_ENDED:
      switch( messageType )
      {
        case ChatMessage::TYPE_APPLICATION_WEBCAM: text = i18n( "<b>%1</b> has ended the webcam session!" ).arg( friendlyName ); break;
        case ChatMessage::TYPE_APPLICATION_FILE: text = i18n( "The file transfer with <b>%1</b> is done!" ).arg( friendlyName ); break;
        default: text = i18n( "<b>%1</b>'s activity has ended!" ).arg( friendlyName ); break;
      }
      break;
    case ChatMessage::CONTENT_APP_FAILED:
      switch( messageType )
      {
        case ChatMessage::TYPE_APPLICATION_WEBCAM: text = i18n( "<b>%1</b>'s webcam session has failed!" ).arg( friendlyName ); break;
        case ChatMessage::TYPE_APPLICATION_FILE: text = i18n( "The file transfer with <b>%1</b> has failed!" ).arg( friendlyName ); break;
        default: text = i18n( "<b>%1</b>'s activity has ended with an error!" ).arg( friendlyName ); break;
      }
      break;

    default:
      break;
  }


  // Display a popup for this event
  PassivePopup::PopupProperties props;
  props.source = message.getContactHandle();
  props.message = text;
  props.icon = message.getContactPicturePath();
  props.status = status;
  props.destMessages = 0;
  props.destObject = chatWindow;

  manager_->notify( this, props );
}



// Enable or disable the chat notifications
void NotificationChat::setEnabled( bool enableChatStart, bool enableChatMessages )
{
  enableChatStart_ = enableChatStart;
  enableChatMessages_ = enableChatMessages;
}



#include "notificationchat.moc"
