/* -*- c++ -*-
 *
 * sharepage.cpp
 *
 * Copyright (C) 2003 Petter Stokke <ummo@hellokitty.com>
 * Copyright (C) 2003,2004,2007 Sebastian Sauer <mail@dipe.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <kdebug.h>
#include <klocale.h>
#include <kconfig.h>
#include <kaction.h>
#include <ktoggleaction.h>
#include <kactioncollection.h>
#include <knuminput.h>
#include <kmimetype.h>
#include <kxmlguifactory.h>

#include <q3intdict.h>
#include <q3popupmenu.h>
#include <qclipboard.h>
#include <qregexp.h>
#include <qtimer.h>
#include <qcheckbox.h>
#include <QShowEvent>
#include <Q3PtrList>

#include "infolist.h"
#include "prefs/prefs.h"
#include "kmldonkey.h"
#include "shareddirs.h"

#include "network.h"
#include "shareinfo.h"
#include "clientinfo.h"

#include "sharepage.h"
#include "sharepage.moc"

// SharedFile

QString SharedFile::xtext(int col) const
{
    ShareInfo* it = KMLDonkey::App->donkey->findShareNo(fileno);
    if (!it) {
        if (!col) return i18n("Unknown share");
        return i18nc("signifies absence of data in list columns", "-");
    }
    switch (col) {
    case 0: // name
        return it->shareName();
    case 1: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->shareNetwork())->networkName();
    case 2: // requests
        return KGlobal::locale()->formatNumber(it->shareRequests(), 0);
    case 3: // bytes
        return humanReadableSize(it->shareUploaded());
    case 4: // File Type
        return KMimeType::findByUrl( KUrl( "file:/" + it->shareName() ), 0, false, true)->comment();
    default:
        return "ERROR!";
    }
}

double SharedFile::numeric(int col) const
{
    ShareInfo* it = KMLDonkey::App->donkey->findShareNo(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 2:
        return (double)it->shareRequests();
    case 3:
        return (double)it->shareUploaded();
    default:
        return 0.0;
    }
}

bool SharedFile::isNumeric(int col) const
{
    switch (col) {
    case 2:
    case 3:
        return true;
    default:
        return false;
    }
}

// UploadersFile

QString UploadersFile::xtext(int col) const
{
    ClientInfo *ci = KMLDonkey::App->donkey->findClientNo(fileno);
    if (ci)
        switch (col) {
            case 0: return ci->clientUpload();
            case 1: return ci->clientName();
            case 2: return humanReadableSize(ci->clientDownloaded());
            case 3: return humanReadableSize(ci->clientUploaded());
            case 4:
                switch (ci->clientState()) {
                    case ClientInfo::NotConnected:
                    case ClientInfo::NotConnected2: return i18nc("not connected to host", "Not connected");
                    case ClientInfo::Connecting: return i18nc("connecting to host", "Connecting");
                    case ClientInfo::Initiating: return i18nc("initiating connection to host", "Initiating");
                    case ClientInfo::Downloading: return i18nc("downloading from host", "Downloading");
                    case ClientInfo::Connected:
                    case ClientInfo::Connected3: return i18nc("connected to host", "Connected");
                    case ClientInfo::Connected2: return i18nc("we're at this position in client's queue", "Queued: %1", KGlobal::locale()->formatNumber(ci->clientQueuePosition(), 0));
                    case ClientInfo::NewHost: return i18nc("new host added to list", "New host");
                    case ClientInfo::Removed: return i18nc("host has been removed", "Removed");
                    case ClientInfo::Blacklisted: return i18nc("host is blacklisted", "Blacklisted");
                    default: return i18nc("unknown host state (unknown ID code)", "Unknown (%1)", ci->clientState());
                }
            case 5: return KMLDonkey::App->donkey->findNetworkNo(ci->clientNetwork())->networkName();
            case 6: return QString::number(ci->clientRating());
            case 7: return humanReadableTime((time_t)ci->clientConnectTime(), true);
            case 8: return ci->clientSoftware() + " " + ci->clientEmuleMod();
            case 9: return KMimeType::findByUrl( KUrl( "file:/" + ci->clientUpload() ), 0, false, true)->comment();
            default: break;
        }

    return "-";
}

double UploadersFile::numeric(int col) const
{
    ClientInfo *ci = KMLDonkey::App->donkey->findClientNo(fileno);
    if (ci)
        switch (col) {
            case 2: return (double)ci->clientDownloaded();
            case 3: return (double)ci->clientUploaded();
            case 6: return ci->clientRating();
            default: break;
       }
    return 0.0;
}

bool UploadersFile::isNumeric(int col) const
{
    switch (col) {
        case 2:
        case 3:
        case 6:
            return true;
        default: break;
    }
    return false;
}

// SharePage

SharePage::SharePage(QWidget* parent)
    : QWidget(parent)
    , KMLDonkeyPage()
    , ClipboardHelper()
{
    setObjectName("sharePage");
    shareList.setAutoDelete(true);
    uploadList.setAutoDelete(true);
    refreshTimer = new QTimer(this, "refreshTimer");

    QVBoxLayout* layout = new QVBoxLayout(this);
    layout->setSpacing(0);
    layout->setMargin(0);
    setLayout(layout);

    uploadSplitter = new QSplitter(this);
    uploadSplitter->setObjectName("uploadSplitter");
    uploadSplitter->setOrientation(Qt::Vertical);

    uploadView = new InfoList(uploadSplitter, "uploadView", true);
    uploadView->addColumn( i18n( "File name" ), 320 );
    uploadView->addColumn( i18n( "Network" ) );
    uploadView->addColumn( i18n( "Requests" ) );
    uploadView->addColumn( i18n( "Bytes" ) );
    uploadView->addColumn( i18n( "File type" ), 320 );

    uploadersView = new InfoList(uploadSplitter, "uploadersView", true);
    uploadersView->addColumn( i18n( "File name" ), 320 );
    uploadersView->addColumn( i18n( "Name" ), 160 );
    uploadersView->addColumn( i18n( "Downloaded" ) );
    uploadersView->addColumn( i18n( "Uploaded" ) );
    uploadersView->addColumn( i18n( "Status" ) );
    uploadersView->addColumn( i18n( "Network" ) );
    uploadersView->addColumn( i18n( "Rating" ) );
    uploadersView->addColumn( i18n( "Time" ) );
    uploadersView->addColumn( i18n( "Software" ) );
    uploadersView->addColumn( i18n( "File type" ), 320 );

    layout->addWidget(uploadSplitter, 1);
    uploadSplitter->setStretchFactor(0,2);
    uploadSplitter->setStretchFactor(1,1);

    connect(uploadView, SIGNAL(contextMenu(K3ListView*, Q3ListViewItem*, const QPoint&)),
            this, SLOT(contextUpload(K3ListView*, Q3ListViewItem*, const QPoint&)));
    connect(uploadView, SIGNAL(selectionChanged()), SLOT(clearUploadersSelection()));

    connect(uploadersView, SIGNAL(contextMenu(K3ListView*, Q3ListViewItem*, const QPoint&)),
            this, SLOT(contextUploaders(K3ListView*, Q3ListViewItem*, const QPoint&)));
    connect(uploadersView, SIGNAL(selectionChanged()), SLOT(clearUploadSelection()));

    connect(KMLDonkey::App->donkey, SIGNAL(shareUpdated(int)), this, SLOT(shareUpdated(int)));
    connect(KMLDonkey::App->donkey, SIGNAL(shareRemoved(int)), this, SLOT(shareRemoved(int)));

    connect(KMLDonkey::App->donkey, SIGNAL(uploadRemoved(int)), this, SLOT(actionRemoveUploader(int)));
    connect(KMLDonkey::App->donkey, SIGNAL(uploadUpdated(int)), this, SLOT(actionUpdateUploader(int)));

    connect(KMLDonkey::App->donkey, SIGNAL(signalConnected()), this, SLOT(actionRefreshUploaders()));

    connect(refreshTimer, SIGNAL(timeout()), SLOT(refreshLists()));
}

void SharePage::handleGenericAction(const QString& action)
{
    if (uploadView->hasFocus()) {
        if (action == "copy_url") copyFileToClipboard(URL);
        else if (action == "copy_html") copyFileToClipboard(HTML);
        else if (action == "copy_hash") copyFileToClipboard(Hash);
    }
    else if (uploadersView->hasFocus()) {
    }
}

void SharePage::showEvent(QShowEvent*)
{
    if (KMLDonkey::App->donkey->isConnected() && ! autoUpdate) {
        actionRefreshShared();
        actionRefreshUploaders();
    }
}

void SharePage::deactivatePageActions()
{
    enableActionList(uploadActions, false);
    enableActionList(uploadersActions, false);
}

QStringList SharePage::supportedGenericActions()
{
    QStringList actions;
    Q3PtrList<Q3ListViewItem> ul = uploadView->selectedItems();
    Q3PtrList<Q3ListViewItem> sl = uploadersView->selectedItems();

    if (!ul.isEmpty()) {
        enableActionList(uploadActions, true);
        actions.append("copy_url");
        actions.append("copy_html");
        actions.append("copy_hash");
    }
    else enableActionList(uploadActions, false);

    if (!sl.isEmpty()) {
        enableActionList(uploadersActions, true);
    }
    else enableActionList(uploadersActions, false);

    return actions;
}

void SharePage::plugGenericActions(QObject* object, const char* slot)
{
    connect(this, SIGNAL(genericActionsChanged(KMLDonkeyPage*)), object, slot);
}

void SharePage::clearUploadSelection()
{
    uploadView->blockSignals(true);
    uploadView->clearSelection();
    uploadView->blockSignals(false);
    emit genericActionsChanged(static_cast<KMLDonkeyPage*>(this));
}

void SharePage::clearUploadersSelection()
{
    uploadersView->blockSignals(true);
    uploadersView->clearSelection();
    uploadersView->blockSignals(false);
    emit genericActionsChanged(static_cast<KMLDonkeyPage*>(this));
}

void SharePage::setupActions(KActionCollection* actionCollection)
{
    refreshShared = new KAction(KIcon("view-refresh"), i18n("&Refresh Shared File List"), this);
    actionCollection->addAction("upload_refresh", refreshShared);
    connect(refreshShared, SIGNAL(triggered()), this, SLOT(actionRefreshShared()));

    refreshUploaders = new KAction(KIcon("view-refresh"), i18n("Refresh U&ploaders List"), this);
    actionCollection->addAction("uploaders_refresh", refreshUploaders);
    connect(refreshUploaders, SIGNAL(triggered()), this, SLOT(actionRefreshUploaders()));

    KAction* action = new KAction(KIcon("document-export"), i18n("&Shared Folders..."), this);
    actionCollection->addAction("shared_dirs", action);
    connect(action, SIGNAL(triggered()), this, SLOT(actionSharedDirs()));
    KMLDonkey::App->addCoreAction( action );

    action = new KAction(KIcon("arrow-up"), i18n("Activate Uploads Page"), this);
    action->setIconText(i18n("Uploads"));
    actionCollection->addAction("activate_page_uploads", action);
    connect(action, SIGNAL(triggered()), this, SLOT(actionActivatePage()));

    deactivatePageActions();
}

void SharePage::clear()
{
    shareList.clear();
    uploadList.clear();
    uploadView->clear();
    uploadersView->clear();
}

void SharePage::configurePrefsDialog(KMLDonkeyPreferences* prefs)
{
    prefs->listsPage->autoUpdateCheckbox->setChecked(autoUpdate);
    prefs->listsPage->updateFrequencyEntry->setValue(updateFrequency);
}

void SharePage::applyPreferences(KMLDonkeyPreferences* prefs)
{
    if (prefs) {
        autoUpdate = prefs->listsPage->autoUpdateCheckbox->isChecked();
        updateFrequency = prefs->listsPage->updateFrequencyEntry->value();
    }

    updateTimer();

    if (KMLDonkey::App->listFont !=  uploadView->font())
        uploadView->setFont(KMLDonkey::App->listFont);
    if (KMLDonkey::App->listFont !=  uploadersView->font())
        uploadersView->setFont(KMLDonkey::App->listFont);
}

void SharePage::restoreState(KSharedConfigPtr conf)
{
    KConfigGroup group = conf->group( "SharePage" );
    autoUpdate = group.readEntry("AutoUpdateShareLists", false);
    updateFrequency = group.readEntry("ShareListUpdateFrequency", 10);

    QList<int> upsizes = group.readEntry( "UploadSplitter", QList<int>() );
    if( upsizes.count() > 0 )
        uploadSplitter->setSizes(upsizes);

    uploadView->initialise();
    uploadersView->initialise();
    applyPreferences();
}

void SharePage::saveState(KSharedConfigPtr conf)
{
    KConfigGroup group = conf->group( "SharePage" );
    group.writeEntry("AutoUpdateShareLists", autoUpdate);
    group.writeEntry("ShareListUpdateFrequency", updateFrequency);
    group.writeEntry("UploadSplitter", uploadSplitter->sizes());
    uploadView->saveLayout();
    uploadersView->saveLayout();
}

void SharePage::contextUpload(K3ListView*,Q3ListViewItem*,const QPoint& pt)
{
    Q3PopupMenu *pop = (Q3PopupMenu*)(KMLDonkey::App->factory())->container("upload_actions", KMLDonkey::App);
    if (!pop)
        KMLDonkey::App->showBadInstallDialog();
    else
        pop->popup(pt);
}

void SharePage::contextUploaders(K3ListView*,Q3ListViewItem*,const QPoint& pt)
{
    Q3PopupMenu *pop = (Q3PopupMenu*)(KMLDonkey::App->factory())->container("uploaders_actions", KMLDonkey::App);
    if (!pop)
        KMLDonkey::App->showBadInstallDialog();
    else
        pop->popup(pt);
}

void SharePage::actionRefreshShared()
{
    KMLDonkey::App->donkey->refreshShared();
}

void SharePage::actionSharedDirs()
{
    if (! KMLDonkey::App->donkey->isConnected()) return;
    SharedDirsDialog *dlg = new SharedDirsDialog(this);
    if (dlg) dlg->show();
}

void SharePage::shareUpdated(int share)
{
    ShareInfo* si = KMLDonkey::App->donkey->findShareNo(share);
    if (!si) return;
    SharedFile* df = shareList[share];
    if (df) {
        df->refresh();
        return;
    }
    df = new SharedFile(uploadView, share);
    shareList.insert(share, df);
}

void SharePage::shareRemoved(int share)
{
    shareList.remove(share);
}

void SharePage::actionActivatePage()
{
    KMLDonkey::App->activatePage(this);
}

void SharePage::copyFileToClipboard(ClipFormat format)
{
    QStringList l;
    Q3PtrList<Q3ListViewItem> list = uploadView->selectedItems();
    SharedFile* it;
    for (it = (SharedFile*)list.first(); it; it = (SharedFile*)list.next()) {
        const ShareInfo* fi = KMLDonkey::App->donkey->findShareNo(it->fileNo());
        if (!fi) continue;
        l.append(fi->shareName());
        l.append(fi->shareUid("ed2k"));
        l.append(QString::number((long int)fi->shareSize()));
    }
    copyToClipboard(l, format);
}

void SharePage::actionRefreshUploaders()
{
    KMLDonkey::App->donkey->updateUploaders();
}

void SharePage::refreshLists()
{
    actionRefreshShared();
    actionRefreshUploaders();
}

void SharePage::updateTimer()
{
    refreshShared->setEnabled(!autoUpdate);
    refreshUploaders->setEnabled(!autoUpdate);
    if (!autoUpdate) {
        if (refreshTimer->isActive()) refreshTimer->stop();
        return;
    }
    if (refreshTimer->isActive()) {
        if (updateFrequency * 1000 != activeFrequency) {
            activeFrequency = updateFrequency * 1000;
            refreshTimer->start(activeFrequency);
        }
    } else {
        activeFrequency = updateFrequency * 1000;
        refreshTimer->start(activeFrequency);
    }
}

void SharePage::actionUpdateUploader(int clientnum)
{
    // kDebug() << "SharePage::actionUpdateUploader" << clientnum;
    ClientInfo *ci = KMLDonkey::App->donkey->findClientNo(clientnum);
    if (! ci) {
        kDebug() << "SharePage::actionUpdateUploader ClientInfo not ready.";
        return;
    }
    UploadersFile *uf = uploadList[clientnum];
    if (! uf) {
        uf = new UploadersFile(uploadersView, clientnum);
        uploadList.insert(clientnum, uf);
    }
    uf->refresh();
}

void SharePage::actionRemoveUploader(int clientnum)
{
    uploadList.remove(clientnum);
}

