/* This file is part of KNemo
   Copyright (C) 2004, 2006 Percy Leonhardt <percy@eris23.de>
   Copyright (C) 2009 John Stamp <jstamp@users.sourceforge.net>

   KNemo is free software; you can redistribute it and/or modify
   it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   KNemo is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <QtDBus/QDBusConnection>
#include <QTimer>

#include <KAboutData>
#include <KConfigGroup>
#include <KLocale>
#include <KStandardDirs>

#include "config.h"
#include "knemodaemon.h"
#include "interface.h"
#include "backends/backendbase.h"
#include "backends/daemonregistry.h"

QString KNemoDaemon::sSelectedInterface = QString::null;

KNemoDaemon::KNemoDaemon()
    : QObject(),
      mConfig( KGlobal::config() ),
      mColorVLines( 0x04FB1D ),
      mColorHLines( 0x04FB1D ),
      mColorIncoming( 0x1889FF ),
      mColorOutgoing( 0xFF7F08 ),
      mColorBackground( 0x313031 )
{
    readConfig();
    QDBusConnection::sessionBus().registerObject("/knemo", this, QDBusConnection::ExportScriptableSlots);

    bool foundBackend = false;
    int i;
    for ( i = 0; DaemonRegistry[i].name != QString::null; i++ )
    {
        if ( DaemonRegistry[i].name == mBackendName )
        {
            foundBackend = true;
            break;
        }
    }

    if ( !foundBackend )
    {
        i = 0; // use the first backend (Nettools)
    }
    mBackend = ( *DaemonRegistry[i].function )( mInterfaceDict );

    //mInterfaceDict.setAutoDelete( true );

    mPollTimer = new QTimer();
    connect( mPollTimer, SIGNAL( timeout() ), this, SLOT( updateInterfaces() ) );
    mPollTimer->start( mGeneralData.pollInterval * 1000 );
}

KNemoDaemon::~KNemoDaemon()
{
    mPollTimer->stop();
    delete mPollTimer;
    delete mBackend;

    foreach ( QString key, mInterfaceDict.keys() )
    {
        Interface *interface = mInterfaceDict.take( key );
        delete interface;
    }
}

void KNemoDaemon::readConfig()
{
    KConfig *config = mConfig.data();

    KConfigGroup generalGroup( config, "General" );
    mBackendName = generalGroup.readEntry( "Backend", "Nettools" );
    mGeneralData.pollInterval = generalGroup.readEntry( "PollInterval", 1 );
    mGeneralData.saveInterval = generalGroup.readEntry( "SaveInterval", 60 );
    mGeneralData.statisticsDir = generalGroup.readEntry( "StatisticsDir", KGlobal::dirs()->saveLocation( "data", "knemo/" ) );
    mGeneralData.toolTipContent = generalGroup.readEntry( "ToolTipContent", 2 );
    QStringList list = generalGroup.readEntry( "Interfaces", QStringList() );
    int numEntries = list.count();

    if ( numEntries == 0 )
        return;

    QString interface;
    foreach ( QString interface, list )
    {
        Interface* iface = new Interface( interface, mGeneralData, mPlotterSettings );
        QString group( "Interface_" );
        group += interface;
        if ( config->hasGroup( group ) )
        {
            KConfigGroup ifaceGroup( config, group );
            InterfaceSettings& settings = iface->getSettings();
            settings.alias = ifaceGroup.readEntry( "Alias" );
            settings.iconSet = ifaceGroup.readEntry( "IconSet", 0 );
            settings.customCommands = ifaceGroup.readEntry( "CustomCommands", false );
            settings.hideWhenNotAvailable = ifaceGroup.readEntry( "HideWhenNotAvailable", false );
            settings.hideWhenNotExisting = ifaceGroup.readEntry( "HideWhenNotExisting", false );
            settings.activateStatistics = ifaceGroup.readEntry( "ActivateStatistics", false );
            settings.trafficThreshold = ifaceGroup.readEntry( "TrafficThreshold", 0 );
            if ( settings.customCommands )
            {
                int numCommands = ifaceGroup.readEntry( "NumCommands", 0 );
                for ( int i = 0; i < numCommands; i++ )
                {
                    QString entry;
                    InterfaceCommand cmd;
                    entry = QString( "RunAsRoot%1" ).arg( i + 1 );
                    cmd.runAsRoot = ifaceGroup.readEntry( entry, false );
                    entry = QString( "Command%1" ).arg( i + 1 );
                    cmd.command = ifaceGroup.readEntry( entry );
                    entry = QString( "MenuText%1" ).arg( i + 1 );
                    cmd.menuText = ifaceGroup.readEntry( entry );
                    settings.commands.append( cmd );
                }
            }
            iface->configChanged(); // important to activate the statistics
        }
        mInterfaceDict.insert( interface, iface );
    }

    // load the settings for the plotter
    KConfigGroup plotterGroup( config, "PlotterSettings" );
    mPlotterSettings.pixel = plotterGroup.readEntry( "Pixel", 1 );
    mPlotterSettings.count = plotterGroup.readEntry( "Count", 5 );
    mPlotterSettings.distance = plotterGroup.readEntry( "Distance", 30 );
    mPlotterSettings.fontSize = plotterGroup.readEntry( "FontSize", 8 );
    mPlotterSettings.minimumValue = plotterGroup.readEntry( "MinimumValue", 0 );
    mPlotterSettings.maximumValue = plotterGroup.readEntry( "MaximumValue", 1 );
    mPlotterSettings.labels = plotterGroup.readEntry( "Labels", true );
    mPlotterSettings.topBar = plotterGroup.readEntry( "TopBar", false );
    mPlotterSettings.showIncoming = plotterGroup.readEntry( "ShowIncoming", true );
    mPlotterSettings.showOutgoing = plotterGroup.readEntry( "ShowOutgoing", true );
    mPlotterSettings.verticalLines = plotterGroup.readEntry( "VerticalLines", true );
    mPlotterSettings.horizontalLines = plotterGroup.readEntry( "HorizontalLines", true );
    mPlotterSettings.automaticDetection = plotterGroup.readEntry( "AutomaticDetection", true );
    mPlotterSettings.verticalLinesScroll = plotterGroup.readEntry( "VerticalLinesScroll", true );
    mPlotterSettings.colorVLines = plotterGroup.readEntry( "ColorVLines", mColorVLines.rgb() );
    mPlotterSettings.colorHLines = plotterGroup.readEntry( "ColorHLines", mColorHLines.rgb() );
    mPlotterSettings.colorIncoming = plotterGroup.readEntry( "ColorIncoming", mColorIncoming.rgb() );
    mPlotterSettings.colorOutgoing = plotterGroup.readEntry( "ColorOutgoing", mColorOutgoing.rgb() );
    mPlotterSettings.colorBackground = plotterGroup.readEntry( "ColorBackground", mColorBackground.rgb() );
}

void KNemoDaemon::reparseConfiguration()
{
    // Read the settings from the config file.
    QHash< QString, InterfaceSettings *> settingsDict;
    KConfig *config = mConfig.data();
    config->reparseConfiguration();

    KConfigGroup generalGroup( config, "General" );
    mGeneralData.pollInterval = generalGroup.readEntry( "PollInterval", 1 );
    mGeneralData.saveInterval = generalGroup.readEntry( "SaveInterval", 60 );
    mGeneralData.statisticsDir = generalGroup.readEntry( "StatisticsDir", KGlobal::dirs()->saveLocation( "data", "knemo/" ) );
    mGeneralData.toolTipContent = generalGroup.readEntry( "ToolTipContent", 2 );

    // restart the timer with the new interval
    mPollTimer->setInterval( mGeneralData.pollInterval * 1000 );

    // select the backend from the config file
    QString backend = generalGroup.readEntry( "Backend", "Nettools" );

    if ( mBackendName != backend )
    {
        bool foundBackend = false;
        mBackendName = backend;
        int i;
        for ( i = 0; DaemonRegistry[i].name != QString::null; i++ )
        {
            if ( DaemonRegistry[i].name == backend )
            {
                foundBackend = true;
                break;
            }
        }

        if ( foundBackend )
        {
            delete mBackend;
            mBackend = ( *DaemonRegistry[i].function )( mInterfaceDict );
        }
    }

    QStringList list;
    list = generalGroup.readEntry( "Interfaces", list );
    int numEntries = list.count();

    if ( numEntries == 0 )
        return;

    foreach ( QString interface, list )
    {
        QString group( "Interface_" );
        group += interface;
        InterfaceSettings* settings = new InterfaceSettings();
        if ( config->hasGroup( group ) )
        {
            KConfigGroup interfaceGroup( config, group );
            settings->alias = interfaceGroup.readEntry( "Alias" );
            settings->iconSet = interfaceGroup.readEntry( "IconSet", 0 );
            settings->customCommands = interfaceGroup.readEntry( "CustomCommands", false );
            settings->hideWhenNotAvailable = interfaceGroup.readEntry( "HideWhenNotAvailable",false );
            settings->hideWhenNotExisting = interfaceGroup.readEntry( "HideWhenNotExisting", false );
            settings->activateStatistics = interfaceGroup.readEntry( "ActivateStatistics", false );
            settings->trafficThreshold = interfaceGroup.readEntry( "TrafficThreshold", 0 );
            if ( settings->customCommands )
            {
                int numCommands = interfaceGroup.readEntry( "NumCommands", 0 );
                for ( int i = 0; i < numCommands; i++ )
                {
                    QString entry;
                    InterfaceCommand cmd;
                    entry = QString( "RunAsRoot%1" ).arg( i + 1 );
                    cmd.runAsRoot = interfaceGroup.readEntry( entry, false );
                    entry = QString( "Command%1" ).arg( i + 1 );
                    cmd.command = interfaceGroup.readEntry( entry );
                    entry = QString( "MenuText%1" ).arg( i + 1 );
                    cmd.menuText = interfaceGroup.readEntry( entry );
                    settings->commands.append( cmd );
                }
            }
        }
        settingsDict.insert( interface, settings );
    }

    // load the settings for the plotter
    KConfigGroup plotterGroup( config, "PlotterSettings" );
    mPlotterSettings.pixel = plotterGroup.readEntry( "Pixel", 1 );
    mPlotterSettings.count = plotterGroup.readEntry( "Count", 5 );
    mPlotterSettings.distance = plotterGroup.readEntry( "Distance", 30 );
    mPlotterSettings.fontSize = plotterGroup.readEntry( "FontSize", 8 );
    mPlotterSettings.minimumValue = plotterGroup.readEntry( "MinimumValue", 0 );
    mPlotterSettings.maximumValue = plotterGroup.readEntry( "MaximumValue", 1 );
    mPlotterSettings.labels = plotterGroup.readEntry( "Labels", true );
    mPlotterSettings.topBar = plotterGroup.readEntry( "TopBar", false );
    mPlotterSettings.showIncoming = plotterGroup.readEntry( "ShowIncoming", true );
    mPlotterSettings.showOutgoing = plotterGroup.readEntry( "ShowOutgoing", true );
    mPlotterSettings.verticalLines = plotterGroup.readEntry( "VerticalLines", true );
    mPlotterSettings.horizontalLines = plotterGroup.readEntry( "HorizontalLines", true );
    mPlotterSettings.automaticDetection = plotterGroup.readEntry( "AutomaticDetection", true );
    mPlotterSettings.verticalLinesScroll = plotterGroup.readEntry( "VerticalLinesScroll", true );
    mPlotterSettings.colorVLines = plotterGroup.readEntry( "ColorVLines", mColorVLines.rgb() );
    mPlotterSettings.colorHLines = plotterGroup.readEntry( "ColorHLines", mColorHLines.rgb() );
    mPlotterSettings.colorIncoming = plotterGroup.readEntry( "ColorIncoming", mColorIncoming.rgb() );
    mPlotterSettings.colorOutgoing = plotterGroup.readEntry( "ColorOutgoing", mColorOutgoing.rgb() );
    mPlotterSettings.colorBackground = plotterGroup.readEntry( "ColorBackground", mColorBackground.rgb() );

    // Remove all interfaces that the user deleted from
    // the internal list.
    foreach ( QString key, mInterfaceDict.keys() )
    {
        if ( !settingsDict.contains( key ) )
        {
            config->deleteGroup( "Interface_" + key );
            Interface *interface = mInterfaceDict.take( key );
            delete interface;
        }
    }
    config->sync();

    // Update all other interfaces and add new ones.
    foreach ( QString key, settingsDict.keys() )
    {
        Interface* iface;
        if ( !mInterfaceDict.contains( key ) )
        {
            iface = new Interface( key, mGeneralData, mPlotterSettings );
            mInterfaceDict.insert( key, iface );
        }
        else
            iface = mInterfaceDict.value( key );

        InterfaceSettings& settings = iface->getSettings();
        settings.alias = settingsDict.value( key )->alias;
        settings.iconSet = settingsDict.value( key )->iconSet;
        settings.customCommands = settingsDict.value( key )->customCommands;
        settings.hideWhenNotAvailable = settingsDict.value( key )->hideWhenNotAvailable;
        settings.hideWhenNotExisting = settingsDict.value( key )->hideWhenNotExisting;
        settings.activateStatistics = settingsDict.value( key )->activateStatistics;
        settings.trafficThreshold = settingsDict.value( key )->trafficThreshold;
        settings.commands = settingsDict.value( key )->commands;
        iface->configChanged();
    }
}

QString KNemoDaemon::getSelectedInterface()
{
    // Reset the variable to avoid preselecting an interface when
    // the user opens the control center module from the control
    // center afterwards.
    QString tmp = sSelectedInterface;
    sSelectedInterface = QString::null;

    return tmp;
}

void KNemoDaemon::updateInterfaces()
{
    mBackend->update();
}

static const char * const description =
    I18N_NOOP( "The KDE Network Monitor" );

void KNemoDaemon::createAboutData()
{
    mAboutData = new KAboutData( "knemo", 0, ki18n( "KNemo" ), KNEMO_VERSION,
                      ki18n( description ),
                      KAboutData::License_GPL_V2,
                      ki18n( "" ),
                      ki18n( "Copyright (C) 2004, 2005, 2006 Percy Leonhardt\nCopyright (C) 2009 John Stamp\n\nSignal plotter taken from KSysGuard\nCopyright (C) 1999 - 2002, Chris Schlaeger\nCopyright (C) 2006 John Tapsell" ),
                      "http://extragear.kde.org/apps/knemo/",
                      "jstamp@users.sourceforge.net");

    mAboutData->addAuthor( ki18n( "Percy Leonhardt" ), ki18n( "Original Author" ),
                    "percy@eris23.de" );
    mAboutData->addAuthor( ki18n( "John Stamp" ), ki18n( "Current maintainer" ),
                    "jstamp@users.sourceforge.net" );
    mAboutData->addCredit( ki18n( "Michael Olbrich" ), ki18n( "Threshold support" ),
                    "michael.olbrich@gmx.net" );
    mAboutData->addCredit( ki18n( "Chris Schlaeger" ), ki18n( "Signal plotter" ),
                    "cs@kde.org" );
    mAboutData->addCredit( ki18n( "John Tapsell" ), ki18n( "Signal plotter" ),
                    "tapsell@kde.org" );
}

void KNemoDaemon::destroyAboutData()
{
    delete mAboutData;
    mAboutData = NULL;
}

KAboutData* KNemoDaemon::mAboutData;

KAboutData* KNemoDaemon::aboutData()
{
    return mAboutData;
}

#include "knemodaemon.moc"
