/*
 *  knockd.c
 * 
 *  Copyright (c) 2004 by Judd Vinet <jvinet@zeroflux.org>
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <linux/if_ether.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/icmp.h>
#include <net/if.h>
#include <bits/time.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <getopt.h>
#include <syslog.h>
#include <pcap.h>
#include "list.h"

static char version[] = "0.3";

#define SEQ_TIMEOUT 25 /* default knock timeout in seconds */
#define CMD_TIMEOUT 10 /* default timeout in seconds between start and stop commands */
#define SEQ_MAX     32 /* maximum number of ports in a knock sequence */

/* knock/event tuples */
typedef struct opendoor {
	char name[128];
	unsigned short seqcount;
	unsigned short sequence[SEQ_MAX];
	unsigned short protocol[SEQ_MAX];
	time_t seq_timeout;
	char *start_command;
	time_t cmd_timeout;
	char *stop_command;
	unsigned short flag_fin;
	unsigned short flag_syn;
	unsigned short flag_rst;
	unsigned short flag_psh;
	unsigned short flag_ack;
	unsigned short flag_urg;
} opendoor_t;
PMList *doors = NULL;

/* we keep one list of knock attempts, one per IP address,
 * and increment the stage as they progress through the sequence.
 */
typedef struct knocker {
	opendoor_t *door;
	short stage;
	char src[16];
	time_t seq_start;
} knocker_t;
PMList *attempts = NULL;

/* function prototypes */
void dprint(char *fmt, ...);
void vprint(char *fmt, ...);
void logprint(char *fmt, ...);
void cleanup(int signum);
void child_exit(int signum);
void read_cfg(int signum);
void ver();
void usage();
char* strtoupper(char *str);
char* trim(char *str);
void runCommand(char *cmd);
int parseconfig(char *configfile);
int get_mac(const char* iface, u_char *addr);
char* parse_cmd(char* command, char* src);
int exec_cmd(char* command, char* name);
void sniff(u_char* arg, const struct pcap_pkthdr* hdr, const u_char* packet);

pcap_t *cap = NULL;
FILE *logfd = NULL;
int lltype = -1;
u_char if_mac[6];

int  o_usesyslog = 0;
int  o_verbose   = 0;
int  o_debug     = 0;
int  o_daemon    = 0;
char o_int[32]           = "eth0";
char o_cfg[PATH_MAX]     = "/etc/knockd.conf";
char o_pidfile[PATH_MAX] = "/var/run/knockd.pid";
char o_logfile[PATH_MAX] = "";

int main(int argc, char **argv)
{
	char pcapErr[PCAP_ERRBUF_SIZE] = "";
	int opt, ret, optidx = 1;
	static struct option opts[] =
	{
		{"verbose",   no_argument,       0, 'v'},
		{"debug",     no_argument,       0, 'D'},
		{"daemon",    no_argument,       0, 'd'},
		{"interface", required_argument, 0, 'i'},
		{"config",    required_argument, 0, 'c'},
		{"help",      no_argument,       0, 'h'},
		{"version",   no_argument,       0, 'V'},
		{0, 0, 0, 0}
	};

	while((opt = getopt_long(argc, argv, "vDdi:c:hV", opts, &optidx))) {
		if(opt < 0) {
			break;
		}
		switch(opt) {
			case 0:   break;
			case 'v': o_verbose = 1; break;
			case 'D': o_debug = 1; break;
			case 'd': o_daemon = 1; break;
			case 'i': strncpy(o_int, optarg, sizeof(o_int)); break;
			case 'c': strncpy(o_cfg, optarg, sizeof(o_cfg)); break;
			case 'V': ver();
			case 'h': /* fallthrough */
			default: usage();
		}
	}

	if(parseconfig(o_cfg)) {
		exit(1);
	}
	if(o_usesyslog) {
		openlog("knockd", 0, LOG_USER);
	}
	if(strlen(o_logfile)) {
		/* open the log file */
		logfd = fopen(o_logfile, "a");
		if(logfd == NULL) {
			perror("warning: cannot open logfile");
		}
	}

	cap = pcap_open_live(o_int, 65535, 0, 0, pcapErr);
	if(strlen(pcapErr)) {
		fprintf(stderr, "could not open %s: %s\n", o_int, pcapErr);
	}
	if(cap == NULL) {
		exit(1);
	}

	lltype = pcap_datalink(cap);
	switch(lltype) {
		case DLT_EN10MB:
			dprint("ethernet interface detected\n");
			break;
		case DLT_LINUX_SLL:
			dprint("ppp interface detected (linux \"cooked\" encapsulation)\n");
			break;
		default: 
			fprintf(stderr, "error: unsupported link-layer type: %s\n",
					pcap_datalink_val_to_name(lltype));
			cleanup(1);
			break;
	}

	if(lltype == DLT_EN10MB && get_mac(o_int, if_mac)) {
		fprintf(stderr, "could not get MAC address for %s\n", o_int);
		cleanup(1);
	}

	if(o_daemon) {
		if(fork() > 0) {
			/* parent */
			exit(0);
		} else {
			/* child */
			FILE *pidfp;
			setsid();
			close(0);			
			close(1);
			close(2);
			/* write our PID to the pidfile*/
			if((pidfp = fopen(o_pidfile, "w"))) {
				fprintf(pidfp, "%d\n", getpid());
				fclose(pidfp);
			}
		}
	}

	signal(SIGINT, cleanup);
	signal(SIGTERM, cleanup);
	signal(SIGCHLD, child_exit);
	signal(SIGHUP, read_cfg);

	vprint("listening on %s...\n", o_int);
	logprint("starting up, listening on %s", o_int);
	ret = 1;
	while(ret >= 0) {
		ret = pcap_dispatch(cap, -1, sniff, NULL);
	}
	dprint("bailed out of main loop! (ret=%d)\n", ret);
	pcap_perror(cap, "pcap");

	cleanup(0);
	/* notreached */
	exit(0);
}

void dprint(char *fmt, ...)
{
	va_list args;
	if(o_debug) {
		va_start(args, fmt);
		vprintf(fmt, args);
		va_end(args);
		fflush(stdout);
	}
}

void vprint(char *fmt, ...)
{
	va_list args;
	if(o_verbose) {
		va_start(args, fmt);
		vprintf(fmt, args);
		va_end(args);
		fflush(stdout);
	}
}

/* Output a message to syslog and/or a logfile */
void logprint(char *fmt, ...)
{
	char msg[1024];
	va_list args;
	va_start(args, fmt);
	vsnprintf(msg, 1024, fmt, args);
	va_end(args);
	if(o_usesyslog) {
		syslog(LOG_NOTICE, "%s", msg);
	}
	if(logfd) {
		time_t t;
		struct tm *tm;
		t = time(NULL);
		tm = localtime(&t);
		
		fprintf(logfd, "[%02d/%02d/%02d %02d:%02d] %s\n", tm->tm_mon+1, tm->tm_mday,
			tm->tm_year-100, tm->tm_hour, tm->tm_min, msg);
		fflush(logfd);
	}
}

/* Signal handlers */
void cleanup(int signum)
{
	int status;

	vprint("waiting for child processes...\n");
	logprint("waiting for child processes...");
	wait(&status);

	vprint("closing...\n");
	logprint("shutting down");
	pcap_close(cap);
	if(o_daemon) {
		unlink(o_pidfile);
	}
	exit(signum);
}

void child_exit(int signum)
{
	/* child wants to exit, let em die */
	wait(NULL);
	return;
}

void read_cfg(int signum){
	vprint("Re-reading config file: %s\n", o_cfg);
	logprint("Re-reading config file: %s\n", o_cfg);
	parseconfig(o_cfg);
	return;
}

void usage() {
	printf("usage: knockd [options]\n");
	printf("options:\n");
	printf("  -i, --interface <int>  network interface to listen on (default \"eth0\")\n");
	printf("  -d, --daemon           run as a daemon\n");
	printf("  -c, --config <file>    use an alternate config file\n");
	printf("  -D, --debug            output debug messages\n");
	printf("  -v, --verbose          be verbose\n");
	printf("  -V, --version          display version\n");
	printf("  -h, --help             this help\n");
	printf("\n");
	exit(1);
}

void ver() {
	printf("knockd %s\n", version);
	printf("Copyright (C) 2004 Judd Vinet <jvinet@zeroflux.org>\n");
	exit(0);
}

/* Convert a string to uppercase
 */
char* strtoupper(char *str)
{
	char *ptr = str;

	while(*ptr) {
		(*ptr) = toupper(*ptr);
		ptr++;
	}
	return str;
}

/* Trim whitespace and newlines from a string
 */
char* trim(char *str)
{
	char *pch = str;
	while(isspace(*pch)) {
		pch++;
	}
	if(pch != str) {
		memmove(str, pch, (strlen(pch) + 1));
	}
	
	pch = (char*)(str + (strlen(str) - 1));
	while(isspace(*pch)) {
		pch--;
	}
	*++pch = '\0';

	return str;
}

/* Parse a config file
 */
int parseconfig(char *configfile)
{
	FILE *fp = NULL;
	char line[PATH_MAX+1];
	char *ptr = NULL;
	char *key = NULL;
	int linenum = 0;
	char section[256] = "";
	opendoor_t *door = NULL;
	PMList *lp;

	if((fp = fopen(configfile, "r")) == NULL) {
		perror(configfile);
		return(1);
	}

	while(fgets(line, PATH_MAX, fp)) {
		linenum++;
		trim(line);
		if(strlen(line) == 0 || line[0] == '#') {
			continue;
		}
		if(line[0] == '[' && line[strlen(line)-1] == ']') {
			/* new config section */
			ptr = line;
			ptr++;
			strncpy(section, ptr, sizeof(section));
			section[strlen(section)-1] = '\0';
			dprint("config: new section: '%s'\n", section);
			if(!strlen(section)) {
				fprintf(stderr, "config: line %d: bad section name\n", linenum);
				return(1);
			}
			if(strcmp(section, "options")) {
				/* start a new knock/event record */
				door = malloc(sizeof(opendoor_t));
				if(door == NULL) {
					perror("malloc");
					exit(1);
				}
				strncpy(door->name, section, sizeof(door->name));
				door->seqcount = 0;
				door->seq_timeout  = SEQ_TIMEOUT; /* default sequence timeout (seconds)  */
				door->start_command = NULL;
				door->cmd_timeout = CMD_TIMEOUT; /* default command timeout (seconds) */
				door->stop_command = NULL;
				doors = list_add(doors, door);
			}
		} else {
			/* directive */
			if(!strlen(section)) {
				fprintf(stderr, "config: line %d: all directives must belong to a section\n", linenum);
				return(1);
			}
			ptr = line;
			key = strsep(&ptr, "=");
			if(key == NULL) {
				fprintf(stderr, "config: line %d: syntax error\n", linenum);
				return(1);
			}
			trim(key);
			key = strtoupper(key);
			if(ptr == NULL) {
				if(!strcmp(key, "USESYSLOG")) {
					o_usesyslog = 1;
					dprint("config: usesyslog\n");
				} else {
					fprintf(stderr, "config: line %d: syntax error\n", linenum);
					return(1);			
				}
			} else {
				trim(ptr);
				if(!strcmp(section, "options")) {
					if(!strcmp(key, "LOGFILE")) {
						strncpy(o_logfile, ptr, PATH_MAX);
						dprint("config: log file: %s\n", o_logfile);
					} else if(!strcmp(key, "PIDFILE")) {
						strncpy(o_pidfile, ptr, PATH_MAX);
						dprint("config: pid file: %s\n", o_pidfile);

					} else {
						fprintf(stderr, "config: line %d: syntax error\n", linenum);
						return(1);
					}
				} else {
					if(door == NULL) {
						fprintf(stderr, "config: line %d: \"%s\" can only be used within a Door section\n",
								linenum, key);
						return(1);
					}
					if(!strcmp(key, "SEQUENCE")) {
						char *num;
						char *protocol;
						char *port;
						int i;
						while((num = strsep(&ptr, ","))) {
							if(door->seqcount >= SEQ_MAX) {
								fprintf(stderr, "config: section %s: too many ports in knock sequence\n",
										door->name);
								return(1);
							}
							port = strsep(&num, ":");
							door->sequence[door->seqcount++] = (unsigned short)atoi(port);
							if((protocol = strsep(&num, ":"))){
								protocol = strtoupper(trim(protocol));
								if(!strcmp(protocol, "TCP")){
									door->protocol[door->seqcount-1] = IPPROTO_TCP;
								} else if(!strcmp(protocol, "UDP")) {
									door->protocol[door->seqcount-1] = IPPROTO_UDP;
								} else {
									fprintf(stderr,"config: line %d: unknown protocol (%s)\n", linenum, protocol);
									return(1);
								}
							} else {
								door->protocol[door->seqcount-1] = IPPROTO_TCP; /* default protocol */
							}
						}
						dprint("config: %s: sequence: ", door->name);
						for(i = 0; i < door->seqcount; i++) {
							switch(door->protocol[i]){
								case IPPROTO_UDP:
									dprint((i == door->seqcount-1 ? "%u:udp\n" : "%u:udp,"), door->sequence[i]);
									break;
								case IPPROTO_TCP: /* fallthrough */
								default: 
									dprint((i == door->seqcount-1 ? "%u:tcp\n" : "%u:tcp,"), door->sequence[i]);
							}
						}
						dprint("config: %s: protocol: %s\n", door->name, ptr);
					} else if(!strcmp(key, "SEQ_TIMEOUT") || !strcmp(key, "TIMEOUT")) {
						door->seq_timeout = (time_t)atoi(ptr);
						dprint("config: %s: seq_timeout: %d\n", door->name, door->seq_timeout);
					} else if(!strcmp(key, "START_COMMAND") || !strcmp(key, "COMMAND")) {
						door->start_command = malloc(sizeof(char) * (strlen(ptr)+1));
						if(door->start_command == NULL) {
							perror("malloc");
							exit(1);
						}
						strcpy(door->start_command, ptr);
						dprint("config: %s: start_command: %s\n", door->name, door->start_command);
					} else if(!strcmp(key, "CMD_TIMEOUT")) {
						door->cmd_timeout = (time_t)atoi(ptr);
						dprint("config: %s: cmd_timeout: %d\n", door->name, door->cmd_timeout);
					} else if(!strcmp(key, "STOP_COMMAND")) {
						door->stop_command = malloc(sizeof(char) * (strlen(ptr)+1));
						if(door->stop_command == NULL) {
							perror("malloc");
							exit(1);
						}
						strcpy(door->stop_command, ptr);
						dprint("config: %s: stop_command: %s\n", door->name, door->stop_command);
					} else if(!strcmp(key, "TCPFLAGS")) {
						char *flag;
						strtoupper(ptr);
						while((flag = strsep(&ptr, ","))) {
							/* allow just some flags to be specified */
							if(!strcmp(flag,"FIN")) {
								door->flag_fin = 1;
							} else if(!strcmp(flag, "SYN")) {
								door->flag_syn = 1;
							} else if(!strcmp(flag, "RST")) {
								door->flag_rst = 1;
							} else if(!strcmp(flag, "PSH")) {
								door->flag_psh = 1;
							} else if(!strcmp(flag, "ACK")) {
								door->flag_ack = 1;
							} else if(!strcmp(flag, "URG")) {
								door->flag_urg = 1;
							} else {
								fprintf(stderr, "config: line %d: unrecognized flag \"%s\"\n",
										linenum, flag);
								return(1);
							}
							dprint("config: tcp flag: %s\n", flag);
						}
					} else {
						fprintf(stderr, "config: line %d: syntax error\n", linenum);
						return(1);
					}
				}
				line[0] = '\0';
			}
		}
	}
	fclose(fp);

	/* sanity checks */
	for(lp = doors; lp; lp = lp->next) {
		door = (opendoor_t*)lp->data;
		if(door->seqcount == 0) {
			fprintf(stderr, "error: section '%s' has an empty knock sequence\n", door->name);
			return(1);
		}
	}

	return(0);
}

/* Get the MAC address of an ethernet interface
 */
int get_mac(const char* iface, u_char *addr)
{
	struct ifreq ifr;
	struct ifreq *ifrptr;
	struct ifconf ifc;
	char buf[1024];
	int s, i;

	s = socket(AF_INET, SOCK_DGRAM, 0);
	if(s == -1) {
		return(-1);
	}

	ifc.ifc_len = sizeof(buf);
	ifc.ifc_buf = buf;
	ioctl(s, SIOCGIFCONF, &ifc);

	ifrptr = ifc.ifc_req;
	for(i = ifc.ifc_len / sizeof(struct ifreq); --i >= 0; ifrptr++) {
		if(!strcmp(ifrptr->ifr_name, iface)) {
			strcpy(ifr.ifr_name, ifrptr->ifr_name);
			if(ioctl(s, SIOCGIFFLAGS, &ifr) == 0) {
				if(!(ifr.ifr_flags & IFF_LOOPBACK)) {
					if(ioctl(s, SIOCGIFHWADDR, &ifr) == 0) {
						close(s);
						bcopy(ifr.ifr_hwaddr.sa_data, addr, 6);
						return(0);
					}
				}
			}
		}
	}
	return(1);
}

char* parse_cmd(char* command, char* src){
	char *ptr1, *ptr2;
	char origCmd[PATH_MAX];
	char parsedCmd[PATH_MAX] = "";
							
	strncpy(origCmd, command, sizeof(origCmd));
	ptr1 = origCmd;
	while((ptr2 = strstr(ptr1, "%IP%"))) {
		ptr2[0] = '\0';
		strcat(parsedCmd, ptr1);
		strcat(parsedCmd, src);
		ptr1 = ptr2 + 4;
	}
	strcat(parsedCmd, ptr1);
	command = parsedCmd;
	return command;
}

int exec_cmd(char* command, char* name){
	int ret;

	logprint("%s: running command: %s\n", name, command);
	vprint("%s: running command: %s\n", name, command);
	ret = system(command);
	if(ret == -1) {
		fprintf(stderr, "error: command fork failed!\n");
		logprint("error: command fork failed!");
	} else if(ret != 0) {
		fprintf(stderr, "%s: command returned non-zero status code (%d)\n", name, WEXITSTATUS(ret));
		logprint("%s: command returned non-zero status code (%d)", name, WEXITSTATUS(ret));
	}
	return ret;
}

/* Sniff an interface, looking for port-knock sequences
 */
void sniff(u_char* arg, const struct pcap_pkthdr* hdr, const u_char* packet)
{
	int i;
	/* packet structs */
	struct ethhdr* eth = NULL;
	struct iphdr* ip   = NULL;
	struct tcphdr* tcp = NULL;
	struct udphdr* udp = NULL;
	char proto[8];
	/* TCP/IP data */	
	struct in_addr inaddr;
	unsigned short sport, dport;
	char srcIP[16], dstIP[16];
	/* timestamp */
	time_t pkt_secs = hdr->ts.tv_sec;
	struct tm* pkt_tm;
	char pkt_date[11];
	char pkt_time[9];
	PMList *lp;
	knocker_t *attempt = NULL;

	if(lltype == DLT_EN10MB) {
		eth = (struct ethhdr*)packet;
		if(ntohs(eth->h_proto) != ETH_P_IP) {
			return;
		}
		/* make sure this packet was sent TO us, not sent BY us */
		for(i = 0; i < 6; i++) {
			if(eth->h_dest[i] != if_mac[i]) {
				return;
			}
		}
		ip = (struct iphdr*)(packet + sizeof(struct ethhdr));
	} else if(lltype == DLT_LINUX_SLL) {
		if(packet[0] == 0 && packet[1] == 0) {
			/* packet was sent TO us, not sent BY us */
			ip = (struct iphdr*)((u_char*)packet + 16);
		} else {
			return;
		}
	}
	
	if(ip->version != 4) {
		/* no IPv6 yet */
		dprint("packet is not IPv4, ignoring...\n");
		return;
	}
	if(ip->protocol == IPPROTO_ICMP) {
		return;
	}
	sport = dport = 0;
	if(ip->protocol == IPPROTO_TCP) {
		strncpy(proto, "tcp", sizeof(proto));
		tcp = (struct tcphdr*)((u_char*)ip + (ip->ihl * 4));
		sport = ntohs(tcp->source);
		dport = ntohs(tcp->dest);
	}
	if(ip->protocol == IPPROTO_UDP) {
		strncpy(proto, "udp", sizeof(proto));
		udp = (struct udphdr*)((u_char*)ip + (ip->ihl * 4));
		sport = ntohs(udp->source);
		dport = ntohs(udp->dest);
	}

	/* get the date/time */
	pkt_tm = localtime(&pkt_secs);
	snprintf(pkt_date, 11, "%04d-%02d-%02d", pkt_tm->tm_year+1900, pkt_tm->tm_mon,
			pkt_tm->tm_mday);
	snprintf(pkt_time, 9, "%02d:%02d:%02d", pkt_tm->tm_hour, pkt_tm->tm_min,
			pkt_tm->tm_sec);

	/* convert IPs from binary to string */
	inaddr.s_addr = ip->saddr;
	strncpy(srcIP, inet_ntoa(inaddr), 16);
	inaddr.s_addr = ip->daddr;
	strncpy(dstIP, inet_ntoa(inaddr), 16);

	dprint("%s %s: %s: %s:%d -> %s:%d %d bytes\n", pkt_date, pkt_time,
			proto, srcIP, sport, dstIP, dport, hdr->len);

	/* clean up expired/completed/failed attempts */
	for(lp = attempts; lp; lp = lp->next) {
		int nix = 0;
		attempt = (knocker_t*)lp->data;
		if(attempt->stage >= attempt->door->seqcount) {
			dprint("removing successful knock attempt (%s)\n", attempt->src);
			nix = 1;
		}
		if(attempt->stage < 0) {
			dprint("removing failed knock attempt (%s)\n", attempt->src);
			nix = 1;
		}
		if(!nix && (pkt_secs - attempt->seq_start) >= attempt->door->seq_timeout) {
			vprint("%s: %s: sequence timeout (stage %d)\n", attempt->src, attempt->door->name, attempt->stage);
			logprint("%s: %s: sequence timeout (stage %d)", attempt->src, attempt->door->name, attempt->stage);
			nix = 1;
		}
		if(nix) {
			/* splice this entry out of the list */
			if(lp->prev) lp->prev->next = lp->next;
			if(lp->next) lp->next->prev = lp->prev;
			if(lp == attempts) attempts = NULL;
			lp->prev = lp->next = NULL;
			list_free(lp);
			continue;
		}
	}

	attempt = NULL;
	/* look for this guy in our attempts list */
	for(lp = attempts; lp; lp = lp->next) {
		if(!strncmp(((knocker_t*)lp->data)->src, srcIP, sizeof(srcIP))) {
			attempt = (knocker_t*)lp->data;
			break;
		}
	}

	if(attempt) {
		int flagsmatch = 1;
		/* if tcp, check the flags to ignore the packets we don't want
		 * (don't even use it to cancel sequences)
		 */
		if(ip->protocol == IPPROTO_TCP) {
			if(attempt->door->flag_fin == 1 && tcp->fin != 1) {
				dprint("packet is not FIN, ignoring...\n");
				flagsmatch = 0;
			}
			if(attempt->door->flag_syn == 1 && tcp->syn != 1) {
				dprint("packet is not SYN, ignoring...\n");
				flagsmatch = 0;
			}
			if(attempt->door->flag_rst == 1 && tcp->rst != 1) {
				dprint("packet is not RST, ignoring...\n");
				flagsmatch = 0;
			}
			if(attempt->door->flag_psh == 1 && tcp->psh != 1) {
				dprint("packet is not PSH, ignoring...\n");
				flagsmatch = 0;
			}
			if(attempt->door->flag_ack == 1 && tcp->ack != 1) {
				dprint("packet is not ACK, ignoring...\n");
				flagsmatch = 0;
			}
			if(attempt->door->flag_urg == 1 && tcp->urg != 1) {
				dprint("packet is not URG, ignoring...\n");
				flagsmatch = 0;
			}
		}
		if(flagsmatch && ip->protocol == attempt->door->protocol[attempt->stage] &&
				dport == attempt->door->sequence[attempt->stage]) {
			/* level up! */
			attempt->stage++;
			vprint("%s: %s: Stage %d\n", attempt->src, attempt->door->name, attempt->stage);
			logprint("%s: %s: Stage %d", attempt->src, attempt->door->name, attempt->stage);
			if(attempt->stage >= attempt->door->seqcount) {
				vprint("%s: %s: OPEN SESAME\n", attempt->src, attempt->door->name);
				logprint("%s: %s: OPEN SESAME", attempt->src, attempt->door->name);
				if(attempt->door->start_command && strlen(attempt->door->start_command)) {
					/* run the associated command */
					if(fork() == 0) {
						/* child */
						setsid();
						/* execute the parsed (%IP% = source IP) command */
						exec_cmd(parse_cmd(attempt->door->start_command, attempt->src), attempt->door->name);
						/* if stop_command is set, sleep for cmd_timeout and run it*/
						if(attempt->door->stop_command){
							sleep(attempt->door->cmd_timeout);
							vprint("%s: %s: command timeout\n", attempt->src, attempt->door->name);
							logprint("%s: %s: command timeout", attempt->src, attempt->door->name);
							exec_cmd(parse_cmd(attempt->door->stop_command, attempt->src), attempt->door->name);
						}
						
						exit(0); /* exit child */
					}				}
			}
		} else if(flagsmatch == 0) {
			/* TCP flags didn't match -- just ignore this packet, don't
			 * invalidate the knock.
			 */
		} else {
			/* invalidate the knock sequence, it will be removed in the
			 * next sniff() call.
			 */
			attempt->stage = -1;
		}
	} else {
		/* did they hit the first port correctly? */
		for(lp = doors; lp; lp = lp->next) {
			opendoor_t *door = (opendoor_t*)lp->data;
			/* if we're working with TCP, try to match the flags */
			if(ip->protocol == IPPROTO_TCP){
				if(door->flag_fin == 1 && tcp->fin != 1){dprint("packet is not FIN, ignoring...\n");continue;}
				if(door->flag_syn == 1 && tcp->syn != 1){dprint("packet is not SYN, ignoring...\n");continue;}
				if(door->flag_rst == 1 && tcp->rst != 1){dprint("packet is not RST, ignoring...\n");continue;}
				if(door->flag_psh == 1 && tcp->psh != 1){dprint("packet is not PSH, ignoring...\n");continue;}
				if(door->flag_ack == 1 && tcp->ack != 1){dprint("packet is not ACK, ignoring...\n");continue;}
				if(door->flag_urg == 1 && tcp->urg != 1){dprint("packet is not URG, ignoring...\n");continue;}
			}

			if(ip->protocol == door->protocol[0] && dport == door->sequence[0]) {
				/* create a new entry */
				attempt = (knocker_t*)malloc(sizeof(knocker_t));
				if(attempt == NULL) {
					perror("malloc");
					exit(1);
				}
				strcpy(attempt->src, srcIP);
				attempt->stage = 1;
				attempt->seq_start = pkt_secs;
				attempt->door = door;
				vprint("%s: %s: Stage 1\n", attempt->src, door->name);
				logprint("%s: %s: Stage 1", attempt->src, door->name);
				attempts = list_add(attempts, attempt);
			}
		}
	}
}

/* vim: set ts=2 sw=2 noet: */
