
#line 1 "./zscanner/scanner.rl"
/*  Copyright (C) 2011 CZ.NIC, z.s.p.o. <knot-dns@labs.nic.cz>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "zscanner/scanner.h"

#include <stdint.h>			// uint32_t
#include <stdlib.h>			// calloc
#include <stdio.h>			// sprintf
#include <libgen.h>			// dirname
#include <stdbool.h>			// bool
#include <math.h>			// pow
#include <string.h>			// strdup
#include <sys/types.h>			// (OpenBSD)
#include <sys/socket.h>			// AF_INET (BSD)
#include <netinet/in.h>			// in_addr (BSD)
#include <arpa/inet.h>			// inet_pton

#include "common/errcode.h"		// error codes
#include "common/descriptor.h"		// KNOT_RRTYPE_A
#include "zscanner/file_loader.h"	// file_loader
#include "zscanner/scanner_functions.h"	// Base64

/*! \brief Shorthand for setting warning data. */
#define SCANNER_WARNING(code) { s->error_code = code; }
/*! \brief Shorthand for setting error data. */
#define SCANNER_ERROR(code)   { s->error_code = code; s->stop = true; }

/*!
 * \brief Writes record type number to r_data.
 *
 * \param type		Type number.
 * \param rdata_tail	Position where to write type number to.
 */
static inline void type_num(const uint16_t type, uint8_t **rdata_tail)
{
	*((uint16_t *)*rdata_tail) = htons(type);
	*rdata_tail += 2;
}

/*!
 * \brief Sets bit to bitmap window.
 *
 * \param type		Type number.
 * \param s		Scanner context.
 */
static inline void window_add_bit(const uint16_t type, scanner_t *s) {
	uint8_t win      = type / 256;
	uint8_t bit_pos  = type % 256;
	uint8_t byte_pos = bit_pos / 8;

	((s->windows[win]).bitmap)[byte_pos] |= 128 >> (bit_pos % 8);

	if ((s->windows[win]).length < byte_pos + 1) {
		(s->windows[win]).length = byte_pos + 1;
	}

	if (s->last_window < win) {
		s->last_window = win;
	}
}

// Include scanner file (in Ragel).

#line 80 "zscanner/scanner.c"
static const short _zone_scanner_actions[] = {
	0, 1, 0, 1, 1, 1, 2, 1, 
	3, 1, 4, 1, 6, 1, 9, 1, 
	10, 1, 11, 1, 13, 1, 14, 1, 
	16, 1, 19, 1, 20, 1, 22, 1, 
	23, 1, 26, 1, 27, 1, 28, 1, 
	30, 1, 31, 1, 40, 1, 41, 1, 
	42, 1, 44, 1, 46, 1, 47, 1, 
	48, 1, 49, 1, 51, 1, 53, 1, 
	54, 1, 56, 1, 58, 1, 60, 1, 
	61, 1, 65, 1, 66, 1, 69, 1, 
	70, 1, 72, 1, 73, 1, 76, 1, 
	78, 1, 79, 1, 80, 1, 81, 1, 
	82, 1, 83, 1, 84, 1, 85, 1, 
	87, 1, 89, 1, 91, 1, 95, 1, 
	96, 1, 100, 1, 101, 1, 105, 1, 
	106, 1, 107, 1, 108, 1, 109, 1, 
	110, 1, 111, 1, 112, 1, 113, 1, 
	114, 1, 115, 1, 116, 1, 117, 1, 
	118, 1, 120, 1, 121, 1, 122, 1, 
	123, 1, 148, 1, 149, 1, 150, 1, 
	151, 1, 152, 1, 153, 1, 154, 1, 
	155, 1, 156, 1, 157, 1, 158, 1, 
	159, 1, 160, 1, 161, 1, 162, 1, 
	163, 1, 164, 1, 165, 1, 166, 1, 
	167, 1, 168, 1, 169, 1, 170, 1, 
	171, 1, 172, 1, 173, 1, 174, 1, 
	175, 1, 176, 1, 177, 1, 178, 1, 
	179, 1, 180, 1, 181, 1, 182, 1, 
	183, 1, 184, 1, 185, 1, 186, 1, 
	187, 1, 188, 1, 189, 1, 190, 1, 
	191, 1, 192, 1, 193, 1, 194, 1, 
	195, 1, 196, 1, 197, 1, 198, 1, 
	199, 1, 200, 1, 201, 1, 202, 1, 
	203, 1, 204, 1, 205, 1, 206, 1, 
	207, 1, 208, 1, 209, 1, 210, 1, 
	211, 1, 212, 1, 213, 1, 214, 1, 
	215, 1, 216, 1, 217, 1, 218, 1, 
	219, 1, 220, 1, 221, 1, 222, 1, 
	223, 1, 224, 1, 225, 1, 227, 1, 
	229, 1, 230, 1, 231, 1, 232, 1, 
	239, 1, 240, 1, 245, 1, 247, 1, 
	252, 1, 255, 1, 256, 1, 257, 1, 
	258, 1, 260, 1, 261, 1, 262, 1, 
	263, 1, 265, 2, 0, 44, 2, 1, 
	0, 2, 1, 27, 2, 1, 261, 2, 
	1, 304, 2, 2, 27, 2, 2, 263, 
	2, 3, 27, 2, 3, 263, 2, 4, 
	65, 2, 4, 69, 2, 5, 6, 2, 
	9, 10, 2, 12, 13, 2, 14, 10, 
	2, 14, 11, 2, 15, 20, 2, 15, 
	69, 2, 15, 76, 2, 16, 0, 2, 
	16, 2, 2, 16, 3, 2, 16, 75, 
	2, 18, 0, 2, 19, 9, 2, 22, 
	63, 2, 22, 100, 2, 22, 111, 2, 
	23, 0, 2, 23, 1, 2, 23, 2, 
	2, 23, 3, 2, 23, 24, 2, 23, 
	228, 2, 25, 21, 2, 26, 1, 2, 
	26, 2, 2, 26, 3, 2, 26, 85, 
	2, 26, 263, 2, 27, 1, 2, 27, 
	2, 2, 27, 3, 2, 28, 4, 2, 
	28, 260, 2, 28, 265, 2, 29, 21, 
	2, 30, 0, 2, 30, 1, 2, 30, 
	2, 2, 30, 3, 2, 30, 24, 2, 
	31, 36, 2, 32, 31, 2, 33, 65, 
	2, 33, 245, 2, 33, 260, 2, 33, 
	265, 2, 37, 233, 2, 37, 234, 2, 
	37, 235, 2, 37, 236, 2, 37, 237, 
	2, 37, 238, 2, 40, 0, 2, 40, 
	1, 2, 40, 2, 2, 40, 3, 2, 
	41, 0, 2, 41, 1, 2, 41, 2, 
	2, 41, 3, 2, 41, 24, 2, 42, 
	0, 2, 42, 1, 2, 42, 2, 2, 
	42, 3, 2, 43, 259, 2, 44, 1, 
	2, 44, 2, 2, 44, 3, 2, 51, 
	2, 2, 51, 3, 2, 51, 42, 2, 
	51, 85, 2, 52, 53, 2, 54, 1, 
	2, 54, 2, 2, 54, 3, 2, 55, 
	260, 2, 56, 0, 2, 57, 58, 2, 
	59, 60, 2, 61, 0, 2, 61, 56, 
	2, 67, 19, 2, 71, 63, 2, 72, 
	2, 2, 72, 3, 2, 73, 4, 2, 
	74, 19, 2, 76, 4, 2, 80, 1, 
	2, 80, 27, 2, 82, 83, 2, 84, 
	1, 2, 84, 2, 2, 84, 3, 2, 
	85, 1, 2, 85, 2, 2, 85, 3, 
	2, 85, 263, 2, 86, 87, 2, 88, 
	260, 2, 89, 90, 2, 91, 92, 2, 
	93, 94, 2, 93, 95, 2, 93, 96, 
	2, 97, 98, 2, 99, 260, 2, 102, 
	260, 2, 103, 0, 2, 119, 260, 2, 
	121, 0, 2, 122, 0, 2, 123, 0, 
	2, 124, 0, 2, 125, 0, 2, 126, 
	0, 2, 127, 0, 2, 128, 0, 2, 
	129, 0, 2, 130, 0, 2, 131, 0, 
	2, 132, 0, 2, 133, 0, 2, 134, 
	0, 2, 135, 0, 2, 136, 0, 2, 
	137, 0, 2, 138, 0, 2, 139, 0, 
	2, 140, 0, 2, 141, 0, 2, 142, 
	0, 2, 143, 0, 2, 144, 0, 2, 
	145, 260, 2, 146, 260, 2, 147, 260, 
	2, 148, 1, 2, 148, 2, 2, 148, 
	3, 2, 149, 1, 2, 149, 2, 2, 
	149, 3, 2, 150, 1, 2, 150, 2, 
	2, 150, 3, 2, 151, 1, 2, 151, 
	2, 2, 151, 3, 2, 152, 1, 2, 
	152, 2, 2, 152, 3, 2, 153, 1, 
	2, 153, 2, 2, 153, 3, 2, 154, 
	1, 2, 154, 2, 2, 154, 3, 2, 
	155, 1, 2, 155, 2, 2, 155, 3, 
	2, 156, 1, 2, 156, 2, 2, 156, 
	3, 2, 157, 1, 2, 157, 2, 2, 
	157, 3, 2, 158, 1, 2, 158, 2, 
	2, 158, 3, 2, 159, 1, 2, 159, 
	2, 2, 159, 3, 2, 160, 1, 2, 
	160, 2, 2, 160, 3, 2, 161, 1, 
	2, 161, 2, 2, 161, 3, 2, 162, 
	1, 2, 162, 2, 2, 162, 3, 2, 
	163, 1, 2, 163, 2, 2, 163, 3, 
	2, 164, 1, 2, 164, 2, 2, 164, 
	3, 2, 165, 1, 2, 165, 2, 2, 
	165, 3, 2, 166, 1, 2, 166, 2, 
	2, 166, 3, 2, 167, 1, 2, 167, 
	2, 2, 167, 3, 2, 168, 1, 2, 
	168, 2, 2, 168, 3, 2, 169, 1, 
	2, 169, 2, 2, 169, 3, 2, 170, 
	1, 2, 170, 2, 2, 170, 3, 2, 
	171, 1, 2, 171, 2, 2, 171, 3, 
	2, 172, 1, 2, 172, 2, 2, 172, 
	3, 2, 173, 1, 2, 173, 2, 2, 
	173, 3, 2, 174, 1, 2, 174, 2, 
	2, 174, 3, 2, 175, 1, 2, 175, 
	2, 2, 175, 3, 2, 176, 1, 2, 
	176, 2, 2, 176, 3, 2, 177, 1, 
	2, 177, 2, 2, 177, 3, 2, 178, 
	1, 2, 178, 2, 2, 178, 3, 2, 
	179, 1, 2, 179, 2, 2, 179, 3, 
	2, 180, 1, 2, 180, 2, 2, 180, 
	3, 2, 181, 1, 2, 181, 2, 2, 
	181, 3, 2, 182, 1, 2, 182, 2, 
	2, 182, 3, 2, 183, 1, 2, 183, 
	2, 2, 183, 3, 2, 184, 1, 2, 
	184, 2, 2, 184, 3, 2, 185, 1, 
	2, 185, 2, 2, 185, 3, 2, 186, 
	1, 2, 186, 2, 2, 186, 3, 2, 
	187, 1, 2, 187, 2, 2, 187, 3, 
	2, 188, 1, 2, 188, 2, 2, 188, 
	3, 2, 189, 1, 2, 189, 2, 2, 
	189, 3, 2, 190, 1, 2, 190, 2, 
	2, 190, 3, 2, 191, 1, 2, 191, 
	2, 2, 191, 3, 2, 192, 1, 2, 
	192, 2, 2, 192, 3, 2, 193, 1, 
	2, 193, 2, 2, 193, 3, 2, 194, 
	1, 2, 194, 2, 2, 194, 3, 2, 
	195, 1, 2, 195, 2, 2, 195, 3, 
	2, 196, 1, 2, 196, 2, 2, 196, 
	3, 2, 197, 1, 2, 197, 2, 2, 
	197, 3, 2, 198, 1, 2, 198, 2, 
	2, 198, 3, 2, 199, 1, 2, 199, 
	2, 2, 199, 3, 2, 200, 1, 2, 
	200, 2, 2, 200, 3, 2, 201, 1, 
	2, 201, 2, 2, 201, 3, 2, 202, 
	1, 2, 202, 2, 2, 202, 3, 2, 
	203, 1, 2, 203, 2, 2, 203, 3, 
	2, 204, 1, 2, 204, 2, 2, 204, 
	3, 2, 205, 1, 2, 205, 2, 2, 
	205, 3, 2, 206, 1, 2, 206, 2, 
	2, 206, 3, 2, 207, 1, 2, 207, 
	2, 2, 207, 3, 2, 208, 1, 2, 
	208, 2, 2, 208, 3, 2, 209, 1, 
	2, 209, 2, 2, 209, 3, 2, 210, 
	1, 2, 210, 2, 2, 210, 3, 2, 
	211, 1, 2, 211, 2, 2, 211, 3, 
	2, 212, 1, 2, 212, 2, 2, 212, 
	3, 2, 213, 1, 2, 213, 2, 2, 
	213, 3, 2, 214, 1, 2, 214, 2, 
	2, 214, 3, 2, 215, 1, 2, 215, 
	2, 2, 215, 3, 2, 216, 1, 2, 
	216, 2, 2, 216, 3, 2, 217, 1, 
	2, 217, 2, 2, 217, 3, 2, 218, 
	1, 2, 218, 2, 2, 218, 3, 2, 
	219, 1, 2, 219, 2, 2, 219, 3, 
	2, 220, 1, 2, 220, 2, 2, 220, 
	3, 2, 221, 1, 2, 221, 2, 2, 
	221, 3, 2, 222, 1, 2, 222, 2, 
	2, 222, 3, 2, 223, 1, 2, 223, 
	2, 2, 223, 3, 2, 224, 1, 2, 
	224, 2, 2, 224, 3, 2, 225, 1, 
	2, 225, 2, 2, 225, 3, 2, 226, 
	0, 2, 229, 1, 2, 229, 2, 2, 
	229, 3, 2, 230, 1, 2, 230, 2, 
	2, 230, 3, 2, 231, 1, 2, 231, 
	2, 2, 231, 3, 2, 232, 1, 2, 
	232, 2, 2, 232, 3, 2, 239, 1, 
	2, 239, 2, 2, 239, 3, 2, 240, 
	1, 2, 240, 2, 2, 240, 3, 2, 
	243, 0, 2, 244, 260, 2, 246, 100, 
	2, 251, 100, 2, 260, 4, 2, 261, 
	1, 2, 261, 27, 2, 263, 2, 2, 
	263, 3, 2, 264, 261, 2, 265, 4, 
	2, 265, 260, 2, 266, 259, 2, 267, 
	259, 2, 268, 259, 2, 269, 259, 2, 
	270, 259, 2, 271, 259, 2, 272, 259, 
	2, 273, 259, 2, 274, 259, 2, 275, 
	259, 2, 276, 259, 2, 277, 259, 2, 
	278, 259, 2, 279, 259, 2, 280, 259, 
	2, 281, 259, 2, 282, 259, 2, 283, 
	259, 2, 284, 259, 2, 285, 259, 2, 
	286, 259, 2, 287, 259, 2, 288, 259, 
	2, 289, 259, 2, 290, 259, 2, 291, 
	259, 2, 292, 259, 2, 293, 259, 2, 
	294, 259, 2, 295, 259, 2, 296, 259, 
	2, 297, 259, 2, 298, 259, 2, 299, 
	259, 2, 300, 259, 2, 301, 259, 2, 
	302, 259, 2, 303, 259, 2, 304, 1, 
	3, 0, 44, 1, 3, 0, 44, 2, 
	3, 0, 44, 3, 3, 1, 77, 0, 
	3, 1, 103, 0, 3, 1, 226, 0, 
	3, 1, 243, 0, 3, 1, 261, 27, 
	3, 1, 261, 304, 3, 1, 263, 304, 
	3, 1, 304, 27, 3, 7, 1, 8, 
	3, 11, 17, 0, 3, 16, 75, 2, 
	3, 16, 75, 3, 3, 19, 9, 10, 
	3, 23, 1, 0, 3, 23, 24, 1, 
	3, 23, 24, 2, 3, 23, 24, 3, 
	3, 26, 1, 85, 3, 26, 2, 85, 
	3, 26, 2, 263, 3, 26, 3, 85, 
	3, 26, 3, 263, 3, 26, 85, 263, 
	3, 26, 264, 261, 3, 26, 266, 259, 
	3, 28, 81, 4, 3, 28, 260, 4, 
	3, 28, 265, 4, 3, 28, 265, 260, 
	3, 30, 1, 0, 3, 30, 24, 1, 
	3, 30, 24, 2, 3, 30, 24, 3, 
	3, 30, 24, 228, 3, 32, 31, 261, 
	3, 33, 145, 260, 3, 33, 147, 260, 
	3, 33, 265, 4, 3, 33, 265, 260, 
	3, 35, 37, 233, 3, 35, 37, 234, 
	3, 35, 37, 235, 3, 35, 37, 236, 
	3, 35, 37, 237, 3, 35, 37, 238, 
	3, 37, 233, 1, 3, 37, 233, 2, 
	3, 37, 233, 3, 3, 37, 234, 1, 
	3, 37, 234, 2, 3, 37, 234, 3, 
	3, 37, 235, 1, 3, 37, 235, 2, 
	3, 37, 235, 3, 3, 37, 236, 1, 
	3, 37, 236, 2, 3, 37, 236, 3, 
	3, 37, 237, 1, 3, 37, 237, 2, 
	3, 37, 237, 3, 3, 37, 238, 1, 
	3, 37, 238, 2, 3, 37, 238, 3, 
	3, 41, 24, 1, 3, 41, 24, 2, 
	3, 41, 24, 3, 3, 43, 259, 1, 
	3, 43, 259, 2, 3, 43, 259, 3, 
	3, 43, 259, 261, 3, 45, 33, 65, 
	3, 45, 33, 260, 3, 45, 33, 265, 
	3, 50, 32, 31, 3, 51, 42, 0, 
	3, 51, 42, 1, 3, 51, 42, 2, 
	3, 51, 42, 3, 3, 51, 85, 1, 
	3, 51, 85, 2, 3, 51, 85, 3, 
	3, 61, 56, 0, 3, 62, 57, 58, 
	3, 64, 0, 1, 3, 67, 19, 9, 
	3, 68, 0, 1, 3, 74, 19, 9, 
	3, 77, 0, 1, 3, 80, 25, 21, 
	3, 80, 27, 1, 3, 80, 27, 2, 
	3, 80, 27, 3, 3, 82, 25, 21, 
	3, 82, 83, 261, 3, 85, 2, 263, 
	3, 85, 3, 263, 3, 88, 99, 260, 
	3, 88, 145, 260, 3, 89, 90, 0, 
	3, 89, 90, 1, 3, 89, 90, 2, 
	3, 89, 90, 3, 3, 91, 92, 0, 
	3, 91, 92, 1, 3, 91, 92, 2, 
	3, 91, 92, 3, 3, 97, 98, 0, 
	3, 97, 98, 1, 3, 97, 98, 2, 
	3, 97, 98, 3, 3, 102, 104, 245, 
	3, 102, 254, 260, 3, 145, 146, 260, 
	3, 186, 226, 0, 3, 187, 226, 0, 
	3, 188, 226, 0, 3, 189, 226, 0, 
	3, 190, 226, 0, 3, 191, 226, 0, 
	3, 192, 226, 0, 3, 193, 226, 0, 
	3, 194, 226, 0, 3, 195, 226, 0, 
	3, 196, 226, 0, 3, 197, 226, 0, 
	3, 198, 226, 0, 3, 199, 226, 0, 
	3, 200, 226, 0, 3, 201, 226, 0, 
	3, 202, 226, 0, 3, 203, 226, 0, 
	3, 204, 226, 0, 3, 205, 226, 0, 
	3, 206, 226, 0, 3, 207, 226, 0, 
	3, 208, 226, 0, 3, 209, 226, 0, 
	3, 210, 226, 0, 3, 211, 226, 0, 
	3, 212, 226, 0, 3, 213, 226, 0, 
	3, 214, 226, 0, 3, 215, 226, 0, 
	3, 216, 226, 0, 3, 217, 226, 0, 
	3, 218, 226, 0, 3, 219, 226, 0, 
	3, 220, 226, 0, 3, 221, 226, 0, 
	3, 222, 226, 0, 3, 223, 226, 0, 
	3, 224, 226, 0, 3, 225, 226, 0, 
	3, 242, 32, 31, 3, 247, 248, 0, 
	3, 247, 249, 0, 3, 250, 102, 260, 
	3, 252, 253, 0, 3, 254, 102, 260, 
	3, 260, 4, 81, 3, 261, 25, 21, 
	3, 261, 304, 1, 3, 263, 304, 1, 
	3, 265, 260, 4, 3, 266, 259, 1, 
	3, 266, 259, 2, 3, 266, 259, 3, 
	3, 266, 259, 261, 3, 266, 259, 263, 
	3, 267, 259, 1, 3, 267, 259, 2, 
	3, 267, 259, 3, 3, 267, 259, 261, 
	3, 268, 259, 1, 3, 268, 259, 2, 
	3, 268, 259, 3, 3, 268, 259, 261, 
	3, 269, 259, 1, 3, 269, 259, 2, 
	3, 269, 259, 3, 3, 269, 259, 261, 
	3, 270, 259, 1, 3, 270, 259, 2, 
	3, 270, 259, 3, 3, 270, 259, 261, 
	3, 271, 259, 1, 3, 271, 259, 2, 
	3, 271, 259, 3, 3, 271, 259, 261, 
	3, 272, 259, 1, 3, 272, 259, 2, 
	3, 272, 259, 3, 3, 272, 259, 261, 
	3, 273, 259, 1, 3, 273, 259, 2, 
	3, 273, 259, 3, 3, 273, 259, 261, 
	3, 274, 259, 1, 3, 274, 259, 2, 
	3, 274, 259, 3, 3, 274, 259, 261, 
	3, 275, 259, 1, 3, 275, 259, 2, 
	3, 275, 259, 3, 3, 275, 259, 261, 
	3, 276, 259, 1, 3, 276, 259, 2, 
	3, 276, 259, 3, 3, 276, 259, 261, 
	3, 277, 259, 1, 3, 277, 259, 2, 
	3, 277, 259, 3, 3, 277, 259, 261, 
	3, 278, 259, 1, 3, 278, 259, 2, 
	3, 278, 259, 3, 3, 278, 259, 261, 
	3, 279, 259, 1, 3, 279, 259, 2, 
	3, 279, 259, 3, 3, 279, 259, 261, 
	3, 280, 259, 1, 3, 280, 259, 2, 
	3, 280, 259, 3, 3, 280, 259, 261, 
	3, 281, 259, 1, 3, 281, 259, 2, 
	3, 281, 259, 3, 3, 281, 259, 261, 
	3, 282, 259, 1, 3, 282, 259, 2, 
	3, 282, 259, 3, 3, 282, 259, 261, 
	3, 283, 259, 1, 3, 283, 259, 2, 
	3, 283, 259, 3, 3, 283, 259, 261, 
	3, 284, 259, 1, 3, 284, 259, 2, 
	3, 284, 259, 3, 3, 284, 259, 261, 
	3, 285, 259, 1, 3, 285, 259, 2, 
	3, 285, 259, 3, 3, 285, 259, 261, 
	3, 286, 259, 1, 3, 286, 259, 2, 
	3, 286, 259, 3, 3, 286, 259, 261, 
	3, 287, 259, 1, 3, 287, 259, 2, 
	3, 287, 259, 3, 3, 287, 259, 261, 
	3, 288, 259, 1, 3, 288, 259, 2, 
	3, 288, 259, 3, 3, 288, 259, 261, 
	3, 289, 259, 1, 3, 289, 259, 2, 
	3, 289, 259, 3, 3, 289, 259, 261, 
	3, 290, 259, 1, 3, 290, 259, 2, 
	3, 290, 259, 3, 3, 290, 259, 261, 
	3, 291, 259, 1, 3, 291, 259, 2, 
	3, 291, 259, 3, 3, 291, 259, 261, 
	3, 292, 259, 1, 3, 292, 259, 2, 
	3, 292, 259, 3, 3, 292, 259, 261, 
	3, 293, 259, 1, 3, 293, 259, 2, 
	3, 293, 259, 3, 3, 293, 259, 261, 
	3, 294, 259, 1, 3, 294, 259, 2, 
	3, 294, 259, 3, 3, 294, 259, 261, 
	3, 295, 259, 1, 3, 295, 259, 2, 
	3, 295, 259, 3, 3, 295, 259, 261, 
	3, 296, 259, 1, 3, 296, 259, 2, 
	3, 296, 259, 3, 3, 296, 259, 261, 
	3, 297, 259, 1, 3, 297, 259, 2, 
	3, 297, 259, 3, 3, 297, 259, 261, 
	3, 298, 259, 1, 3, 298, 259, 2, 
	3, 298, 259, 3, 3, 298, 259, 261, 
	3, 299, 259, 1, 3, 299, 259, 2, 
	3, 299, 259, 3, 3, 299, 259, 261, 
	3, 300, 259, 1, 3, 300, 259, 2, 
	3, 300, 259, 3, 3, 300, 259, 261, 
	3, 301, 259, 1, 3, 301, 259, 2, 
	3, 301, 259, 3, 3, 301, 259, 261, 
	3, 302, 259, 1, 3, 302, 259, 2, 
	3, 302, 259, 3, 3, 302, 259, 261, 
	3, 303, 259, 1, 3, 303, 259, 2, 
	3, 303, 259, 3, 3, 303, 259, 261, 
	3, 304, 1, 27, 4, 5, 7, 1, 
	8, 4, 14, 11, 17, 0, 4, 16, 
	68, 0, 1, 4, 26, 1, 263, 304, 
	4, 26, 1, 264, 261, 4, 26, 1, 
	266, 259, 4, 26, 2, 85, 263, 4, 
	26, 2, 264, 261, 4, 26, 2, 266, 
	259, 4, 26, 3, 85, 263, 4, 26, 
	3, 264, 261, 4, 26, 3, 266, 259, 
	4, 26, 266, 259, 261, 4, 26, 266, 
	259, 263, 4, 28, 33, 265, 4, 4, 
	28, 45, 33, 265, 4, 28, 265, 260, 
	4, 4, 32, 31, 25, 21, 4, 33, 
	265, 260, 4, 4, 35, 32, 31, 36, 
	4, 35, 37, 233, 1, 4, 35, 37, 
	233, 2, 4, 35, 37, 233, 3, 4, 
	35, 37, 234, 1, 4, 35, 37, 234, 
	2, 4, 35, 37, 234, 3, 4, 35, 
	37, 235, 1, 4, 35, 37, 235, 2, 
	4, 35, 37, 235, 3, 4, 35, 37, 
	236, 1, 4, 35, 37, 236, 2, 4, 
	35, 37, 236, 3, 4, 35, 37, 237, 
	1, 4, 35, 37, 237, 2, 4, 35, 
	37, 237, 3, 4, 35, 37, 238, 1, 
	4, 35, 37, 238, 2, 4, 35, 37, 
	238, 3, 4, 37, 235, 243, 0, 4, 
	37, 236, 243, 0, 4, 37, 237, 243, 
	0, 4, 37, 238, 243, 0, 4, 38, 
	34, 32, 31, 4, 39, 34, 32, 31, 
	4, 43, 259, 1, 261, 4, 45, 33, 
	4, 65, 4, 51, 50, 32, 31, 4, 
	51, 64, 0, 1, 4, 67, 19, 9, 
	10, 4, 72, 77, 0, 1, 4, 74, 
	19, 9, 10, 4, 82, 83, 25, 21, 
	4, 85, 1, 263, 304, 4, 88, 145, 
	146, 260, 4, 89, 90, 1, 0, 4, 
	91, 92, 1, 0, 4, 97, 98, 1, 
	0, 4, 186, 1, 226, 0, 4, 187, 
	1, 226, 0, 4, 188, 1, 226, 0, 
	4, 189, 1, 226, 0, 4, 190, 1, 
	226, 0, 4, 191, 1, 226, 0, 4, 
	192, 1, 226, 0, 4, 193, 1, 226, 
	0, 4, 194, 1, 226, 0, 4, 195, 
	1, 226, 0, 4, 196, 1, 226, 0, 
	4, 197, 1, 226, 0, 4, 198, 1, 
	226, 0, 4, 199, 1, 226, 0, 4, 
	200, 1, 226, 0, 4, 201, 1, 226, 
	0, 4, 202, 1, 226, 0, 4, 203, 
	1, 226, 0, 4, 204, 1, 226, 0, 
	4, 205, 1, 226, 0, 4, 206, 1, 
	226, 0, 4, 207, 1, 226, 0, 4, 
	208, 1, 226, 0, 4, 209, 1, 226, 
	0, 4, 210, 1, 226, 0, 4, 211, 
	1, 226, 0, 4, 212, 1, 226, 0, 
	4, 213, 1, 226, 0, 4, 214, 1, 
	226, 0, 4, 215, 1, 226, 0, 4, 
	216, 1, 226, 0, 4, 217, 1, 226, 
	0, 4, 218, 1, 226, 0, 4, 219, 
	1, 226, 0, 4, 220, 1, 226, 0, 
	4, 221, 1, 226, 0, 4, 222, 1, 
	226, 0, 4, 223, 1, 226, 0, 4, 
	224, 1, 226, 0, 4, 225, 1, 226, 
	0, 4, 266, 259, 1, 261, 4, 266, 
	259, 2, 263, 4, 266, 259, 3, 263, 
	4, 266, 259, 263, 261, 4, 267, 259, 
	1, 261, 4, 268, 259, 1, 261, 4, 
	269, 259, 1, 261, 4, 270, 259, 1, 
	261, 4, 271, 259, 1, 261, 4, 272, 
	259, 1, 261, 4, 273, 259, 1, 261, 
	4, 274, 259, 1, 261, 4, 275, 259, 
	1, 261, 4, 276, 259, 1, 261, 4, 
	277, 259, 1, 261, 4, 278, 259, 1, 
	261, 4, 279, 259, 1, 261, 4, 280, 
	259, 1, 261, 4, 281, 259, 1, 261, 
	4, 282, 259, 1, 261, 4, 283, 259, 
	1, 261, 4, 284, 259, 1, 261, 4, 
	285, 259, 1, 261, 4, 286, 259, 1, 
	261, 4, 287, 259, 1, 261, 4, 288, 
	259, 1, 261, 4, 289, 259, 1, 261, 
	4, 290, 259, 1, 261, 4, 291, 259, 
	1, 261, 4, 292, 259, 1, 261, 4, 
	293, 259, 1, 261, 4, 294, 259, 1, 
	261, 4, 295, 259, 1, 261, 4, 296, 
	259, 1, 261, 4, 297, 259, 1, 261, 
	4, 298, 259, 1, 261, 4, 299, 259, 
	1, 261, 4, 300, 259, 1, 261, 4, 
	301, 259, 1, 261, 4, 302, 259, 1, 
	261, 4, 303, 259, 1, 261, 5, 16, 
	75, 77, 0, 1, 5, 26, 1, 85, 
	263, 304, 5, 26, 1, 266, 259, 261, 
	5, 26, 2, 266, 259, 263, 5, 26, 
	3, 266, 259, 263, 5, 26, 266, 259, 
	263, 261, 5, 28, 33, 265, 260, 4, 
	5, 32, 31, 261, 25, 21, 5, 35, 
	37, 235, 243, 0, 5, 35, 37, 236, 
	243, 0, 5, 35, 37, 237, 243, 0, 
	5, 35, 37, 238, 243, 0, 5, 37, 
	235, 1, 243, 0, 5, 37, 236, 1, 
	243, 0, 5, 37, 237, 1, 243, 0, 
	5, 37, 238, 1, 243, 0, 5, 45, 
	33, 265, 260, 4, 5, 82, 83, 261, 
	25, 21, 5, 241, 38, 34, 32, 31, 
	5, 266, 259, 1, 263, 304, 6, 26, 
	1, 266, 259, 263, 304, 6, 28, 45, 
	33, 265, 260, 4, 6, 35, 37, 235, 
	1, 243, 0, 6, 35, 37, 236, 1, 
	243, 0, 6, 35, 37, 237, 1, 243, 
	0, 6, 35, 37, 238, 1, 243, 0, 
	6, 266, 259, 1, 263, 261, 304, 6, 
	266, 259, 263, 261, 304, 1, 7, 26, 
	1, 266, 259, 263, 261, 304
};

static const short _zone_scanner_cond_offsets[] = {
	0, 0, 2, 4, 6, 8, 10, 12, 
	14, 14, 14, 16, 19, 21, 24, 26, 
	28, 30, 30, 30, 30, 32, 37, 37, 
	37, 37, 39, 39, 39, 39, 39, 41, 
	41, 41, 41, 43, 43, 43, 43, 45, 
	47, 47, 47, 47, 47, 49, 49, 51, 
	51, 51, 51, 51, 53, 53, 55, 57, 
	57, 57, 57, 57, 57, 57, 57, 59, 
	59, 59, 61, 63, 63, 63, 65, 65, 
	67, 67, 69, 71, 71, 71, 71, 71, 
	73, 75, 75, 75, 75, 75, 77, 77, 
	79, 81, 81, 83, 85, 85, 85, 85, 
	85, 87, 87, 87, 89, 89, 91, 91, 
	91, 91, 93, 95, 95, 95, 97, 97, 
	99, 99, 101, 101, 101, 101, 103, 103, 
	103, 103, 105, 105, 107, 107, 107, 107, 
	109, 112, 114, 114, 116, 118, 120, 122, 
	122, 124, 127, 127, 127, 127, 127, 127, 
	127, 127, 127, 127, 129, 131, 134, 136, 
	138, 141, 143, 143, 145, 148, 150, 152, 
	154, 156, 158, 161, 163, 165, 167, 169, 
	171, 173, 175, 177, 179, 181, 183, 185, 
	187, 189, 191, 193, 196, 198, 198, 198, 
	198, 198, 198, 198, 198, 198, 198, 200, 
	202, 204, 207, 209, 211, 213, 215, 217, 
	219, 221, 223, 225, 227, 229, 231, 233, 
	235, 237, 239, 241, 246, 251, 256, 256, 
	256, 258, 258, 258, 258, 260, 260, 262, 
	265, 267, 269, 274, 279, 282, 287, 289, 
	291, 293, 295, 297, 299, 301, 303, 305, 
	307, 309, 311, 313, 316, 319, 321, 324, 
	324, 324, 324, 324, 324, 324, 324, 324, 
	324, 324, 324, 324, 324, 324, 324, 324, 
	324, 324, 325, 325, 325, 325, 325, 326, 
	328, 330, 332, 334, 334, 336, 336, 338, 
	341, 343, 345, 345, 347, 349, 349, 349, 
	349, 349, 349, 349, 351, 354, 356, 358, 
	360, 362, 362, 364, 366, 366, 366, 366, 
	366, 366, 366, 368, 371, 373, 376, 379, 
	379, 379, 379, 379, 381, 384, 384, 386, 
	388, 390, 390, 390, 392, 395, 395, 395, 
	395, 397, 397, 397, 397, 399, 399, 399, 
	399, 399, 401, 401, 401, 401, 403, 403, 
	403, 403, 405, 407, 407, 407, 407, 407, 
	409, 409, 411, 411, 411, 411, 411, 413, 
	413, 413, 413, 413, 413, 413, 413, 415, 
	415, 415, 417, 419, 419, 419, 421, 421, 
	423, 423, 425, 427, 427, 427, 427, 427, 
	429, 431, 431, 431, 431, 431, 433, 433, 
	435, 437, 437, 439, 441, 441, 441, 441, 
	441, 443, 443, 443, 445, 445, 447, 447, 
	447, 447, 449, 451, 451, 451, 453, 453, 
	455, 455, 457, 457, 457, 457, 459, 459, 
	459, 459, 461, 461, 463, 463, 463, 463, 
	465, 465, 465, 465, 467, 467, 469, 471, 
	473, 475, 477, 477, 479, 482, 485, 488, 
	490, 492, 494, 496, 496, 498, 501, 504, 
	506, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 509, 
	509, 509, 509, 509, 509, 509, 509, 511, 
	513, 515, 517, 519, 521, 523, 525, 527, 
	529, 531, 533, 533, 533, 533, 533, 536, 
	538, 538, 540, 543, 545, 545, 547, 550, 
	552, 552, 554, 557, 560, 563, 563, 565, 
	567, 567, 570, 570, 572, 574, 574, 577, 
	577, 579, 581, 581, 584, 584, 586, 588, 
	591, 591, 591, 591, 593, 595, 597, 599, 
	601, 603, 605, 607, 609, 611, 613, 615, 
	617, 619, 621, 623, 623, 625, 627, 629, 
	631, 633, 635, 637, 639, 642, 644, 646, 
	649, 651, 653, 655, 658, 660, 662, 664, 
	667, 669, 671, 673, 676, 678, 681, 683, 
	685, 688, 691, 694, 696, 699, 701, 703, 
	706, 709, 709, 711, 713, 715, 717, 719, 
	721, 721, 724, 727, 730, 730, 732, 734, 
	736, 738, 740, 742, 744, 746, 748, 750, 
	750, 753, 756, 759, 762, 765, 765, 767, 
	769, 771, 773, 775, 777, 779, 782, 785, 
	788, 790, 790, 790, 790, 790, 790, 792, 
	795, 795, 795, 795, 795, 797, 799, 801, 
	803, 805, 807, 807, 809, 812, 815, 818, 
	821, 821, 823, 825, 827, 829, 829, 831, 
	834, 837, 840, 840, 842, 844, 846, 848, 
	850, 852, 858, 869, 871, 874, 880, 883, 
	894, 897, 900, 903, 905, 907, 909, 911, 
	917, 920, 923, 925, 927, 929, 931, 937, 
	940, 943, 945, 947, 949, 951, 957, 960, 
	963, 966, 966, 968, 970, 972, 974, 976, 
	978, 980, 982, 984, 986, 988, 990, 992, 
	994, 996, 998, 1000, 1003, 1006, 1009, 1012, 
	1015, 1018, 1021, 1024, 1024, 1024, 1026, 1026, 
	1026, 1026, 1028, 1028, 1030, 1030, 1030, 1030, 
	1032, 1032, 1032, 1032, 1034, 1034, 1034, 1034, 
	1034, 1036, 1036, 1036, 1036, 1038, 1038, 1038, 
	1038, 1040, 1042, 1042, 1042, 1042, 1042, 1044, 
	1044, 1046, 1046, 1046, 1046, 1046, 1048, 1048, 
	1048, 1048, 1048, 1048, 1048, 1048, 1050, 1050, 
	1050, 1052, 1054, 1054, 1054, 1056, 1056, 1058, 
	1058, 1060, 1062, 1062, 1062, 1062, 1062, 1064, 
	1066, 1066, 1066, 1066, 1066, 1068, 1068, 1070, 
	1072, 1072, 1074, 1076, 1076, 1076, 1076, 1076, 
	1078, 1078, 1078, 1080, 1080, 1082, 1082, 1082, 
	1082, 1084, 1086, 1086, 1086, 1088, 1088, 1090, 
	1090, 1092, 1092, 1092, 1092, 1094, 1094, 1094, 
	1094, 1096, 1096, 1098, 1098, 1098, 1098, 1100, 
	1100, 1100, 1100, 1100, 1102, 1104, 1106, 1108, 
	1110, 1112, 1114, 1117, 1120, 1123, 1123, 1125, 
	1125, 1127, 1129, 1131, 1133, 1135, 1137, 1139, 
	1141, 1141, 1141, 1141, 1141, 1141, 1141, 1141, 
	1141, 1141, 1141, 1141, 1141, 1141, 1141, 1144, 
	1144, 1146, 1149, 1152, 1155, 1155, 1157, 1159, 
	1161, 1163, 1165, 1167, 1167, 1167, 1167, 1170, 
	1173, 1176, 1176, 1178, 1180, 1182, 1184, 1186, 
	1188, 1188, 1190, 1193, 1196, 1199, 1202, 1202, 
	1204, 1206, 1206, 1209, 1209, 1211, 1213, 1213, 
	1213, 1213, 1213, 1213, 1213, 1213, 1213, 1213, 
	1216, 1216, 1216, 1216, 1216, 1216, 1216, 1216, 
	1216, 1216, 1216, 1216, 1216, 1218, 1220, 1222, 
	1224, 1226, 1228, 1230, 1232, 1234, 1236, 1236, 
	1236, 1236, 1236, 1236, 1236, 1236, 1238, 1238, 
	1240, 1243, 1243, 1245, 1248, 1248, 1250, 1253, 
	1255, 1255, 1257, 1260, 1263, 1263, 1263, 1263, 
	1263, 1263, 1263, 1263, 1263, 1263, 1265, 1268, 
	1268, 1268, 1270, 1273, 1275, 1278, 1280, 1283, 
	1285, 1288, 1288, 1288, 1288, 1288, 1290, 1293, 
	1293, 1295, 1298, 1298, 1300, 1303, 1303, 1309, 
	1312, 1323, 1326, 1337, 1340, 1340, 1340, 1340, 
	1340, 1340, 1340, 1340, 1342, 1345, 1345, 1345, 
	1345
};

static const char _zone_scanner_cond_lengths[] = {
	0, 2, 2, 2, 2, 2, 2, 2, 
	0, 0, 2, 3, 2, 3, 2, 2, 
	2, 0, 0, 0, 2, 5, 0, 0, 
	0, 2, 0, 0, 0, 0, 2, 0, 
	0, 0, 2, 0, 0, 0, 2, 2, 
	0, 0, 0, 0, 2, 0, 2, 0, 
	0, 0, 0, 2, 0, 2, 2, 0, 
	0, 0, 0, 0, 0, 0, 2, 0, 
	0, 2, 2, 0, 0, 2, 0, 2, 
	0, 2, 2, 0, 0, 0, 0, 2, 
	2, 0, 0, 0, 0, 2, 0, 2, 
	2, 0, 2, 2, 0, 0, 0, 0, 
	2, 0, 0, 2, 0, 2, 0, 0, 
	0, 2, 2, 0, 0, 2, 0, 2, 
	0, 2, 0, 0, 0, 2, 0, 0, 
	0, 2, 0, 2, 0, 0, 0, 2, 
	3, 2, 0, 2, 2, 2, 2, 0, 
	2, 3, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 2, 2, 3, 2, 2, 
	3, 2, 0, 2, 3, 2, 2, 2, 
	2, 2, 3, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 3, 2, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 2, 2, 
	2, 3, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 5, 5, 5, 0, 0, 
	2, 0, 0, 0, 2, 0, 2, 3, 
	2, 2, 5, 5, 3, 5, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 3, 3, 2, 3, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 1, 0, 0, 0, 0, 1, 2, 
	2, 2, 2, 0, 2, 0, 2, 3, 
	2, 2, 0, 2, 2, 0, 0, 0, 
	0, 0, 0, 2, 3, 2, 2, 2, 
	2, 0, 2, 2, 0, 0, 0, 0, 
	0, 0, 2, 3, 2, 3, 3, 0, 
	0, 0, 0, 2, 3, 0, 2, 2, 
	2, 0, 0, 2, 3, 0, 0, 0, 
	2, 0, 0, 0, 2, 0, 0, 0, 
	0, 2, 0, 0, 0, 2, 0, 0, 
	0, 2, 2, 0, 0, 0, 0, 2, 
	0, 2, 0, 0, 0, 0, 2, 0, 
	0, 0, 0, 0, 0, 0, 2, 0, 
	0, 2, 2, 0, 0, 2, 0, 2, 
	0, 2, 2, 0, 0, 0, 0, 2, 
	2, 0, 0, 0, 0, 2, 0, 2, 
	2, 0, 2, 2, 0, 0, 0, 0, 
	2, 0, 0, 2, 0, 2, 0, 0, 
	0, 2, 2, 0, 0, 2, 0, 2, 
	0, 2, 0, 0, 0, 2, 0, 0, 
	0, 2, 0, 2, 0, 0, 0, 2, 
	0, 0, 0, 2, 0, 2, 2, 2, 
	2, 2, 0, 2, 3, 3, 3, 2, 
	2, 2, 2, 0, 2, 3, 3, 2, 
	3, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 0, 0, 0, 0, 3, 2, 
	0, 2, 3, 2, 0, 2, 3, 2, 
	0, 2, 3, 3, 3, 0, 2, 2, 
	0, 3, 0, 2, 2, 0, 3, 0, 
	2, 2, 0, 3, 0, 2, 2, 3, 
	0, 0, 0, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 0, 2, 2, 2, 2, 
	2, 2, 2, 2, 3, 2, 2, 3, 
	2, 2, 2, 3, 2, 2, 2, 3, 
	2, 2, 2, 3, 2, 3, 2, 2, 
	3, 3, 3, 2, 3, 2, 2, 3, 
	3, 0, 2, 2, 2, 2, 2, 2, 
	0, 3, 3, 3, 0, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 0, 
	3, 3, 3, 3, 3, 0, 2, 2, 
	2, 2, 2, 2, 2, 3, 3, 3, 
	2, 0, 0, 0, 0, 0, 2, 3, 
	0, 0, 0, 0, 2, 2, 2, 2, 
	2, 2, 0, 2, 3, 3, 3, 3, 
	0, 2, 2, 2, 2, 0, 2, 3, 
	3, 3, 0, 2, 2, 2, 2, 2, 
	2, 6, 11, 2, 3, 6, 3, 11, 
	3, 3, 3, 2, 2, 2, 2, 6, 
	3, 3, 2, 2, 2, 2, 6, 3, 
	3, 2, 2, 2, 2, 6, 3, 3, 
	3, 0, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 3, 3, 3, 3, 3, 
	3, 3, 3, 0, 0, 2, 0, 0, 
	0, 2, 0, 2, 0, 0, 0, 2, 
	0, 0, 0, 2, 0, 0, 0, 0, 
	2, 0, 0, 0, 2, 0, 0, 0, 
	2, 2, 0, 0, 0, 0, 2, 0, 
	2, 0, 0, 0, 0, 2, 0, 0, 
	0, 0, 0, 0, 0, 2, 0, 0, 
	2, 2, 0, 0, 2, 0, 2, 0, 
	2, 2, 0, 0, 0, 0, 2, 2, 
	0, 0, 0, 0, 2, 0, 2, 2, 
	0, 2, 2, 0, 0, 0, 0, 2, 
	0, 0, 2, 0, 2, 0, 0, 0, 
	2, 2, 0, 0, 2, 0, 2, 0, 
	2, 0, 0, 0, 2, 0, 0, 0, 
	2, 0, 2, 0, 0, 0, 2, 0, 
	0, 0, 0, 2, 2, 2, 2, 2, 
	2, 2, 3, 3, 3, 0, 2, 0, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 3, 0, 
	2, 3, 3, 3, 0, 2, 2, 2, 
	2, 2, 2, 0, 0, 0, 3, 3, 
	3, 0, 2, 2, 2, 2, 2, 2, 
	0, 2, 3, 3, 3, 3, 0, 2, 
	2, 0, 3, 0, 2, 2, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 3, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 0, 0, 
	0, 0, 0, 0, 0, 2, 0, 2, 
	3, 0, 2, 3, 0, 2, 3, 2, 
	0, 2, 3, 3, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 2, 3, 0, 
	0, 2, 3, 2, 3, 2, 3, 2, 
	3, 0, 0, 0, 0, 2, 3, 0, 
	2, 3, 0, 2, 3, 0, 6, 3, 
	11, 3, 11, 3, 0, 0, 0, 0, 
	0, 0, 0, 2, 3, 0, 0, 0, 
	0
};

static const short _zone_scanner_cond_keys[] = {
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 58, 59, 59, 
	60, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 58, 
	59, 59, 60, 127, -128, 9, 10, 10, 
	11, 58, 59, 59, 60, 127, -128, 9, 
	10, 10, 11, 58, 59, 59, 60, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 58, 
	59, 59, 60, 127, -128, 9, 10, 10, 
	11, 58, 59, 59, 60, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 58, 59, 59, 60, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 10, 10, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	9, 9, 10, 10, 32, 32, 40, 40, 
	41, 41, 59, 59, 9, 9, 10, 10, 
	32, 32, 40, 40, 41, 41, 43, 43, 
	47, 47, 48, 57, 59, 59, 65, 90, 
	97, 122, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 9, 9, 10, 10, 
	32, 32, 40, 40, 41, 41, 59, 59, 
	-128, 9, 10, 10, 11, 127, 9, 9, 
	10, 10, 32, 32, 40, 40, 41, 41, 
	43, 43, 47, 47, 48, 57, 59, 59, 
	65, 90, 97, 122, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 9, 9, 
	10, 10, 32, 32, 40, 40, 41, 41, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 9, 9, 
	10, 10, 32, 32, 40, 40, 41, 41, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 9, 9, 
	10, 10, 32, 32, 40, 40, 41, 41, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 9, 9, 
	10, 10, 32, 32, 40, 40, 41, 41, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	9, 9, 10, 10, 32, 32, 40, 40, 
	41, 41, 43, 43, 47, 47, 48, 57, 
	59, 59, 65, 90, 97, 122, -128, 9, 
	10, 10, 11, 127, 9, 9, 10, 10, 
	32, 32, 40, 40, 41, 41, 43, 43, 
	47, 47, 48, 57, 59, 59, 65, 90, 
	97, 122, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 0
};

static const char _zone_scanner_cond_spaces[] = {
	0, 0, 0, 0, 0, 0, 5, 5, 
	5, 5, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 5, 5, 0, 0, 0, 
	5, 5, 0, 0, 0, 0, 5, 5, 
	0, 0, 0, 0, 0, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 0, 0, 0, 
	0, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 5, 5, 5, 5, 
	5, 5, 5, 5, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 5, 5, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 5, 
	5, 5, 5, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	5, 5, 5, 5, 5, 5, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 2, 2, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 5, 5, 0, 0, 0, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	0, 0, 0, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 0, 0, 0, 
	0, 0, 0, 0, 0, 5, 5, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	5, 5, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	5, 5, 5, 5, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 0, 
	0, 0, 5, 5, 5, 5, 0, 0, 
	0, 5, 5, 5, 5, 5, 5, 0, 
	0, 0, 5, 5, 5, 5, 5, 5, 
	0, 0, 0, 5, 5, 5, 5, 5, 
	5, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 1, 1, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 5, 5, 5, 5, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 5, 
	5, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 5, 5, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 8, 11, 8, 8, 
	8, 11, 3, 6, 3, 3, 3, 3, 
	3, 3, 6, 3, 3, 1, 1, 6, 
	6, 6, 4, 9, 4, 4, 4, 9, 
	7, 7, 7, 8, 11, 8, 8, 8, 
	3, 3, 3, 11, 3, 3, 10, 10, 
	10, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 8, 
	11, 8, 8, 8, 11, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 8, 11, 8, 8, 8, 
	11, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 8, 
	11, 8, 8, 8, 11, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 1, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	1, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 1, 1, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 5, 5, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 5, 5, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 5, 5, 
	5, 5, 5, 5, 0, 0, 5, 5, 
	0, 0, 0, 5, 5, 0, 0, 0, 
	5, 5, 0, 0, 0, 0, 0, 5, 
	5, 0, 0, 0, 0, 0, 0, 5, 
	5, 0, 0, 0, 5, 5, 0, 0, 
	0, 5, 5, 0, 0, 0, 5, 5, 
	0, 0, 0, 5, 5, 0, 0, 0, 
	5, 5, 0, 0, 0, 5, 5, 0, 
	0, 0, 5, 5, 0, 0, 0, 4, 
	9, 4, 4, 4, 9, 7, 7, 7, 
	3, 6, 3, 3, 3, 3, 3, 3, 
	6, 3, 3, 6, 6, 6, 8, 11, 
	8, 8, 8, 3, 3, 3, 11, 3, 
	3, 10, 10, 10, 5, 5, 0, 0, 
	0, 0
};

static const short _zone_scanner_key_offsets[] = {
	0, 0, 38, 56, 90, 106, 123, 131, 
	139, 140, 141, 149, 152, 171, 177, 227, 
	247, 283, 287, 289, 291, 301, 315, 317, 
	319, 321, 331, 337, 339, 341, 343, 353, 
	357, 359, 361, 371, 373, 375, 377, 387, 
	397, 399, 401, 403, 404, 414, 415, 425, 
	427, 429, 431, 433, 443, 447, 453, 487, 
	489, 491, 493, 495, 497, 499, 501, 511, 
	515, 517, 527, 537, 543, 544, 554, 555, 
	565, 567, 577, 587, 591, 593, 595, 597, 
	607, 617, 623, 625, 627, 629, 639, 641, 
	651, 663, 665, 676, 688, 690, 692, 694, 
	696, 706, 708, 710, 720, 726, 736, 738, 
	740, 742, 752, 762, 770, 772, 782, 784, 
	794, 796, 806, 808, 810, 812, 822, 828, 
	830, 832, 842, 844, 854, 856, 858, 860, 
	872, 875, 913, 917, 923, 959, 977, 985, 
	997, 1005, 1008, 1009, 1015, 1017, 1019, 1021, 
	1023, 1025, 1027, 1033, 1039, 1075, 1078, 1096, 
	1132, 1135, 1143, 1155, 1163, 1166, 1182, 1230, 
	1248, 1267, 1305, 1311, 1325, 1339, 1389, 1401, 
	1415, 1425, 1435, 1447, 1459, 1473, 1485, 1499, 
	1509, 1523, 1539, 1553, 1559, 1573, 1575, 1577, 
	1579, 1581, 1583, 1589, 1591, 1593, 1599, 1607, 
	1627, 1665, 1671, 1689, 1739, 1751, 1765, 1775, 
	1785, 1797, 1809, 1823, 1835, 1849, 1859, 1873, 
	1889, 1903, 1918, 1956, 1970, 1984, 1998, 2000, 
	2002, 2012, 2014, 2016, 2018, 2028, 2030, 2040, 
	2046, 2060, 2074, 2090, 2104, 2107, 2121, 2131, 
	2143, 2151, 2159, 2169, 2179, 2191, 2201, 2213, 
	2221, 2233, 2247, 2259, 2265, 2271, 2281, 2287, 
	2288, 2289, 2300, 2307, 2323, 2338, 2340, 2342, 
	2344, 2359, 2365, 2371, 2377, 2389, 2391, 2393, 
	2395, 2407, 2413, 2419, 2421, 2423, 2425, 2431, 
	2437, 2445, 2465, 2473, 2474, 2484, 2496, 2506, 
	2509, 2515, 2531, 2532, 2550, 2558, 2559, 2568, 
	2570, 2572, 2574, 2583, 2591, 2594, 2600, 2612, 
	2620, 2638, 2639, 2657, 2665, 2666, 2675, 2677, 
	2679, 2681, 2690, 2698, 2704, 2720, 2723, 2726, 
	2733, 2740, 2748, 2756, 2773, 2776, 2777, 2787, 
	2825, 2841, 2843, 2845, 2855, 2858, 2862, 2864, 
	2866, 2876, 2878, 2880, 2882, 2892, 2898, 2900, 
	2902, 2904, 2914, 2918, 2920, 2922, 2932, 2934, 
	2936, 2938, 2948, 2958, 2960, 2962, 2964, 2965, 
	2975, 2976, 2986, 2988, 2990, 2992, 2994, 3004, 
	3006, 3008, 3010, 3012, 3014, 3016, 3018, 3028, 
	3032, 3034, 3044, 3054, 3060, 3061, 3071, 3072, 
	3082, 3084, 3094, 3104, 3108, 3110, 3112, 3114, 
	3124, 3134, 3140, 3142, 3144, 3146, 3156, 3158, 
	3168, 3180, 3182, 3193, 3205, 3207, 3209, 3211, 
	3213, 3223, 3225, 3227, 3237, 3243, 3253, 3255, 
	3257, 3259, 3269, 3279, 3287, 3289, 3299, 3301, 
	3311, 3313, 3323, 3325, 3327, 3329, 3339, 3345, 
	3347, 3349, 3359, 3361, 3371, 3373, 3375, 3377, 
	3389, 3391, 3393, 3395, 3405, 3407, 3417, 3423, 
	3431, 3439, 3451, 3457, 3473, 3476, 3479, 3482, 
	3488, 3497, 3507, 3519, 3525, 3541, 3544, 3547, 
	3555, 3558, 3570, 3578, 3582, 3588, 3590, 3597, 
	3599, 3601, 3603, 3605, 3606, 3607, 3609, 3611, 
	3613, 3614, 3620, 3622, 3626, 3627, 3629, 3631, 
	3633, 3635, 3641, 3643, 3645, 3647, 3649, 3650, 
	3651, 3653, 3655, 3657, 3658, 3659, 3660, 3666, 
	3667, 3668, 3670, 3672, 3674, 3675, 3676, 3677, 
	3683, 3685, 3687, 3689, 3691, 3693, 3695, 3697, 
	3703, 3705, 3707, 3709, 3711, 3713, 3715, 3719, 
	3721, 3723, 3729, 3731, 3733, 3739, 3741, 3743, 
	3747, 3749, 3750, 3756, 3758, 3760, 3763, 3770, 
	3772, 3774, 3776, 3778, 3779, 3780, 3782, 3784, 
	3786, 3787, 3793, 3794, 3795, 3801, 3802, 3803, 
	3809, 3823, 3831, 3833, 3835, 3837, 3839, 3841, 
	3847, 3853, 3855, 3857, 3859, 3861, 3863, 3869, 
	3873, 3875, 3881, 3883, 3885, 3891, 3893, 3895, 
	3897, 3903, 3905, 3907, 3913, 3917, 3919, 3925, 
	3927, 3929, 3935, 3937, 3939, 3941, 3947, 3949, 
	3951, 3957, 3960, 3969, 3978, 3984, 3993, 3999, 
	4014, 4020, 4028, 4036, 4044, 4062, 4070, 4088, 
	4096, 4114, 4122, 4140, 4148, 4160, 4168, 4171, 
	4179, 4191, 4199, 4202, 4210, 4222, 4230, 4233, 
	4241, 4253, 4261, 4264, 4267, 4270, 4276, 4282, 
	4294, 4300, 4303, 4312, 4318, 4333, 4339, 4342, 
	4344, 4352, 4367, 4373, 4376, 4382, 4392, 4408, 
	4411, 4418, 4431, 4433, 4441, 4451, 4459, 4469, 
	4478, 4486, 4492, 4500, 4508, 4518, 4526, 4536, 
	4545, 4553, 4559, 4568, 4570, 4584, 4596, 4610, 
	4622, 4636, 4648, 4662, 4672, 4675, 4688, 4701, 
	4704, 4717, 4730, 4740, 4743, 4756, 4769, 4779, 
	4782, 4795, 4808, 4818, 4821, 4827, 4830, 4838, 
	4846, 4849, 4852, 4855, 4861, 4864, 4872, 4880, 
	4883, 4886, 4888, 4896, 4904, 4912, 4920, 4928, 
	4943, 4949, 4952, 4955, 4958, 4960, 4968, 4976, 
	4984, 4996, 5002, 5014, 5020, 5032, 5038, 5053, 
	5059, 5062, 5065, 5068, 5071, 5074, 5080, 5086, 
	5094, 5102, 5114, 5120, 5133, 5135, 5138, 5141, 
	5144, 5157, 5159, 5160, 5163, 5166, 5168, 5180, 
	5183, 5184, 5191, 5198, 5200, 5208, 5220, 5226, 
	5234, 5242, 5254, 5260, 5276, 5279, 5282, 5285, 
	5288, 5290, 5298, 5306, 5314, 5326, 5332, 5348, 
	5351, 5354, 5357, 5359, 5367, 5377, 5383, 5391, 
	5399, 5406, 5440, 5453, 5455, 5458, 5472, 5475, 
	5516, 5525, 5528, 5531, 5537, 5545, 5553, 5562, 
	5599, 5602, 5605, 5611, 5619, 5627, 5640, 5681, 
	5684, 5687, 5693, 5701, 5709, 5724, 5758, 5761, 
	5764, 5767, 5795, 5807, 5819, 5825, 5833, 5841, 
	5849, 5857, 5865, 5873, 5881, 5889, 5897, 5905, 
	5920, 5926, 5939, 5941, 5944, 5947, 5950, 5953, 
	5956, 5959, 5962, 5965, 5967, 5969, 5975, 5977, 
	5979, 5981, 5987, 5989, 5995, 5999, 6001, 6003, 
	6009, 6011, 6013, 6015, 6021, 6027, 6029, 6031, 
	6033, 6039, 6043, 6045, 6047, 6053, 6055, 6057, 
	6059, 6065, 6071, 6073, 6075, 6077, 6078, 6084, 
	6085, 6091, 6093, 6095, 6097, 6099, 6105, 6107, 
	6109, 6111, 6113, 6115, 6117, 6119, 6125, 6129, 
	6131, 6137, 6143, 6149, 6150, 6156, 6157, 6163, 
	6165, 6171, 6177, 6181, 6183, 6185, 6187, 6193, 
	6199, 6205, 6207, 6209, 6211, 6217, 6219, 6225, 
	6233, 6235, 6242, 6250, 6252, 6254, 6256, 6258, 
	6264, 6266, 6268, 6274, 6280, 6286, 6288, 6290, 
	6292, 6298, 6304, 6312, 6314, 6320, 6322, 6328, 
	6330, 6336, 6338, 6340, 6342, 6348, 6354, 6356, 
	6358, 6364, 6366, 6372, 6374, 6376, 6378, 6386, 
	6395, 6401, 6407, 6409, 6417, 6425, 6433, 6445, 
	6451, 6464, 6466, 6469, 6472, 6475, 6482, 6484, 
	6486, 6494, 6502, 6510, 6518, 6526, 6539, 6545, 
	6557, 6563, 6570, 6576, 6583, 6590, 6596, 6603, 
	6615, 6621, 6622, 6623, 6624, 6625, 6626, 6629, 
	6635, 6647, 6650, 6653, 6656, 6658, 6666, 6674, 
	6682, 6690, 6698, 6711, 6717, 6723, 6735, 6738, 
	6741, 6744, 6746, 6754, 6762, 6770, 6778, 6786, 
	6798, 6804, 6820, 6823, 6826, 6829, 6832, 6834, 
	6842, 6851, 6860, 6863, 6865, 6873, 6885, 6891, 
	6897, 6903, 6904, 6910, 6916, 6922, 6928, 6935, 
	6938, 6944, 6950, 6951, 6957, 6963, 6970, 6976, 
	6982, 6983, 6989, 6995, 7002, 7020, 7053, 7099, 
	7147, 7165, 7213, 7231, 7264, 7327, 7390, 7390, 
	7390, 7390, 7402, 7402, 7402, 7402, 7418, 7418, 
	7435, 7438, 7438, 7476, 7479, 7479, 7495, 7498, 
	7510, 7510, 7526, 7529, 7532, 7532, 7532, 7532, 
	7532, 7532, 7532, 7532, 7532, 7532, 7548, 7551, 
	7551, 7551, 7561, 7564, 7576, 7579, 7591, 7594, 
	7606, 7609, 7609, 7609, 7609, 7609, 7622, 7625, 
	7625, 7641, 7644, 7644, 7660, 7663, 7663, 7677, 
	7680, 7693, 7696, 7737, 7746, 7746, 7746, 7746, 
	7746, 7746, 7746, 7746, 7762, 7765, 7765, 7765, 
	7765
};

static const short _zone_scanner_trans_keys[] = {
	9, 32, 40, 41, 65, 67, 68, 69, 
	72, 73, 75, 76, 77, 78, 80, 82, 
	83, 84, 97, 99, 100, 101, 104, 105, 
	107, 108, 109, 110, 112, 114, 115, 116, 
	778, 827, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 68, 72, 77, 83, 87, 100, 
	104, 109, 115, 119, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 65, 67, 68, 69, 
	72, 73, 75, 76, 77, 78, 80, 82, 
	83, 84, 97, 99, 100, 101, 104, 105, 
	107, 108, 109, 110, 112, 114, 115, 116, 
	1034, 1083, 9, 32, 40, 41, 65, 70, 
	80, 97, 102, 112, 2058, 2107, 2314, 2363, 
	2570, 2619, 9, 32, 40, 41, 92, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	58, 60, 127, 9, 32, 40, 41, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 778, 
	827, 1034, 1083, 10, 35, 9, 32, 40, 
	41, 778, 827, 1034, 1083, 1034, 896, 1151, 
	9, 32, 40, 41, 92, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	58, 60, 127, 778, 1034, 640, 895, 896, 
	1151, 9, 32, 40, 41, 58, 65, 67, 
	68, 69, 72, 73, 75, 76, 77, 78, 
	80, 82, 83, 84, 92, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 1802, 1851, 2058, 2107, 2314, 
	2363, 2570, 2619, -128, 8, 11, 47, 48, 
	57, 60, 127, 9, 32, 40, 41, 68, 
	72, 77, 83, 87, 100, 104, 109, 115, 
	119, 778, 827, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 97, 99, 100, 101, 104, 105, 107, 
	108, 109, 110, 112, 114, 115, 116, 778, 
	827, 1034, 1083, 69, 78, 101, 110, 82, 
	114, 84, 116, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 777, 778, 800, 
	808, 809, 827, 1033, 1034, 1056, 1064, 1065, 
	1083, 896, 1151, 65, 97, 77, 109, 69, 
	101, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 72, 78, 83, 104, 110, 
	115, 67, 99, 73, 105, 68, 100, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 65, 83, 97, 115, 77, 109, 69, 
	101, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 75, 107, 69, 101, 89, 
	121, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 85, 117, 73, 
	105, 52, 54, 56, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 52, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 73, 105, 78, 110, 70, 102, 79, 
	111, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 78, 80, 110, 112, 9, 
	32, 40, 41, 1034, 1083, 9, 32, 40, 
	41, 65, 67, 68, 69, 72, 73, 75, 
	76, 77, 78, 80, 82, 83, 84, 97, 
	99, 100, 101, 104, 105, 107, 108, 109, 
	110, 112, 114, 115, 116, 1034, 1083, 80, 
	112, 83, 115, 69, 101, 67, 99, 75, 
	107, 69, 101, 89, 121, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 69, 
	88, 101, 120, 89, 121, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 51, 54, 79, 80, 111, 112, 50, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 52, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 67, 99, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 73, 88, 105, 120, 78, 
	110, 70, 102, 79, 111, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 65, 73, 83, 97, 105, 115, 80, 
	112, 84, 116, 82, 114, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 68, 
	100, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 9, 32, 40, 41, 69, 
	101, 2058, 2107, 2314, 2363, 2570, 2619, 67, 
	99, 9, 32, 40, 41, 51, 2058, 2107, 
	2314, 2363, 2570, 2619, 9, 32, 40, 41, 
	80, 112, 2058, 2107, 2314, 2363, 2570, 2619, 
	65, 97, 82, 114, 65, 97, 77, 109, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 84, 116, 82, 114, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	80, 82, 84, 112, 114, 116, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	83, 115, 73, 105, 71, 103, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 79, 80, 82, 83, 111, 112, 
	114, 115, 65, 97, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 70, 102, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 86, 118, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 72, 104, 
	70, 102, 80, 112, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 76, 88, 
	89, 108, 120, 121, 83, 115, 65, 97, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 84, 116, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 80, 112, 
	69, 101, 48, 57, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	1034, 896, 1151, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 778, 827, 1034, 1083, 48, 
	57, 78, 80, 110, 112, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 68, 72, 77, 83, 87, 
	100, 104, 109, 115, 119, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 68, 72, 77, 83, 87, 100, 104, 
	109, 115, 119, 48, 57, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 1034, 896, 1151, 
	10, 73, 79, 84, 105, 111, 116, 78, 
	110, 67, 99, 76, 108, 85, 117, 68, 
	100, 69, 101, 32, 59, 9, 10, 40, 
	41, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 97, 99, 100, 101, 104, 105, 107, 
	108, 109, 110, 112, 114, 115, 116, 1034, 
	1083, 48, 57, 1034, 896, 1151, 9, 32, 
	40, 41, 68, 72, 77, 83, 87, 100, 
	104, 109, 115, 119, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 65, 67, 68, 69, 
	72, 73, 75, 76, 77, 78, 80, 82, 
	83, 84, 97, 99, 100, 101, 104, 105, 
	107, 108, 109, 110, 112, 114, 115, 116, 
	1034, 1083, 48, 57, 1034, 896, 1151, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 68, 
	72, 77, 83, 87, 100, 104, 109, 115, 
	119, 48, 57, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 1034, 896, 1151, 9, 32, 
	40, 41, 65, 70, 80, 97, 102, 112, 
	2058, 2107, 2314, 2363, 2570, 2619, 9, 32, 
	40, 41, 58, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 92, 97, 99, 100, 101, 104, 105, 
	107, 108, 109, 110, 112, 114, 115, 116, 
	2058, 2107, 2314, 2363, 2570, 2619, -128, 8, 
	11, 47, 48, 57, 60, 127, 9, 32, 
	40, 41, 65, 70, 80, 97, 102, 112, 
	1802, 1851, 2058, 2107, 2314, 2363, 2570, 2619, 
	9, 32, 40, 41, 92, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	58, 60, 127, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 778, 827, 1034, 1083, 48, 
	57, 778, 1034, 640, 895, 896, 1151, 9, 
	32, 40, 41, 778, 827, 1034, 1083, -128, 
	8, 11, 58, 60, 127, 9, 32, 40, 
	41, 778, 827, 1034, 1083, -128, 8, 11, 
	58, 60, 127, 9, 32, 40, 41, 58, 
	65, 67, 68, 69, 72, 73, 75, 76, 
	77, 78, 80, 82, 83, 84, 92, 97, 
	99, 100, 101, 104, 105, 107, 108, 109, 
	110, 112, 114, 115, 116, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	47, 48, 57, 60, 127, 9, 32, 40, 
	41, 69, 78, 101, 110, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 72, 78, 83, 
	104, 110, 115, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 85, 117, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 73, 105, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 78, 
	80, 110, 112, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 69, 88, 101, 120, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 51, 
	54, 79, 80, 111, 112, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 73, 88, 105, 
	120, 778, 827, 1034, 1083, 9, 32, 40, 
	41, 65, 73, 83, 97, 105, 115, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 84, 
	116, 778, 827, 1034, 1083, 9, 32, 40, 
	41, 80, 82, 84, 112, 114, 116, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 79, 
	80, 82, 83, 111, 112, 114, 115, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 76, 
	88, 89, 108, 120, 121, 778, 827, 1034, 
	1083, 778, 1034, 640, 895, 896, 1151, 9, 
	32, 40, 41, 73, 79, 84, 105, 111, 
	116, 778, 827, 1034, 1083, 82, 114, 73, 
	105, 71, 103, 73, 105, 78, 110, 32, 
	59, 9, 10, 40, 41, 84, 116, 76, 
	108, 32, 59, 9, 10, 40, 41, 9, 
	32, 40, 41, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 68, 72, 77, 83, 87, 
	100, 104, 109, 115, 119, 778, 827, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 778, 827, 1034, 1083, 48, 
	57, 778, 1034, 640, 895, 896, 1151, 9, 
	32, 40, 41, 65, 70, 80, 97, 102, 
	112, 1802, 1851, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 58, 65, 67, 
	68, 69, 72, 73, 75, 76, 77, 78, 
	80, 82, 83, 84, 92, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 1802, 1851, 2058, 2107, 2314, 
	2363, 2570, 2619, -128, 8, 11, 47, 48, 
	57, 60, 127, 9, 32, 40, 41, 69, 
	78, 101, 110, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 72, 78, 83, 104, 110, 
	115, 778, 827, 1034, 1083, 9, 32, 40, 
	41, 85, 117, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 73, 105, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 78, 80, 110, 
	112, 778, 827, 1034, 1083, 9, 32, 40, 
	41, 69, 88, 101, 120, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 51, 54, 79, 
	80, 111, 112, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 73, 88, 105, 120, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 65, 
	73, 83, 97, 105, 115, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 84, 116, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 80, 
	82, 84, 112, 114, 116, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 79, 80, 82, 
	83, 111, 112, 114, 115, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 76, 88, 89, 
	108, 120, 121, 778, 827, 1034, 1083, 9, 
	32, 35, 40, 41, 778, 827, 1034, 1083, 
	-128, 8, 11, 58, 60, 127, 9, 32, 
	40, 41, 65, 67, 68, 69, 72, 73, 
	75, 76, 77, 78, 80, 82, 83, 84, 
	97, 99, 100, 101, 104, 105, 107, 108, 
	109, 110, 112, 114, 115, 116, 778, 827, 
	1034, 1083, 48, 57, 777, 778, 800, 808, 
	809, 827, 1033, 1034, 1056, 1064, 1065, 1083, 
	896, 1151, 777, 778, 800, 808, 809, 827, 
	1033, 1034, 1056, 1064, 1065, 1083, 896, 1151, 
	777, 778, 800, 808, 809, 1033, 1034, 1056, 
	1064, 1065, 640, 895, 896, 1151, 65, 97, 
	65, 97, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 83, 115, 68, 100, 
	66, 98, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 76, 108, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	778, 1034, 640, 895, 896, 1151, 9, 32, 
	40, 41, 778, 827, 1034, 1083, -128, 8, 
	11, 58, 60, 127, 9, 32, 40, 41, 
	778, 827, 1034, 1083, -128, 8, 11, 58, 
	60, 127, 777, 778, 800, 808, 809, 827, 
	1033, 1034, 1056, 1064, 1065, 1083, 640, 895, 
	896, 1151, 777, 778, 800, 808, 809, 1033, 
	1034, 1056, 1064, 1065, 640, 895, 896, 1151, 
	1034, 896, 1151, 777, 778, 800, 808, 809, 
	827, 1033, 1034, 1056, 1064, 1065, 1083, 896, 
	1151, 9, 32, 40, 41, 69, 78, 101, 
	110, 1034, 1083, 9, 32, 40, 41, 72, 
	78, 83, 104, 110, 115, 1034, 1083, 9, 
	32, 40, 41, 85, 117, 1034, 1083, 9, 
	32, 40, 41, 73, 105, 1034, 1083, 9, 
	32, 40, 41, 78, 80, 110, 112, 1034, 
	1083, 9, 32, 40, 41, 69, 88, 101, 
	120, 1034, 1083, 9, 32, 40, 41, 51, 
	54, 79, 80, 111, 112, 1034, 1083, 9, 
	32, 40, 41, 73, 88, 105, 120, 1034, 
	1083, 9, 32, 40, 41, 65, 73, 83, 
	97, 105, 115, 1034, 1083, 9, 32, 40, 
	41, 84, 116, 1034, 1083, 9, 32, 40, 
	41, 80, 82, 84, 112, 114, 116, 1034, 
	1083, 9, 32, 40, 41, 79, 80, 82, 
	83, 111, 112, 114, 115, 1034, 1083, 9, 
	32, 40, 41, 76, 88, 89, 108, 120, 
	121, 1034, 1083, 778, 1034, 640, 895, 896, 
	1151, 778, 1034, 640, 895, 896, 1151, 9, 
	32, 40, 41, 78, 80, 110, 112, 1034, 
	1083, 778, 1034, 640, 895, 896, 1151, 10, 
	10, 42, 46, 64, 92, 95, 45, 57, 
	65, 90, 97, 122, 32, 46, 59, 9, 
	10, 40, 41, 32, 42, 45, 59, 92, 
	95, 9, 10, 40, 41, 47, 57, 65, 
	90, 97, 122, 32, 46, 59, 92, 95, 
	9, 10, 40, 41, 45, 57, 65, 90, 
	97, 122, 48, 57, 48, 57, 48, 57, 
	32, 46, 59, 92, 95, 9, 10, 40, 
	41, 45, 57, 65, 90, 97, 122, 32, 
	59, 9, 10, 40, 41, 32, 59, 9, 
	10, 40, 41, 34, 92, 33, 58, 60, 
	126, 32, 33, 59, 92, 9, 10, 35, 
	39, 40, 41, 42, 126, 48, 57, 48, 
	57, 48, 57, 32, 33, 59, 92, 9, 
	10, 35, 39, 40, 41, 42, 126, 9, 
	34, 92, 522, 32, 126, 32, 59, 9, 
	10, 40, 41, 48, 57, 48, 57, 48, 
	57, 9, 34, 92, 522, 32, 126, 9, 
	32, 40, 41, 1034, 1083, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 9, 32, 40, 
	41, 68, 72, 77, 83, 87, 100, 104, 
	109, 115, 119, 778, 827, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 778, 827, 1034, 
	1083, 10, 9, 32, 40, 41, 778, 827, 
	1034, 1083, 48, 57, 68, 72, 77, 83, 
	87, 100, 104, 109, 115, 119, 48, 57, 
	9, 32, 40, 41, 778, 827, 1034, 1083, 
	48, 57, 1034, 896, 1151, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 42, 
	46, 92, 95, 1034, 1083, 45, 57, 65, 
	90, 97, 122, 46, 9, 32, 40, 41, 
	42, 45, 92, 95, 778, 827, 1034, 1083, 
	47, 57, 65, 90, 97, 122, 9, 32, 
	40, 41, 778, 827, 1034, 1083, 10, 46, 
	92, 95, 45, 57, 65, 90, 97, 122, 
	48, 57, 48, 57, 48, 57, 46, 92, 
	95, 45, 57, 65, 90, 97, 122, 9, 
	32, 40, 41, 778, 827, 1034, 1083, 1034, 
	896, 1151, 9, 32, 40, 41, 1034, 1083, 
	9, 32, 40, 41, 1034, 1083, -128, 8, 
	11, 58, 60, 127, 9, 32, 40, 41, 
	778, 827, 1034, 1083, 9, 32, 40, 41, 
	42, 46, 92, 95, 778, 827, 1034, 1083, 
	45, 57, 65, 90, 97, 122, 46, 9, 
	32, 40, 41, 42, 45, 92, 95, 778, 
	827, 1034, 1083, 47, 57, 65, 90, 97, 
	122, 9, 32, 40, 41, 778, 827, 1034, 
	1083, 10, 46, 92, 95, 45, 57, 65, 
	90, 97, 122, 48, 57, 48, 57, 48, 
	57, 46, 92, 95, 45, 57, 65, 90, 
	97, 122, 9, 32, 40, 41, 778, 827, 
	1034, 1083, 778, 1034, 640, 895, 896, 1151, 
	9, 32, 40, 41, 42, 46, 92, 95, 
	1034, 1083, 45, 57, 65, 90, 97, 122, 
	1034, 896, 1151, 1034, 896, 1151, 43, 47, 
	57, 65, 90, 97, 122, 43, 47, 57, 
	65, 90, 97, 122, 43, 61, 47, 57, 
	65, 90, 97, 122, 43, 61, 47, 57, 
	65, 90, 97, 122, 9, 32, 40, 41, 
	43, 2058, 2107, 2314, 2363, 2570, 2619, 47, 
	57, 65, 90, 97, 122, 1034, 896, 1151, 
	61, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 65, 70, 80, 
	97, 102, 112, 2058, 2107, 2314, 2363, 2570, 
	2619, 65, 97, 65, 97, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 1034, 
	896, 1151, 69, 78, 101, 110, 82, 114, 
	84, 116, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 65, 97, 77, 109, 
	69, 101, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 72, 78, 83, 104, 
	110, 115, 67, 99, 73, 105, 68, 100, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 65, 83, 97, 115, 77, 109, 
	69, 101, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 75, 107, 69, 101, 
	89, 121, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 85, 117, 
	73, 105, 52, 54, 56, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 52, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 73, 105, 78, 110, 70, 102, 
	79, 111, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 80, 112, 83, 115, 
	69, 101, 67, 99, 75, 107, 69, 101, 
	89, 121, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 69, 88, 101, 120, 
	89, 121, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 51, 54, 
	79, 80, 111, 112, 50, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 52, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 67, 99, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	73, 88, 105, 120, 78, 110, 70, 102, 
	79, 111, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 65, 73, 
	83, 97, 105, 115, 80, 112, 84, 116, 
	82, 114, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 68, 100, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	9, 32, 40, 41, 69, 101, 2058, 2107, 
	2314, 2363, 2570, 2619, 67, 99, 9, 32, 
	40, 41, 51, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 80, 112, 2058, 
	2107, 2314, 2363, 2570, 2619, 65, 97, 82, 
	114, 65, 97, 77, 109, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 84, 
	116, 82, 114, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 80, 82, 84, 
	112, 114, 116, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 83, 115, 73, 
	105, 71, 103, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 79, 
	80, 82, 83, 111, 112, 114, 115, 65, 
	97, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 70, 102, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 86, 
	118, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 72, 104, 70, 102, 80, 
	112, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 76, 88, 89, 108, 120, 
	121, 83, 115, 65, 97, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 84, 
	116, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 80, 112, 69, 101, 48, 
	57, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 83, 115, 68, 
	100, 66, 98, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 76, 108, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 65, 
	70, 97, 102, 48, 57, 65, 70, 97, 
	102, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 65, 70, 97, 
	102, 1034, 896, 1151, 1034, 896, 1151, 1034, 
	896, 1151, 9, 32, 40, 41, 1034, 1083, 
	9, 32, 40, 41, 48, 1034, 1083, 49, 
	57, 9, 32, 40, 41, 778, 827, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 65, 70, 97, 102, 48, 
	57, 65, 70, 97, 102, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 48, 
	57, 65, 70, 97, 102, 1034, 896, 1151, 
	1034, 896, 1151, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 1034, 896, 1151, 68, 69, 
	73, 80, 82, 100, 101, 105, 112, 114, 
	48, 57, 32, 59, 9, 10, 40, 41, 
	48, 57, 72, 83, 104, 115, 32, 59, 
	9, 10, 40, 41, 65, 97, 32, 45, 
	59, 9, 10, 40, 41, 78, 110, 83, 
	115, 69, 101, 67, 99, 51, 45, 83, 
	115, 72, 104, 65, 97, 49, 32, 59, 
	9, 10, 40, 41, 67, 99, 67, 68, 
	99, 100, 45, 71, 103, 79, 111, 83, 
	115, 84, 116, 32, 59, 9, 10, 40, 
	41, 83, 115, 65, 97, 80, 112, 50, 
	51, 53, 54, 83, 115, 72, 104, 65, 
	97, 50, 53, 54, 32, 59, 9, 10, 
	40, 41, 56, 52, 83, 115, 72, 104, 
	65, 97, 51, 56, 52, 32, 59, 9, 
	10, 40, 41, 78, 110, 68, 100, 73, 
	105, 82, 114, 69, 101, 67, 99, 84, 
	116, 32, 59, 9, 10, 40, 41, 82, 
	114, 73, 105, 86, 118, 65, 97, 84, 
	116, 69, 101, 68, 79, 100, 111, 78, 
	110, 83, 115, 32, 59, 9, 10, 40, 
	41, 73, 105, 68, 100, 32, 59, 9, 
	10, 40, 41, 83, 115, 65, 97, 77, 
	83, 109, 115, 68, 100, 53, 32, 59, 
	9, 10, 40, 41, 72, 104, 65, 97, 
	49, 50, 53, 32, 45, 59, 9, 10, 
	40, 41, 78, 110, 83, 115, 69, 101, 
	67, 99, 51, 45, 83, 115, 72, 104, 
	65, 97, 49, 32, 59, 9, 10, 40, 
	41, 53, 54, 32, 59, 9, 10, 40, 
	41, 49, 50, 32, 59, 9, 10, 40, 
	41, 65, 73, 79, 80, 83, 85, 97, 
	105, 111, 112, 115, 117, 48, 57, 32, 
	59, 9, 10, 40, 41, 48, 57, 67, 
	99, 80, 112, 75, 107, 73, 105, 88, 
	120, 32, 59, 9, 10, 40, 41, 65, 
	80, 83, 97, 112, 115, 67, 99, 80, 
	112, 75, 107, 73, 105, 88, 120, 32, 
	59, 9, 10, 40, 41, 71, 75, 103, 
	107, 80, 112, 32, 59, 9, 10, 40, 
	41, 73, 105, 88, 120, 32, 59, 9, 
	10, 40, 41, 80, 112, 75, 107, 73, 
	105, 32, 59, 9, 10, 40, 41, 73, 
	105, 68, 100, 32, 59, 9, 10, 40, 
	41, 71, 75, 103, 107, 80, 112, 32, 
	59, 9, 10, 40, 41, 73, 105, 88, 
	120, 32, 59, 9, 10, 40, 41, 80, 
	112, 75, 107, 73, 105, 32, 59, 9, 
	10, 40, 41, 82, 114, 73, 105, 32, 
	59, 9, 10, 40, 41, 46, 48, 57, 
	32, 46, 59, 9, 10, 40, 41, 48, 
	57, 42, 92, 95, 45, 57, 64, 90, 
	97, 122, 32, 59, 9, 10, 40, 41, 
	42, 92, 95, 45, 57, 64, 90, 97, 
	122, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 42, 92, 95, 1034, 1083, 
	45, 57, 64, 90, 97, 122, 9, 32, 
	40, 41, 1034, 1083, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	68, 72, 77, 83, 87, 100, 104, 109, 
	115, 119, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 68, 72, 77, 83, 87, 100, 
	104, 109, 115, 119, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 68, 72, 77, 83, 
	87, 100, 104, 109, 115, 119, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 32, 59, 68, 72, 77, 83, 
	87, 100, 104, 109, 115, 119, 9, 10, 
	40, 41, 48, 57, 32, 59, 9, 10, 
	40, 41, 48, 57, 68, 72, 77, 83, 
	87, 100, 104, 109, 115, 119, 48, 57, 
	32, 59, 9, 10, 40, 41, 48, 57, 
	1034, 896, 1151, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 68, 72, 77, 83, 87, 
	100, 104, 109, 115, 119, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 1034, 
	896, 1151, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 68, 72, 77, 83, 87, 100, 
	104, 109, 115, 119, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 1034, 896, 
	1151, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 68, 72, 77, 83, 87, 100, 104, 
	109, 115, 119, 48, 57, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 1034, 896, 1151, 
	1034, 896, 1151, 1034, 896, 1151, 32, 59, 
	9, 10, 40, 41, 9, 32, 40, 41, 
	1034, 1083, 9, 32, 40, 41, 1034, 1083, 
	-128, 8, 11, 58, 60, 127, 32, 59, 
	9, 10, 40, 41, 1034, 896, 1151, 42, 
	92, 95, 45, 57, 64, 90, 97, 122, 
	9, 32, 40, 41, 1034, 1083, 9, 32, 
	40, 41, 42, 92, 95, 1034, 1083, 45, 
	57, 64, 90, 97, 122, 32, 59, 9, 
	10, 40, 41, 1034, 896, 1151, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 42, 92, 95, 1034, 
	1083, 45, 57, 64, 90, 97, 122, 32, 
	59, 9, 10, 40, 41, 1034, 896, 1151, 
	32, 59, 9, 10, 40, 41, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, -128, 8, 11, 58, 60, 127, 
	1034, 896, 1151, 46, 48, 58, 65, 70, 
	97, 102, 32, 46, 59, 9, 10, 40, 
	41, 48, 58, 65, 70, 97, 102, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 78, 83, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 78, 
	83, 1034, 1083, 48, 57, 9, 32, 40, 
	41, 46, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 78, 83, 1034, 1083, 9, 32, 
	40, 41, 1034, 1083, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	69, 87, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 69, 87, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 46, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 69, 87, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 45, 1034, 1083, 48, 57, 
	48, 57, 9, 32, 40, 41, 46, 109, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 48, 57, 9, 32, 40, 41, 
	46, 109, 2058, 2107, 2314, 2363, 2570, 2619, 
	48, 57, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 48, 57, 9, 32, 
	40, 41, 46, 109, 2058, 2107, 2314, 2363, 
	2570, 2619, 48, 57, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	9, 32, 40, 41, 46, 109, 2058, 2107, 
	2314, 2363, 2570, 2619, 48, 57, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	1034, 896, 1151, 9, 32, 40, 41, 109, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	9, 32, 40, 41, 109, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 1034, 896, 1151, 
	9, 32, 40, 41, 109, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 9, 32, 40, 
	41, 109, 2058, 2107, 2314, 2363, 2570, 2619, 
	48, 57, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 1034, 896, 1151, 9, 
	32, 40, 41, 109, 2058, 2107, 2314, 2363, 
	2570, 2619, 48, 57, 9, 32, 40, 41, 
	109, 2058, 2107, 2314, 2363, 2570, 2619, 48, 
	57, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 1034, 896, 1151, 9, 32, 
	40, 41, 109, 2058, 2107, 2314, 2363, 2570, 
	2619, 48, 57, 9, 32, 40, 41, 109, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 1034, 896, 1151, 9, 32, 40, 
	41, 1034, 1083, 1034, 896, 1151, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 1034, 896, 
	1151, 1034, 896, 1151, 1034, 896, 1151, 9, 
	32, 40, 41, 1034, 1083, 1034, 896, 1151, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	1034, 896, 1151, 1034, 896, 1151, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 42, 92, 95, 1034, 
	1083, 45, 57, 64, 90, 97, 122, 32, 
	59, 9, 10, 40, 41, 1034, 896, 1151, 
	1034, 896, 1151, 1034, 896, 1151, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, -128, 8, 
	11, 58, 60, 127, 9, 32, 40, 41, 
	1034, 1083, 9, 32, 40, 41, 1034, 1083, 
	-128, 8, 11, 58, 60, 127, 9, 32, 
	40, 41, 1034, 1083, 9, 32, 40, 41, 
	1034, 1083, -128, 8, 11, 58, 60, 127, 
	9, 32, 40, 41, 1034, 1083, 9, 32, 
	40, 41, 42, 92, 95, 1034, 1083, 45, 
	57, 64, 90, 97, 122, 32, 59, 9, 
	10, 40, 41, 1034, 896, 1151, 1034, 896, 
	1151, 1034, 896, 1151, 1034, 896, 1151, 1034, 
	896, 1151, 48, 57, 65, 90, 97, 122, 
	9, 32, 40, 41, 1034, 1083, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 65, 90, 
	97, 122, 9, 32, 40, 41, 1034, 1083, 
	9, 32, 40, 41, 43, 1034, 1083, 47, 
	57, 65, 90, 97, 122, 1546, 1595, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	9, 32, 33, 40, 41, 49, 50, 2058, 
	2107, 2314, 2363, 2570, 2619, 49, 50, 58, 
	46, 48, 57, 47, 46, 57, 48, 57, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 48, 57, 1034, 896, 1151, 58, 
	46, 48, 58, 65, 70, 97, 102, 47, 
	46, 58, 65, 70, 97, 102, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	65, 90, 97, 122, 9, 32, 40, 41, 
	1034, 1083, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	65, 70, 97, 102, 1034, 896, 1151, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	65, 70, 97, 102, 1034, 896, 1151, 1034, 
	896, 1151, 1034, 896, 1151, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 48, 49, 50, 51, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 46, 1034, 1083, 6153, 6176, 
	6184, 6185, 6409, 6432, 6440, 6441, 6665, 6688, 
	6696, 6697, 9482, 9531, 9738, 9787, 10250, 10299, 
	10506, 10555, 10762, 10811, 11274, 11323, 11530, 11579, 
	11786, 11835, 12298, 12347, 12554, 12603, 12810, 12859, 
	3081, 3104, 3112, 3113, 3115, 4106, 4155, 3119, 
	3129, 3137, 3162, 3169, 3194, 1546, 1595, 4106, 
	3968, 4223, 4617, 4640, 4648, 4649, 13578, 13627, 
	13834, 13883, 14346, 14395, 14602, 14651, 14858, 14907, 
	5642, 5504, 5759, 3115, 6153, 6176, 6184, 6185, 
	6409, 6432, 6440, 6441, 6665, 6688, 6696, 6697, 
	9482, 9531, 9738, 9787, 10250, 10299, 10506, 10555, 
	10762, 10811, 11274, 11323, 11530, 11579, 11786, 11835, 
	12298, 12347, 12554, 12603, 12810, 12859, 3119, 3129, 
	3137, 3162, 3169, 3194, 7690, 8202, 8714, 7552, 
	7807, 8064, 8319, 8576, 8831, 1034, 896, 1151, 
	1034, 896, 1151, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 46, 1034, 1083, 
	48, 57, 46, 6153, 6176, 6184, 6185, 6409, 
	6432, 6440, 6441, 6665, 6688, 6696, 6697, 9482, 
	9531, 9738, 9787, 10250, 10299, 10506, 10555, 10762, 
	10811, 11274, 11323, 11530, 11579, 11786, 11835, 12298, 
	12347, 12554, 12603, 12810, 12859, 48, 57, 1034, 
	896, 1151, 1034, 896, 1151, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 46, 
	1034, 1083, 48, 58, 65, 70, 97, 102, 
	46, 6153, 6176, 6184, 6185, 6409, 6432, 6440, 
	6441, 6665, 6688, 6696, 6697, 9482, 9531, 9738, 
	9787, 10250, 10299, 10506, 10555, 10762, 10811, 11274, 
	11323, 11530, 11579, 11786, 11835, 12298, 12347, 12554, 
	12603, 12810, 12859, 48, 58, 65, 70, 97, 
	102, 1034, 896, 1151, 1034, 896, 1151, 9, 
	32, 40, 41, 1034, 1083, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 9, 32, 40, 
	41, 42, 92, 95, 1034, 1083, 45, 57, 
	64, 90, 97, 122, 6153, 6176, 6184, 6185, 
	6409, 6432, 6440, 6441, 6665, 6688, 6696, 6697, 
	9482, 9531, 9738, 9787, 10250, 10299, 10506, 10555, 
	10762, 10811, 11274, 11323, 11530, 11579, 11786, 11835, 
	12298, 12347, 12554, 12603, 12810, 12859, 1034, 896, 
	1151, 1034, 896, 1151, 1034, 896, 1151, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 9, 32, 40, 41, 65, 
	70, 80, 97, 102, 112, 1034, 1083, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 65, 
	90, 97, 122, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 42, 92, 95, 
	1034, 1083, 45, 57, 64, 90, 97, 122, 
	9, 32, 40, 41, 1034, 1083, 9, 32, 
	40, 41, 43, 1034, 1083, 47, 57, 65, 
	90, 97, 122, 1546, 1595, 1034, 896, 1151, 
	1034, 896, 1151, 1034, 896, 1151, 1034, 896, 
	1151, 1034, 896, 1151, 1034, 896, 1151, 1034, 
	896, 1151, 1034, 896, 1151, 65, 97, 65, 
	97, 9, 32, 40, 41, 1034, 1083, 83, 
	115, 68, 100, 66, 98, 9, 32, 40, 
	41, 1034, 1083, 76, 108, 9, 32, 40, 
	41, 1034, 1083, 69, 78, 101, 110, 82, 
	114, 84, 116, 9, 32, 40, 41, 1034, 
	1083, 65, 97, 77, 109, 69, 101, 9, 
	32, 40, 41, 1034, 1083, 72, 78, 83, 
	104, 110, 115, 67, 99, 73, 105, 68, 
	100, 9, 32, 40, 41, 1034, 1083, 65, 
	83, 97, 115, 77, 109, 69, 101, 9, 
	32, 40, 41, 1034, 1083, 75, 107, 69, 
	101, 89, 121, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 85, 
	117, 73, 105, 52, 54, 56, 9, 32, 
	40, 41, 1034, 1083, 52, 9, 32, 40, 
	41, 1034, 1083, 73, 105, 78, 110, 70, 
	102, 79, 111, 9, 32, 40, 41, 1034, 
	1083, 80, 112, 83, 115, 69, 101, 67, 
	99, 75, 107, 69, 101, 89, 121, 9, 
	32, 40, 41, 1034, 1083, 69, 88, 101, 
	120, 89, 121, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 51, 
	54, 79, 80, 111, 112, 50, 9, 32, 
	40, 41, 1034, 1083, 52, 9, 32, 40, 
	41, 1034, 1083, 67, 99, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 1034, 
	1083, 73, 88, 105, 120, 78, 110, 70, 
	102, 79, 111, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 65, 
	73, 83, 97, 105, 115, 80, 112, 84, 
	116, 82, 114, 9, 32, 40, 41, 1034, 
	1083, 68, 100, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 69, 101, 1034, 
	1083, 67, 99, 9, 32, 40, 41, 51, 
	1034, 1083, 9, 32, 40, 41, 80, 112, 
	1034, 1083, 65, 97, 82, 114, 65, 97, 
	77, 109, 9, 32, 40, 41, 1034, 1083, 
	84, 116, 82, 114, 9, 32, 40, 41, 
	1034, 1083, 80, 82, 84, 112, 114, 116, 
	9, 32, 40, 41, 1034, 1083, 83, 115, 
	73, 105, 71, 103, 9, 32, 40, 41, 
	1034, 1083, 9, 32, 40, 41, 1034, 1083, 
	79, 80, 82, 83, 111, 112, 114, 115, 
	65, 97, 9, 32, 40, 41, 1034, 1083, 
	70, 102, 9, 32, 40, 41, 1034, 1083, 
	86, 118, 9, 32, 40, 41, 1034, 1083, 
	72, 104, 70, 102, 80, 112, 9, 32, 
	40, 41, 1034, 1083, 76, 88, 89, 108, 
	120, 121, 83, 115, 65, 97, 9, 32, 
	40, 41, 1034, 1083, 84, 116, 9, 32, 
	40, 41, 1034, 1083, 80, 112, 69, 101, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 42, 92, 95, 45, 57, 64, 
	90, 97, 122, 32, 59, 9, 10, 40, 
	41, 32, 59, 9, 10, 40, 41, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 65, 90, 97, 122, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 43, 
	1034, 1083, 47, 57, 65, 90, 97, 122, 
	1546, 1595, 1034, 896, 1151, 1034, 896, 1151, 
	1034, 896, 1151, 43, 47, 57, 65, 90, 
	97, 122, 1546, 1595, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 45, 1034, 1083, 48, 57, 65, 
	70, 97, 102, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 65, 86, 97, 118, 48, 57, 65, 
	86, 97, 118, 61, 48, 57, 65, 86, 
	97, 118, 48, 57, 65, 86, 97, 118, 
	61, 48, 57, 65, 86, 97, 118, 61, 
	48, 57, 65, 86, 97, 118, 48, 57, 
	65, 86, 97, 118, 61, 48, 57, 65, 
	86, 97, 118, 32, 59, 9, 10, 40, 
	41, 48, 57, 65, 86, 97, 118, 32, 
	59, 9, 10, 40, 41, 61, 61, 61, 
	61, 61, 1034, 896, 1151, 48, 57, 65, 
	70, 97, 102, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 65, 70, 97, 102, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 45, 1034, 
	1083, 48, 57, 65, 70, 97, 102, 32, 
	59, 9, 10, 40, 41, 48, 57, 65, 
	70, 97, 102, 32, 59, 9, 10, 40, 
	41, 48, 57, 65, 70, 97, 102, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	65, 70, 97, 102, 1034, 896, 1151, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 46, 1034, 
	1083, 48, 57, 32, 46, 59, 9, 10, 
	40, 41, 48, 57, 1034, 896, 1151, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 65, 70, 97, 102, 48, 57, 65, 
	70, 97, 102, 48, 57, 65, 70, 97, 
	102, 48, 57, 65, 70, 97, 102, 58, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	32, 58, 59, 9, 10, 40, 41, 1034, 
	896, 1151, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 45, 48, 
	57, 65, 70, 97, 102, 48, 57, 65, 
	70, 97, 102, 32, 45, 59, 9, 10, 
	40, 41, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 45, 48, 
	57, 65, 70, 97, 102, 48, 57, 65, 
	70, 97, 102, 32, 45, 59, 9, 10, 
	40, 41, 9, 32, 36, 40, 41, 42, 
	92, 95, 778, 827, 1034, 1083, 45, 57, 
	64, 90, 97, 122, 9, 32, 36, 40, 
	41, 42, 58, 92, 95, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	44, 45, 57, 60, 63, 64, 90, 91, 
	96, 97, 122, 123, 127, 9, 32, 36, 
	40, 41, 42, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 92, 95, 97, 99, 100, 101, 104, 
	105, 107, 108, 109, 110, 112, 114, 115, 
	116, 778, 827, 1034, 1083, 45, 57, 64, 
	90, 98, 122, 9, 32, 36, 40, 41, 
	42, 65, 67, 68, 69, 72, 73, 75, 
	76, 77, 78, 80, 82, 83, 84, 92, 
	95, 97, 99, 100, 101, 104, 105, 107, 
	108, 109, 110, 112, 114, 115, 116, 778, 
	827, 1034, 1083, 45, 47, 48, 57, 64, 
	90, 98, 122, 9, 32, 36, 40, 41, 
	42, 92, 95, 778, 827, 1034, 1083, 45, 
	57, 64, 90, 97, 122, 9, 32, 36, 
	40, 41, 42, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 92, 95, 97, 99, 100, 101, 104, 
	105, 107, 108, 109, 110, 112, 114, 115, 
	116, 778, 827, 1034, 1083, 45, 47, 48, 
	57, 64, 90, 98, 122, 9, 32, 36, 
	40, 41, 42, 92, 95, 778, 827, 1034, 
	1083, 45, 57, 64, 90, 97, 122, 9, 
	32, 36, 40, 41, 42, 58, 92, 95, 
	1802, 1851, 2058, 2107, 2314, 2363, 2570, 2619, 
	-128, 8, 11, 44, 45, 57, 60, 63, 
	64, 90, 91, 96, 97, 122, 123, 127, 
	9, 32, 36, 40, 41, 42, 58, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 92, 95, 97, 
	99, 100, 101, 104, 105, 107, 108, 109, 
	110, 112, 114, 115, 116, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	44, 45, 47, 48, 57, 60, 63, 64, 
	90, 91, 96, 98, 122, 123, 127, 9, 
	32, 36, 40, 41, 42, 58, 65, 67, 
	68, 69, 72, 73, 75, 76, 77, 78, 
	80, 82, 83, 84, 92, 95, 97, 99, 
	100, 101, 104, 105, 107, 108, 109, 110, 
	112, 114, 115, 116, 1802, 1851, 2058, 2107, 
	2314, 2363, 2570, 2619, -128, 8, 11, 44, 
	45, 47, 48, 57, 60, 63, 64, 90, 
	91, 96, 98, 122, 123, 127, 32, 33, 
	59, 92, 9, 10, 35, 39, 40, 41, 
	42, 126, 9, 32, 40, 41, 42, 46, 
	92, 95, 1034, 1083, 45, 57, 65, 90, 
	97, 122, 9, 32, 40, 41, 43, 2058, 
	2107, 2314, 2363, 2570, 2619, 47, 57, 65, 
	90, 97, 122, 1034, 896, 1151, 9, 32, 
	40, 41, 65, 67, 68, 69, 72, 73, 
	75, 76, 77, 78, 80, 82, 83, 84, 
	97, 99, 100, 101, 104, 105, 107, 108, 
	109, 110, 112, 114, 115, 116, 2058, 2107, 
	2314, 2363, 2570, 2619, 1034, 896, 1151, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 48, 57, 65, 70, 97, 102, 1034, 
	896, 1151, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 65, 70, 97, 102, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	48, 57, 65, 70, 97, 102, 1034, 896, 
	1151, 1034, 896, 1151, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, -128, 8, 
	11, 58, 60, 127, 1034, 896, 1151, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 1034, 896, 1151, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	1034, 896, 1151, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 48, 57, 1034, 
	896, 1151, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 48, 57, 1034, 896, 
	1151, 9, 32, 33, 40, 41, 49, 50, 
	2058, 2107, 2314, 2363, 2570, 2619, 1034, 896, 
	1151, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 65, 70, 97, 
	102, 1034, 896, 1151, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	65, 70, 97, 102, 1034, 896, 1151, 4617, 
	4640, 4648, 4649, 13578, 13627, 13834, 13883, 14346, 
	14395, 14602, 14651, 14858, 14907, 5642, 5504, 5759, 
	3081, 3104, 3112, 3113, 3115, 4106, 4155, 3119, 
	3129, 3137, 3162, 3169, 3194, 4106, 3968, 4223, 
	3115, 6153, 6176, 6184, 6185, 6409, 6432, 6440, 
	6441, 6665, 6688, 6696, 6697, 9482, 9531, 9738, 
	9787, 10250, 10299, 10506, 10555, 10762, 10811, 11274, 
	11323, 11530, 11579, 11786, 11835, 12298, 12347, 12554, 
	12603, 12810, 12859, 3119, 3129, 3137, 3162, 3169, 
	3194, 7690, 8202, 8714, 7552, 7807, 8064, 8319, 
	8576, 8831, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 48, 57, 65, 70, 
	97, 102, 1034, 896, 1151, 0
};

static const char _zone_scanner_single_lengths[] = {
	0, 36, 16, 34, 16, 11, 8, 8, 
	1, 1, 8, 1, 13, 2, 42, 18, 
	36, 4, 2, 2, 10, 12, 2, 2, 
	2, 10, 6, 2, 2, 2, 10, 4, 
	2, 2, 10, 2, 2, 2, 10, 10, 
	2, 2, 2, 1, 10, 1, 10, 2, 
	2, 2, 2, 10, 4, 6, 34, 2, 
	2, 2, 2, 2, 2, 2, 10, 4, 
	2, 10, 10, 6, 1, 10, 1, 10, 
	2, 10, 10, 4, 2, 2, 2, 10, 
	10, 6, 2, 2, 2, 10, 2, 10, 
	12, 2, 11, 12, 2, 2, 2, 2, 
	10, 2, 2, 10, 6, 10, 2, 2, 
	2, 10, 10, 8, 2, 10, 2, 10, 
	2, 10, 2, 2, 2, 10, 6, 2, 
	2, 10, 2, 10, 2, 2, 0, 10, 
	1, 36, 4, 6, 34, 16, 6, 10, 
	6, 1, 1, 6, 2, 2, 2, 2, 
	2, 2, 2, 6, 34, 1, 16, 34, 
	1, 6, 10, 6, 1, 16, 40, 18, 
	13, 36, 2, 8, 8, 42, 12, 14, 
	10, 10, 12, 12, 14, 12, 14, 10, 
	14, 16, 14, 2, 14, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 8, 18, 
	36, 2, 18, 42, 12, 14, 10, 10, 
	12, 12, 14, 12, 14, 10, 14, 16, 
	14, 9, 36, 12, 12, 10, 2, 2, 
	10, 2, 2, 2, 10, 2, 10, 2, 
	8, 8, 12, 10, 1, 12, 10, 12, 
	8, 8, 10, 10, 12, 10, 12, 8, 
	12, 14, 12, 2, 2, 10, 2, 1, 
	1, 5, 3, 6, 5, 0, 0, 0, 
	5, 2, 2, 2, 4, 0, 0, 0, 
	4, 4, 2, 0, 0, 0, 4, 6, 
	6, 18, 8, 1, 8, 10, 8, 1, 
	6, 10, 1, 12, 8, 1, 3, 0, 
	0, 0, 3, 8, 1, 6, 6, 8, 
	12, 1, 12, 8, 1, 3, 0, 0, 
	0, 3, 8, 2, 10, 1, 1, 1, 
	1, 2, 2, 11, 1, 1, 10, 38, 
	16, 2, 2, 10, 1, 4, 2, 2, 
	10, 2, 2, 2, 10, 6, 2, 2, 
	2, 10, 4, 2, 2, 10, 2, 2, 
	2, 10, 10, 2, 2, 2, 1, 10, 
	1, 10, 2, 2, 2, 2, 10, 2, 
	2, 2, 2, 2, 2, 2, 10, 4, 
	2, 10, 10, 6, 1, 10, 1, 10, 
	2, 10, 10, 4, 2, 2, 2, 10, 
	10, 6, 2, 2, 2, 10, 2, 10, 
	12, 2, 11, 12, 2, 2, 2, 2, 
	10, 2, 2, 10, 6, 10, 2, 2, 
	2, 10, 10, 8, 2, 10, 2, 10, 
	2, 10, 2, 2, 2, 10, 6, 2, 
	2, 10, 2, 10, 2, 2, 0, 10, 
	2, 2, 2, 10, 2, 10, 6, 6, 
	6, 6, 0, 10, 1, 1, 1, 6, 
	7, 8, 6, 0, 10, 1, 1, 6, 
	1, 10, 2, 4, 2, 2, 3, 2, 
	2, 2, 2, 1, 1, 2, 2, 2, 
	1, 2, 2, 4, 1, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 1, 1, 
	2, 2, 2, 1, 1, 1, 2, 1, 
	1, 2, 2, 2, 1, 1, 1, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 4, 2, 
	2, 2, 2, 2, 2, 2, 2, 4, 
	2, 1, 2, 2, 2, 3, 3, 2, 
	2, 2, 2, 1, 1, 2, 2, 2, 
	1, 2, 1, 1, 2, 1, 1, 2, 
	12, 2, 2, 2, 2, 2, 2, 2, 
	6, 2, 2, 2, 2, 2, 2, 4, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 4, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 1, 3, 3, 2, 3, 6, 9, 
	6, 6, 6, 6, 16, 6, 16, 6, 
	16, 6, 12, 2, 10, 2, 1, 6, 
	10, 6, 1, 6, 10, 6, 1, 6, 
	10, 6, 1, 1, 1, 2, 6, 6, 
	2, 1, 3, 6, 9, 2, 1, 0, 
	6, 9, 2, 1, 2, 10, 10, 1, 
	1, 3, 0, 6, 8, 6, 8, 7, 
	8, 6, 6, 6, 8, 6, 8, 7, 
	8, 6, 7, 0, 12, 10, 12, 10, 
	12, 10, 12, 10, 1, 11, 11, 1, 
	11, 11, 10, 1, 11, 11, 10, 1, 
	11, 11, 10, 1, 6, 1, 6, 6, 
	1, 1, 1, 6, 1, 6, 6, 1, 
	1, 0, 6, 6, 6, 6, 6, 9, 
	2, 1, 1, 1, 0, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 9, 2, 
	1, 1, 1, 1, 1, 0, 6, 6, 
	6, 6, 6, 7, 2, 1, 1, 1, 
	13, 2, 1, 1, 1, 0, 10, 1, 
	1, 1, 1, 0, 6, 6, 6, 6, 
	6, 6, 0, 10, 1, 1, 1, 1, 
	0, 6, 6, 6, 6, 0, 10, 1, 
	1, 1, 0, 6, 10, 6, 6, 6, 
	7, 34, 7, 2, 1, 14, 1, 35, 
	3, 1, 1, 6, 6, 6, 7, 35, 
	1, 1, 6, 6, 6, 7, 35, 1, 
	1, 6, 6, 6, 9, 34, 1, 1, 
	1, 28, 12, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 6, 9, 
	6, 7, 2, 1, 1, 1, 1, 1, 
	1, 1, 1, 2, 2, 6, 2, 2, 
	2, 6, 2, 6, 4, 2, 2, 6, 
	2, 2, 2, 6, 6, 2, 2, 2, 
	6, 4, 2, 2, 6, 2, 2, 2, 
	6, 6, 2, 2, 2, 1, 6, 1, 
	6, 2, 2, 2, 2, 6, 2, 2, 
	2, 2, 2, 2, 2, 6, 4, 2, 
	6, 6, 6, 1, 6, 1, 6, 2, 
	6, 6, 4, 2, 2, 2, 6, 6, 
	6, 2, 2, 2, 6, 2, 6, 8, 
	2, 7, 8, 2, 2, 2, 2, 6, 
	2, 2, 6, 6, 6, 2, 2, 2, 
	6, 6, 8, 2, 6, 2, 6, 2, 
	6, 2, 2, 2, 6, 6, 2, 2, 
	6, 2, 6, 2, 2, 0, 6, 3, 
	2, 2, 0, 6, 6, 6, 6, 6, 
	7, 2, 1, 1, 1, 1, 2, 0, 
	6, 6, 6, 6, 6, 7, 6, 6, 
	0, 1, 0, 1, 1, 0, 1, 2, 
	2, 1, 1, 1, 1, 1, 1, 0, 
	6, 1, 1, 1, 0, 6, 6, 6, 
	6, 6, 7, 2, 0, 2, 1, 1, 
	1, 0, 6, 6, 6, 6, 6, 6, 
	0, 10, 1, 1, 1, 1, 0, 6, 
	7, 3, 1, 0, 6, 6, 0, 0, 
	0, 1, 0, 0, 0, 0, 3, 1, 
	0, 0, 1, 0, 0, 3, 0, 0, 
	1, 0, 0, 3, 12, 17, 40, 40, 
	12, 40, 12, 17, 45, 45, 0, 0, 
	0, 4, 0, 0, 0, 10, 0, 11, 
	1, 0, 38, 1, 0, 10, 1, 6, 
	0, 10, 1, 1, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 10, 1, 0, 
	0, 10, 1, 10, 1, 10, 1, 10, 
	1, 0, 0, 0, 0, 13, 1, 0, 
	10, 1, 0, 10, 1, 0, 14, 1, 
	7, 1, 35, 3, 0, 0, 0, 0, 
	0, 0, 0, 10, 1, 0, 0, 0, 
	0
};

static const char _zone_scanner_range_lengths[] = {
	0, 1, 1, 0, 0, 3, 0, 0, 
	0, 0, 0, 1, 3, 2, 4, 1, 
	0, 0, 0, 0, 0, 1, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 1, 
	1, 1, 0, 0, 1, 1, 1, 1, 
	1, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 2, 0, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 0, 4, 0, 
	3, 1, 2, 3, 3, 4, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 2, 0, 0, 0, 0, 
	0, 0, 2, 0, 0, 2, 0, 1, 
	1, 2, 0, 4, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 3, 1, 1, 1, 2, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 2, 
	3, 3, 2, 2, 1, 1, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 2, 2, 0, 2, 0, 
	0, 3, 2, 5, 5, 1, 1, 1, 
	5, 2, 2, 2, 4, 1, 1, 1, 
	4, 1, 2, 1, 1, 1, 1, 0, 
	1, 1, 0, 0, 1, 1, 1, 1, 
	0, 3, 0, 3, 0, 0, 3, 1, 
	1, 1, 3, 0, 1, 0, 3, 0, 
	3, 0, 3, 0, 0, 3, 1, 1, 
	1, 3, 0, 2, 3, 1, 1, 3, 
	3, 3, 3, 3, 1, 0, 0, 0, 
	0, 0, 0, 0, 1, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 1, 
	0, 0, 0, 0, 0, 0, 0, 1, 
	1, 3, 3, 3, 1, 1, 1, 0, 
	1, 1, 3, 3, 3, 1, 1, 1, 
	1, 1, 3, 0, 2, 0, 2, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 2, 0, 0, 0, 0, 0, 0, 
	0, 2, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 2, 0, 
	0, 0, 0, 0, 0, 0, 0, 2, 
	0, 0, 0, 0, 0, 0, 0, 2, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 2, 0, 0, 2, 0, 0, 0, 
	0, 0, 2, 0, 0, 0, 2, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 2, 0, 0, 2, 0, 0, 2, 
	1, 3, 0, 0, 0, 0, 0, 2, 
	0, 0, 0, 0, 0, 0, 2, 0, 
	0, 2, 0, 0, 2, 0, 0, 0, 
	2, 0, 0, 2, 0, 0, 2, 0, 
	0, 2, 0, 0, 0, 2, 0, 0, 
	2, 1, 3, 3, 2, 3, 0, 3, 
	0, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 3, 3, 1, 3, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 2, 0, 3, 
	2, 1, 3, 0, 3, 2, 1, 1, 
	1, 3, 2, 1, 2, 0, 3, 1, 
	3, 5, 1, 1, 1, 1, 1, 1, 
	0, 0, 1, 1, 1, 1, 1, 1, 
	0, 0, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 0, 1, 1, 1, 1, 
	1, 1, 0, 1, 1, 1, 0, 1, 
	1, 1, 0, 1, 0, 1, 1, 1, 
	1, 1, 1, 0, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 3, 
	2, 1, 1, 1, 1, 1, 1, 1, 
	3, 0, 3, 0, 3, 0, 3, 2, 
	1, 1, 1, 1, 1, 3, 0, 1, 
	1, 3, 0, 3, 0, 1, 1, 1, 
	0, 0, 0, 1, 1, 1, 1, 1, 
	0, 3, 3, 1, 1, 3, 0, 1, 
	1, 3, 3, 3, 1, 1, 1, 1, 
	1, 1, 1, 1, 3, 3, 3, 1, 
	1, 1, 1, 1, 0, 0, 1, 1, 
	0, 0, 3, 0, 1, 0, 1, 3, 
	3, 1, 1, 0, 1, 1, 1, 1, 
	1, 1, 0, 1, 1, 3, 3, 1, 
	1, 0, 1, 1, 3, 0, 1, 1, 
	1, 0, 0, 3, 0, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 3, 
	0, 3, 0, 1, 1, 1, 1, 1, 
	1, 1, 1, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 1, 3, 
	2, 2, 1, 1, 1, 1, 3, 0, 
	3, 0, 1, 1, 1, 3, 0, 1, 
	1, 1, 1, 1, 1, 3, 0, 3, 
	3, 3, 3, 3, 3, 3, 3, 5, 
	2, 0, 0, 0, 0, 0, 1, 3, 
	3, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 3, 2, 3, 5, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 3, 
	3, 3, 1, 1, 1, 1, 1, 1, 
	1, 3, 1, 1, 1, 3, 3, 3, 
	3, 0, 3, 3, 3, 3, 2, 1, 
	3, 3, 0, 3, 3, 2, 3, 3, 
	0, 3, 3, 2, 3, 8, 3, 4, 
	3, 4, 3, 8, 9, 9, 0, 0, 
	0, 4, 0, 0, 0, 3, 0, 3, 
	1, 0, 0, 1, 0, 3, 1, 3, 
	0, 3, 1, 1, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 3, 1, 0, 
	0, 0, 1, 1, 1, 1, 1, 1, 
	1, 0, 0, 0, 0, 0, 1, 0, 
	3, 1, 0, 3, 1, 0, 0, 1, 
	3, 1, 3, 3, 0, 0, 0, 0, 
	0, 0, 0, 3, 1, 0, 0, 0, 
	0
};

static const short _zone_scanner_index_offsets[] = {
	0, 0, 38, 56, 91, 108, 123, 132, 
	141, 143, 145, 154, 157, 174, 179, 226, 
	246, 283, 288, 291, 294, 305, 319, 322, 
	325, 328, 339, 346, 349, 352, 355, 366, 
	371, 374, 377, 388, 391, 394, 397, 408, 
	419, 422, 425, 428, 430, 441, 443, 454, 
	457, 460, 463, 466, 477, 482, 489, 524, 
	527, 530, 533, 536, 539, 542, 545, 556, 
	561, 564, 575, 586, 593, 595, 606, 608, 
	619, 622, 633, 644, 649, 652, 655, 658, 
	669, 680, 687, 690, 693, 696, 707, 710, 
	721, 734, 737, 749, 762, 765, 768, 771, 
	774, 785, 788, 791, 802, 809, 820, 823, 
	826, 829, 840, 851, 860, 863, 874, 877, 
	888, 891, 902, 905, 908, 911, 922, 929, 
	932, 935, 946, 949, 960, 963, 966, 968, 
	980, 983, 1021, 1026, 1033, 1069, 1087, 1095, 
	1107, 1115, 1118, 1120, 1127, 1130, 1133, 1136, 
	1139, 1142, 1145, 1150, 1157, 1193, 1196, 1214, 
	1250, 1253, 1261, 1273, 1281, 1284, 1301, 1346, 
	1365, 1382, 1420, 1425, 1437, 1449, 1496, 1509, 
	1524, 1535, 1546, 1559, 1572, 1587, 1600, 1615, 
	1626, 1641, 1658, 1673, 1678, 1693, 1696, 1699, 
	1702, 1705, 1708, 1713, 1716, 1719, 1724, 1733, 
	1753, 1791, 1796, 1815, 1862, 1875, 1890, 1901, 
	1912, 1925, 1938, 1953, 1966, 1981, 1992, 2007, 
	2024, 2039, 2052, 2090, 2104, 2118, 2131, 2134, 
	2137, 2148, 2151, 2154, 2157, 2168, 2171, 2182, 
	2187, 2199, 2211, 2226, 2239, 2242, 2256, 2267, 
	2280, 2289, 2298, 2309, 2320, 2333, 2344, 2357, 
	2366, 2379, 2394, 2407, 2412, 2417, 2428, 2433, 
	2435, 2437, 2446, 2452, 2464, 2475, 2477, 2479, 
	2481, 2492, 2497, 2502, 2507, 2516, 2518, 2520, 
	2522, 2531, 2537, 2542, 2544, 2546, 2548, 2554, 
	2561, 2569, 2589, 2598, 2600, 2610, 2622, 2632, 
	2635, 2642, 2656, 2658, 2674, 2683, 2685, 2692, 
	2694, 2696, 2698, 2705, 2714, 2717, 2724, 2734, 
	2743, 2759, 2761, 2777, 2786, 2788, 2795, 2797, 
	2799, 2801, 2808, 2817, 2822, 2836, 2839, 2842, 
	2847, 2852, 2858, 2864, 2879, 2882, 2884, 2895, 
	2934, 2951, 2954, 2957, 2968, 2971, 2976, 2979, 
	2982, 2993, 2996, 2999, 3002, 3013, 3020, 3023, 
	3026, 3029, 3040, 3045, 3048, 3051, 3062, 3065, 
	3068, 3071, 3082, 3093, 3096, 3099, 3102, 3104, 
	3115, 3117, 3128, 3131, 3134, 3137, 3140, 3151, 
	3154, 3157, 3160, 3163, 3166, 3169, 3172, 3183, 
	3188, 3191, 3202, 3213, 3220, 3222, 3233, 3235, 
	3246, 3249, 3260, 3271, 3276, 3279, 3282, 3285, 
	3296, 3307, 3314, 3317, 3320, 3323, 3334, 3337, 
	3348, 3361, 3364, 3376, 3389, 3392, 3395, 3398, 
	3401, 3412, 3415, 3418, 3429, 3436, 3447, 3450, 
	3453, 3456, 3467, 3478, 3487, 3490, 3501, 3504, 
	3515, 3518, 3529, 3532, 3535, 3538, 3549, 3556, 
	3559, 3562, 3573, 3576, 3587, 3590, 3593, 3595, 
	3607, 3610, 3613, 3616, 3627, 3630, 3641, 3648, 
	3656, 3664, 3674, 3678, 3692, 3695, 3698, 3701, 
	3708, 3717, 3727, 3737, 3741, 3755, 3758, 3761, 
	3769, 3772, 3784, 3790, 3795, 3800, 3803, 3809, 
	3812, 3815, 3818, 3821, 3823, 3825, 3828, 3831, 
	3834, 3836, 3841, 3844, 3849, 3851, 3854, 3857, 
	3860, 3863, 3868, 3871, 3874, 3877, 3880, 3882, 
	3884, 3887, 3890, 3893, 3895, 3897, 3899, 3904, 
	3906, 3908, 3911, 3914, 3917, 3919, 3921, 3923, 
	3928, 3931, 3934, 3937, 3940, 3943, 3946, 3949, 
	3954, 3957, 3960, 3963, 3966, 3969, 3972, 3977, 
	3980, 3983, 3988, 3991, 3994, 3999, 4002, 4005, 
	4010, 4013, 4015, 4020, 4023, 4026, 4030, 4036, 
	4039, 4042, 4045, 4048, 4050, 4052, 4055, 4058, 
	4061, 4063, 4068, 4070, 4072, 4077, 4079, 4081, 
	4086, 4100, 4106, 4109, 4112, 4115, 4118, 4121, 
	4126, 4133, 4136, 4139, 4142, 4145, 4148, 4153, 
	4158, 4161, 4166, 4169, 4172, 4177, 4180, 4183, 
	4186, 4191, 4194, 4197, 4202, 4207, 4210, 4215, 
	4218, 4221, 4226, 4229, 4232, 4235, 4240, 4243, 
	4246, 4251, 4254, 4261, 4268, 4273, 4280, 4287, 
	4300, 4307, 4315, 4323, 4331, 4349, 4357, 4375, 
	4383, 4401, 4409, 4425, 4431, 4443, 4449, 4452, 
	4460, 4472, 4480, 4483, 4491, 4503, 4511, 4514, 
	4522, 4534, 4542, 4545, 4548, 4551, 4556, 4563, 
	4573, 4578, 4581, 4588, 4595, 4608, 4613, 4616, 
	4618, 4626, 4639, 4644, 4647, 4652, 4663, 4677, 
	4680, 4685, 4694, 4696, 4704, 4714, 4722, 4732, 
	4741, 4750, 4757, 4765, 4773, 4783, 4791, 4801, 
	4810, 4819, 4826, 4835, 4837, 4851, 4863, 4877, 
	4889, 4903, 4915, 4929, 4940, 4943, 4956, 4969, 
	4972, 4985, 4998, 5009, 5012, 5025, 5038, 5049, 
	5052, 5065, 5078, 5089, 5092, 5099, 5102, 5110, 
	5118, 5121, 5124, 5127, 5134, 5137, 5145, 5153, 
	5156, 5159, 5161, 5169, 5177, 5185, 5193, 5201, 
	5214, 5219, 5222, 5225, 5228, 5230, 5238, 5246, 
	5254, 5264, 5271, 5281, 5288, 5298, 5305, 5318, 
	5323, 5326, 5329, 5332, 5335, 5338, 5342, 5349, 
	5357, 5365, 5375, 5382, 5393, 5396, 5399, 5402, 
	5405, 5419, 5422, 5424, 5427, 5430, 5432, 5444, 
	5447, 5449, 5454, 5459, 5461, 5469, 5479, 5486, 
	5494, 5502, 5512, 5516, 5530, 5533, 5536, 5539, 
	5542, 5544, 5552, 5560, 5568, 5578, 5582, 5596, 
	5599, 5602, 5605, 5607, 5615, 5626, 5633, 5641, 
	5649, 5657, 5692, 5703, 5706, 5709, 5724, 5727, 
	5766, 5773, 5776, 5779, 5786, 5794, 5802, 5811, 
	5848, 5851, 5854, 5861, 5869, 5877, 5888, 5927, 
	5930, 5933, 5940, 5948, 5956, 5969, 6004, 6007, 
	6010, 6013, 6042, 6055, 6065, 6072, 6080, 6088, 
	6096, 6104, 6112, 6120, 6128, 6136, 6144, 6152, 
	6165, 6172, 6183, 6186, 6189, 6192, 6195, 6198, 
	6201, 6204, 6207, 6210, 6213, 6216, 6223, 6226, 
	6229, 6232, 6239, 6242, 6249, 6254, 6257, 6260, 
	6267, 6270, 6273, 6276, 6283, 6290, 6293, 6296, 
	6299, 6306, 6311, 6314, 6317, 6324, 6327, 6330, 
	6333, 6340, 6347, 6350, 6353, 6356, 6358, 6365, 
	6367, 6374, 6377, 6380, 6383, 6386, 6393, 6396, 
	6399, 6402, 6405, 6408, 6411, 6414, 6421, 6426, 
	6429, 6436, 6443, 6450, 6452, 6459, 6461, 6468, 
	6471, 6478, 6485, 6490, 6493, 6496, 6499, 6506, 
	6513, 6520, 6523, 6526, 6529, 6536, 6539, 6546, 
	6555, 6558, 6566, 6575, 6578, 6581, 6584, 6587, 
	6594, 6597, 6600, 6607, 6614, 6621, 6624, 6627, 
	6630, 6637, 6644, 6653, 6656, 6663, 6666, 6673, 
	6676, 6683, 6686, 6689, 6692, 6699, 6706, 6709, 
	6712, 6719, 6722, 6729, 6732, 6735, 6737, 6745, 
	6752, 6757, 6762, 6764, 6772, 6780, 6788, 6798, 
	6805, 6816, 6819, 6822, 6825, 6828, 6833, 6836, 
	6838, 6846, 6854, 6862, 6870, 6878, 6889, 6896, 
	6906, 6910, 6915, 6919, 6924, 6929, 6933, 6938, 
	6946, 6951, 6953, 6955, 6957, 6959, 6961, 6964, 
	6968, 6978, 6981, 6984, 6987, 6989, 6997, 7005, 
	7013, 7021, 7029, 7040, 7045, 7049, 7057, 7060, 
	7063, 7066, 7068, 7076, 7084, 7092, 7100, 7108, 
	7118, 7122, 7136, 7139, 7142, 7145, 7148, 7150, 
	7158, 7167, 7174, 7177, 7179, 7187, 7197, 7201, 
	7205, 7209, 7211, 7215, 7219, 7223, 7227, 7233, 
	7236, 7240, 7244, 7246, 7250, 7254, 7260, 7264, 
	7268, 7270, 7274, 7278, 7284, 7300, 7326, 7370, 
	7415, 7431, 7476, 7492, 7518, 7573, 7628, 7629, 
	7630, 7631, 7640, 7641, 7642, 7643, 7657, 7658, 
	7673, 7676, 7677, 7716, 7719, 7720, 7734, 7737, 
	7747, 7748, 7762, 7765, 7768, 7769, 7770, 7771, 
	7772, 7773, 7774, 7775, 7776, 7777, 7791, 7794, 
	7795, 7796, 7807, 7810, 7822, 7825, 7837, 7840, 
	7852, 7855, 7856, 7857, 7858, 7859, 7873, 7876, 
	7877, 7891, 7894, 7895, 7909, 7912, 7913, 7928, 
	7931, 7942, 7945, 7984, 7991, 7992, 7993, 7994, 
	7995, 7996, 7997, 7998, 8012, 8015, 8016, 8017, 
	8018
};

static const short _zone_scanner_indicies[] = {
	1, 1, 2, 3, 5, 6, 7, 8, 
	9, 10, 11, 12, 13, 14, 15, 16, 
	17, 18, 5, 6, 7, 8, 9, 10, 
	11, 12, 13, 14, 15, 16, 17, 18, 
	19, 20, 21, 22, 4, 0, 24, 24, 
	25, 26, 28, 29, 30, 31, 32, 28, 
	29, 30, 31, 32, 33, 34, 27, 23, 
	36, 36, 37, 38, 39, 40, 41, 42, 
	43, 44, 45, 46, 47, 48, 49, 50, 
	51, 52, 39, 40, 41, 42, 43, 44, 
	45, 46, 47, 48, 49, 50, 51, 52, 
	53, 54, 35, 56, 56, 57, 58, 59, 
	60, 61, 59, 60, 61, 62, 63, 64, 
	64, 65, 66, 55, 68, 68, 70, 71, 
	72, 73, 74, 75, 75, 76, 77, 67, 
	67, 67, 69, 79, 79, 80, 81, 82, 
	83, 82, 83, 78, 85, 85, 86, 87, 
	88, 89, 88, 89, 84, 88, 89, 91, 
	90, 85, 85, 86, 87, 88, 89, 88, 
	89, 78, 73, 74, 69, 92, 92, 93, 
	94, 72, 88, 89, 88, 95, 96, 97, 
	88, 98, 67, 67, 67, 78, 88, 99, 
	89, 95, 78, 100, 100, 102, 103, 67, 
	105, 106, 107, 108, 109, 110, 111, 112, 
	113, 114, 115, 116, 117, 118, 72, 105, 
	106, 107, 108, 109, 110, 111, 112, 113, 
	114, 115, 116, 117, 118, 19, 20, 119, 
	120, 121, 122, 123, 124, 67, 67, 104, 
	67, 101, 126, 126, 127, 128, 28, 29, 
	30, 31, 32, 28, 29, 30, 31, 32, 
	82, 83, 129, 130, 27, 125, 132, 132, 
	133, 134, 39, 40, 41, 42, 43, 44, 
	45, 46, 47, 48, 49, 50, 51, 52, 
	39, 40, 41, 42, 43, 44, 45, 46, 
	47, 48, 49, 50, 51, 52, 88, 89, 
	135, 136, 131, 137, 138, 137, 138, 35, 
	139, 139, 35, 140, 140, 35, 141, 141, 
	142, 143, 144, 145, 146, 146, 147, 148, 
	55, 85, 88, 85, 86, 87, 89, 149, 
	88, 149, 150, 151, 95, 74, 78, 152, 
	152, 35, 153, 153, 35, 154, 154, 35, 
	155, 155, 156, 157, 158, 159, 160, 160, 
	161, 162, 55, 163, 164, 165, 163, 164, 
	165, 35, 166, 166, 35, 167, 167, 35, 
	168, 168, 35, 169, 169, 170, 171, 172, 
	173, 174, 174, 175, 176, 55, 177, 178, 
	177, 178, 35, 179, 179, 35, 180, 180, 
	35, 181, 181, 182, 183, 184, 185, 186, 
	186, 187, 188, 55, 189, 189, 35, 190, 
	190, 35, 191, 191, 35, 192, 192, 193, 
	194, 195, 196, 197, 197, 198, 199, 55, 
	200, 200, 201, 202, 203, 204, 205, 205, 
	206, 207, 55, 208, 208, 35, 209, 209, 
	35, 210, 211, 35, 212, 35, 213, 213, 
	214, 215, 216, 217, 218, 218, 219, 220, 
	55, 221, 35, 222, 222, 223, 224, 225, 
	226, 227, 227, 228, 229, 55, 230, 230, 
	35, 231, 231, 35, 232, 232, 35, 233, 
	233, 35, 234, 234, 235, 236, 237, 238, 
	239, 239, 240, 241, 55, 242, 243, 242, 
	243, 35, 244, 244, 245, 246, 247, 248, 
	35, 249, 249, 250, 251, 252, 253, 254, 
	255, 256, 257, 258, 259, 260, 261, 262, 
	263, 264, 265, 252, 253, 254, 255, 256, 
	257, 258, 259, 260, 261, 262, 263, 264, 
	265, 266, 267, 35, 243, 243, 35, 268, 
	268, 35, 269, 269, 35, 270, 270, 35, 
	271, 271, 35, 272, 272, 35, 273, 273, 
	35, 274, 274, 275, 276, 277, 278, 279, 
	279, 280, 281, 55, 282, 283, 282, 283, 
	35, 284, 284, 35, 285, 285, 286, 287, 
	288, 289, 290, 290, 291, 292, 55, 293, 
	293, 294, 295, 296, 297, 298, 298, 299, 
	300, 55, 301, 302, 303, 304, 303, 304, 
	35, 305, 35, 306, 306, 307, 308, 309, 
	310, 311, 311, 312, 313, 55, 314, 35, 
	315, 315, 316, 317, 318, 319, 320, 320, 
	321, 322, 55, 323, 323, 35, 324, 324, 
	325, 326, 327, 328, 329, 329, 330, 331, 
	55, 332, 332, 333, 334, 335, 336, 337, 
	337, 338, 339, 55, 340, 341, 340, 341, 
	35, 342, 342, 35, 343, 343, 35, 344, 
	344, 35, 345, 345, 346, 347, 348, 349, 
	350, 350, 351, 352, 55, 353, 353, 354, 
	355, 356, 357, 358, 358, 359, 360, 55, 
	361, 362, 363, 361, 362, 363, 35, 364, 
	364, 35, 365, 365, 35, 366, 366, 35, 
	367, 367, 368, 369, 370, 371, 372, 372, 
	373, 374, 55, 375, 375, 35, 376, 376, 
	377, 378, 379, 380, 381, 381, 382, 383, 
	55, 384, 384, 385, 386, 387, 387, 388, 
	389, 390, 390, 391, 392, 55, 393, 393, 
	35, 394, 394, 395, 396, 397, 398, 399, 
	400, 400, 401, 402, 55, 403, 403, 404, 
	405, 406, 406, 407, 408, 409, 409, 410, 
	411, 55, 412, 412, 35, 413, 413, 35, 
	414, 414, 35, 415, 415, 35, 416, 416, 
	417, 418, 419, 420, 421, 421, 422, 423, 
	55, 424, 424, 35, 425, 425, 35, 426, 
	426, 427, 428, 429, 430, 431, 431, 432, 
	433, 55, 434, 435, 436, 434, 435, 436, 
	35, 437, 437, 438, 439, 440, 441, 442, 
	442, 443, 444, 55, 445, 445, 35, 446, 
	446, 35, 447, 447, 35, 448, 448, 449, 
	450, 451, 452, 453, 453, 454, 455, 55, 
	456, 456, 457, 458, 459, 460, 461, 461, 
	462, 463, 55, 464, 465, 466, 467, 464, 
	465, 466, 467, 35, 468, 468, 35, 469, 
	469, 470, 471, 472, 473, 474, 474, 475, 
	476, 55, 477, 477, 35, 478, 478, 479, 
	480, 481, 482, 483, 483, 484, 485, 55, 
	486, 486, 35, 487, 487, 488, 489, 490, 
	491, 492, 492, 493, 494, 55, 495, 495, 
	35, 496, 496, 35, 497, 497, 35, 498, 
	498, 499, 500, 501, 502, 503, 503, 504, 
	505, 55, 506, 507, 508, 506, 507, 508, 
	35, 509, 509, 35, 510, 510, 35, 511, 
	511, 512, 513, 514, 515, 516, 516, 517, 
	518, 55, 519, 519, 35, 520, 520, 521, 
	522, 523, 524, 525, 525, 526, 527, 55, 
	528, 528, 35, 529, 529, 35, 531, 530, 
	533, 533, 534, 535, 537, 538, 539, 539, 
	540, 541, 536, 532, 266, 267, 35, 542, 
	542, 543, 544, 5, 6, 7, 8, 9, 
	10, 11, 12, 13, 14, 15, 16, 17, 
	18, 5, 6, 7, 8, 9, 10, 11, 
	12, 13, 14, 15, 16, 17, 18, 19, 
	20, 545, 546, 4, 0, 547, 243, 547, 
	243, 35, 548, 548, 549, 550, 551, 552, 
	35, 553, 553, 554, 555, 557, 558, 559, 
	560, 561, 562, 563, 564, 565, 566, 567, 
	568, 569, 570, 557, 558, 559, 560, 561, 
	562, 563, 564, 565, 566, 567, 568, 569, 
	570, 571, 572, 556, 530, 573, 573, 574, 
	575, 577, 578, 579, 580, 581, 577, 578, 
	579, 580, 581, 582, 583, 576, 23, 573, 
	573, 574, 575, 582, 583, 584, 23, 586, 
	587, 588, 589, 590, 586, 587, 588, 589, 
	590, 585, 23, 591, 591, 592, 593, 595, 
	596, 594, 23, 571, 572, 35, 19, 20, 
	598, 599, 600, 598, 599, 600, 597, 601, 
	601, 597, 602, 602, 597, 603, 603, 597, 
	604, 604, 597, 605, 605, 597, 606, 606, 
	597, 607, 607, 607, 607, 597, 609, 609, 
	610, 611, 612, 613, 608, 614, 614, 615, 
	616, 5, 6, 7, 8, 9, 10, 11, 
	12, 13, 14, 15, 16, 17, 18, 5, 
	6, 7, 8, 9, 10, 11, 12, 13, 
	14, 15, 16, 17, 18, 617, 618, 4, 
	530, 617, 618, 619, 621, 621, 622, 623, 
	28, 29, 30, 31, 32, 28, 29, 30, 
	31, 32, 624, 625, 27, 620, 626, 626, 
	627, 628, 5, 6, 7, 8, 9, 10, 
	11, 12, 13, 14, 15, 16, 17, 18, 
	5, 6, 7, 8, 9, 10, 11, 12, 
	13, 14, 15, 16, 17, 18, 629, 630, 
	4, 530, 629, 630, 35, 24, 24, 25, 
	26, 33, 34, 631, 23, 633, 634, 635, 
	636, 637, 633, 634, 635, 636, 637, 632, 
	23, 638, 638, 639, 640, 642, 643, 641, 
	23, 53, 54, 35, 645, 645, 646, 647, 
	59, 60, 61, 59, 60, 61, 648, 649, 
	64, 64, 650, 651, 644, 652, 652, 653, 
	654, 67, 105, 106, 107, 108, 109, 110, 
	111, 112, 113, 114, 115, 116, 117, 118, 
	72, 105, 106, 107, 108, 109, 110, 111, 
	112, 113, 114, 115, 116, 117, 118, 655, 
	656, 75, 75, 657, 658, 67, 67, 104, 
	67, 532, 660, 660, 661, 662, 59, 60, 
	61, 59, 60, 61, 82, 83, 663, 664, 
	665, 666, 667, 668, 659, 669, 669, 670, 
	671, 72, 88, 89, 99, 95, 96, 97, 
	672, 98, 67, 67, 67, 78, 673, 673, 
	674, 675, 5, 6, 7, 8, 9, 10, 
	11, 12, 13, 14, 15, 16, 17, 18, 
	5, 6, 7, 8, 9, 10, 11, 12, 
	13, 14, 15, 16, 17, 18, 88, 89, 
	676, 677, 4, 0, 88, 676, 89, 677, 
	84, 678, 678, 679, 680, 88, 89, 88, 
	89, 89, 89, 89, 78, 678, 678, 679, 
	680, 88, 89, 88, 89, 89, 89, 89, 
	84, 681, 681, 682, 683, 67, 105, 106, 
	107, 108, 109, 110, 111, 112, 113, 114, 
	115, 116, 117, 118, 72, 105, 106, 107, 
	108, 109, 110, 111, 112, 113, 114, 115, 
	116, 117, 118, 88, 89, 676, 684, 96, 
	97, 676, 685, 67, 67, 104, 67, 101, 
	79, 79, 80, 81, 137, 138, 137, 138, 
	82, 83, 82, 83, 659, 79, 79, 80, 
	81, 163, 164, 165, 163, 164, 165, 82, 
	83, 82, 83, 659, 79, 79, 80, 81, 
	208, 208, 82, 83, 82, 83, 659, 79, 
	79, 80, 81, 230, 230, 82, 83, 82, 
	83, 659, 79, 79, 80, 81, 547, 243, 
	547, 243, 82, 83, 82, 83, 659, 79, 
	79, 80, 81, 282, 283, 282, 283, 82, 
	83, 82, 83, 659, 79, 79, 80, 81, 
	301, 302, 303, 304, 303, 304, 82, 83, 
	82, 83, 659, 79, 79, 80, 81, 340, 
	341, 340, 341, 82, 83, 82, 83, 659, 
	79, 79, 80, 81, 361, 362, 363, 361, 
	362, 363, 82, 83, 82, 83, 659, 79, 
	79, 80, 81, 424, 424, 82, 83, 82, 
	83, 659, 79, 79, 80, 81, 434, 435, 
	436, 434, 435, 436, 82, 83, 82, 83, 
	659, 79, 79, 80, 81, 464, 465, 466, 
	467, 464, 465, 466, 467, 82, 83, 82, 
	83, 659, 79, 79, 80, 81, 506, 507, 
	508, 506, 507, 508, 82, 83, 82, 83, 
	659, 88, 686, 89, 684, 78, 79, 79, 
	80, 81, 598, 599, 600, 598, 599, 600, 
	82, 83, 82, 83, 687, 688, 688, 597, 
	689, 689, 597, 690, 690, 597, 691, 691, 
	597, 692, 692, 597, 693, 693, 693, 693, 
	597, 694, 694, 597, 695, 695, 597, 696, 
	696, 696, 696, 597, 698, 698, 699, 700, 
	82, 83, 701, 702, 697, 704, 704, 705, 
	706, 28, 29, 30, 31, 32, 28, 29, 
	30, 31, 32, 82, 83, 707, 708, 27, 
	703, 709, 709, 710, 711, 5, 6, 7, 
	8, 9, 10, 11, 12, 13, 14, 15, 
	16, 17, 18, 5, 6, 7, 8, 9, 
	10, 11, 12, 13, 14, 15, 16, 17, 
	18, 88, 89, 712, 713, 4, 0, 88, 
	712, 89, 713, 131, 715, 715, 716, 717, 
	59, 60, 61, 59, 60, 61, 82, 83, 
	718, 719, 665, 666, 720, 721, 714, 722, 
	722, 723, 724, 67, 105, 106, 107, 108, 
	109, 110, 111, 112, 113, 114, 115, 116, 
	117, 118, 72, 105, 106, 107, 108, 109, 
	110, 111, 112, 113, 114, 115, 116, 117, 
	118, 88, 89, 686, 684, 96, 97, 725, 
	685, 67, 67, 104, 67, 101, 698, 698, 
	699, 700, 137, 138, 137, 138, 82, 83, 
	701, 702, 714, 698, 698, 699, 700, 163, 
	164, 165, 163, 164, 165, 82, 83, 701, 
	702, 714, 698, 698, 699, 700, 208, 208, 
	82, 83, 701, 702, 714, 698, 698, 699, 
	700, 230, 230, 82, 83, 701, 702, 714, 
	698, 698, 699, 700, 547, 243, 547, 243, 
	82, 83, 701, 702, 714, 698, 698, 699, 
	700, 282, 283, 282, 283, 82, 83, 701, 
	702, 714, 698, 698, 699, 700, 301, 302, 
	303, 304, 303, 304, 82, 83, 701, 702, 
	714, 698, 698, 699, 700, 340, 341, 340, 
	341, 82, 83, 701, 702, 714, 698, 698, 
	699, 700, 361, 362, 363, 361, 362, 363, 
	82, 83, 701, 702, 714, 698, 698, 699, 
	700, 424, 424, 82, 83, 701, 702, 714, 
	698, 698, 699, 700, 434, 435, 436, 434, 
	435, 436, 82, 83, 701, 702, 714, 698, 
	698, 699, 700, 464, 465, 466, 467, 464, 
	465, 466, 467, 82, 83, 701, 702, 714, 
	698, 698, 699, 700, 506, 507, 508, 506, 
	507, 508, 82, 83, 701, 702, 714, 726, 
	726, 91, 728, 729, 90, 90, 730, 731, 
	90, 90, 90, 727, 732, 732, 733, 734, 
	5, 6, 7, 8, 9, 10, 11, 12, 
	13, 14, 15, 16, 17, 18, 5, 6, 
	7, 8, 9, 10, 11, 12, 13, 14, 
	15, 16, 17, 18, 82, 83, 735, 736, 
	4, 101, 79, 82, 79, 80, 81, 83, 
	737, 735, 737, 738, 739, 736, 618, 78, 
	85, 88, 85, 86, 87, 89, 740, 676, 
	740, 741, 742, 677, 618, 84, 678, 88, 
	678, 679, 680, 743, 676, 743, 744, 745, 
	89, 684, 78, 746, 746, 35, 747, 747, 
	35, 748, 748, 749, 750, 751, 752, 753, 
	753, 754, 755, 55, 756, 756, 35, 757, 
	757, 35, 758, 758, 35, 759, 759, 760, 
	761, 762, 763, 764, 764, 765, 766, 55, 
	767, 767, 35, 768, 768, 769, 770, 771, 
	772, 773, 773, 774, 775, 55, 19, 119, 
	20, 120, 78, 776, 776, 777, 778, 88, 
	89, 88, 89, 20, 20, 20, 78, 776, 
	776, 777, 778, 88, 89, 88, 89, 20, 
	20, 20, 84, 776, 88, 776, 777, 778, 
	89, 779, 676, 779, 780, 781, 684, 20, 
	120, 78, 678, 88, 678, 679, 680, 782, 
	88, 782, 783, 784, 89, 95, 78, 655, 
	656, 69, 85, 88, 85, 86, 87, 89, 
	785, 676, 785, 786, 787, 684, 656, 78, 
	609, 609, 610, 611, 137, 138, 137, 138, 
	612, 613, 788, 609, 609, 610, 611, 163, 
	164, 165, 163, 164, 165, 612, 613, 788, 
	609, 609, 610, 611, 208, 208, 612, 613, 
	788, 609, 609, 610, 611, 230, 230, 612, 
	613, 788, 609, 609, 610, 611, 547, 243, 
	547, 243, 612, 613, 788, 609, 609, 610, 
	611, 282, 283, 282, 283, 612, 613, 788, 
	609, 609, 610, 611, 301, 302, 303, 304, 
	303, 304, 612, 613, 788, 609, 609, 610, 
	611, 340, 341, 340, 341, 612, 613, 788, 
	609, 609, 610, 611, 361, 362, 363, 361, 
	362, 363, 612, 613, 788, 609, 609, 610, 
	611, 424, 424, 612, 613, 788, 609, 609, 
	610, 611, 434, 435, 436, 434, 435, 436, 
	612, 613, 788, 609, 609, 610, 611, 464, 
	465, 466, 467, 464, 465, 466, 467, 612, 
	613, 788, 609, 609, 610, 611, 506, 507, 
	508, 506, 507, 508, 612, 613, 788, 19, 
	21, 20, 22, 84, 19, 545, 20, 546, 
	131, 609, 609, 610, 611, 242, 243, 242, 
	243, 612, 613, 788, 88, 135, 89, 136, 
	131, 790, 789, 792, 791, 794, 796, 797, 
	798, 795, 795, 795, 795, 793, 799, 800, 
	799, 799, 799, 793, 801, 802, 803, 801, 
	804, 803, 801, 801, 803, 803, 803, 793, 
	799, 800, 799, 806, 805, 799, 799, 805, 
	805, 805, 793, 807, 805, 809, 808, 810, 
	808, 811, 813, 811, 814, 812, 811, 811, 
	812, 812, 812, 808, 801, 801, 801, 801, 
	793, 815, 815, 815, 815, 793, 818, 819, 
	817, 817, 816, 820, 817, 820, 819, 820, 
	817, 821, 817, 816, 822, 817, 824, 823, 
	825, 823, 826, 827, 826, 829, 826, 827, 
	828, 827, 823, 830, 831, 832, 830, 830, 
	816, 820, 820, 820, 820, 833, 834, 830, 
	835, 823, 836, 823, 837, 838, 839, 837, 
	837, 823, 841, 841, 842, 843, 844, 845, 
	840, 841, 841, 842, 843, 844, 845, 847, 
	846, 849, 849, 850, 851, 853, 854, 855, 
	856, 857, 853, 854, 855, 856, 857, 858, 
	859, 858, 859, 852, 848, 849, 849, 850, 
	851, 858, 859, 858, 859, 860, 858, 859, 
	849, 849, 850, 851, 858, 859, 858, 859, 
	861, 848, 864, 865, 866, 867, 868, 864, 
	865, 866, 867, 868, 863, 862, 869, 869, 
	870, 871, 873, 874, 873, 874, 872, 848, 
	844, 845, 840, 876, 876, 877, 878, 879, 
	880, 875, 876, 876, 877, 878, 881, 883, 
	884, 882, 879, 880, 882, 882, 882, 875, 
	885, 875, 887, 887, 888, 889, 890, 891, 
	892, 891, 893, 894, 893, 894, 891, 891, 
	891, 886, 895, 895, 896, 897, 898, 899, 
	898, 899, 886, 898, 899, 885, 901, 900, 
	900, 900, 900, 875, 902, 900, 904, 903, 
	905, 903, 907, 908, 906, 906, 906, 906, 
	903, 887, 887, 888, 889, 893, 894, 893, 
	894, 886, 879, 880, 875, 909, 909, 910, 
	911, 912, 913, 619, 909, 909, 910, 911, 
	912, 913, 914, 914, 914, 915, 917, 917, 
	918, 919, 920, 921, 920, 922, 916, 924, 
	924, 925, 926, 927, 929, 930, 928, 931, 
	932, 931, 933, 928, 928, 928, 923, 935, 
	934, 936, 936, 937, 938, 939, 940, 941, 
	940, 942, 943, 942, 943, 940, 940, 940, 
	923, 944, 944, 945, 946, 931, 932, 931, 
	932, 84, 931, 932, 935, 948, 947, 947, 
	947, 947, 934, 949, 947, 951, 950, 952, 
	950, 954, 955, 953, 953, 953, 953, 950, 
	936, 936, 937, 938, 942, 943, 942, 943, 
	923, 931, 956, 932, 933, 84, 957, 957, 
	958, 959, 927, 929, 930, 928, 960, 961, 
	928, 928, 928, 934, 960, 961, 619, 912, 
	913, 619, 963, 963, 963, 963, 962, 964, 
	964, 964, 964, 962, 965, 966, 965, 965, 
	965, 962, 967, 968, 967, 967, 967, 962, 
	968, 968, 969, 970, 963, 971, 972, 973, 
	973, 974, 975, 963, 963, 963, 962, 971, 
	972, 962, 968, 962, 977, 977, 978, 979, 
	980, 981, 982, 982, 983, 984, 976, 985, 
	985, 986, 987, 988, 989, 990, 991, 992, 
	993, 994, 995, 996, 997, 998, 999, 1000, 
	1001, 988, 989, 990, 991, 992, 993, 994, 
	995, 996, 997, 998, 999, 1000, 1001, 1002, 
	1003, 1004, 1004, 1005, 1006, 976, 1007, 1007, 
	1008, 1009, 1010, 1011, 1012, 1010, 1011, 1012, 
	1013, 1014, 1015, 1015, 1016, 1017, 976, 1018, 
	1018, 976, 1019, 1019, 976, 1020, 1020, 1021, 
	1022, 1023, 1024, 1025, 1025, 1026, 1027, 976, 
	1002, 1003, 976, 1028, 1029, 1028, 1029, 976, 
	1030, 1030, 976, 1031, 1031, 976, 1032, 1032, 
	1033, 1034, 1035, 1036, 1037, 1037, 1038, 1039, 
	976, 1040, 1040, 976, 1041, 1041, 976, 1042, 
	1042, 976, 1043, 1043, 1044, 1045, 1046, 1047, 
	1048, 1048, 1049, 1050, 976, 1051, 1052, 1053, 
	1051, 1052, 1053, 976, 1054, 1054, 976, 1055, 
	1055, 976, 1056, 1056, 976, 1057, 1057, 1058, 
	1059, 1060, 1061, 1062, 1062, 1063, 1064, 976, 
	1065, 1066, 1065, 1066, 976, 1067, 1067, 976, 
	1068, 1068, 976, 1069, 1069, 1070, 1071, 1072, 
	1073, 1074, 1074, 1075, 1076, 976, 1077, 1077, 
	976, 1078, 1078, 976, 1079, 1079, 976, 1080, 
	1080, 1081, 1082, 1083, 1084, 1085, 1085, 1086, 
	1087, 976, 1088, 1088, 1089, 1090, 1091, 1092, 
	1093, 1093, 1094, 1095, 976, 1096, 1096, 976, 
	1097, 1097, 976, 1098, 1099, 976, 1100, 976, 
	1101, 1101, 1102, 1103, 1104, 1105, 1106, 1106, 
	1107, 1108, 976, 1109, 976, 1110, 1110, 1111, 
	1112, 1113, 1114, 1115, 1115, 1116, 1117, 976, 
	1118, 1118, 976, 1119, 1119, 976, 1120, 1120, 
	976, 1121, 1121, 976, 1122, 1122, 1123, 1124, 
	1125, 1126, 1127, 1127, 1128, 1129, 976, 1130, 
	1130, 976, 1131, 1131, 976, 1132, 1132, 976, 
	1133, 1133, 976, 1134, 1134, 976, 1135, 1135, 
	976, 1136, 1136, 976, 1137, 1137, 1138, 1139, 
	1140, 1141, 1142, 1142, 1143, 1144, 976, 1145, 
	1146, 1145, 1146, 976, 1147, 1147, 976, 1148, 
	1148, 1149, 1150, 1151, 1152, 1153, 1153, 1154, 
	1155, 976, 1156, 1156, 1157, 1158, 1159, 1160, 
	1161, 1161, 1162, 1163, 976, 1164, 1165, 1166, 
	1167, 1166, 1167, 976, 1168, 976, 1169, 1169, 
	1170, 1171, 1172, 1173, 1174, 1174, 1175, 1176, 
	976, 1177, 976, 1178, 1178, 1179, 1180, 1181, 
	1182, 1183, 1183, 1184, 1185, 976, 1186, 1186, 
	976, 1187, 1187, 1188, 1189, 1190, 1191, 1192, 
	1192, 1193, 1194, 976, 1195, 1195, 1196, 1197, 
	1198, 1199, 1200, 1200, 1201, 1202, 976, 1203, 
	1204, 1203, 1204, 976, 1205, 1205, 976, 1206, 
	1206, 976, 1207, 1207, 976, 1208, 1208, 1209, 
	1210, 1211, 1212, 1213, 1213, 1214, 1215, 976, 
	1216, 1216, 1217, 1218, 1219, 1220, 1221, 1221, 
	1222, 1223, 976, 1224, 1225, 1226, 1224, 1225, 
	1226, 976, 1227, 1227, 976, 1228, 1228, 976, 
	1229, 1229, 976, 1230, 1230, 1231, 1232, 1233, 
	1234, 1235, 1235, 1236, 1237, 976, 1238, 1238, 
	976, 1239, 1239, 1240, 1241, 1242, 1243, 1244, 
	1244, 1245, 1246, 976, 1247, 1247, 1248, 1249, 
	1250, 1250, 1251, 1252, 1253, 1253, 1254, 1255, 
	976, 1256, 1256, 976, 1257, 1257, 1258, 1259, 
	1260, 1261, 1262, 1263, 1263, 1264, 1265, 976, 
	1266, 1266, 1267, 1268, 1269, 1269, 1270, 1271, 
	1272, 1272, 1273, 1274, 976, 1275, 1275, 976, 
	1276, 1276, 976, 1277, 1277, 976, 1278, 1278, 
	976, 1279, 1279, 1280, 1281, 1282, 1283, 1284, 
	1284, 1285, 1286, 976, 1287, 1287, 976, 1288, 
	1288, 976, 1289, 1289, 1290, 1291, 1292, 1293, 
	1294, 1294, 1295, 1296, 976, 1297, 1298, 1299, 
	1297, 1298, 1299, 976, 1300, 1300, 1301, 1302, 
	1303, 1304, 1305, 1305, 1306, 1307, 976, 1308, 
	1308, 976, 1309, 1309, 976, 1310, 1310, 976, 
	1311, 1311, 1312, 1313, 1314, 1315, 1316, 1316, 
	1317, 1318, 976, 1319, 1319, 1320, 1321, 1322, 
	1323, 1324, 1324, 1325, 1326, 976, 1327, 1328, 
	1329, 1330, 1327, 1328, 1329, 1330, 976, 1331, 
	1331, 976, 1332, 1332, 1333, 1334, 1335, 1336, 
	1337, 1337, 1338, 1339, 976, 1340, 1340, 976, 
	1341, 1341, 1342, 1343, 1344, 1345, 1346, 1346, 
	1347, 1348, 976, 1349, 1349, 976, 1350, 1350, 
	1351, 1352, 1353, 1354, 1355, 1355, 1356, 1357, 
	976, 1358, 1358, 976, 1359, 1359, 976, 1360, 
	1360, 976, 1361, 1361, 1362, 1363, 1364, 1365, 
	1366, 1366, 1367, 1368, 976, 1369, 1370, 1371, 
	1369, 1370, 1371, 976, 1372, 1372, 976, 1373, 
	1373, 976, 1374, 1374, 1375, 1376, 1377, 1378, 
	1379, 1379, 1380, 1381, 976, 1382, 1382, 976, 
	1383, 1383, 1384, 1385, 1386, 1387, 1388, 1388, 
	1389, 1390, 976, 1391, 1391, 976, 1392, 1392, 
	976, 1393, 976, 1394, 1394, 1395, 1396, 1398, 
	1399, 1400, 1400, 1401, 1402, 1397, 976, 1403, 
	1403, 976, 1404, 1404, 976, 1405, 1405, 976, 
	1406, 1406, 1407, 1408, 1409, 1410, 1411, 1411, 
	1412, 1413, 976, 1414, 1414, 976, 1415, 1415, 
	1416, 1417, 1418, 1419, 1420, 1420, 1421, 1422, 
	976, 1424, 1424, 1425, 1426, 1427, 1428, 1423, 
	1424, 1424, 1425, 1426, 1427, 1428, 1430, 1429, 
	1431, 1431, 1432, 1433, 1435, 1436, 1434, 1429, 
	1438, 1438, 1439, 1440, 1442, 1443, 1441, 1441, 
	1441, 1437, 1444, 1444, 1444, 1437, 1445, 1445, 
	1446, 1447, 1448, 1449, 1450, 1450, 1451, 1452, 
	1441, 1441, 1441, 1437, 1448, 1449, 1437, 1442, 
	1443, 1423, 1427, 1428, 1423, 1453, 1453, 1454, 
	1455, 1456, 1457, 1423, 1453, 1453, 1454, 1455, 
	1458, 1456, 1457, 1459, 1429, 1460, 1460, 1461, 
	1462, 1464, 1464, 1465, 1466, 1463, 1429, 1467, 
	1467, 1468, 1469, 1471, 1472, 1470, 1470, 1470, 
	1437, 1473, 1473, 1473, 1437, 1474, 1474, 1475, 
	1476, 1477, 1478, 1479, 1479, 1480, 1481, 1470, 
	1470, 1470, 1437, 1477, 1478, 1437, 1471, 1472, 
	1423, 1482, 1482, 1483, 1484, 1485, 1486, 1463, 
	1429, 1456, 1457, 1423, 1489, 1490, 1491, 1492, 
	1493, 1489, 1490, 1491, 1492, 1493, 1488, 1487, 
	1494, 1494, 1494, 1494, 1495, 1487, 1496, 1497, 
	1496, 1497, 1487, 1498, 1498, 1498, 1498, 1487, 
	1499, 1499, 1487, 1500, 1501, 1500, 1500, 1500, 
	1487, 1502, 1502, 1487, 1503, 1503, 1487, 1504, 
	1504, 1487, 1505, 1505, 1487, 1506, 1487, 1507, 
	1487, 1508, 1508, 1487, 1509, 1509, 1487, 1510, 
	1510, 1487, 1511, 1487, 1512, 1512, 1512, 1512, 
	1487, 1513, 1513, 1487, 1514, 1515, 1514, 1515, 
	1487, 1516, 1487, 1517, 1517, 1487, 1518, 1518, 
	1487, 1519, 1519, 1487, 1520, 1520, 1487, 1521, 
	1521, 1521, 1521, 1487, 1522, 1522, 1487, 1523, 
	1523, 1487, 1524, 1524, 1487, 1525, 1526, 1487, 
	1527, 1487, 1528, 1487, 1529, 1529, 1487, 1530, 
	1530, 1487, 1531, 1531, 1487, 1532, 1487, 1533, 
	1487, 1534, 1487, 1535, 1535, 1535, 1535, 1487, 
	1536, 1487, 1537, 1487, 1538, 1538, 1487, 1539, 
	1539, 1487, 1540, 1540, 1487, 1541, 1487, 1542, 
	1487, 1543, 1487, 1544, 1544, 1544, 1544, 1487, 
	1545, 1545, 1487, 1546, 1546, 1487, 1547, 1547, 
	1487, 1548, 1548, 1487, 1549, 1549, 1487, 1550, 
	1550, 1487, 1551, 1551, 1487, 1552, 1552, 1552, 
	1552, 1487, 1553, 1553, 1487, 1554, 1554, 1487, 
	1555, 1555, 1487, 1556, 1556, 1487, 1557, 1557, 
	1487, 1558, 1558, 1487, 1559, 1560, 1559, 1560, 
	1487, 1561, 1561, 1487, 1562, 1562, 1487, 1563, 
	1563, 1563, 1563, 1487, 1564, 1564, 1487, 1565, 
	1565, 1487, 1566, 1566, 1566, 1566, 1487, 1567, 
	1567, 1487, 1568, 1568, 1487, 1569, 1570, 1569, 
	1570, 1487, 1571, 1571, 1487, 1572, 1487, 1573, 
	1573, 1573, 1573, 1487, 1574, 1574, 1487, 1575, 
	1575, 1487, 1576, 1577, 1578, 1487, 1579, 1580, 
	1579, 1579, 1579, 1487, 1581, 1581, 1487, 1582, 
	1582, 1487, 1583, 1583, 1487, 1584, 1584, 1487, 
	1585, 1487, 1586, 1487, 1587, 1587, 1487, 1588, 
	1588, 1487, 1589, 1589, 1487, 1590, 1487, 1591, 
	1591, 1591, 1591, 1487, 1592, 1487, 1593, 1487, 
	1594, 1594, 1594, 1594, 1487, 1595, 1487, 1596, 
	1487, 1597, 1597, 1597, 1597, 1487, 1600, 1601, 
	1602, 1603, 1604, 1605, 1600, 1601, 1602, 1603, 
	1604, 1605, 1599, 1598, 1606, 1606, 1606, 1606, 
	1607, 1598, 1608, 1608, 1598, 1609, 1609, 1598, 
	1610, 1610, 1598, 1611, 1611, 1598, 1612, 1612, 
	1598, 1613, 1613, 1613, 1613, 1598, 1614, 1615, 
	1616, 1614, 1615, 1616, 1598, 1617, 1617, 1598, 
	1618, 1618, 1598, 1619, 1619, 1598, 1620, 1620, 
	1598, 1621, 1621, 1598, 1622, 1622, 1622, 1622, 
	1598, 1623, 1624, 1623, 1624, 1598, 1625, 1625, 
	1598, 1626, 1626, 1626, 1626, 1598, 1627, 1627, 
	1598, 1628, 1628, 1598, 1629, 1629, 1629, 1629, 
	1598, 1630, 1630, 1598, 1631, 1631, 1598, 1632, 
	1632, 1598, 1633, 1633, 1633, 1633, 1598, 1634, 
	1634, 1598, 1635, 1635, 1598, 1636, 1636, 1636, 
	1636, 1598, 1637, 1638, 1637, 1638, 1598, 1639, 
	1639, 1598, 1640, 1640, 1640, 1640, 1598, 1641, 
	1641, 1598, 1642, 1642, 1598, 1643, 1643, 1643, 
	1643, 1598, 1644, 1644, 1598, 1645, 1645, 1598, 
	1646, 1646, 1598, 1647, 1647, 1647, 1647, 1598, 
	1648, 1648, 1598, 1649, 1649, 1598, 1650, 1650, 
	1650, 1650, 1598, 1652, 1652, 1651, 1653, 1654, 
	1653, 1653, 1653, 1654, 1651, 1655, 1655, 1655, 
	1655, 1655, 1655, 69, 1656, 1656, 1656, 1656, 
	69, 1657, 1657, 1657, 1657, 1657, 1657, 69, 
	1658, 1658, 1659, 1660, 1661, 1662, 69, 1663, 
	1663, 1664, 1665, 1666, 1666, 1666, 1667, 1668, 
	1666, 1666, 1666, 69, 1669, 1669, 1670, 1671, 
	1672, 1673, 69, 1675, 1675, 1676, 1677, 1679, 
	1680, 1678, 1674, 1681, 1681, 1682, 1683, 1685, 
	1686, 1684, 1674, 1687, 1687, 1688, 1689, 1691, 
	1692, 1690, 1674, 1694, 1694, 1695, 1696, 1698, 
	1699, 1700, 1701, 1702, 1698, 1699, 1700, 1701, 
	1702, 1703, 1704, 1697, 1693, 1705, 1705, 1706, 
	1707, 1709, 1710, 1708, 1674, 1711, 1711, 1712, 
	1713, 1715, 1716, 1717, 1718, 1719, 1715, 1716, 
	1717, 1718, 1719, 1720, 1721, 1714, 1693, 1722, 
	1722, 1723, 1724, 1726, 1727, 1725, 1674, 1728, 
	1728, 1729, 1730, 1732, 1733, 1734, 1735, 1736, 
	1732, 1733, 1734, 1735, 1736, 1737, 1738, 1731, 
	1693, 1739, 1739, 1740, 1741, 1743, 1744, 1742, 
	1674, 1745, 1745, 1747, 1748, 1749, 1750, 1751, 
	1747, 1748, 1749, 1750, 1751, 1745, 1745, 1746, 
	1693, 1745, 1745, 1745, 1745, 1752, 1693, 1754, 
	1755, 1756, 1757, 1758, 1754, 1755, 1756, 1757, 
	1758, 1753, 1693, 1759, 1759, 1759, 1759, 1760, 
	1693, 1743, 1744, 69, 1728, 1728, 1729, 1730, 
	1737, 1738, 1761, 1693, 1763, 1764, 1765, 1766, 
	1767, 1763, 1764, 1765, 1766, 1767, 1762, 1693, 
	1768, 1768, 1769, 1770, 1772, 1773, 1771, 1693, 
	1726, 1727, 69, 1711, 1711, 1712, 1713, 1720, 
	1721, 1774, 1693, 1776, 1777, 1778, 1779, 1780, 
	1776, 1777, 1778, 1779, 1780, 1775, 1693, 1781, 
	1781, 1782, 1783, 1785, 1786, 1784, 1693, 1709, 
	1710, 69, 1694, 1694, 1695, 1696, 1703, 1704, 
	1787, 1693, 1789, 1790, 1791, 1792, 1793, 1789, 
	1790, 1791, 1792, 1793, 1788, 1693, 1794, 1794, 
	1795, 1796, 1798, 1799, 1797, 1693, 1691, 1692, 
	69, 1679, 1680, 69, 1667, 1668, 69, 69, 
	69, 69, 69, 1800, 1801, 1801, 1802, 1803, 
	1804, 1805, 69, 1807, 1807, 1808, 1809, 1810, 
	1811, 1806, 1806, 1806, 69, 1812, 1812, 1812, 
	1812, 69, 1810, 1811, 69, 1813, 1813, 1813, 
	1813, 1813, 1813, 69, 1814, 1814, 1815, 1816, 
	1817, 1818, 69, 1819, 1819, 1820, 1821, 1822, 
	1822, 1822, 1823, 1824, 1822, 1822, 1822, 69, 
	1825, 1825, 1825, 1825, 69, 1823, 1824, 69, 
	1826, 1674, 1827, 1827, 1828, 1829, 1831, 1832, 
	1830, 1674, 1833, 1833, 1834, 1835, 1836, 1836, 
	1836, 1837, 1838, 1836, 1836, 1836, 69, 1839, 
	1839, 1839, 1839, 69, 1837, 1838, 69, 69, 
	69, 69, 69, 1840, 1841, 1841, 1842, 1843, 
	1844, 1845, 1846, 1846, 1847, 1848, 69, 1849, 
	1849, 1850, 1851, 1852, 1853, 1854, 1854, 1855, 
	1856, 1840, 1840, 1840, 69, 1852, 1853, 69, 
	1857, 1857, 1857, 1857, 1651, 1858, 1859, 1858, 
	1858, 1858, 1859, 1859, 1859, 1651, 1861, 1860, 
	1862, 1862, 1863, 1864, 1866, 1867, 1865, 1860, 
	1868, 1868, 1869, 1870, 1872, 1873, 1874, 1875, 
	1871, 1860, 1876, 1876, 1877, 1878, 1880, 1881, 
	1879, 1860, 1882, 1882, 1883, 1884, 1872, 1873, 
	1886, 1887, 1885, 1860, 1888, 1888, 1889, 1890, 
	1891, 1893, 1894, 1892, 1860, 1895, 1895, 1896, 
	1897, 1872, 1873, 1898, 1899, 1860, 1900, 1900, 
	1901, 1902, 1903, 1904, 1860, 1900, 1900, 1901, 
	1902, 1903, 1904, 1905, 1860, 1906, 1906, 1907, 
	1908, 1910, 1911, 1909, 1860, 1912, 1912, 1913, 
	1914, 1916, 1917, 1918, 1919, 1915, 1860, 1920, 
	1920, 1921, 1922, 1924, 1925, 1923, 1860, 1926, 
	1926, 1927, 1928, 1916, 1917, 1930, 1931, 1929, 
	1860, 1932, 1932, 1933, 1934, 1935, 1937, 1938, 
	1936, 1860, 1939, 1939, 1940, 1941, 1916, 1917, 
	1942, 1943, 1860, 1944, 1944, 1945, 1946, 1947, 
	1948, 1860, 1944, 1944, 1945, 1946, 1949, 1947, 
	1948, 1950, 1860, 1951, 1860, 1952, 1952, 1953, 
	1954, 1955, 1957, 1958, 1959, 1960, 1960, 1961, 
	1962, 1956, 1860, 1963, 1963, 1964, 1965, 1967, 
	1968, 1969, 1969, 1970, 1971, 1966, 1860, 1972, 
	1972, 1973, 1974, 1975, 1977, 1978, 1979, 1980, 
	1980, 1981, 1982, 1976, 1860, 1983, 1983, 1984, 
	1985, 1987, 1988, 1969, 1969, 1989, 1990, 1986, 
	1860, 1991, 1991, 1992, 1993, 1994, 1996, 1997, 
	1998, 1999, 1999, 2000, 2001, 1995, 1860, 2002, 
	2002, 2003, 2004, 2006, 2007, 1969, 1969, 2008, 
	2009, 2005, 1860, 2010, 2010, 2011, 2012, 2013, 
	2010, 2015, 2016, 2017, 2017, 2018, 2019, 2014, 
	1860, 2020, 2020, 2021, 2022, 2023, 2024, 1969, 
	1969, 2025, 2026, 1860, 2023, 2024, 1860, 2027, 
	2027, 2028, 2029, 2027, 2031, 2032, 2033, 2033, 
	2034, 2035, 2030, 1860, 2010, 2010, 2011, 2012, 
	2010, 2015, 2016, 2017, 2017, 2018, 2019, 2036, 
	1860, 2006, 2007, 1860, 2037, 2037, 2038, 2039, 
	2041, 2042, 2043, 2044, 2044, 2045, 2046, 2040, 
	1860, 1991, 1991, 1992, 1993, 1996, 1997, 1998, 
	1999, 1999, 2000, 2001, 2047, 1860, 2002, 2002, 
	2003, 2004, 2006, 2007, 1969, 1969, 2008, 2009, 
	1860, 1987, 1988, 1860, 2048, 2048, 2049, 2050, 
	2052, 2053, 2054, 2055, 2055, 2056, 2057, 2051, 
	1860, 1972, 1972, 1973, 1974, 1977, 1978, 1979, 
	1980, 1980, 1981, 1982, 2058, 1860, 1983, 1983, 
	1984, 1985, 1987, 1988, 1969, 1969, 1989, 1990, 
	1860, 1967, 1968, 1860, 2059, 2059, 2060, 2061, 
	2063, 2064, 2065, 2066, 2066, 2067, 2068, 2062, 
	1860, 1952, 1952, 1953, 1954, 1957, 1958, 1959, 
	1960, 1960, 1961, 1962, 2069, 1860, 1963, 1963, 
	1964, 1965, 1967, 1968, 1969, 1969, 1970, 1971, 
	1860, 1947, 1948, 1860, 2070, 2070, 2071, 2072, 
	2073, 2074, 1860, 1942, 1943, 1860, 2075, 2075, 
	2076, 2077, 2079, 2080, 2078, 1860, 1932, 1932, 
	1933, 1934, 1937, 1938, 2081, 1860, 1930, 1931, 
	1860, 1918, 1919, 1860, 1903, 1904, 1860, 2082, 
	2082, 2083, 2084, 2085, 2086, 1860, 1898, 1899, 
	1860, 2087, 2087, 2088, 2089, 2091, 2092, 2090, 
	1860, 1888, 1888, 1889, 1890, 1893, 1894, 2093, 
	1860, 1886, 1887, 1860, 1874, 1875, 1860, 2094, 
	1674, 2095, 2095, 2096, 2097, 2099, 2100, 2098, 
	1674, 2101, 2101, 2102, 2103, 2105, 2106, 2104, 
	1674, 2107, 2107, 2108, 2109, 2111, 2112, 2110, 
	1674, 2113, 2113, 2114, 2115, 2117, 2118, 2116, 
	1674, 2119, 2119, 2120, 2121, 2123, 2124, 2122, 
	1674, 2125, 2125, 2126, 2127, 2128, 2128, 2128, 
	2129, 2130, 2128, 2128, 2128, 69, 2131, 2131, 
	2131, 2131, 69, 2129, 2130, 69, 2117, 2118, 
	69, 2105, 2106, 69, 2132, 1674, 2133, 2133, 
	2134, 2135, 2137, 2138, 2136, 1674, 2139, 2139, 
	2140, 2141, 2143, 2144, 2142, 1674, 2145, 2145, 
	2146, 2147, 2149, 2150, 2148, 1674, 2152, 2152, 
	2153, 2154, 2155, 2156, 2151, 2151, 2151, 69, 
	2157, 2157, 2158, 2159, 2160, 2161, 69, 2163, 
	2163, 2164, 2165, 2166, 2167, 2162, 2162, 2162, 
	69, 2168, 2168, 2169, 2170, 2171, 2172, 69, 
	2174, 2174, 2175, 2176, 2177, 2178, 2173, 2173, 
	2173, 69, 2179, 2179, 2180, 2181, 2182, 2183, 
	69, 2184, 2184, 2185, 2186, 2187, 2187, 2187, 
	2188, 2189, 2187, 2187, 2187, 69, 2190, 2190, 
	2190, 2190, 69, 2188, 2189, 69, 2177, 2178, 
	69, 2166, 2167, 69, 2155, 2156, 69, 2143, 
	2144, 69, 2191, 2191, 2191, 69, 2192, 2192, 
	2193, 2194, 2195, 2196, 69, 2192, 2192, 2193, 
	2194, 2195, 2196, 2197, 1674, 2198, 2198, 2199, 
	2200, 2202, 2203, 2201, 1674, 2204, 2204, 2205, 
	2206, 2208, 2209, 2207, 2207, 2207, 69, 2210, 
	2210, 2211, 2212, 2213, 2214, 69, 2210, 2210, 
	2211, 2212, 2215, 2213, 2214, 2215, 2215, 2215, 
	69, 2216, 2216, 69, 2213, 2214, 69, 2208, 
	2209, 69, 2195, 2196, 69, 2218, 2218, 2219, 
	2220, 2221, 2222, 2223, 2224, 2225, 2226, 2226, 
	2227, 2228, 2217, 2229, 2230, 2217, 2231, 2217, 
	2233, 2233, 2232, 2235, 2234, 2232, 2236, 2217, 
	2237, 2237, 2238, 2239, 2241, 2242, 2243, 2243, 
	2244, 2245, 2240, 2217, 2224, 2225, 69, 2246, 
	2217, 2247, 2247, 2247, 2247, 2232, 2249, 2248, 
	2248, 2248, 2232, 2250, 1674, 2251, 2251, 2252, 
	2253, 2255, 2256, 2254, 1674, 2257, 2257, 2258, 
	2259, 2261, 2262, 2260, 2260, 2260, 69, 2263, 
	2263, 2264, 2265, 2266, 2267, 69, 2263, 2263, 
	2264, 2265, 2266, 2267, 2268, 1674, 2269, 2269, 
	2270, 2271, 2273, 2274, 2272, 1674, 2276, 2276, 
	2277, 2278, 2280, 2281, 2279, 2279, 2279, 2275, 
	2282, 2282, 2282, 2275, 2283, 2283, 2284, 2285, 
	2286, 2287, 2288, 2288, 2289, 2290, 2279, 2279, 
	2279, 2275, 2286, 2287, 2275, 2280, 2281, 69, 
	2266, 2267, 69, 2261, 2262, 69, 2291, 1674, 
	2292, 2292, 2293, 2294, 2296, 2297, 2295, 1674, 
	2298, 2298, 2299, 2300, 2302, 2303, 2301, 1674, 
	2304, 2304, 2305, 2306, 2308, 2309, 2307, 1674, 
	2310, 2310, 2311, 2312, 2314, 2315, 2313, 2313, 
	2313, 2275, 2316, 2316, 2316, 2275, 2317, 2317, 
	2318, 2319, 2320, 2321, 2322, 2322, 2323, 2324, 
	2313, 2313, 2313, 2275, 2320, 2321, 2275, 2314, 
	2315, 69, 2302, 2303, 69, 2325, 1674, 2326, 
	2326, 2327, 2328, 2330, 2331, 2329, 1674, 2333, 
	2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340, 
	2341, 2332, 2342, 2342, 2343, 2344, 2345, 2346, 
	2332, 2342, 2342, 2343, 2344, 2345, 2346, 2348, 
	2347, 2349, 2349, 2350, 2351, 2353, 2354, 2352, 
	2347, 2355, 2355, 2356, 2357, 2358, 2359, 2360, 
	2332, 2362, 2362, 2363, 2364, 2365, 2365, 2366, 
	2367, 2368, 2368, 2369, 2370, 2371, 2371, 2371, 
	2371, 2372, 2373, 2371, 2371, 2374, 2375, 2376, 
	2377, 2371, 2371, 2378, 2379, 2380, 2381, 2371, 
	2371, 2382, 2383, 2361, 2362, 2362, 2363, 2364, 
	2385, 2372, 2373, 2385, 2385, 2385, 2384, 2371, 
	2371, 2384, 2372, 2373, 2384, 2365, 2365, 2366, 
	2367, 2371, 2371, 2371, 2371, 2376, 2377, 2371, 
	2371, 2378, 2379, 2384, 2376, 2377, 2384, 2385, 
	2362, 2362, 2363, 2364, 2365, 2365, 2366, 2367, 
	2368, 2368, 2369, 2370, 2371, 2371, 2371, 2371, 
	2372, 2373, 2371, 2371, 2374, 2375, 2376, 2377, 
	2371, 2371, 2378, 2379, 2380, 2381, 2371, 2371, 
	2382, 2383, 2385, 2385, 2385, 2384, 2372, 2376, 
	2380, 2373, 2377, 2381, 2384, 2359, 2360, 2332, 
	2345, 2346, 2332, 2386, 2386, 2387, 2388, 2389, 
	2390, 2332, 2386, 2386, 2387, 2388, 2389, 2390, 
	2391, 2347, 2392, 2392, 2393, 2394, 2396, 2397, 
	2395, 2347, 2399, 2399, 2400, 2401, 2402, 2403, 
	2404, 2402, 2398, 2406, 2407, 2407, 2408, 2409, 
	2410, 2410, 2411, 2412, 2413, 2413, 2414, 2415, 
	2416, 2416, 2416, 2416, 2417, 2418, 2416, 2416, 
	2419, 2420, 2421, 2422, 2416, 2416, 2423, 2424, 
	2425, 2426, 2416, 2416, 2427, 2428, 2406, 2405, 
	2403, 2404, 2332, 2389, 2390, 2332, 2429, 2429, 
	2430, 2431, 2432, 2433, 2332, 2429, 2429, 2430, 
	2431, 2432, 2433, 2434, 2347, 2435, 2435, 2436, 
	2437, 2439, 2440, 2438, 2347, 2441, 2441, 2442, 
	2443, 2444, 2445, 2446, 2444, 2444, 2444, 2398, 
	2447, 2448, 2448, 2449, 2450, 2451, 2451, 2452, 
	2453, 2454, 2454, 2455, 2456, 2457, 2457, 2457, 
	2457, 2458, 2459, 2457, 2457, 2460, 2461, 2462, 
	2463, 2457, 2457, 2464, 2465, 2466, 2467, 2457, 
	2457, 2468, 2469, 2447, 2447, 2447, 2405, 2445, 
	2446, 2332, 2432, 2433, 2332, 2470, 2470, 2471, 
	2472, 2473, 2474, 2332, 2470, 2470, 2471, 2472, 
	2473, 2474, 2475, 2347, 2476, 2476, 2477, 2478, 
	2480, 2481, 2479, 2347, 2482, 2482, 2483, 2484, 
	2485, 2485, 2485, 2486, 2487, 2485, 2485, 2485, 
	2332, 2488, 2488, 2489, 2490, 2491, 2491, 2492, 
	2493, 2494, 2494, 2495, 2496, 2497, 2497, 2497, 
	2497, 2498, 2499, 2497, 2497, 2500, 2501, 2502, 
	2503, 2497, 2497, 2504, 2505, 2506, 2507, 2497, 
	2497, 2508, 2509, 2361, 2486, 2487, 2332, 2473, 
	2474, 2332, 2340, 2341, 69, 2511, 2512, 2513, 
	2514, 2515, 2516, 2517, 2518, 2519, 2520, 2521, 
	2522, 2523, 2524, 2511, 2512, 2513, 2514, 2515, 
	2516, 2517, 2518, 2519, 2520, 2521, 2522, 2523, 
	2524, 2510, 2525, 2525, 2526, 2527, 2528, 2529, 
	2530, 2528, 2529, 2530, 2531, 2532, 2510, 2533, 
	2533, 2534, 2535, 2537, 2538, 2536, 2536, 2536, 
	69, 2539, 2539, 2540, 2541, 2542, 2543, 69, 
	2539, 2539, 2540, 2541, 2542, 2543, 2544, 1674, 
	2545, 2545, 2546, 2547, 2549, 2550, 2548, 1674, 
	2551, 2551, 2552, 2553, 2555, 2556, 2554, 1674, 
	2557, 2557, 2558, 2559, 2561, 2562, 2560, 1674, 
	2564, 2564, 2565, 2566, 2568, 2569, 2567, 2563, 
	2570, 2570, 2571, 2572, 2574, 2575, 2573, 2563, 
	2576, 2576, 2577, 2578, 2580, 2581, 2579, 2563, 
	2582, 2582, 2583, 2584, 2586, 2587, 2585, 2563, 
	2588, 2588, 2589, 2590, 2592, 2593, 2591, 1674, 
	2594, 2594, 2595, 2596, 2598, 2599, 2597, 1674, 
	2600, 2600, 2601, 2602, 2603, 2603, 2603, 2604, 
	2605, 2603, 2603, 2603, 69, 2606, 2606, 2607, 
	2608, 2609, 2610, 69, 2611, 2611, 2612, 2613, 
	2614, 2615, 2616, 2614, 2614, 2614, 69, 2617, 
	2617, 69, 2615, 2616, 69, 2604, 2605, 69, 
	2592, 2593, 69, 2580, 2581, 69, 2568, 2569, 
	69, 2555, 2556, 69, 2542, 2543, 69, 2537, 
	2538, 69, 2618, 2618, 2510, 2619, 2619, 2510, 
	2620, 2620, 2621, 2622, 2623, 2624, 2510, 2625, 
	2625, 2510, 2626, 2626, 2510, 2627, 2627, 2510, 
	2628, 2628, 2629, 2630, 2631, 2632, 2510, 2633, 
	2633, 2510, 2634, 2634, 2635, 2636, 2637, 2638, 
	2510, 2639, 2640, 2639, 2640, 2510, 2641, 2641, 
	2510, 2642, 2642, 2510, 2643, 2643, 2644, 2645, 
	2646, 2647, 2510, 2648, 2648, 2510, 2649, 2649, 
	2510, 2650, 2650, 2510, 2651, 2651, 2652, 2653, 
	2654, 2655, 2510, 2656, 2657, 2658, 2656, 2657, 
	2658, 2510, 2659, 2659, 2510, 2660, 2660, 2510, 
	2661, 2661, 2510, 2662, 2662, 2663, 2664, 2665, 
	2666, 2510, 2667, 2668, 2667, 2668, 2510, 2669, 
	2669, 2510, 2670, 2670, 2510, 2671, 2671, 2672, 
	2673, 2674, 2675, 2510, 2676, 2676, 2510, 2677, 
	2677, 2510, 2678, 2678, 2510, 2679, 2679, 2680, 
	2681, 2682, 2683, 2510, 2684, 2684, 2685, 2686, 
	2687, 2688, 2510, 2689, 2689, 2510, 2690, 2690, 
	2510, 2691, 2692, 2510, 2693, 2510, 2694, 2694, 
	2695, 2696, 2697, 2698, 2510, 2699, 2510, 2700, 
	2700, 2701, 2702, 2703, 2704, 2510, 2705, 2705, 
	2510, 2706, 2706, 2510, 2707, 2707, 2510, 2708, 
	2708, 2510, 2709, 2709, 2710, 2711, 2712, 2713, 
	2510, 2714, 2714, 2510, 2715, 2715, 2510, 2716, 
	2716, 2510, 2717, 2717, 2510, 2718, 2718, 2510, 
	2719, 2719, 2510, 2720, 2720, 2510, 2721, 2721, 
	2722, 2723, 2724, 2725, 2510, 2726, 2727, 2726, 
	2727, 2510, 2728, 2728, 2510, 2729, 2729, 2730, 
	2731, 2732, 2733, 2510, 2734, 2734, 2735, 2736, 
	2737, 2738, 2510, 2739, 2740, 2741, 2742, 2741, 
	2742, 2510, 2743, 2510, 2744, 2744, 2745, 2746, 
	2747, 2748, 2510, 2749, 2510, 2750, 2750, 2751, 
	2752, 2753, 2754, 2510, 2755, 2755, 2510, 2756, 
	2756, 2757, 2758, 2759, 2760, 2510, 2761, 2761, 
	2762, 2763, 2764, 2765, 2510, 2766, 2767, 2766, 
	2767, 2510, 2768, 2768, 2510, 2769, 2769, 2510, 
	2770, 2770, 2510, 2771, 2771, 2772, 2773, 2774, 
	2775, 2510, 2776, 2776, 2777, 2778, 2779, 2780, 
	2510, 2781, 2782, 2783, 2781, 2782, 2783, 2510, 
	2784, 2784, 2510, 2785, 2785, 2510, 2786, 2786, 
	2510, 2787, 2787, 2788, 2789, 2790, 2791, 2510, 
	2792, 2792, 2510, 2793, 2793, 2794, 2795, 2796, 
	2797, 2510, 2798, 2798, 2799, 2800, 2801, 2801, 
	2802, 2803, 2510, 2804, 2804, 2510, 2805, 2805, 
	2806, 2807, 2808, 2809, 2810, 2510, 2811, 2811, 
	2812, 2813, 2814, 2814, 2815, 2816, 2510, 2817, 
	2817, 2510, 2818, 2818, 2510, 2819, 2819, 2510, 
	2820, 2820, 2510, 2821, 2821, 2822, 2823, 2824, 
	2825, 2510, 2826, 2826, 2510, 2827, 2827, 2510, 
	2828, 2828, 2829, 2830, 2831, 2832, 2510, 2833, 
	2834, 2835, 2833, 2834, 2835, 2510, 2836, 2836, 
	2837, 2838, 2839, 2840, 2510, 2841, 2841, 2510, 
	2842, 2842, 2510, 2843, 2843, 2510, 2844, 2844, 
	2845, 2846, 2847, 2848, 2510, 2849, 2849, 2850, 
	2851, 2852, 2853, 2510, 2854, 2855, 2856, 2857, 
	2854, 2855, 2856, 2857, 2510, 2858, 2858, 2510, 
	2859, 2859, 2860, 2861, 2862, 2863, 2510, 2864, 
	2864, 2510, 2865, 2865, 2866, 2867, 2868, 2869, 
	2510, 2870, 2870, 2510, 2871, 2871, 2872, 2873, 
	2874, 2875, 2510, 2876, 2876, 2510, 2877, 2877, 
	2510, 2878, 2878, 2510, 2879, 2879, 2880, 2881, 
	2882, 2883, 2510, 2884, 2885, 2886, 2884, 2885, 
	2886, 2510, 2887, 2887, 2510, 2888, 2888, 2510, 
	2889, 2889, 2890, 2891, 2892, 2893, 2510, 2894, 
	2894, 2510, 2895, 2895, 2896, 2897, 2898, 2899, 
	2510, 2900, 2900, 2510, 2901, 2901, 2510, 2903, 
	2902, 2904, 2904, 2905, 2906, 2908, 2909, 2907, 
	2902, 2910, 2910, 2910, 2910, 2910, 2910, 69, 
	2911, 2911, 2911, 2911, 69, 2912, 2912, 2912, 
	2912, 69, 2913, 1674, 2914, 2914, 2915, 2916, 
	2918, 2919, 2917, 1674, 2920, 2920, 2921, 2922, 
	2924, 2925, 2923, 1674, 2926, 2926, 2927, 2928, 
	2930, 2931, 2929, 1674, 2932, 2932, 2933, 2934, 
	2936, 2937, 2935, 2935, 2935, 69, 2938, 2938, 
	2939, 2940, 2941, 2942, 69, 2938, 2938, 2939, 
	2940, 2943, 2941, 2942, 2943, 2943, 2943, 69, 
	2944, 2944, 69, 2941, 2942, 69, 2936, 2937, 
	69, 2924, 2925, 69, 2945, 2945, 2945, 2945, 
	69, 2946, 2946, 69, 2947, 1674, 2948, 2948, 
	2949, 2950, 2952, 2953, 2951, 1674, 2954, 2954, 
	2955, 2956, 2958, 2959, 2957, 1674, 2960, 2960, 
	2961, 2962, 2964, 2965, 2963, 1674, 2966, 2966, 
	2967, 2968, 2970, 2971, 2969, 1674, 2972, 2972, 
	2973, 2974, 2976, 2977, 2975, 1674, 2978, 2978, 
	2979, 2980, 2981, 2983, 2984, 2982, 2982, 2982, 
	2275, 2985, 2985, 2986, 2987, 2988, 2989, 2275, 
	2991, 2991, 2992, 2993, 2995, 2996, 2994, 2994, 
	2994, 2990, 2997, 2997, 2997, 2990, 2999, 2998, 
	2998, 2998, 2990, 3000, 3000, 3000, 2990, 3002, 
	3001, 3001, 3001, 2990, 3004, 3003, 3003, 3003, 
	2990, 3005, 3005, 3005, 2990, 3007, 3006, 3006, 
	3006, 2990, 3008, 3008, 3008, 3008, 3009, 3009, 
	3009, 2990, 3010, 3010, 3010, 3010, 69, 3011, 
	2990, 3007, 2990, 3004, 2990, 3012, 2990, 3002, 
	2990, 2995, 2996, 69, 3013, 3013, 3013, 2275, 
	2985, 2985, 2986, 2987, 2988, 2989, 3014, 3014, 
	3014, 2275, 2983, 2984, 69, 2970, 2971, 69, 
	2958, 2959, 69, 3015, 1674, 3016, 3016, 3017, 
	3018, 3020, 3021, 3019, 1674, 3022, 3022, 3023, 
	3024, 3026, 3027, 3025, 1674, 3028, 3028, 3029, 
	3030, 3032, 3033, 3031, 1674, 3034, 3034, 3035, 
	3036, 3038, 3039, 3037, 1674, 3040, 3040, 3041, 
	3042, 3044, 3045, 3043, 1674, 3046, 3046, 3047, 
	3048, 3049, 3051, 3052, 3050, 3050, 3050, 2275, 
	3053, 3053, 3053, 3053, 2275, 3054, 3054, 3054, 
	2275, 3053, 3053, 3053, 3053, 3055, 3055, 3055, 
	2275, 3051, 3052, 69, 3038, 3039, 69, 3026, 
	3027, 69, 3056, 1674, 3057, 3057, 3058, 3059, 
	3061, 3062, 3060, 1674, 3063, 3063, 3064, 3065, 
	3067, 3068, 3066, 1674, 3069, 3069, 3070, 3071, 
	3073, 3074, 3072, 1674, 3075, 3075, 3076, 3077, 
	3079, 3080, 3078, 1674, 3081, 3081, 3082, 3083, 
	3085, 3086, 3084, 1674, 3087, 3087, 3088, 3089, 
	3091, 3092, 3090, 3090, 3090, 2275, 3093, 3093, 
	3093, 2275, 3094, 3094, 3095, 3096, 3097, 3098, 
	3099, 3099, 3100, 3101, 3090, 3090, 3090, 2275, 
	3097, 3098, 2275, 3091, 3092, 69, 3079, 3080, 
	69, 3067, 3068, 69, 3102, 1674, 3103, 3103, 
	3104, 3105, 3107, 3108, 3106, 1674, 3109, 3109, 
	3110, 3111, 3112, 3113, 3114, 3112, 1651, 3115, 
	3116, 3115, 3115, 3115, 3116, 1651, 3113, 3114, 
	69, 3117, 1674, 3118, 3118, 3119, 3120, 3122, 
	3123, 3121, 1674, 3124, 3124, 3125, 3126, 3128, 
	3129, 3127, 3127, 3127, 2275, 3130, 3130, 3130, 
	2275, 3131, 3131, 3131, 2275, 3132, 3132, 3132, 
	2275, 3134, 3133, 3135, 3135, 3135, 2275, 3136, 
	3136, 3136, 2275, 3137, 3137, 3137, 2275, 3138, 
	3138, 3138, 2275, 3140, 3134, 3140, 3140, 3140, 
	3139, 3128, 3129, 69, 3141, 3141, 3141, 2275, 
	3142, 3142, 3142, 2275, 3144, 3143, 3145, 3145, 
	3145, 2275, 3146, 3146, 3146, 2275, 3147, 3144, 
	3147, 3147, 3147, 3143, 3148, 3148, 3148, 2275, 
	3149, 3149, 3149, 2275, 3150, 3143, 3151, 3151, 
	3151, 2275, 3152, 3152, 3152, 2275, 3153, 3150, 
	3153, 3153, 3153, 3143, 3155, 3155, 3156, 3157, 
	3158, 3159, 3159, 3159, 19, 20, 3160, 3161, 
	3159, 3159, 3159, 3154, 3162, 3162, 3163, 3164, 
	3165, 3166, 67, 3167, 3166, 19, 20, 3168, 
	3169, 121, 122, 3170, 3171, 67, 67, 3166, 
	67, 3166, 67, 3166, 67, 697, 3173, 3173, 
	3156, 3174, 3175, 3159, 3176, 3177, 3178, 3179, 
	3180, 3181, 3182, 3183, 3184, 3185, 3186, 3187, 
	3188, 3189, 3159, 3159, 3176, 3177, 3178, 3179, 
	3180, 3181, 3182, 3183, 3184, 3185, 3186, 3187, 
	3188, 3189, 19, 20, 3190, 3191, 3159, 3159, 
	3159, 3172, 3173, 3173, 3156, 3174, 3175, 3159, 
	3194, 3195, 3196, 3197, 3198, 3199, 3200, 3201, 
	3202, 3203, 3204, 3205, 3206, 3207, 3159, 3159, 
	3194, 3195, 3196, 3197, 3198, 3199, 3200, 3201, 
	3202, 3203, 3204, 3205, 3206, 3207, 19, 20, 
	3190, 3191, 3159, 3193, 3159, 3159, 3192, 3209, 
	3209, 3210, 3211, 3212, 3213, 3213, 3213, 3214, 
	3215, 3216, 3217, 3213, 3213, 3213, 3208, 3155, 
	3155, 3156, 3218, 3219, 3159, 3194, 3195, 3196, 
	3197, 3198, 3199, 3200, 3201, 3202, 3203, 3204, 
	3205, 3206, 3207, 3159, 3159, 3194, 3195, 3196, 
	3197, 3198, 3199, 3200, 3201, 3202, 3203, 3204, 
	3205, 3206, 3207, 19, 20, 3220, 3161, 3159, 
	3193, 3159, 3159, 3192, 3221, 3221, 3156, 3222, 
	3223, 3159, 3159, 3159, 88, 89, 3224, 3225, 
	3159, 3159, 3159, 697, 3226, 3226, 3163, 3227, 
	3228, 3166, 67, 3167, 3166, 88, 89, 3224, 
	3229, 96, 97, 3224, 3230, 67, 67, 3166, 
	67, 3166, 67, 3166, 67, 697, 3162, 3162, 
	3163, 3164, 3165, 3166, 67, 3233, 3234, 3235, 
	3236, 3237, 3238, 3239, 3240, 3241, 3242, 3243, 
	3244, 3245, 3246, 3167, 3166, 3233, 3234, 3235, 
	3236, 3237, 3238, 3239, 3240, 3241, 3242, 3243, 
	3244, 3245, 3246, 19, 20, 3168, 3169, 121, 
	122, 3170, 3171, 67, 67, 3166, 3232, 67, 
	3166, 67, 3166, 67, 3231, 3226, 3226, 3163, 
	3227, 3228, 3166, 67, 3233, 3234, 3235, 3236, 
	3237, 3238, 3239, 3240, 3241, 3242, 3243, 3244, 
	3245, 3246, 3167, 3166, 3233, 3234, 3235, 3236, 
	3237, 3238, 3239, 3240, 3241, 3242, 3243, 3244, 
	3245, 3246, 88, 89, 3247, 3229, 96, 97, 
	3247, 3230, 67, 67, 3166, 3232, 67, 3166, 
	67, 3166, 67, 3231, 619, 619, 619, 820, 
	817, 820, 819, 820, 817, 821, 817, 816, 
	619, 619, 619, 957, 957, 958, 959, 927, 
	929, 930, 928, 960, 961, 928, 928, 928, 
	934, 619, 968, 968, 969, 970, 963, 971, 
	972, 973, 973, 974, 975, 963, 963, 963, 
	962, 971, 972, 962, 619, 985, 985, 986, 
	987, 988, 989, 990, 991, 992, 993, 994, 
	995, 996, 997, 998, 999, 1000, 1001, 988, 
	989, 990, 991, 992, 993, 994, 995, 996, 
	997, 998, 999, 1000, 1001, 1002, 1003, 1004, 
	1004, 1005, 1006, 976, 1002, 1003, 976, 619, 
	1445, 1445, 1446, 1447, 1448, 1449, 1450, 1450, 
	1451, 1452, 1441, 1441, 1441, 1437, 1448, 1449, 
	1437, 1467, 1467, 1468, 1469, 1471, 1472, 1470, 
	1470, 1470, 1437, 1423, 1474, 1474, 1475, 1476, 
	1477, 1478, 1479, 1479, 1480, 1481, 1470, 1470, 
	1470, 1437, 1477, 1478, 1437, 1471, 1472, 1423, 
	619, 619, 619, 619, 619, 619, 619, 619, 
	619, 1849, 1849, 1850, 1851, 1852, 1853, 1854, 
	1854, 1855, 1856, 1840, 1840, 1840, 69, 1852, 
	1853, 69, 619, 619, 2020, 2020, 2021, 2022, 
	2023, 2024, 1969, 1969, 2025, 2026, 1860, 2023, 
	2024, 1860, 2002, 2002, 2003, 2004, 2006, 2007, 
	1969, 1969, 2008, 2009, 2005, 1860, 2006, 2007, 
	1860, 1983, 1983, 1984, 1985, 1987, 1988, 1969, 
	1969, 1989, 1990, 1986, 1860, 1987, 1988, 1860, 
	1963, 1963, 1964, 1965, 1967, 1968, 1969, 1969, 
	1970, 1971, 1966, 1860, 1967, 1968, 1860, 619, 
	619, 619, 619, 2218, 2218, 2219, 2220, 2221, 
	2222, 2223, 2224, 2225, 2226, 2226, 2227, 2228, 
	2217, 2224, 2225, 69, 619, 2283, 2283, 2284, 
	2285, 2286, 2287, 2288, 2288, 2289, 2290, 2279, 
	2279, 2279, 2275, 2286, 2287, 2275, 619, 2317, 
	2317, 2318, 2319, 2320, 2321, 2322, 2322, 2323, 
	2324, 2313, 2313, 2313, 2275, 2320, 2321, 2275, 
	619, 2365, 2365, 2366, 2367, 2371, 2371, 2371, 
	2371, 2376, 2377, 2371, 2371, 2378, 2379, 2384, 
	2376, 2377, 2384, 2362, 2362, 2363, 2364, 2385, 
	2372, 2373, 2385, 2385, 2385, 2384, 2372, 2373, 
	2384, 2385, 2362, 2362, 2363, 2364, 2365, 2365, 
	2366, 2367, 2368, 2368, 2369, 2370, 2371, 2371, 
	2371, 2371, 2372, 2373, 2371, 2371, 2374, 2375, 
	2376, 2377, 2371, 2371, 2378, 2379, 2380, 2381, 
	2371, 2371, 2382, 2383, 2385, 2385, 2385, 2384, 
	2372, 2376, 2380, 2373, 2377, 2381, 2384, 619, 
	619, 619, 619, 619, 619, 619, 3094, 3094, 
	3095, 3096, 3097, 3098, 3099, 3099, 3100, 3101, 
	3090, 3090, 3090, 2275, 3097, 3098, 2275, 619, 
	619, 619, 619, 0
};

static const short _zone_scanner_trans_targs[] = {
	0, 1, 1, 1, 2, 4, 17, 26, 
	40, 47, 130, 63, 67, 75, 81, 97, 
	100, 107, 118, 1060, 138, 1065, 243, 0, 
	3, 3, 3, 2, 153, 153, 153, 153, 
	153, 3, 156, 0, 3, 3, 3, 4, 
	17, 26, 40, 47, 52, 63, 67, 75, 
	81, 97, 100, 107, 118, 3, 156, 0, 
	5, 5, 5, 214, 217, 221, 5, 11, 
	10, 12, 21, 6, 5, 0, 5, 5, 
	9, 5, 11, 10, 12, 21, 0, 7, 
	7, 7, 1060, 8, 0, 7, 7, 7, 
	1060, 8, 6, 10, 12, 12, 12, 13, 
	1066, 163, 227, 1061, 14, 0, 14, 14, 
	15, 159, 166, 167, 168, 169, 170, 171, 
	172, 173, 174, 175, 176, 177, 178, 1068, 
	223, 1066, 224, 1069, 226, 0, 16, 16, 
	16, 1062, 246, 0, 16, 16, 16, 1062, 
	246, 18, 22, 19, 20, 5, 5, 5, 
	5, 11, 10, 12, 21, 21, 21, 21, 
	23, 24, 25, 5, 5, 5, 5, 11, 
	10, 12, 21, 27, 31, 39, 28, 29, 
	30, 5, 5, 5, 5, 11, 10, 12, 
	21, 32, 35, 33, 34, 5, 5, 5, 
	5, 11, 10, 12, 21, 36, 37, 38, 
	5, 5, 5, 5, 11, 10, 12, 21, 
	5, 5, 5, 5, 11, 10, 12, 21, 
	41, 42, 43, 45, 44, 5, 5, 5, 
	5, 11, 10, 12, 21, 46, 5, 5, 
	5, 5, 11, 10, 12, 21, 48, 49, 
	50, 51, 5, 5, 5, 5, 11, 10, 
	12, 21, 53, 56, 54, 54, 54, 54, 
	128, 54, 54, 54, 4, 17, 26, 40, 
	47, 55, 63, 67, 75, 81, 97, 100, 
	107, 118, 54, 128, 57, 58, 59, 60, 
	61, 62, 5, 5, 5, 5, 11, 10, 
	12, 21, 64, 66, 65, 5, 5, 5, 
	5, 11, 10, 12, 21, 5, 5, 5, 
	5, 11, 10, 12, 21, 68, 70, 72, 
	74, 69, 5, 5, 5, 5, 11, 10, 
	12, 21, 71, 5, 5, 5, 5, 11, 
	10, 12, 21, 73, 5, 5, 5, 5, 
	11, 10, 12, 21, 5, 5, 5, 5, 
	11, 10, 12, 21, 76, 80, 77, 78, 
	79, 5, 5, 5, 5, 11, 10, 12, 
	21, 5, 5, 5, 5, 11, 10, 12, 
	21, 82, 86, 88, 83, 84, 85, 5, 
	5, 5, 5, 11, 10, 12, 21, 87, 
	5, 5, 5, 5, 11, 10, 12, 21, 
	5, 5, 5, 89, 5, 11, 10, 12, 
	21, 90, 5, 5, 5, 91, 5, 11, 
	10, 12, 21, 5, 5, 5, 92, 5, 
	11, 10, 12, 21, 93, 94, 95, 96, 
	5, 5, 5, 5, 11, 10, 12, 21, 
	98, 99, 5, 5, 5, 5, 11, 10, 
	12, 21, 101, 102, 106, 5, 5, 5, 
	5, 11, 10, 12, 21, 103, 104, 105, 
	5, 5, 5, 5, 11, 10, 12, 21, 
	5, 5, 5, 5, 11, 10, 12, 21, 
	108, 110, 112, 114, 109, 5, 5, 5, 
	5, 11, 10, 12, 21, 111, 5, 5, 
	5, 5, 11, 10, 12, 21, 113, 5, 
	5, 5, 5, 11, 10, 12, 21, 115, 
	116, 117, 5, 5, 5, 5, 11, 10, 
	12, 21, 119, 122, 124, 120, 121, 5, 
	5, 5, 5, 11, 10, 12, 21, 123, 
	5, 5, 5, 5, 11, 10, 12, 21, 
	125, 126, 0, 127, 0, 5, 5, 5, 
	127, 5, 11, 10, 12, 21, 129, 129, 
	129, 1063, 244, 131, 132, 132, 132, 132, 
	137, 132, 132, 132, 133, 4, 17, 26, 
	40, 47, 55, 63, 67, 75, 81, 97, 
	100, 107, 118, 132, 137, 54, 54, 54, 
	133, 134, 134, 134, 134, 134, 54, 128, 
	135, 135, 136, 136, 136, 136, 136, 54, 
	54, 54, 135, 54, 128, 0, 140, 181, 
	187, 141, 142, 143, 144, 145, 146, 1064, 
	0, 148, 148, 148, 148, 149, 148, 148, 
	148, 148, 149, 0, 0, 151, 151, 151, 
	151, 152, 151, 151, 151, 151, 152, 154, 
	154, 155, 155, 155, 155, 155, 3, 3, 
	3, 154, 3, 156, 0, 158, 158, 158, 
	158, 228, 165, 229, 158, 158, 158, 158, 
	228, 165, 229, 0, 160, 160, 160, 1061, 
	13, 1066, 163, 1067, 227, 160, 160, 160, 
	1067, 161, 161, 161, 1065, 162, 164, 164, 
	164, 165, 165, 165, 179, 213, 1068, 0, 
	182, 183, 184, 185, 186, 1064, 188, 189, 
	1064, 0, 161, 161, 161, 1065, 162, 0, 
	192, 192, 192, 1063, 193, 192, 192, 192, 
	1063, 193, 0, 195, 195, 195, 1068, 179, 
	1069, 213, 195, 195, 195, 1069, 210, 0, 
	210, 210, 210, 211, 161, 161, 161, 1065, 
	162, 212, 212, 212, 212, 212, 212, 213, 
	213, 213, 215, 216, 5, 5, 5, 5, 
	11, 10, 12, 21, 218, 219, 220, 5, 
	5, 5, 5, 11, 10, 12, 21, 222, 
	5, 5, 5, 5, 11, 10, 12, 21, 
	225, 225, 225, 226, 226, 226, 227, 227, 
	227, 229, 229, 229, 0, 248, 1070, 248, 
	1070, 0, 250, 252, 257, 258, 253, 1071, 
	251, 1071, 250, 252, 253, 252, 253, 254, 
	0, 255, 256, 1071, 252, 251, 253, 1071, 
	0, 260, 265, 261, 1072, 1073, 262, 0, 
	263, 264, 1072, 260, 1073, 261, 265, 266, 
	267, 0, 268, 269, 270, 265, 266, 267, 
	0, 272, 272, 272, 272, 279, 0, 273, 
	0, 274, 274, 274, 273, 276, 276, 276, 
	276, 276, 1074, 275, 0, 277, 0, 277, 
	278, 278, 278, 278, 278, 274, 274, 274, 
	277, 1074, 275, 0, 281, 281, 281, 281, 
	292, 282, 286, 291, 287, 283, 0, 284, 
	284, 284, 282, 286, 287, 1075, 285, 284, 
	284, 284, 1075, 285, 286, 287, 288, 0, 
	289, 290, 286, 283, 287, 294, 294, 294, 
	294, 310, 295, 0, 0, 296, 296, 296, 
	1076, 300, 307, 0, 296, 296, 296, 297, 
	301, 306, 302, 1076, 300, 307, 0, 298, 
	299, 299, 299, 297, 301, 302, 1076, 300, 
	299, 299, 299, 301, 302, 303, 0, 304, 
	305, 301, 298, 302, 1077, 308, 308, 308, 
	308, 309, 0, 312, 313, 314, 317, 315, 
	315, 315, 315, 315, 316, 1078, 1079, 1080, 
	0, 319, 319, 319, 319, 324, 1081, 1082, 
	1083, 319, 319, 319, 320, 325, 333, 347, 
	354, 359, 367, 371, 379, 385, 401, 404, 
	411, 422, 319, 324, 1081, 1082, 1083, 319, 
	319, 319, 321, 432, 436, 319, 324, 1081, 
	1082, 1083, 322, 323, 319, 319, 319, 319, 
	324, 1081, 1082, 1083, 326, 329, 327, 328, 
	319, 319, 319, 319, 324, 1081, 1082, 1083, 
	330, 331, 332, 319, 319, 319, 319, 324, 
	1081, 1082, 1083, 334, 338, 346, 335, 336, 
	337, 319, 319, 319, 319, 324, 1081, 1082, 
	1083, 339, 342, 340, 341, 319, 319, 319, 
	319, 324, 1081, 1082, 1083, 343, 344, 345, 
	319, 319, 319, 319, 324, 1081, 1082, 1083, 
	319, 319, 319, 319, 324, 1081, 1082, 1083, 
	348, 349, 350, 352, 351, 319, 319, 319, 
	319, 324, 1081, 1082, 1083, 353, 319, 319, 
	319, 319, 324, 1081, 1082, 1083, 355, 356, 
	357, 358, 319, 319, 319, 319, 324, 1081, 
	1082, 1083, 360, 361, 362, 363, 364, 365, 
	366, 319, 319, 319, 319, 324, 1081, 1082, 
	1083, 368, 370, 369, 319, 319, 319, 319, 
	324, 1081, 1082, 1083, 319, 319, 319, 319, 
	324, 1081, 1082, 1083, 372, 374, 376, 378, 
	373, 319, 319, 319, 319, 324, 1081, 1082, 
	1083, 375, 319, 319, 319, 319, 324, 1081, 
	1082, 1083, 377, 319, 319, 319, 319, 324, 
	1081, 1082, 1083, 319, 319, 319, 319, 324, 
	1081, 1082, 1083, 380, 384, 381, 382, 383, 
	319, 319, 319, 319, 324, 1081, 1082, 1083, 
	319, 319, 319, 319, 324, 1081, 1082, 1083, 
	386, 390, 392, 387, 388, 389, 319, 319, 
	319, 319, 324, 1081, 1082, 1083, 391, 319, 
	319, 319, 319, 324, 1081, 1082, 1083, 319, 
	319, 319, 393, 319, 324, 1081, 1082, 1083, 
	394, 319, 319, 319, 395, 319, 324, 1081, 
	1082, 1083, 319, 319, 319, 396, 319, 324, 
	1081, 1082, 1083, 397, 398, 399, 400, 319, 
	319, 319, 319, 324, 1081, 1082, 1083, 402, 
	403, 319, 319, 319, 319, 324, 1081, 1082, 
	1083, 405, 406, 410, 319, 319, 319, 319, 
	324, 1081, 1082, 1083, 407, 408, 409, 319, 
	319, 319, 319, 324, 1081, 1082, 1083, 319, 
	319, 319, 319, 324, 1081, 1082, 1083, 412, 
	414, 416, 418, 413, 319, 319, 319, 319, 
	324, 1081, 1082, 1083, 415, 319, 319, 319, 
	319, 324, 1081, 1082, 1083, 417, 319, 319, 
	319, 319, 324, 1081, 1082, 1083, 419, 420, 
	421, 319, 319, 319, 319, 324, 1081, 1082, 
	1083, 423, 426, 428, 424, 425, 319, 319, 
	319, 319, 324, 1081, 1082, 1083, 427, 319, 
	319, 319, 319, 324, 1081, 1082, 1083, 429, 
	430, 431, 319, 319, 319, 431, 319, 324, 
	1081, 1082, 1083, 433, 434, 435, 319, 319, 
	319, 319, 324, 1081, 1082, 1083, 437, 319, 
	319, 319, 319, 324, 1081, 1082, 1083, 0, 
	439, 439, 439, 439, 446, 0, 440, 441, 
	441, 441, 440, 441, 445, 0, 441, 441, 
	441, 442, 441, 445, 443, 443, 443, 443, 
	443, 444, 1084, 1085, 1086, 448, 448, 448, 
	448, 456, 449, 455, 1087, 1087, 1087, 455, 
	1088, 1087, 1091, 450, 450, 450, 451, 450, 
	454, 452, 452, 452, 452, 452, 453, 1088, 
	1089, 1090, 450, 450, 450, 450, 454, 0, 
	458, 459, 474, 504, 512, 525, 1092, 458, 
	460, 461, 1092, 462, 1092, 463, 464, 465, 
	466, 467, 468, 469, 470, 471, 472, 473, 
	1092, 475, 476, 482, 477, 478, 479, 480, 
	481, 1092, 483, 484, 485, 486, 495, 487, 
	488, 489, 490, 491, 492, 493, 494, 1092, 
	496, 497, 498, 499, 500, 501, 502, 503, 
	1092, 505, 506, 507, 508, 509, 510, 511, 
	1092, 513, 514, 515, 516, 517, 518, 519, 
	522, 520, 521, 1092, 523, 524, 1092, 526, 
	527, 528, 531, 529, 530, 1092, 532, 533, 
	534, 546, 549, 1092, 535, 536, 537, 538, 
	539, 540, 541, 542, 543, 544, 545, 1092, 
	547, 548, 1092, 550, 551, 1092, 0, 553, 
	554, 560, 577, 580, 586, 590, 1093, 553, 
	555, 556, 557, 558, 559, 1093, 561, 567, 
	573, 562, 563, 564, 565, 566, 1093, 568, 
	570, 569, 1093, 571, 572, 1093, 574, 575, 
	576, 1093, 578, 579, 1093, 581, 583, 582, 
	1093, 584, 585, 1093, 587, 588, 589, 1093, 
	591, 592, 1093, 0, 594, 1094, 594, 596, 
	1095, 598, 599, 599, 599, 599, 628, 599, 
	599, 599, 600, 599, 628, 601, 601, 601, 
	601, 627, 0, 601, 601, 601, 602, 601, 
	627, 603, 603, 603, 602, 603, 626, 603, 
	603, 603, 604, 603, 626, 0, 605, 605, 
	605, 604, 623, 623, 623, 623, 623, 605, 
	622, 605, 605, 605, 606, 605, 622, 607, 
	607, 607, 606, 619, 619, 619, 619, 619, 
	607, 618, 607, 607, 607, 608, 607, 618, 
	609, 609, 609, 608, 615, 615, 615, 615, 
	615, 609, 614, 609, 609, 609, 610, 609, 
	614, 1096, 610, 611, 611, 611, 611, 611, 
	612, 612, 613, 613, 613, 613, 613, 1096, 
	612, 616, 616, 617, 617, 617, 617, 617, 
	609, 609, 609, 616, 609, 614, 620, 620, 
	621, 621, 621, 621, 621, 607, 607, 607, 
	620, 607, 618, 624, 624, 625, 625, 625, 
	625, 625, 605, 605, 605, 624, 605, 622, 
	630, 631, 631, 631, 631, 633, 632, 631, 
	631, 631, 631, 633, 1097, 635, 636, 636, 
	636, 636, 638, 636, 636, 636, 637, 636, 
	638, 1098, 640, 641, 641, 641, 640, 641, 
	643, 641, 641, 641, 642, 641, 643, 1099, 
	645, 646, 646, 646, 646, 647, 1100, 1101, 
	1102, 646, 646, 646, 646, 647, 1100, 1101, 
	1102, 649, 1103, 649, 0, 651, 652, 652, 
	652, 651, 652, 704, 652, 652, 652, 653, 
	657, 699, 652, 704, 654, 654, 654, 653, 
	654, 703, 654, 654, 654, 655, 654, 703, 
	656, 656, 656, 701, 655, 656, 700, 656, 
	656, 656, 656, 700, 658, 658, 658, 658, 
	698, 659, 660, 660, 660, 659, 660, 697, 
	660, 660, 660, 661, 665, 692, 660, 697, 
	662, 662, 662, 661, 662, 696, 662, 662, 
	662, 663, 662, 696, 664, 664, 664, 694, 
	663, 664, 693, 664, 664, 664, 664, 693, 
	666, 666, 666, 666, 691, 667, 668, 668, 
	669, 669, 669, 688, 668, 690, 669, 687, 
	1104, 1111, 1112, 669, 669, 669, 670, 669, 
	687, 1104, 1111, 1112, 671, 671, 671, 684, 
	670, 686, 671, 683, 1104, 1109, 1110, 671, 
	671, 671, 672, 671, 683, 1109, 1110, 673, 
	673, 673, 680, 672, 682, 673, 679, 1104, 
	1107, 1108, 673, 673, 673, 674, 673, 679, 
	1107, 1108, 675, 675, 675, 677, 674, 675, 
	676, 1104, 1105, 1106, 675, 675, 675, 675, 
	676, 1105, 1106, 675, 675, 675, 678, 675, 
	676, 1104, 1105, 1106, 678, 673, 673, 673, 
	681, 682, 673, 679, 1104, 1107, 1108, 681, 
	671, 671, 671, 685, 686, 671, 683, 1104, 
	1109, 1110, 685, 669, 669, 669, 689, 690, 
	669, 687, 1104, 1111, 1112, 689, 666, 666, 
	666, 666, 691, 664, 664, 664, 695, 664, 
	693, 695, 658, 658, 658, 658, 698, 656, 
	656, 656, 702, 656, 700, 702, 706, 707, 
	707, 707, 706, 707, 715, 707, 707, 707, 
	708, 707, 715, 709, 709, 709, 708, 709, 
	714, 709, 709, 709, 710, 709, 714, 711, 
	711, 711, 710, 711, 713, 711, 711, 711, 
	712, 711, 713, 1113, 717, 718, 718, 718, 
	717, 718, 732, 718, 718, 718, 719, 718, 
	732, 720, 720, 720, 719, 720, 731, 721, 
	720, 720, 720, 720, 731, 722, 722, 722, 
	722, 730, 723, 722, 722, 722, 722, 730, 
	724, 724, 724, 724, 729, 725, 724, 724, 
	724, 724, 729, 726, 726, 726, 726, 728, 
	726, 726, 726, 727, 726, 728, 1114, 734, 
	735, 735, 735, 735, 743, 736, 737, 737, 
	737, 736, 737, 742, 737, 737, 737, 738, 
	737, 742, 739, 739, 739, 739, 741, 740, 
	1115, 0, 744, 745, 744, 744, 746, 752, 
	744, 751, 1116, 1117, 1118, 746, 752, 747, 
	0, 748, 748, 749, 750, 744, 744, 744, 
	750, 744, 751, 1116, 1117, 1118, 753, 754, 
	754, 749, 756, 757, 757, 757, 756, 757, 
	767, 757, 757, 757, 758, 757, 767, 759, 
	759, 759, 759, 766, 760, 761, 761, 761, 
	760, 761, 765, 0, 761, 761, 761, 762, 
	761, 765, 763, 763, 763, 763, 763, 764, 
	1119, 1120, 1121, 769, 770, 770, 770, 769, 
	770, 777, 770, 770, 770, 771, 770, 777, 
	772, 772, 772, 771, 772, 776, 772, 772, 
	772, 773, 772, 776, 774, 774, 774, 774, 
	774, 775, 1122, 1123, 1124, 779, 780, 780, 
	780, 779, 780, 816, 0, 780, 780, 780, 
	781, 795, 802, 809, 780, 816, 782, 782, 
	782, 782, 794, 0, 783, 784, 784, 784, 
	783, 784, 793, 784, 784, 784, 785, 784, 
	793, 0, 786, 786, 786, 789, 789, 789, 
	791, 791, 791, 1125, 786, 788, 1128, 1129, 
	789, 790, 1126, 1127, 791, 792, 1130, 1131, 
	0, 787, 796, 796, 796, 796, 801, 797, 
	798, 798, 798, 797, 798, 800, 0, 798, 
	798, 798, 799, 798, 800, 0, 799, 786, 
	786, 786, 789, 789, 789, 791, 791, 791, 
	1125, 786, 788, 1128, 1129, 789, 790, 1126, 
	1127, 791, 792, 1130, 1131, 803, 803, 803, 
	803, 808, 804, 805, 805, 805, 804, 805, 
	807, 805, 805, 805, 806, 805, 807, 806, 
	786, 786, 786, 789, 789, 789, 791, 791, 
	791, 1125, 786, 788, 1128, 1129, 789, 790, 
	1126, 1127, 791, 792, 1130, 1131, 810, 810, 
	810, 810, 815, 811, 812, 812, 812, 811, 
	812, 814, 812, 812, 812, 813, 812, 814, 
	786, 786, 786, 789, 789, 789, 791, 791, 
	791, 1125, 786, 788, 1128, 1129, 789, 790, 
	1126, 1127, 791, 792, 1130, 1131, 0, 818, 
	852, 860, 874, 881, 886, 894, 898, 906, 
	912, 928, 931, 938, 949, 819, 819, 819, 
	843, 846, 850, 819, 842, 819, 819, 819, 
	820, 819, 842, 821, 821, 821, 821, 841, 
	822, 823, 823, 823, 822, 823, 840, 823, 
	823, 823, 824, 823, 840, 825, 825, 825, 
	824, 825, 839, 0, 825, 825, 825, 826, 
	825, 839, 827, 827, 827, 826, 827, 838, 
	827, 827, 827, 828, 827, 838, 829, 829, 
	829, 828, 829, 837, 829, 829, 829, 830, 
	829, 837, 831, 831, 831, 830, 831, 836, 
	831, 831, 831, 832, 831, 836, 833, 833, 
	833, 833, 835, 833, 833, 833, 834, 833, 
	835, 1132, 844, 845, 819, 819, 819, 819, 
	842, 847, 848, 849, 819, 819, 819, 819, 
	842, 851, 819, 819, 819, 819, 842, 853, 
	856, 854, 855, 819, 819, 819, 819, 842, 
	857, 858, 859, 819, 819, 819, 819, 842, 
	861, 865, 873, 862, 863, 864, 819, 819, 
	819, 819, 842, 866, 869, 867, 868, 819, 
	819, 819, 819, 842, 870, 871, 872, 819, 
	819, 819, 819, 842, 819, 819, 819, 819, 
	842, 875, 876, 877, 879, 878, 819, 819, 
	819, 819, 842, 880, 819, 819, 819, 819, 
	842, 882, 883, 884, 885, 819, 819, 819, 
	819, 842, 887, 888, 889, 890, 891, 892, 
	893, 819, 819, 819, 819, 842, 895, 897, 
	896, 819, 819, 819, 819, 842, 819, 819, 
	819, 819, 842, 899, 901, 903, 905, 900, 
	819, 819, 819, 819, 842, 902, 819, 819, 
	819, 819, 842, 904, 819, 819, 819, 819, 
	842, 819, 819, 819, 819, 842, 907, 911, 
	908, 909, 910, 819, 819, 819, 819, 842, 
	819, 819, 819, 819, 842, 913, 917, 919, 
	914, 915, 916, 819, 819, 819, 819, 842, 
	918, 819, 819, 819, 819, 842, 819, 819, 
	819, 920, 819, 842, 921, 819, 819, 819, 
	922, 819, 842, 819, 819, 819, 923, 819, 
	842, 924, 925, 926, 927, 819, 819, 819, 
	819, 842, 929, 930, 819, 819, 819, 819, 
	842, 932, 933, 937, 819, 819, 819, 819, 
	842, 934, 935, 936, 819, 819, 819, 819, 
	842, 819, 819, 819, 819, 842, 939, 941, 
	943, 945, 940, 819, 819, 819, 819, 842, 
	942, 819, 819, 819, 819, 842, 944, 819, 
	819, 819, 819, 842, 946, 947, 948, 819, 
	819, 819, 819, 842, 950, 953, 955, 951, 
	952, 819, 819, 819, 819, 842, 954, 819, 
	819, 819, 819, 842, 956, 957, 0, 958, 
	819, 819, 819, 958, 819, 842, 960, 961, 
	1133, 963, 964, 964, 964, 963, 964, 972, 
	964, 964, 964, 965, 964, 972, 966, 966, 
	966, 965, 966, 971, 966, 966, 966, 967, 
	966, 971, 968, 968, 968, 968, 970, 969, 
	1134, 974, 1135, 976, 977, 977, 977, 976, 
	977, 1003, 977, 977, 977, 978, 977, 1003, 
	979, 979, 979, 978, 979, 1002, 979, 979, 
	979, 980, 979, 1002, 981, 981, 981, 980, 
	981, 1001, 981, 981, 981, 982, 999, 981, 
	1001, 983, 983, 983, 983, 998, 0, 983, 
	983, 983, 984, 983, 998, 985, 986, 996, 
	987, 988, 995, 989, 993, 990, 991, 991, 
	992, 984, 1136, 994, 997, 1000, 999, 1005, 
	1006, 1006, 1006, 1005, 1006, 1016, 1006, 1006, 
	1006, 1007, 1006, 1016, 1008, 1008, 1008, 1007, 
	1008, 1015, 1008, 1008, 1008, 1009, 1008, 1015, 
	1010, 1010, 1010, 1009, 1010, 1014, 1010, 1010, 
	1010, 1011, 1012, 1010, 1014, 1137, 1013, 1012, 
	1018, 1019, 1019, 1019, 1018, 1019, 1029, 1019, 
	1019, 1019, 1020, 1019, 1029, 1021, 1021, 1021, 
	1020, 1021, 1028, 1021, 1021, 1021, 1022, 1021, 
	1028, 1023, 1023, 1023, 1022, 1023, 1027, 1023, 
	1023, 1023, 1024, 1023, 1027, 1025, 1025, 1025, 
	1025, 1025, 1026, 1138, 1139, 1140, 1031, 1032, 
	1032, 1032, 1031, 1032, 1034, 1032, 1032, 1032, 
	1033, 1032, 1034, 1141, 1033, 1036, 1037, 1037, 
	1037, 1036, 1037, 1047, 1037, 1037, 1037, 1038, 
	1037, 1047, 1039, 1040, 1041, 0, 1042, 1043, 
	1044, 1045, 1046, 0, 1142, 1049, 1050, 0, 
	1051, 1052, 1053, 1143, 1055, 1056, 1057, 1058, 
	1059, 1144, 0, 1, 139, 1, 1, 147, 
	1065, 243, 14, 180, 14, 14, 190, 209, 
	1068, 223, 1069, 226, 0, 129, 129, 129, 
	157, 230, 231, 232, 233, 245, 235, 236, 
	237, 238, 239, 240, 241, 242, 1063, 244, 
	0, 150, 157, 230, 231, 232, 233, 234, 
	235, 236, 237, 238, 239, 240, 241, 242, 
	0, 1, 139, 1, 1, 147, 1060, 138, 
	1065, 243, 1, 1, 1065, 161, 161, 161, 
	1065, 162, 165, 165, 165, 179, 213, 0, 
	191, 194, 196, 197, 198, 199, 200, 201, 
	202, 203, 204, 205, 206, 207, 208, 1065
};

static const short _zone_scanner_trans_actions[] = {
	1872, 0, 5, 7, 506, 659, 659, 659, 
	659, 659, 659, 659, 659, 659, 659, 659, 
	659, 659, 659, 3, 0, 3, 0, 2012, 
	101, 674, 677, 41, 55, 53, 51, 0, 
	57, 671, 101, 345, 0, 5, 7, 95, 
	95, 95, 95, 95, 95, 95, 95, 95, 
	95, 95, 95, 95, 95, 3, 0, 1592, 
	1595, 2384, 2388, 0, 0, 0, 2380, 1595, 
	2392, 3481, 2392, 339, 0, 337, 5, 7, 
	0, 3, 0, 339, 356, 339, 1571, 343, 
	1580, 1583, 2372, 343, 9, 0, 5, 7, 
	1709, 0, 1586, 341, 0, 5, 7, 0, 
	2368, 339, 339, 359, 0, 3086, 5, 7, 
	1860, 2096, 2096, 2096, 2096, 2096, 2096, 2096, 
	2096, 2096, 2096, 2096, 2096, 2096, 2096, 3, 
	0, 1574, 339, 356, 339, 3782, 680, 2100, 
	2104, 3256, 680, 1589, 0, 5, 7, 359, 
	0, 0, 0, 0, 0, 1649, 2676, 2680, 
	2672, 1649, 2684, 3586, 2684, 0, 5, 7, 
	0, 0, 0, 1601, 2420, 2424, 2416, 1601, 
	2428, 3506, 2428, 0, 0, 0, 0, 0, 
	0, 1676, 2820, 2824, 2816, 1676, 2828, 3631, 
	2828, 0, 0, 0, 0, 1652, 2692, 2696, 
	2688, 1652, 2700, 3591, 2700, 0, 0, 0, 
	1673, 2804, 2808, 2800, 1673, 2812, 3626, 2812, 
	1658, 2724, 2728, 2720, 1658, 2732, 3601, 2732, 
	0, 0, 0, 0, 0, 1703, 2964, 2968, 
	2960, 1703, 2972, 3676, 2972, 0, 1706, 2980, 
	2984, 2976, 1706, 2988, 3681, 2988, 0, 0, 
	0, 0, 1610, 2468, 2472, 2464, 1610, 2476, 
	3521, 2476, 0, 0, 99, 665, 668, 662, 
	99, 0, 5, 7, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 3, 0, 0, 0, 0, 0, 
	0, 0, 1664, 2756, 2760, 2752, 1664, 2764, 
	3611, 2764, 0, 0, 0, 1631, 2580, 2584, 
	2576, 1631, 2588, 3556, 2588, 1646, 2660, 2664, 
	2656, 1646, 2668, 3581, 2668, 0, 0, 0, 
	0, 0, 1694, 2916, 2920, 2912, 1694, 2924, 
	3661, 2924, 0, 1697, 2932, 2936, 2928, 1697, 
	2940, 3666, 2940, 0, 1637, 2612, 2616, 2608, 
	1637, 2620, 3566, 2620, 1700, 2948, 2952, 2944, 
	1700, 2956, 3671, 2956, 0, 0, 0, 0, 
	0, 1613, 2484, 2488, 2480, 1613, 2492, 3526, 
	2492, 1616, 2500, 2504, 2496, 1616, 2508, 3531, 
	2508, 0, 0, 0, 0, 0, 0, 1643, 
	2644, 2648, 2640, 1643, 2652, 3576, 2652, 0, 
	1691, 2900, 2904, 2896, 1691, 2908, 3656, 2908, 
	1598, 2404, 2408, 0, 2400, 1598, 2412, 3501, 
	2412, 0, 1670, 2788, 2792, 0, 2784, 1670, 
	2796, 3621, 2796, 1679, 2836, 2840, 0, 2832, 
	1679, 2844, 3636, 2844, 0, 0, 0, 0, 
	1682, 2852, 2856, 2848, 1682, 2860, 3641, 2860, 
	0, 0, 1607, 2452, 2456, 2448, 1607, 2460, 
	3516, 2460, 0, 0, 0, 1622, 2532, 2536, 
	2528, 1622, 2540, 3541, 2540, 0, 0, 0, 
	1667, 2772, 2776, 2768, 1667, 2780, 3616, 2780, 
	1628, 2564, 2568, 2560, 1628, 2572, 3551, 2572, 
	0, 0, 0, 0, 0, 1604, 2436, 2440, 
	2432, 1604, 2444, 3511, 2444, 0, 1688, 2884, 
	2888, 2880, 1688, 2892, 3651, 2892, 0, 1640, 
	2628, 2632, 2624, 1640, 2636, 3571, 2636, 0, 
	0, 0, 1661, 2740, 2744, 2736, 1661, 2748, 
	3606, 2748, 0, 0, 0, 0, 0, 1685, 
	2868, 2872, 2864, 1685, 2876, 3646, 2876, 0, 
	1619, 2516, 2520, 2512, 1619, 2524, 3536, 2524, 
	0, 0, 518, 506, 1876, 578, 1992, 1996, 
	41, 1988, 578, 2000, 3216, 2000, 0, 5, 
	7, 3, 0, 0, 99, 665, 668, 662, 
	99, 0, 5, 7, 506, 97, 97, 97, 
	97, 97, 97, 97, 97, 97, 97, 97, 
	97, 97, 97, 3, 0, 101, 674, 677, 
	41, 55, 53, 51, 0, 57, 671, 101, 
	2016, 41, 55, 53, 51, 0, 57, 599, 
	2040, 2044, 3226, 2036, 599, 93, 89, 89, 
	89, 0, 0, 0, 0, 0, 0, 87, 
	37, 33, 455, 458, 452, 33, 0, 5, 
	7, 3, 0, 0, 3071, 461, 1796, 1804, 
	1792, 461, 0, 5, 7, 3, 0, 2016, 
	41, 55, 53, 51, 0, 57, 599, 2040, 
	2044, 3226, 2036, 599, 1836, 1820, 3036, 3051, 
	3021, 1820, 3698, 3056, 0, 5, 7, 3, 
	0, 356, 339, 2376, 2396, 3486, 3491, 3800, 
	2396, 3855, 3496, 3848, 3496, 0, 5, 7, 
	1744, 0, 5, 7, 359, 0, 0, 5, 
	7, 0, 5, 7, 0, 339, 359, 2360, 
	0, 0, 0, 0, 0, 79, 0, 0, 
	75, 1828, 464, 1800, 1808, 3011, 464, 3813, 
	1812, 3026, 3041, 3692, 1812, 0, 5, 7, 
	359, 0, 3076, 3061, 3704, 3710, 3806, 3061, 
	3862, 3716, 0, 5, 7, 1744, 1816, 479, 
	3031, 3046, 3016, 1816, 343, 365, 371, 1748, 
	343, 343, 1580, 1583, 0, 5, 7, 0, 
	5, 7, 0, 0, 1634, 2596, 2600, 2592, 
	1634, 2604, 3561, 2604, 0, 0, 0, 1625, 
	2548, 2552, 2544, 1625, 2556, 3546, 2556, 0, 
	1655, 2708, 2712, 2704, 1655, 2716, 3596, 2716, 
	0, 5, 7, 0, 5, 7, 0, 5, 
	7, 0, 5, 7, 482, 380, 2996, 11, 
	1756, 27, 1772, 1772, 25, 0, 419, 1760, 
	17, 404, 383, 383, 13, 15, 0, 386, 
	395, 19, 19, 3001, 389, 392, 21, 416, 
	620, 65, 0, 0, 1, 617, 623, 2052, 
	69, 69, 626, 629, 2048, 71, 65, 0, 
	0, 67, 623, 69, 69, 629, 71, 71, 
	73, 0, 5, 7, 3, 0, 509, 506, 
	3221, 0, 5, 7, 41, 55, 53, 51, 
	0, 57, 2056, 0, 374, 2016, 2004, 41, 
	55, 53, 51, 0, 57, 59, 590, 593, 
	3226, 3231, 59, 77, 0, 5, 7, 3, 
	0, 3236, 3236, 632, 2060, 17, 377, 23, 
	407, 410, 383, 383, 13, 3006, 23, 0, 
	5, 7, 2064, 0, 15, 0, 386, 398, 
	19, 19, 389, 392, 21, 0, 5, 7, 
	3, 0, 635, 83, 644, 81, 638, 641, 
	3241, 81, 81, 650, 0, 5, 7, 3246, 
	3246, 647, 2068, 2072, 0, 0, 85, 17, 
	413, 1764, 1768, 383, 383, 13, 3686, 413, 
	0, 5, 7, 15, 0, 386, 401, 19, 
	19, 389, 392, 21, 1724, 0, 5, 7, 
	3, 0, 125, 117, 119, 121, 0, 123, 
	0, 5, 7, 3, 0, 1, 350, 1, 
	309, 307, 1496, 1499, 1493, 307, 2332, 3476, 
	2332, 0, 5, 7, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 3, 0, 1502, 1732, 1502, 231, 
	1154, 1157, 0, 0, 0, 1151, 231, 2180, 
	3286, 2180, 0, 0, 257, 1271, 1274, 1268, 
	257, 2232, 3351, 2232, 0, 0, 0, 0, 
	267, 1316, 1319, 1313, 267, 2252, 3376, 2252, 
	0, 0, 0, 235, 1172, 1175, 1169, 235, 
	2188, 3296, 2188, 0, 0, 0, 0, 0, 
	0, 285, 1397, 1400, 1394, 285, 2288, 3421, 
	2288, 0, 0, 0, 0, 269, 1325, 1328, 
	1322, 269, 2256, 3381, 2256, 0, 0, 0, 
	283, 1388, 1391, 1385, 283, 2284, 3416, 2284, 
	273, 1343, 1346, 1340, 273, 2264, 3391, 2264, 
	0, 0, 0, 0, 0, 303, 1478, 1481, 
	1475, 303, 2324, 3466, 2324, 0, 305, 1487, 
	1490, 1484, 305, 2328, 3471, 2328, 0, 0, 
	0, 0, 241, 1199, 1202, 1196, 241, 2200, 
	3311, 2200, 0, 0, 0, 0, 0, 0, 
	0, 277, 1361, 1364, 1358, 277, 2272, 3401, 
	2272, 0, 0, 0, 255, 1262, 1265, 1259, 
	255, 2228, 3346, 2228, 265, 1307, 1310, 1304, 
	265, 2248, 3371, 2248, 0, 0, 0, 0, 
	0, 297, 1451, 1454, 1448, 297, 2312, 3451, 
	2312, 0, 299, 1460, 1463, 1457, 299, 2316, 
	3456, 2316, 0, 259, 1280, 1283, 1277, 259, 
	2236, 3356, 2236, 301, 1469, 1472, 1466, 301, 
	2320, 3461, 2320, 0, 0, 0, 0, 0, 
	243, 1208, 1211, 1205, 243, 2204, 3316, 2204, 
	245, 1217, 1220, 1214, 245, 2208, 3321, 2208, 
	0, 0, 0, 0, 0, 0, 263, 1298, 
	1301, 1295, 263, 2244, 3366, 2244, 0, 295, 
	1442, 1445, 1439, 295, 2308, 3446, 2308, 233, 
	1163, 1166, 0, 1160, 233, 2184, 3291, 2184, 
	0, 281, 1379, 1382, 0, 1376, 281, 2280, 
	3411, 2280, 287, 1406, 1409, 0, 1403, 287, 
	2292, 3426, 2292, 0, 0, 0, 0, 289, 
	1415, 1418, 1412, 289, 2296, 3431, 2296, 0, 
	0, 239, 1190, 1193, 1187, 239, 2196, 3306, 
	2196, 0, 0, 0, 249, 1235, 1238, 1232, 
	249, 2216, 3331, 2216, 0, 0, 0, 279, 
	1370, 1373, 1367, 279, 2276, 3406, 2276, 253, 
	1253, 1256, 1250, 253, 2224, 3341, 2224, 0, 
	0, 0, 0, 0, 237, 1181, 1184, 1178, 
	237, 2192, 3301, 2192, 0, 293, 1433, 1436, 
	1430, 293, 2304, 3441, 2304, 0, 261, 1289, 
	1292, 1286, 261, 2240, 3361, 2240, 0, 0, 
	0, 275, 1352, 1355, 1349, 275, 2268, 3396, 
	2268, 0, 0, 0, 0, 0, 291, 1424, 
	1427, 1421, 291, 2300, 3436, 2300, 0, 247, 
	1226, 1229, 1223, 247, 2212, 3326, 2212, 0, 
	0, 506, 229, 1145, 1148, 41, 1142, 229, 
	2176, 3281, 2176, 0, 0, 0, 251, 1244, 
	1247, 1241, 251, 2220, 3336, 2220, 0, 271, 
	1334, 1337, 1331, 271, 2260, 3386, 2260, 323, 
	0, 5, 7, 3, 0, 512, 506, 49, 
	584, 587, 41, 581, 49, 2164, 0, 5, 
	7, 113, 3, 0, 115, 0, 5, 7, 
	3, 0, 713, 1728, 713, 0, 5, 7, 
	3, 0, 506, 506, 347, 1716, 1720, 41, 
	1, 1712, 347, 0, 5, 7, 113, 3, 
	0, 115, 0, 5, 7, 3, 0, 713, 
	1728, 713, 49, 584, 587, 581, 49, 329, 
	506, 0, 0, 0, 0, 0, 539, 41, 
	0, 0, 722, 0, 725, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	731, 0, 0, 0, 0, 0, 0, 0, 
	0, 743, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 746, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	749, 0, 0, 0, 0, 0, 0, 0, 
	752, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 755, 0, 0, 758, 0, 
	0, 0, 0, 0, 0, 719, 0, 0, 
	0, 0, 0, 728, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 734, 
	0, 0, 737, 0, 0, 740, 331, 506, 
	0, 0, 0, 0, 0, 0, 551, 41, 
	0, 0, 0, 0, 0, 779, 0, 0, 
	0, 0, 0, 0, 0, 0, 782, 0, 
	0, 0, 776, 0, 0, 770, 0, 0, 
	0, 773, 0, 0, 788, 0, 0, 0, 
	767, 0, 0, 761, 0, 0, 0, 764, 
	0, 0, 785, 686, 683, 2116, 103, 485, 
	488, 485, 500, 1848, 1852, 1844, 500, 0, 
	5, 7, 485, 3, 0, 500, 1848, 1852, 
	1844, 500, 515, 0, 5, 7, 506, 3, 
	0, 47, 572, 575, 41, 569, 47, 0, 
	5, 7, 506, 3, 0, 2008, 47, 572, 
	575, 41, 55, 53, 51, 0, 57, 569, 
	47, 0, 5, 7, 506, 3, 0, 47, 
	572, 575, 41, 55, 53, 51, 0, 57, 
	569, 47, 0, 5, 7, 506, 3, 0, 
	47, 572, 575, 41, 55, 53, 51, 0, 
	57, 569, 47, 0, 5, 7, 506, 3, 
	0, 566, 41, 55, 53, 51, 0, 57, 
	2016, 41, 55, 53, 51, 0, 57, 2020, 
	3226, 2016, 41, 55, 53, 51, 0, 57, 
	596, 2028, 2032, 3226, 2024, 596, 2016, 41, 
	55, 53, 51, 0, 57, 596, 2028, 2032, 
	3226, 2024, 596, 2016, 41, 55, 53, 51, 
	0, 57, 596, 2028, 2032, 3226, 2024, 596, 
	422, 31, 437, 440, 434, 31, 422, 0, 
	5, 7, 3, 0, 431, 485, 500, 1848, 
	1852, 1844, 500, 0, 5, 7, 485, 3, 
	0, 488, 506, 563, 1980, 1984, 41, 1976, 
	563, 0, 5, 7, 485, 3, 0, 488, 
	422, 31, 437, 440, 434, 31, 431, 1776, 
	431, 0, 5, 7, 3, 0, 1, 350, 
	1, 683, 2132, 103, 1562, 2336, 311, 1508, 
	1511, 41, 1505, 311, 0, 5, 7, 506, 
	0, 0, 3, 0, 315, 1526, 1529, 41, 
	1523, 315, 0, 5, 7, 3211, 3, 0, 
	521, 1908, 1912, 0, 41, 1904, 521, 0, 
	5, 7, 3, 0, 0, 5, 7, 3, 
	0, 506, 313, 1517, 1520, 41, 1514, 313, 
	0, 5, 7, 506, 0, 0, 3, 0, 
	317, 1535, 1538, 41, 1532, 317, 0, 5, 
	7, 3211, 3, 0, 524, 1920, 1924, 0, 
	41, 1916, 524, 0, 5, 7, 3, 0, 
	0, 5, 7, 3, 0, 0, 3206, 3794, 
	527, 1932, 1936, 0, 41, 527, 1928, 527, 
	3186, 3758, 3186, 0, 5, 7, 3206, 3, 
	0, 1559, 1736, 1559, 530, 1944, 1948, 0, 
	41, 530, 1940, 530, 3191, 3764, 3191, 0, 
	5, 7, 3206, 3, 0, 1736, 1559, 533, 
	1956, 1960, 0, 41, 533, 1952, 533, 3196, 
	3770, 3196, 0, 5, 7, 3206, 3, 0, 
	1736, 1559, 536, 1968, 1972, 0, 41, 1964, 
	536, 3201, 3776, 3201, 0, 5, 7, 3, 
	0, 1736, 1559, 1900, 3176, 3181, 3091, 3171, 
	1900, 3752, 3841, 3752, 503, 1896, 3161, 3166, 
	3091, 1896, 3156, 1896, 3746, 3834, 3746, 503, 
	1892, 3146, 3151, 3091, 1892, 3141, 1892, 3740, 
	3827, 3740, 503, 1888, 3131, 3136, 3091, 1888, 
	3126, 1888, 3734, 3820, 3734, 503, 321, 1553, 
	1556, 1550, 321, 1884, 3116, 3121, 3091, 3111, 
	1884, 503, 319, 1544, 1547, 1541, 319, 1880, 
	3101, 3106, 3091, 3096, 1880, 503, 506, 45, 
	557, 560, 41, 554, 45, 0, 5, 7, 
	506, 3, 0, 45, 557, 560, 41, 554, 
	45, 0, 5, 7, 506, 3, 0, 563, 
	1980, 1984, 41, 1976, 563, 0, 5, 7, 
	485, 3, 0, 488, 506, 45, 557, 560, 
	41, 554, 45, 0, 5, 7, 506, 3, 
	0, 45, 557, 560, 41, 554, 45, 422, 
	0, 5, 7, 3, 0, 31, 437, 440, 
	434, 31, 422, 0, 5, 7, 3, 0, 
	31, 437, 440, 434, 31, 422, 0, 5, 
	7, 3, 0, 443, 1784, 1788, 1780, 443, 
	0, 5, 7, 485, 3, 0, 488, 335, 
	0, 5, 7, 3, 0, 506, 45, 557, 
	560, 41, 554, 45, 0, 5, 7, 333, 
	3, 0, 0, 5, 7, 3, 0, 127, 
	1, 707, 0, 695, 5, 7, 698, 701, 
	3, 0, 1, 350, 1, 109, 111, 0, 
	2108, 683, 103, 105, 506, 704, 2156, 2160, 
	41, 2152, 704, 2148, 3276, 2148, 0, 683, 
	103, 107, 506, 45, 557, 560, 41, 554, 
	45, 0, 5, 7, 333, 3, 0, 0, 
	5, 7, 3, 0, 506, 43, 545, 548, 
	41, 542, 43, 710, 0, 5, 7, 113, 
	3, 0, 115, 0, 5, 7, 3, 0, 
	1, 350, 1, 506, 43, 545, 548, 41, 
	542, 43, 0, 5, 7, 506, 3, 0, 
	43, 545, 548, 41, 542, 43, 0, 5, 
	7, 113, 3, 0, 115, 0, 5, 7, 
	3, 0, 1, 350, 1, 506, 43, 545, 
	548, 41, 542, 43, 791, 0, 5, 7, 
	145, 147, 149, 151, 3, 0, 0, 5, 
	7, 3, 0, 1864, 506, 43, 545, 548, 
	41, 542, 43, 0, 5, 7, 0, 3, 
	0, 2172, 0, 5, 7, 0, 5, 7, 
	0, 5, 7, 1, 3, 0, 350, 1, 
	3, 0, 350, 1, 3, 0, 350, 1, 
	794, 127, 0, 5, 7, 3, 0, 506, 
	43, 545, 548, 41, 542, 43, 2112, 0, 
	5, 7, 683, 3, 0, 3261, 103, 689, 
	2124, 2128, 689, 2124, 2128, 689, 2124, 2128, 
	2116, 2120, 689, 3266, 2116, 2120, 689, 3266, 
	2116, 2120, 689, 3266, 2116, 0, 5, 7, 
	3, 0, 506, 43, 545, 548, 41, 542, 
	43, 0, 5, 7, 683, 3, 0, 103, 
	692, 2140, 2144, 692, 2140, 2144, 692, 2140, 
	2144, 2132, 2136, 692, 3271, 2132, 2136, 692, 
	3271, 2132, 2136, 692, 3271, 2132, 0, 5, 
	7, 3, 0, 506, 43, 545, 548, 41, 
	542, 43, 0, 5, 7, 485, 3, 0, 
	39, 494, 497, 39, 494, 497, 39, 494, 
	497, 488, 491, 39, 1840, 488, 491, 39, 
	1840, 488, 491, 39, 1840, 488, 797, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 153, 803, 806, 
	0, 0, 0, 800, 153, 0, 5, 7, 
	333, 3, 0, 0, 5, 7, 3, 0, 
	506, 43, 545, 548, 41, 542, 43, 0, 
	5, 7, 506, 3, 0, 47, 572, 575, 
	41, 569, 47, 614, 0, 5, 7, 602, 
	3, 0, 63, 608, 611, 61, 605, 63, 
	0, 5, 7, 602, 3, 0, 63, 608, 
	611, 61, 605, 63, 0, 5, 7, 506, 
	3, 0, 563, 1980, 1984, 41, 1976, 563, 
	0, 5, 7, 485, 3, 0, 500, 1848, 
	1852, 1844, 500, 0, 5, 7, 127, 3, 
	0, 1, 0, 0, 179, 920, 923, 917, 
	179, 0, 0, 0, 173, 893, 896, 890, 
	173, 0, 193, 983, 986, 980, 193, 0, 
	0, 0, 0, 189, 965, 968, 962, 189, 
	0, 0, 0, 157, 821, 824, 818, 157, 
	0, 0, 0, 0, 0, 0, 207, 1046, 
	1049, 1043, 207, 0, 0, 0, 0, 191, 
	974, 977, 971, 191, 0, 0, 0, 205, 
	1037, 1040, 1034, 205, 195, 992, 995, 989, 
	195, 0, 0, 0, 0, 0, 225, 1127, 
	1130, 1124, 225, 0, 227, 1136, 1139, 1133, 
	227, 0, 0, 0, 0, 163, 848, 851, 
	845, 163, 0, 0, 0, 0, 0, 0, 
	0, 199, 1010, 1013, 1007, 199, 0, 0, 
	0, 177, 911, 914, 908, 177, 187, 956, 
	959, 953, 187, 0, 0, 0, 0, 0, 
	219, 1100, 1103, 1097, 219, 0, 221, 1109, 
	1112, 1106, 221, 0, 181, 929, 932, 926, 
	181, 223, 1118, 1121, 1115, 223, 0, 0, 
	0, 0, 0, 165, 857, 860, 854, 165, 
	167, 866, 869, 863, 167, 0, 0, 0, 
	0, 0, 0, 185, 947, 950, 944, 185, 
	0, 217, 1091, 1094, 1088, 217, 155, 812, 
	815, 0, 809, 155, 0, 203, 1028, 1031, 
	0, 1025, 203, 209, 1055, 1058, 0, 1052, 
	209, 0, 0, 0, 0, 211, 1064, 1067, 
	1061, 211, 0, 0, 161, 839, 842, 836, 
	161, 0, 0, 0, 171, 884, 887, 881, 
	171, 0, 0, 0, 201, 1019, 1022, 1016, 
	201, 175, 902, 905, 899, 175, 0, 0, 
	0, 0, 0, 159, 830, 833, 827, 159, 
	0, 215, 1082, 1085, 1079, 215, 0, 183, 
	938, 941, 935, 183, 0, 0, 0, 197, 
	1001, 1004, 998, 197, 0, 0, 0, 0, 
	0, 213, 1073, 1076, 1070, 213, 0, 169, 
	875, 878, 872, 169, 0, 0, 1868, 506, 
	45, 557, 560, 41, 554, 45, 485, 1856, 
	1, 506, 45, 557, 560, 41, 554, 45, 
	0, 5, 7, 506, 3, 0, 43, 545, 
	548, 41, 542, 43, 0, 5, 7, 333, 
	3, 0, 0, 5, 7, 3, 0, 127, 
	1, 127, 1, 506, 43, 545, 548, 41, 
	542, 43, 0, 5, 7, 506, 3, 0, 
	43, 545, 548, 41, 542, 43, 0, 5, 
	7, 506, 3, 0, 45, 557, 560, 41, 
	554, 45, 0, 5, 7, 29, 425, 3, 
	0, 31, 437, 440, 434, 31, 716, 0, 
	5, 7, 428, 3, 0, 131, 133, 0, 
	135, 137, 0, 139, 0, 141, 143, 0, 
	446, 129, 1, 0, 0, 115, 113, 506, 
	43, 545, 548, 41, 542, 43, 0, 5, 
	7, 506, 3, 0, 43, 545, 548, 41, 
	542, 43, 0, 5, 7, 506, 3, 0, 
	45, 557, 560, 41, 554, 45, 0, 5, 
	7, 29, 425, 3, 0, 431, 115, 113, 
	506, 43, 545, 548, 41, 542, 43, 0, 
	5, 7, 506, 3, 0, 43, 545, 548, 
	41, 542, 43, 0, 5, 7, 506, 3, 
	0, 43, 545, 548, 41, 542, 43, 0, 
	5, 7, 113, 3, 0, 115, 0, 5, 
	7, 3, 0, 1, 350, 1, 506, 45, 
	557, 560, 41, 554, 45, 0, 5, 7, 
	683, 3, 0, 2116, 103, 506, 45, 557, 
	560, 41, 554, 45, 0, 5, 7, 1568, 
	3, 0, 115, 113, 115, 2168, 327, 113, 
	115, 113, 115, 2356, 2352, 1565, 115, 2348, 
	325, 113, 115, 2340, 1565, 115, 325, 113, 
	115, 2344, 476, 35, 0, 470, 473, 449, 
	467, 35, 35, 339, 362, 368, 2364, 449, 
	353, 35, 1740, 1577, 1832, 35, 362, 368, 
	2092, 2092, 2092, 2092, 2092, 2092, 2092, 2092, 
	2092, 2092, 2092, 2092, 2092, 2092, 353, 35, 
	3066, 3081, 3251, 3251, 3251, 3251, 3251, 3251, 
	3251, 3251, 3251, 3251, 3251, 3251, 3251, 3251, 
	1824, 656, 91, 2084, 2088, 2076, 653, 91, 
	2080, 656, 362, 368, 353, 35, 362, 368, 
	2992, 35, 35, 362, 368, 35, 1577, 3722, 
	3728, 3788, 3788, 3788, 3788, 3788, 3788, 3788, 
	3788, 3788, 3788, 3788, 3788, 3788, 3788, 1752
};

static const short _zone_scanner_eof_actions[] = {
	0, 1872, 2012, 345, 1592, 337, 1571, 9, 
	9, 337, 1571, 337, 1571, 1571, 3086, 3782, 
	1589, 345, 345, 345, 1592, 1571, 345, 345, 
	345, 1592, 345, 345, 345, 345, 1592, 345, 
	345, 345, 1592, 345, 345, 345, 1592, 1592, 
	345, 345, 345, 345, 1592, 345, 1592, 345, 
	345, 345, 345, 1592, 345, 345, 345, 345, 
	345, 345, 345, 345, 345, 345, 1592, 345, 
	345, 1592, 1592, 345, 345, 1592, 345, 1592, 
	345, 1592, 1592, 345, 345, 345, 345, 1592, 
	1592, 345, 345, 345, 345, 1592, 345, 1592, 
	1592, 345, 1592, 1592, 345, 345, 345, 345, 
	1592, 345, 345, 1592, 345, 1592, 345, 345, 
	345, 1592, 1592, 345, 345, 1592, 345, 1592, 
	345, 1592, 345, 345, 345, 1592, 345, 345, 
	345, 1592, 345, 1592, 345, 345, 518, 1876, 
	345, 1872, 345, 345, 518, 2012, 2012, 2012, 
	2012, 345, 9, 93, 93, 93, 93, 93, 
	93, 93, 93, 37, 518, 0, 3071, 518, 
	345, 2012, 2012, 2012, 345, 1836, 1876, 2376, 
	1571, 1872, 9, 1571, 9, 3086, 2376, 2376, 
	2376, 2376, 2376, 2376, 2376, 2376, 2376, 2376, 
	2376, 2376, 2376, 1571, 2360, 93, 93, 93, 
	93, 93, 93, 93, 93, 93, 1828, 3813, 
	1872, 1589, 3076, 3086, 3076, 3076, 3076, 3076, 
	3076, 3076, 3076, 3076, 3076, 3076, 3076, 3076, 
	3076, 479, 3086, 1571, 9, 1571, 345, 345, 
	1592, 345, 345, 345, 1592, 345, 1592, 1571, 
	1571, 9, 1571, 1571, 337, 1571, 482, 482, 
	482, 482, 482, 482, 482, 482, 482, 482, 
	482, 482, 482, 9, 1589, 482, 1589, 0, 
	0, 27, 27, 27, 27, 395, 395, 395, 
	395, 27, 27, 620, 620, 2052, 2052, 2052, 
	2052, 620, 67, 2052, 2052, 2052, 2052, 73, 
	509, 3221, 374, 374, 3221, 2004, 3221, 73, 
	77, 77, 77, 377, 377, 377, 77, 398, 
	398, 398, 398, 377, 77, 0, 83, 644, 
	650, 85, 650, 9, 9, 85, 401, 401, 
	401, 401, 650, 9, 85, 0, 0, 125, 
	125, 125, 125, 125, 125, 125, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 323, 512, 
	512, 2164, 2164, 2164, 2164, 323, 323, 323, 
	512, 512, 2164, 2164, 2164, 2164, 323, 512, 
	323, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	331, 686, 686, 337, 337, 337, 337, 337, 
	337, 515, 515, 515, 2008, 515, 2008, 515, 
	2008, 515, 2008, 2008, 2008, 2008, 337, 2008, 
	2008, 2008, 337, 2008, 2008, 2008, 337, 2008, 
	2008, 2008, 337, 337, 337, 337, 337, 337, 
	337, 337, 337, 337, 337, 337, 337, 515, 
	515, 337, 337, 337, 337, 337, 337, 337, 
	686, 686, 1562, 1562, 1562, 1562, 1562, 1562, 
	1562, 1562, 1562, 1562, 1562, 1562, 1562, 1562, 
	1562, 1562, 1562, 1562, 1562, 1562, 1562, 1562, 
	1562, 1562, 1562, 1562, 1562, 1562, 1562, 1562, 
	1562, 1562, 1562, 1562, 1562, 1562, 1562, 1562, 
	1562, 1562, 1562, 1562, 1562, 1562, 1562, 1562, 
	1562, 1562, 1562, 1562, 1562, 1562, 1562, 1562, 
	1562, 515, 515, 515, 515, 515, 515, 337, 
	337, 337, 337, 337, 515, 515, 515, 515, 
	337, 337, 337, 337, 337, 337, 337, 337, 
	337, 337, 337, 337, 337, 337, 337, 515, 
	515, 337, 337, 337, 337, 337, 337, 337, 
	707, 707, 707, 2108, 2108, 707, 707, 337, 
	707, 2108, 2108, 515, 515, 337, 337, 515, 
	515, 710, 710, 710, 710, 337, 337, 337, 
	515, 515, 515, 515, 710, 710, 710, 710, 
	337, 337, 515, 515, 791, 791, 1864, 1864, 
	791, 2172, 794, 794, 794, 794, 794, 794, 
	794, 791, 791, 791, 1864, 1864, 2112, 3261, 
	791, 791, 791, 1864, 1864, 2112, 3261, 791, 
	791, 791, 1864, 1864, 791, 2172, 791, 791, 
	337, 797, 797, 337, 337, 515, 515, 515, 
	515, 614, 614, 614, 614, 515, 515, 337, 
	337, 337, 337, 337, 337, 337, 337, 337, 
	337, 337, 337, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 797, 797, 797, 
	797, 797, 797, 797, 797, 1868, 1868, 337, 
	337, 337, 515, 515, 515, 515, 337, 337, 
	337, 337, 337, 337, 337, 337, 337, 515, 
	515, 515, 515, 515, 515, 710, 710, 716, 
	716, 716, 716, 716, 716, 716, 716, 716, 
	337, 716, 716, 716, 716, 716, 337, 710, 
	710, 337, 337, 337, 515, 515, 515, 515, 
	515, 515, 710, 710, 710, 710, 337, 337, 
	337, 515, 515, 515, 515, 515, 515, 710, 
	710, 710, 710, 337, 337, 337, 515, 515, 
	686, 686, 337, 515, 515, 710, 710, 710, 
	710, 2168, 710, 710, 710, 710, 2356, 337, 
	710, 710, 2348, 710, 710, 2348, 710, 710, 
	2348, 710, 710, 2348, 0, 0, 0, 0, 
	91, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0
};

static const int zone_scanner_start = 1060;
static const int zone_scanner_first_final = 1060;
static const int zone_scanner_error = 0;

static const int zone_scanner_en_err_line = 247;
static const int zone_scanner_en_dname_ = 249;
static const int zone_scanner_en_text_ = 259;
static const int zone_scanner_en_default_ttl_ = 271;
static const int zone_scanner_en_zone_origin_ = 280;
static const int zone_scanner_en_include_file_ = 293;
static const int zone_scanner_en_base64_ = 311;
static const int zone_scanner_en_bitmap_ = 318;
static const int zone_scanner_en_nonempty_hex_r_data = 438;
static const int zone_scanner_en_hex_r_data = 447;
static const int zone_scanner_en_dns_alg_ = 457;
static const int zone_scanner_en_cert_type_ = 552;
static const int zone_scanner_en_r_data_a = 593;
static const int zone_scanner_en_r_data_ns = 595;
static const int zone_scanner_en_r_data_soa = 597;
static const int zone_scanner_en_r_data_hinfo = 629;
static const int zone_scanner_en_r_data_minfo = 634;
static const int zone_scanner_en_r_data_mx = 639;
static const int zone_scanner_en_r_data_txt = 644;
static const int zone_scanner_en_r_data_aaaa = 648;
static const int zone_scanner_en_r_data_loc = 650;
static const int zone_scanner_en_r_data_srv = 705;
static const int zone_scanner_en_r_data_naptr = 716;
static const int zone_scanner_en_r_data_cert = 733;
static const int zone_scanner_en_r_data_apl = 744;
static const int zone_scanner_en_r_data_ds = 755;
static const int zone_scanner_en_r_data_sshfp = 768;
static const int zone_scanner_en_r_data_ipseckey = 778;
static const int zone_scanner_en_r_data_rrsig = 817;
static const int zone_scanner_en_r_data_nsec = 959;
static const int zone_scanner_en_r_data_dnskey = 962;
static const int zone_scanner_en_r_data_dhcid = 973;
static const int zone_scanner_en_r_data_nsec3 = 975;
static const int zone_scanner_en_r_data_nsec3param = 1004;
static const int zone_scanner_en_r_data_tlsa = 1017;
static const int zone_scanner_en_r_data_l32 = 1030;
static const int zone_scanner_en_r_data_l64 = 1035;
static const int zone_scanner_en_r_data_eui48 = 1048;
static const int zone_scanner_en_r_data_eui64 = 1054;
static const int zone_scanner_en_main = 1060;


#line 82 "./zscanner/scanner.rl"


scanner_t* scanner_create(const char *file_name)
{
	scanner_t *s = calloc(1, sizeof(scanner_t));
	if (s == NULL) {
		return NULL;
	}

	s->file_name = strdup(file_name);
	s->line_counter = 1;

	// Nonzero initial scanner state.
	s->cs = zone_scanner_start;

	return s;
}

void scanner_free(scanner_t *s)
{
	if (s != NULL) {
		free(s->file_name);
		free(s);
	}
}

int scanner_process(const char *start,
		    const char *end,
		    const bool is_last_block,
		    scanner_t  *s)
{
	// Necessary scanner variables.
	const char *p = start;
	const char *pe = end;
	char       *eof = NULL;
	int        stack[RAGEL_STACK_SIZE];

	// Auxiliary variables which are used in scanner body.
	struct in_addr  addr4;
	struct in6_addr addr6;
	uint32_t timestamp;
	int16_t  window;
	int	 ret;

	// Next 2 variables are for better performance.
	// Restoring r_data pointer to next free space.
	uint8_t *rdata_tail = s->r_data + s->r_data_tail;
	// Initialization of the last r_data byte.
	uint8_t *rdata_stop = s->r_data + MAX_RDATA_LENGTH - 1;

	// Restoring scanner states.
	int cs  = s->cs;
	int top = s->top;
	memcpy(stack, s->stack, sizeof(stack));

	// End of file check.
	if (is_last_block == true) {
		eof = (char *)pe;
	}

	// Writing scanner body (in C).
	
#line 5016 "zscanner/scanner.c"
	{
	int _klen;
	unsigned int _trans;
	short _widec;
	const short *_acts;
	unsigned int _nacts;
	const short *_keys;

	if ( p == pe )
		goto _test_eof;
	if ( cs == 0 )
		goto _out;
_resume:
	_widec = (*p);
	_klen = _zone_scanner_cond_lengths[cs];
	_keys = _zone_scanner_cond_keys + (_zone_scanner_cond_offsets[cs]*2);
	if ( _klen > 0 ) {
		const short *_lower = _keys;
		const short *_mid;
		const short *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( _widec < _mid[0] )
				_upper = _mid - 2;
			else if ( _widec > _mid[1] )
				_lower = _mid + 2;
			else {
				switch ( _zone_scanner_cond_spaces[_zone_scanner_cond_offsets[cs] + ((_mid - _keys)>>1)] ) {
	case 0: {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./zscanner/scanner_body.rl"
 s->multiline  ) _widec += 256;
		break;
	}
	case 1: {
		_widec = (short)(1152 + ((*p) - -128));
		if ( 
#line 63 "./zscanner/scanner_body.rl"
 !s->multiline  ) _widec += 256;
		break;
	}
	case 2: {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 559 "./zscanner/scanner_body.rl"
 s->multiline  ) _widec += 256;
		break;
	}
	case 3: {
		_widec = (short)(2688 + ((*p) - -128));
		if ( 
#line 1197 "./zscanner/scanner_body.rl"
 s->number64 != 0  ) _widec += 256;
		break;
	}
	case 4: {
		_widec = (short)(4224 + ((*p) - -128));
		if ( 
#line 1198 "./zscanner/scanner_body.rl"
 s->number64 == 0  ) _widec += 256;
		break;
	}
	case 5: {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./zscanner/scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./zscanner/scanner_body.rl"
 !s->multiline  ) _widec += 512;
		break;
	}
	case 6: {
		_widec = (short)(3200 + ((*p) - -128));
		if ( 
#line 54 "./zscanner/scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 1197 "./zscanner/scanner_body.rl"
 s->number64 != 0  ) _widec += 512;
		break;
	}
	case 7: {
		_widec = (short)(4736 + ((*p) - -128));
		if ( 
#line 54 "./zscanner/scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 1198 "./zscanner/scanner_body.rl"
 s->number64 == 0  ) _widec += 512;
		break;
	}
	case 8: {
		_widec = (short)(5760 + ((*p) - -128));
		if ( 
#line 1197 "./zscanner/scanner_body.rl"
 s->number64 != 0  ) _widec += 256;
		if ( 
#line 1198 "./zscanner/scanner_body.rl"
 s->number64 == 0  ) _widec += 512;
		break;
	}
	case 9: {
		_widec = (short)(12928 + ((*p) - -128));
		if ( 
#line 54 "./zscanner/scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./zscanner/scanner_body.rl"
 !s->multiline  ) _widec += 512;
		if ( 
#line 1198 "./zscanner/scanner_body.rl"
 s->number64 == 0  ) _widec += 1024;
		break;
	}
	case 10: {
		_widec = (short)(6784 + ((*p) - -128));
		if ( 
#line 54 "./zscanner/scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 1197 "./zscanner/scanner_body.rl"
 s->number64 != 0  ) _widec += 512;
		if ( 
#line 1198 "./zscanner/scanner_body.rl"
 s->number64 == 0  ) _widec += 1024;
		break;
	}
	case 11: {
		_widec = (short)(8832 + ((*p) - -128));
		if ( 
#line 54 "./zscanner/scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./zscanner/scanner_body.rl"
 !s->multiline  ) _widec += 512;
		if ( 
#line 1197 "./zscanner/scanner_body.rl"
 s->number64 != 0  ) _widec += 1024;
		if ( 
#line 1198 "./zscanner/scanner_body.rl"
 s->number64 == 0  ) _widec += 2048;
		break;
	}
				}
				break;
			}
		}
	}

	_keys = _zone_scanner_trans_keys + _zone_scanner_key_offsets[cs];
	_trans = _zone_scanner_index_offsets[cs];

	_klen = _zone_scanner_single_lengths[cs];
	if ( _klen > 0 ) {
		const short *_lower = _keys;
		const short *_mid;
		const short *_upper = _keys + _klen - 1;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( _widec < *_mid )
				_upper = _mid - 1;
			else if ( _widec > *_mid )
				_lower = _mid + 1;
			else {
				_trans += (unsigned int)(_mid - _keys);
				goto _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _zone_scanner_range_lengths[cs];
	if ( _klen > 0 ) {
		const short *_lower = _keys;
		const short *_mid;
		const short *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( _widec < _mid[0] )
				_upper = _mid - 2;
			else if ( _widec > _mid[1] )
				_lower = _mid + 2;
			else {
				_trans += (unsigned int)((_mid - _keys)>>1);
				goto _match;
			}
		}
		_trans += _klen;
	}

_match:
	_trans = _zone_scanner_indicies[_trans];
	cs = _zone_scanner_trans_targs[_trans];

	if ( _zone_scanner_trans_actions[_trans] == 0 )
		goto _again;

	_acts = _zone_scanner_actions + _zone_scanner_trans_actions[_trans];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 )
	{
		switch ( *_acts++ )
		{
	case 0:
#line 20 "./zscanner/scanner_body.rl"
	{
		p--; {cs = stack[--top]; goto _again;}
	}
	break;
	case 1:
#line 25 "./zscanner/scanner_body.rl"
	{
		s->line_counter++;
	}
	break;
	case 2:
#line 29 "./zscanner/scanner_body.rl"
	{
		if (s->multiline == true) {
			SCANNER_ERROR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {cs = 247; goto _again;}
		}
		s->multiline = true;
	}
	break;
	case 3:
#line 36 "./zscanner/scanner_body.rl"
	{
		if (s->multiline == false) {
			SCANNER_ERROR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {cs = 247; goto _again;}
		}
		s->multiline = false;
	}
	break;
	case 4:
#line 44 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_REST);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 5:
#line 67 "./zscanner/scanner_body.rl"
	{
		s->buffer_length = 0;
	}
	break;
	case 6:
#line 70 "./zscanner/scanner_body.rl"
	{
		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	break;
	case 7:
#line 75 "./zscanner/scanner_body.rl"
	{
		// Ending string in buffer.
		s->buffer[s->buffer_length++] = 0;

		// Error counter incrementation.
		s->error_counter++;

		// Initialization of fcall stack.
		top = 0;

		// Process error message.
		s->process_error(s);

		// Reset.
		s->error_code = KNOT_EOK;
		s->multiline = false;

		// In case of serious error, stop scanner.
		if (s->stop == true) {
			return -1;
		}
	}
	break;
	case 8:
#line 100 "./zscanner/scanner_body.rl"
	{ {cs = 1060; goto _again;} }
	break;
	case 9:
#line 104 "./zscanner/scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	break;
	case 10:
#line 108 "./zscanner/scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			SCANNER_WARNING(ZSCANNER_ELABEL_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 11:
#line 117 "./zscanner/scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			SCANNER_WARNING(ZSCANNER_EDNAME_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 12:
#line 127 "./zscanner/scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length] = 0;
			s->item_length++;
		} else {
			SCANNER_WARNING(ZSCANNER_ELABEL_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 13:
#line 136 "./zscanner/scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	break;
	case 14:
#line 140 "./zscanner/scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
	break;
	case 15:
#line 143 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 16:
#line 162 "./zscanner/scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	break;
	case 17:
#line 165 "./zscanner/scanner_body.rl"
	{
		memcpy(s->dname + s->dname_tmp_length,
		       s->zone_origin,
		       s->zone_origin_length);

		s->dname_tmp_length += s->zone_origin_length;

		if (s->dname_tmp_length > MAX_DNAME_LENGTH) {
			SCANNER_WARNING(ZSCANNER_EDNAME_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 18:
#line 177 "./zscanner/scanner_body.rl"
	{
		memcpy(s->dname,
		       s->zone_origin,
		       s->zone_origin_length);

		s->dname_tmp_length = s->zone_origin_length;
	}
	break;
	case 19:
#line 185 "./zscanner/scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	break;
	case 20:
#line 189 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_DNAME_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 21:
#line 201 "./zscanner/scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 249; goto _again;} }
	break;
	case 22:
#line 205 "./zscanner/scanner_body.rl"
	{
		s->item_length_location = rdata_tail++;
	}
	break;
	case 23:
#line 208 "./zscanner/scanner_body.rl"
	{
		s->item_length = rdata_tail - s->item_length_location - 1;

		if (s->item_length <= MAX_ITEM_LENGTH) {
			*(s->item_length_location) = (uint8_t)(s->item_length);
		} else {
			SCANNER_WARNING(ZSCANNER_EITEM_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 24:
#line 219 "./zscanner/scanner_body.rl"
	{
		s->r_data_blocks[++(s->r_data_blocks_count)] =
			rdata_tail - s->r_data;
	}
	break;
	case 25:
#line 229 "./zscanner/scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	break;
	case 26:
#line 233 "./zscanner/scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	break;
	case 27:
#line 236 "./zscanner/scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			SCANNER_WARNING(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 28:
#line 242 "./zscanner/scanner_body.rl"
	{
		s->r_owner_length = 0;
		SCANNER_WARNING(ZSCANNER_EBAD_OWNER);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 29:
#line 254 "./zscanner/scanner_body.rl"
	{
		s->dname = rdata_tail;
	}
	break;
	case 30:
#line 257 "./zscanner/scanner_body.rl"
	{
		rdata_tail += s->dname_tmp_length;
	}
	break;
	case 31:
#line 265 "./zscanner/scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 32:
#line 282 "./zscanner/scanner_body.rl"
	{
		s->number64 = 0;
	}
	break;
	case 33:
#line 285 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 34:
#line 293 "./zscanner/scanner_body.rl"
	{
		s->decimal_counter = 0;
	}
	break;
	case 35:
#line 296 "./zscanner/scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
	break;
	case 36:
#line 299 "./zscanner/scanner_body.rl"
	{
		s->decimal_counter++;
	}
	break;
	case 37:
#line 303 "./zscanner/scanner_body.rl"
	{
		if (s->decimal_counter == 0 && s->number64 < UINT32_MAX) {
			s->number64 *= pow(10, s->decimals);
		} else if (s->decimal_counter <= s->decimals &&
				 s->number64_tmp < UINT32_MAX) {
			s->number64 *= pow(10, s->decimals - s->decimal_counter);
			s->number64 += s->number64_tmp * pow(10, s->decimals);
		} else {
			SCANNER_WARNING(ZSCANNER_EFLOAT_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 38:
#line 320 "./zscanner/scanner_body.rl"
	{
		s->decimals = 2;
	}
	break;
	case 39:
#line 323 "./zscanner/scanner_body.rl"
	{
		s->decimals = 3;
	}
	break;
	case 40:
#line 332 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= UINT8_MAX) {
			*rdata_tail = (uint8_t)(s->number64);
			rdata_tail += 1;
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 41:
#line 341 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			*((uint16_t *)rdata_tail) = htons((uint16_t)(s->number64));
			rdata_tail += 2;
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 42:
#line 350 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			*((uint32_t *)rdata_tail) = htonl((uint32_t)(s->number64));
			rdata_tail += 4;
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 43:
#line 360 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 44:
#line 369 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_data_length = (uint16_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 45:
#line 386 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_TIME_UNIT);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 46:
#line 393 "./zscanner/scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  SCANNER_WARNING(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {cs = 247; goto _again;}
	              }
	            }
	break;
	case 47:
#line 400 "./zscanner/scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  SCANNER_WARNING(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {cs = 247; goto _again;}
	              }
	            }
	break;
	case 48:
#line 407 "./zscanner/scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  SCANNER_WARNING(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {cs = 247; goto _again;}
	              }
	            }
	break;
	case 49:
#line 414 "./zscanner/scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  SCANNER_WARNING(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {cs = 247; goto _again;}
	              }
	            }
	break;
	case 50:
#line 424 "./zscanner/scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
	break;
	case 51:
#line 427 "./zscanner/scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 52:
#line 445 "./zscanner/scanner_body.rl"
	{
		s->buffer_length = 0;
	}
	break;
	case 53:
#line 448 "./zscanner/scanner_body.rl"
	{
		if (s->buffer_length < MAX_RDATA_LENGTH) {
			s->buffer[s->buffer_length++] = (*p);
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 54:
#line 456 "./zscanner/scanner_body.rl"
	{
		s->buffer[s->buffer_length] = 0;

		if (s->buffer_length == 14) { // Date; 14 = len("YYYYMMDDHHmmSS").
			ret = date_to_timestamp(s->buffer, &timestamp);

			if (ret == KNOT_EOK) {
				*((uint32_t *)rdata_tail) = htonl(timestamp);
				rdata_tail += 4;
			} else {
				SCANNER_WARNING(ret);
				p--; {cs = 247; goto _again;}
			}
		} else if (s->buffer_length <= 10) { // Timestamp format.
			char *end;

			s->number64 = strtoull((char *)(s->buffer), &end,  10);

			if (end == (char *)(s->buffer) || *end != '\0') {
				SCANNER_WARNING(ZSCANNER_EBAD_TIMESTAMP);
				p--; {cs = 247; goto _again;}
			}

			if (s->number64 <= UINT32_MAX) {
				*((uint32_t *)rdata_tail) = htonl((uint32_t)s->number64);
				rdata_tail += 4;
			} else {
				SCANNER_WARNING(ZSCANNER_ENUMBER32_OVERFLOW);
				p--; {cs = 247; goto _again;}
			}
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_TIMESTAMP_LENGTH);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 55:
#line 491 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_TIMESTAMP_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 56:
#line 501 "./zscanner/scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*(rdata_tail++) = (*p);
		} else {
			SCANNER_WARNING(ZSCANNER_ETEXT_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 57:
#line 509 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_TEXT_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 58:
#line 513 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_TEXT);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 59:
#line 518 "./zscanner/scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = 0;
			s->item_length++;
		} else {
			SCANNER_WARNING(ZSCANNER_ETEXT_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 60:
#line 527 "./zscanner/scanner_body.rl"
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + ASCII_0)
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 61:
#line 540 "./zscanner/scanner_body.rl"
	{
		rdata_tail++;
	}
	break;
	case 62:
#line 543 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 63:
#line 565 "./zscanner/scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 259; goto _again;} }
	break;
	case 64:
#line 575 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->default_ttl = (uint32_t)(s->number64);
		} else {
			SCANNER_ERROR(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 65:
#line 583 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_TTL);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 66:
#line 590 "./zscanner/scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 271; goto _again;} }
	break;
	case 67:
#line 594 "./zscanner/scanner_body.rl"
	{
		s->dname = s->zone_origin;
	}
	break;
	case 68:
#line 597 "./zscanner/scanner_body.rl"
	{
		s->zone_origin_length = s->dname_tmp_length;
	}
	break;
	case 69:
#line 600 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_ORIGIN);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 70:
#line 607 "./zscanner/scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 280; goto _again;} }
	break;
	case 71:
#line 611 "./zscanner/scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	break;
	case 72:
#line 614 "./zscanner/scanner_body.rl"
	{
		*rdata_tail = 0; // Ending filename string.
		strncpy((char*)(s->include_filename), (char*)(s->r_data),
		        sizeof(s->include_filename));

		// Check for correct string copy.
		if (strlen(s->include_filename) !=
		    (size_t)(rdata_tail - s->r_data)) {
			SCANNER_ERROR(ZSCANNER_EBAD_INCLUDE_FILENAME);
			p--; {cs = 247; goto _again;}
		}

		// For detection whether origin is not present.
		s->dname = NULL;
	}
	break;
	case 73:
#line 629 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_INCLUDE_FILENAME);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 74:
#line 634 "./zscanner/scanner_body.rl"
	{
		s->dname = s->r_data;
	}
	break;
	case 75:
#line 637 "./zscanner/scanner_body.rl"
	{
		s->r_data_length = s->dname_tmp_length;
	}
	break;
	case 76:
#line 640 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_INCLUDE_ORIGIN);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 77:
#line 645 "./zscanner/scanner_body.rl"
	{
		char text_origin[MAX_DNAME_LENGTH];

		// Origin conversion from wire to text form.
		if (s->dname == NULL) { // Use current origin.
			wire_dname_to_str(s->zone_origin,
			                  s->zone_origin_length,
					  text_origin);
		} else { // Use specified origin.
			wire_dname_to_str(s->r_data,
			                  s->r_data_length,
					  text_origin);
		}

		if (s->include_filename[0] != '/') { // Relative file path.
			// Get absolute path of the current zone file.
			if (realpath(s->file_name, (char*)(s->buffer)) != NULL) {
				char *full_current_zone_file_name =
					strdup((char*)(s->buffer));

				// Creating full include file name.
				snprintf((char*)(s->buffer), sizeof(s->buffer),
				        "%s/%s",
				        dirname(full_current_zone_file_name),
				        s->include_filename);

				free(full_current_zone_file_name);
			} else {
				SCANNER_ERROR(ZSCANNER_EUNPROCESSED_INCLUDE);
				p--; {cs = 247; goto _again;}
			}
		} else {
			strncpy((char*)(s->buffer), (char*)(s->include_filename),
			        sizeof(s->buffer));
		}

		// Create new file loader for included zone file.
		file_loader_t *fl = file_loader_create((char*)(s->buffer),
		                                       text_origin,
		                                       DEFAULT_CLASS,
		                                       DEFAULT_TTL,
		                                       s->process_record,
		                                       s->process_error,
		                                       s->data);
		if (fl != NULL) {
			// Process included zone file.
			ret = file_loader_process(fl);
			file_loader_free(fl);

			if (ret != 0) {
				SCANNER_ERROR(ZSCANNER_EUNPROCESSED_INCLUDE);
				p--; {cs = 247; goto _again;}
			}
		} else {
			SCANNER_ERROR(ZSCANNER_EUNOPENED_INCLUDE);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 78:
#line 711 "./zscanner/scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 293; goto _again;} }
	break;
	case 79:
#line 717 "./zscanner/scanner_body.rl"
	{
		s->stop = true;
	}
	break;
	case 80:
#line 721 "./zscanner/scanner_body.rl"
	{
		s->stop = false;
	}
	break;
	case 81:
#line 724 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_DIRECTIVE);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 82:
#line 736 "./zscanner/scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	break;
	case 83:
#line 740 "./zscanner/scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	break;
	case 84:
#line 744 "./zscanner/scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
	break;
	case 85:
#line 748 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 86:
#line 763 "./zscanner/scanner_body.rl"
	{
		s->buffer_length = 0;
	}
	break;
	case 87:
#line 766 "./zscanner/scanner_body.rl"
	{
		if (s->buffer_length < MAX_RDATA_LENGTH) {
			s->buffer[s->buffer_length++] = (*p);
		}
		else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 88:
#line 775 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_ADDRESS_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 89:
#line 780 "./zscanner/scanner_body.rl"
	{
		s->buffer[s->buffer_length] = 0;

		if (inet_pton(AF_INET, (char *)s->buffer, &addr4) <= 0) {
			SCANNER_WARNING(ZSCANNER_EBAD_IPV4);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 90:
#line 788 "./zscanner/scanner_body.rl"
	{
		memcpy(rdata_tail, &(addr4.s_addr), INET4_ADDR_LENGTH);
		rdata_tail += INET4_ADDR_LENGTH;
	}
	break;
	case 91:
#line 793 "./zscanner/scanner_body.rl"
	{
		s->buffer[s->buffer_length] = 0;

		if (inet_pton(AF_INET6, (char *)s->buffer, &addr6) <= 0) {
			SCANNER_WARNING(ZSCANNER_EBAD_IPV6);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 92:
#line 801 "./zscanner/scanner_body.rl"
	{
		memcpy(rdata_tail, &(addr6.s6_addr), INET6_ADDR_LENGTH);
		rdata_tail += INET6_ADDR_LENGTH;
	}
	break;
	case 93:
#line 818 "./zscanner/scanner_body.rl"
	{
		memset(&(s->apl), 0, sizeof(s->apl));
	}
	break;
	case 94:
#line 821 "./zscanner/scanner_body.rl"
	{
		s->apl.excl_flag = 128; // dec 128  = bin 10000000.
	}
	break;
	case 95:
#line 824 "./zscanner/scanner_body.rl"
	{
		s->apl.addr_family = 1;
	}
	break;
	case 96:
#line 827 "./zscanner/scanner_body.rl"
	{
		s->apl.addr_family = 2;
	}
	break;
	case 97:
#line 830 "./zscanner/scanner_body.rl"
	{
		if ((s->apl.addr_family == 1 && s->number64 <= 32) ||
		    (s->apl.addr_family == 2 && s->number64 <= 128)) {
			s->apl.prefix_length = (uint8_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_APL);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 98:
#line 839 "./zscanner/scanner_body.rl"
	{
		// Write address family.
		*((uint16_t *)rdata_tail) = htons(s->apl.addr_family);
		rdata_tail += 2;
		// Write prefix length in bites.
		*(rdata_tail) = s->apl.prefix_length;
		rdata_tail += 1;
		// Copy address to buffer.
		uint8_t len;
		switch (s->apl.addr_family) {
		case 1:
			len = INET4_ADDR_LENGTH;
			memcpy(s->buffer, &(addr4.s_addr), len);
			break;
		case 2:
			len = INET6_ADDR_LENGTH;
			memcpy(s->buffer, &(addr6.s6_addr), len);
			break;
		default:
			SCANNER_WARNING(ZSCANNER_EBAD_APL);
			p--; {cs = 247; goto _again;}
		}
		// Find prefix without trailing zeroes.
		while (len > 0) {
			if ((s->buffer[len - 1] & 255) != 0) {
				break;
			}
			len--;
		}
		// Write negation flag + prefix length in bytes.
		*(rdata_tail) = len + s->apl.excl_flag;
		rdata_tail += 1;
		// Write address prefix non-null data.
		memcpy(rdata_tail, s->buffer, len);
		rdata_tail += len;
	}
	break;
	case 99:
#line 875 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_APL);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 100:
#line 893 "./zscanner/scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = first_hex_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 101:
#line 901 "./zscanner/scanner_body.rl"
	{
		*rdata_tail += second_hex_to_num[(uint8_t)(*p)];
		rdata_tail++;
	}
	break;
	case 102:
#line 905 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_HEX_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 103:
#line 919 "./zscanner/scanner_body.rl"
	{
		if ((rdata_tail - s->r_data) != s->r_data_length) {
			SCANNER_WARNING(ZSCANNER_EBAD_RDATA_LENGTH);
			p--; {cs = 247; goto _again;}
		}

		ret = find_rdata_blocks(s);
		if (ret != KNOT_EOK) {
			SCANNER_WARNING(ret);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 104:
#line 932 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_HEX_RDATA);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 105:
#line 942 "./zscanner/scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = first_base64_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 106:
#line 950 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) += second_left_base64_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = second_right_base64_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 107:
#line 960 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) += third_left_base64_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = third_right_base64_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 108:
#line 970 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) += fourth_base64_to_num[(uint8_t)(*p)];
	}
	break;
	case 109:
#line 974 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_BASE64_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 110:
#line 996 "./zscanner/scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 311; goto _again;} }
	break;
	case 111:
#line 1000 "./zscanner/scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = first_base32hex_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 112:
#line 1008 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) += second_left_base32hex_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = second_right_base32hex_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 113:
#line 1018 "./zscanner/scanner_body.rl"
	{
		*rdata_tail += third_base32hex_to_num[(uint8_t)(*p)];
	}
	break;
	case 114:
#line 1021 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) += fourth_left_base32hex_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = fourth_right_base32hex_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 115:
#line 1031 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) += fifth_left_base32hex_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = fifth_right_base32hex_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 116:
#line 1041 "./zscanner/scanner_body.rl"
	{
		*rdata_tail += sixth_base32hex_to_num[(uint8_t)(*p)];
	}
	break;
	case 117:
#line 1044 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) += seventh_left_base32hex_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = seventh_right_base32hex_to_num[(uint8_t)(*p)];
		} else {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 118:
#line 1054 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) += eighth_base32hex_to_num[(uint8_t)(*p)];
	}
	break;
	case 119:
#line 1058 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_BASE32HEX_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 120:
#line 1093 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 0;
	}
	break;
	case 121:
#line 1096 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 1;
	}
	break;
	case 122:
#line 1099 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 2;
	}
	break;
	case 123:
#line 1102 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 3;
	}
	break;
	case 124:
#line 1105 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 5;
	}
	break;
	case 125:
#line 1108 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 6;
	}
	break;
	case 126:
#line 1111 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 7;
	}
	break;
	case 127:
#line 1114 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 8;
	}
	break;
	case 128:
#line 1117 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 10;
	}
	break;
	case 129:
#line 1120 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 12;
	}
	break;
	case 130:
#line 1123 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 13;
	}
	break;
	case 131:
#line 1126 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 14;
	}
	break;
	case 132:
#line 1129 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 252;
	}
	break;
	case 133:
#line 1132 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 253;
	}
	break;
	case 134:
#line 1135 "./zscanner/scanner_body.rl"
	{
		*(rdata_tail++) = 254;
	}
	break;
	case 135:
#line 1139 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(1);
		rdata_tail += 2;
	}
	break;
	case 136:
#line 1143 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(2);
		rdata_tail += 2;
	}
	break;
	case 137:
#line 1147 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(3);
		rdata_tail += 2;
	}
	break;
	case 138:
#line 1151 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(4);
		rdata_tail += 2;
	}
	break;
	case 139:
#line 1155 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(5);
		rdata_tail += 2;
	}
	break;
	case 140:
#line 1159 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(6);
		rdata_tail += 2;
	}
	break;
	case 141:
#line 1163 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(7);
		rdata_tail += 2;
	}
	break;
	case 142:
#line 1167 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(8);
		rdata_tail += 2;
	}
	break;
	case 143:
#line 1171 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(253);
		rdata_tail += 2;
	}
	break;
	case 144:
#line 1175 "./zscanner/scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(254);
		rdata_tail += 2;
	}
	break;
	case 145:
#line 1182 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_GATEWAY);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 146:
#line 1186 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_GATEWAY_KEY);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 147:
#line 1204 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 148:
#line 1210 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_A, &rdata_tail); }
	break;
	case 149:
#line 1211 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NS, &rdata_tail); }
	break;
	case 150:
#line 1212 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_CNAME, &rdata_tail); }
	break;
	case 151:
#line 1213 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_SOA, &rdata_tail); }
	break;
	case 152:
#line 1214 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_PTR, &rdata_tail); }
	break;
	case 153:
#line 1215 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_HINFO, &rdata_tail); }
	break;
	case 154:
#line 1216 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_MINFO, &rdata_tail); }
	break;
	case 155:
#line 1217 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_MX, &rdata_tail); }
	break;
	case 156:
#line 1218 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_TXT, &rdata_tail); }
	break;
	case 157:
#line 1219 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_RP, &rdata_tail); }
	break;
	case 158:
#line 1220 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_AFSDB, &rdata_tail); }
	break;
	case 159:
#line 1221 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_RT, &rdata_tail); }
	break;
	case 160:
#line 1222 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_KEY, &rdata_tail); }
	break;
	case 161:
#line 1223 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_AAAA, &rdata_tail); }
	break;
	case 162:
#line 1224 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_LOC, &rdata_tail); }
	break;
	case 163:
#line 1225 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_SRV, &rdata_tail); }
	break;
	case 164:
#line 1226 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NAPTR, &rdata_tail); }
	break;
	case 165:
#line 1227 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_KX, &rdata_tail); }
	break;
	case 166:
#line 1228 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_CERT, &rdata_tail); }
	break;
	case 167:
#line 1229 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_DNAME, &rdata_tail); }
	break;
	case 168:
#line 1230 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_APL, &rdata_tail); }
	break;
	case 169:
#line 1231 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_DS, &rdata_tail); }
	break;
	case 170:
#line 1232 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_SSHFP, &rdata_tail); }
	break;
	case 171:
#line 1233 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_IPSECKEY, &rdata_tail); }
	break;
	case 172:
#line 1234 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_RRSIG, &rdata_tail); }
	break;
	case 173:
#line 1235 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NSEC, &rdata_tail); }
	break;
	case 174:
#line 1236 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_DNSKEY, &rdata_tail); }
	break;
	case 175:
#line 1237 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_DHCID, &rdata_tail); }
	break;
	case 176:
#line 1238 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NSEC3, &rdata_tail); }
	break;
	case 177:
#line 1239 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NSEC3PARAM, &rdata_tail); }
	break;
	case 178:
#line 1240 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_TLSA, &rdata_tail); }
	break;
	case 179:
#line 1241 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_SPF, &rdata_tail); }
	break;
	case 180:
#line 1242 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NID, &rdata_tail); }
	break;
	case 181:
#line 1243 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_L32, &rdata_tail); }
	break;
	case 182:
#line 1244 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_L64, &rdata_tail); }
	break;
	case 183:
#line 1245 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_LP, &rdata_tail); }
	break;
	case 184:
#line 1246 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_EUI48, &rdata_tail); }
	break;
	case 185:
#line 1247 "./zscanner/scanner_body.rl"
	{ type_num(KNOT_RRTYPE_EUI64, &rdata_tail); }
	break;
	case 186:
#line 1253 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			SCANNER_WARNING(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 187:
#line 1266 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
	break;
	case 188:
#line 1267 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
	break;
	case 189:
#line 1268 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
	break;
	case 190:
#line 1269 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
	break;
	case 191:
#line 1270 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
	break;
	case 192:
#line 1271 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
	break;
	case 193:
#line 1272 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
	break;
	case 194:
#line 1273 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
	break;
	case 195:
#line 1274 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
	break;
	case 196:
#line 1275 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
	break;
	case 197:
#line 1276 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
	break;
	case 198:
#line 1277 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
	break;
	case 199:
#line 1278 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
	break;
	case 200:
#line 1279 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
	break;
	case 201:
#line 1280 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
	break;
	case 202:
#line 1281 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
	break;
	case 203:
#line 1282 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
	break;
	case 204:
#line 1283 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
	break;
	case 205:
#line 1284 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
	break;
	case 206:
#line 1285 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
	break;
	case 207:
#line 1286 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
	break;
	case 208:
#line 1287 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
	break;
	case 209:
#line 1288 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
	break;
	case 210:
#line 1289 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
	break;
	case 211:
#line 1290 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
	break;
	case 212:
#line 1291 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
	break;
	case 213:
#line 1292 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
	break;
	case 214:
#line 1293 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
	break;
	case 215:
#line 1294 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
	break;
	case 216:
#line 1295 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
	break;
	case 217:
#line 1296 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
	break;
	case 218:
#line 1297 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
	break;
	case 219:
#line 1298 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
	break;
	case 220:
#line 1299 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
	break;
	case 221:
#line 1300 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
	break;
	case 222:
#line 1301 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
	break;
	case 223:
#line 1302 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
	break;
	case 224:
#line 1303 "./zscanner/scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
	break;
	case 225:
#line 1307 "./zscanner/scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
	break;
	case 226:
#line 1311 "./zscanner/scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
					p--; {cs = 247; goto _again;}
				}
			}
		}
	}
	break;
	case 227:
#line 1334 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_BITMAP);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 228:
#line 1342 "./zscanner/scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 318; goto _again;} }
	break;
	case 229:
#line 1346 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= 90) {
			s->loc.d1 = (uint32_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 230:
#line 1354 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= 180) {
			s->loc.d2 = (uint32_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 231:
#line 1362 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= 59) {
			s->loc.m1 = (uint32_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 232:
#line 1370 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= 59) {
			s->loc.m2 = (uint32_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 233:
#line 1378 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= 59999) {
			s->loc.s1 = (uint32_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 234:
#line 1386 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= 59999) {
			s->loc.s2 = (uint32_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 235:
#line 1394 "./zscanner/scanner_body.rl"
	{
		if ((s->loc.alt_sign ==  1 && s->number64 <= 4284967295) ||
		    (s->loc.alt_sign == -1 && s->number64 <=   10000000))
		{
			s->loc.alt = (uint32_t)(s->number64);
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 236:
#line 1404 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= 9000000000ULL) {
			s->loc.siz = s->number64;
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 237:
#line 1412 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= 9000000000ULL) {
			s->loc.hp = s->number64;
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 238:
#line 1420 "./zscanner/scanner_body.rl"
	{
		if (s->number64 <= 9000000000ULL) {
			s->loc.vp = s->number64;
		} else {
			SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 239:
#line 1428 "./zscanner/scanner_body.rl"
	{
		s->loc.lat_sign = -1;
	}
	break;
	case 240:
#line 1431 "./zscanner/scanner_body.rl"
	{
		s->loc.long_sign = -1;
	}
	break;
	case 241:
#line 1434 "./zscanner/scanner_body.rl"
	{
		s->loc.alt_sign = -1;
	}
	break;
	case 242:
#line 1451 "./zscanner/scanner_body.rl"
	{
		memset(&(s->loc), 0, sizeof(s->loc));
		// Defaults.
		s->loc.siz = 100;
		s->loc.vp  = 1000;
		s->loc.hp  = 1000000;
		s->loc.lat_sign  = 1;
		s->loc.long_sign = 1;
		s->loc.alt_sign  = 1;
	}
	break;
	case 243:
#line 1461 "./zscanner/scanner_body.rl"
	{
		// Write version.
		*(rdata_tail) = 0;
		rdata_tail += 1;
		// Write size.
		*(rdata_tail) = loc64to8(s->loc.siz);
		rdata_tail += 1;
		// Write horizontal precision.
		*(rdata_tail) = loc64to8(s->loc.hp);
		rdata_tail += 1;
		// Write vertical precision.
		*(rdata_tail) = loc64to8(s->loc.vp);
		rdata_tail += 1;
		// Write latitude.
		*((uint32_t *)rdata_tail) = htonl(LOC_LAT_ZERO + s->loc.lat_sign *
			(3600000 * s->loc.d1 + 60000 * s->loc.m1 + s->loc.s1));
		rdata_tail += 4;
		// Write longitude.
		*((uint32_t *)rdata_tail) = htonl(LOC_LONG_ZERO + s->loc.long_sign *
			(3600000 * s->loc.d2 + 60000 * s->loc.m2 + s->loc.s2));
		rdata_tail += 4;
		// Write altitude.
		*((uint32_t *)rdata_tail) = htonl(LOC_ALT_ZERO + s->loc.alt_sign *
			(s->loc.alt));
		rdata_tail += 4;
	}
	break;
	case 244:
#line 1487 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_LOC_DATA);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 245:
#line 1500 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_HEX_RDATA);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 246:
#line 1518 "./zscanner/scanner_body.rl"
	{
		s->item_length = 0;
	}
	break;
	case 247:
#line 1521 "./zscanner/scanner_body.rl"
	{
		s->item_length++;
	}
	break;
	case 248:
#line 1524 "./zscanner/scanner_body.rl"
	{
		if (s->item_length != 6) {
			SCANNER_WARNING(ZSCANNER_EBAD_EUI_LENGTH);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 249:
#line 1530 "./zscanner/scanner_body.rl"
	{
		if (s->item_length != 8) {
			SCANNER_WARNING(ZSCANNER_EBAD_EUI_LENGTH);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 250:
#line 1536 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_CHAR_DASH);                       
		p--; {cs = 247; goto _again;}
	}
	break;
	case 251:
#line 1551 "./zscanner/scanner_body.rl"
	{
		s->item_length = 0;
	}
	break;
	case 252:
#line 1554 "./zscanner/scanner_body.rl"
	{
		s->item_length++;
	}
	break;
	case 253:
#line 1557 "./zscanner/scanner_body.rl"
	{
		if (s->item_length != 4) {
			SCANNER_WARNING(ZSCANNER_EBAD_L64_LENGTH);
			p--; {cs = 247; goto _again;}
		}
	}
	break;
	case 254:
#line 1563 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_CHAR_COLON);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 255:
#line 1576 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_ALGORITHM);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 256:
#line 1580 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_CERT_TYPE);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 257:
#line 1602 "./zscanner/scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 457; goto _again;} }
	break;
	case 258:
#line 1617 "./zscanner/scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 552; goto _again;} }
	break;
	case 259:
#line 1621 "./zscanner/scanner_body.rl"
	{
		s->r_data_blocks[0] = 0;
		s->r_data_blocks_count = 0;
		rdata_tail = s->r_data;
	}
	break;
	case 260:
#line 1626 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_RDATA);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 261:
#line 1744 "./zscanner/scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = cs; cs = 593; goto _again;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = cs; cs = 595; goto _again;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = cs; cs = 597; goto _again;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = cs; cs = 629; goto _again;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = cs; cs = 634; goto _again;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = cs; cs = 639; goto _again;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = cs; cs = 644; goto _again;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = cs; cs = 648; goto _again;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = cs; cs = 650; goto _again;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = cs; cs = 705; goto _again;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = cs; cs = 716; goto _again;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = cs; cs = 733; goto _again;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = cs; cs = 744; goto _again;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = cs; cs = 755; goto _again;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = cs; cs = 768; goto _again;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = cs; cs = 778; goto _again;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = cs; cs = 817; goto _again;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = cs; cs = 959; goto _again;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = cs; cs = 962; goto _again;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = cs; cs = 973; goto _again;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = cs; cs = 975; goto _again;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = cs; cs = 1004; goto _again;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = cs; cs = 1017; goto _again;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = cs; cs = 1035; goto _again;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = cs; cs = 1030; goto _again;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = cs; cs = 1048; goto _again;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = cs; cs = 1054; goto _again;}
		default:
			SCANNER_WARNING(ZSCANNER_ECANNOT_TEXT_DATA);
			{cs = 247; goto _again;}
		}
	}
	break;
	case 262:
#line 1817 "./zscanner/scanner_body.rl"
	{
		switch (s->r_type) {
		// Next types must not have empty rdata.
		case KNOT_RRTYPE_A:
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
		case KNOT_RRTYPE_SOA:
		case KNOT_RRTYPE_HINFO:
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
		case KNOT_RRTYPE_RP:
		case KNOT_RRTYPE_AAAA:
		case KNOT_RRTYPE_LOC:
		case KNOT_RRTYPE_SRV:
		case KNOT_RRTYPE_NAPTR:
		case KNOT_RRTYPE_CERT:
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_SSHFP:
		case KNOT_RRTYPE_IPSECKEY:
		case KNOT_RRTYPE_RRSIG:
		case KNOT_RRTYPE_NSEC:
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_NSEC3:
		case KNOT_RRTYPE_NSEC3PARAM:
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L32:
		case KNOT_RRTYPE_L64:
		case KNOT_RRTYPE_LP:
		case KNOT_RRTYPE_EUI48:
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = cs; cs = 438; goto _again;}
		// Next types can have empty rdata.
		case KNOT_RRTYPE_APL:
		default:
			{stack[top++] = cs; cs = 447; goto _again;}
		}
	}
	break;
	case 263:
#line 1865 "./zscanner/scanner_body.rl"
	{
		s->r_data_blocks[++(s->r_data_blocks_count)] =
			(uint16_t)(rdata_tail - s->r_data);
	}
	break;
	case 264:
#line 1873 "./zscanner/scanner_body.rl"
	{ p--; }
	break;
	case 265:
#line 1880 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 266:
#line 1886 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
	break;
	case 267:
#line 1887 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
	break;
	case 268:
#line 1888 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	break;
	case 269:
#line 1889 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
	break;
	case 270:
#line 1890 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
	break;
	case 271:
#line 1891 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	break;
	case 272:
#line 1892 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	break;
	case 273:
#line 1893 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
	break;
	case 274:
#line 1894 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
	break;
	case 275:
#line 1895 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
	break;
	case 276:
#line 1896 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	break;
	case 277:
#line 1897 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
	break;
	case 278:
#line 1898 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
	break;
	case 279:
#line 1899 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	break;
	case 280:
#line 1900 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
	break;
	case 281:
#line 1901 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
	break;
	case 282:
#line 1902 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	break;
	case 283:
#line 1903 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
	break;
	case 284:
#line 1904 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
	break;
	case 285:
#line 1905 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	break;
	case 286:
#line 1906 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
	break;
	case 287:
#line 1907 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
	break;
	case 288:
#line 1908 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	break;
	case 289:
#line 1909 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	break;
	case 290:
#line 1910 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	break;
	case 291:
#line 1911 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	break;
	case 292:
#line 1912 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	break;
	case 293:
#line 1913 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	break;
	case 294:
#line 1914 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	break;
	case 295:
#line 1915 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	break;
	case 296:
#line 1916 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	break;
	case 297:
#line 1917 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
	break;
	case 298:
#line 1918 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
	break;
	case 299:
#line 1919 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
	break;
	case 300:
#line 1920 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
	break;
	case 301:
#line 1921 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
	break;
	case 302:
#line 1922 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	break;
	case 303:
#line 1923 "./zscanner/scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	break;
	case 304:
#line 1929 "./zscanner/scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			SCANNER_WARNING(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 247; goto _again;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	break;
#line 7499 "zscanner/scanner.c"
		}
	}

_again:
	if ( cs == 0 )
		goto _out;
	if ( ++p != pe )
		goto _resume;
	_test_eof: {}
	if ( p == eof )
	{
	const short *__acts = _zone_scanner_actions + _zone_scanner_eof_actions[cs];
	unsigned int __nacts = (unsigned int) *__acts++;
	while ( __nacts-- > 0 ) {
		switch ( *__acts++ ) {
	case 4:
#line 44 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_REST);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 15:
#line 143 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 20:
#line 189 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_DNAME_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 28:
#line 242 "./zscanner/scanner_body.rl"
	{
		s->r_owner_length = 0;
		SCANNER_WARNING(ZSCANNER_EBAD_OWNER);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 33:
#line 285 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 45:
#line 386 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_TIME_UNIT);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 55:
#line 491 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_TIMESTAMP_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 57:
#line 509 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_TEXT_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 58:
#line 513 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_TEXT);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 62:
#line 543 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 65:
#line 583 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_TTL);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 69:
#line 600 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_ORIGIN);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 73:
#line 629 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_INCLUDE_FILENAME);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 76:
#line 640 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_INCLUDE_ORIGIN);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 80:
#line 721 "./zscanner/scanner_body.rl"
	{
		s->stop = false;
	}
	break;
	case 81:
#line 724 "./zscanner/scanner_body.rl"
	{
		SCANNER_ERROR(ZSCANNER_EBAD_DIRECTIVE);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 88:
#line 775 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_ADDRESS_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 99:
#line 875 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_APL);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 102:
#line 905 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_HEX_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 104:
#line 932 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_HEX_RDATA);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 109:
#line 974 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_BASE64_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 119:
#line 1058 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_BASE32HEX_CHAR);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 145:
#line 1182 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_GATEWAY);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 146:
#line 1186 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_GATEWAY_KEY);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 147:
#line 1204 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 227:
#line 1334 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_BITMAP);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 244:
#line 1487 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_LOC_DATA);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 245:
#line 1500 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_HEX_RDATA);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 250:
#line 1536 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_CHAR_DASH);                       
		p--; {cs = 247; goto _again;}
	}
	break;
	case 254:
#line 1563 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_CHAR_COLON);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 255:
#line 1576 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_ALGORITHM);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 256:
#line 1580 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_CERT_TYPE);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 260:
#line 1626 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EBAD_RDATA);
		p--; {cs = 247; goto _again;}
	}
	break;
	case 265:
#line 1880 "./zscanner/scanner_body.rl"
	{
		SCANNER_WARNING(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {cs = 247; goto _again;}
	}
	break;
#line 7753 "zscanner/scanner.c"
		}
	}
	}

	_out: {}
	}

#line 144 "./zscanner/scanner.rl"

	// Check if scanner state machine is in uncovered state.
	if (cs == zone_scanner_error) {
		SCANNER_ERROR(ZSCANNER_UNCOVERED_STATE);
		s->error_counter++;

		// Fill error context data.
		for (s->buffer_length = 0;
		     ((p + s->buffer_length) < pe) &&
		     (s->buffer_length < sizeof(s->buffer) - 1);
		     s->buffer_length++)
		{
			// Only rest of the current line.
			if (*(p + s->buffer_length) == '\n') {
				break;
			}
			s->buffer[s->buffer_length] = *(p + s->buffer_length);
		}

		// Ending string in buffer.
		s->buffer[s->buffer_length++] = 0;

		// Processing error.
		s->process_error(s);

		return -1;
	}

	// Storing scanner states.
	s->cs  = cs;
	s->top = top;
	memcpy(s->stack, stack, sizeof(stack));

	// Storing r_data pointer.
	s->r_data_tail = rdata_tail - s->r_data;

	// Check if any errors has occured.
	if (s->error_counter > 0) {
		return -1;
	}

	return 0;
}
