/* This file is part of the KDE project
 * Copyright ( C ) 2007 Thorsten Zachmann <zachmann@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (  at your option ) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef KPRANIMATIONDIRECTOR_H
#define KPRANIMATIONDIRECTOR_H

#include <QList>
#include <QMap>
#include <QObject>
#include <QPair>
#include <QTimeLine>

#include <KoZoomHandler.h>

// TODO remove see where it is used
#include "pageeffects/KPrCoverDownEffect.h"

class QPainter;
class QPaintEvent;
class KoViewConverter;
class KoShape;
class KoShapeManager;
class KoPACanvas;
class KoPAPageBase;
class KoPAView;
class KPrPageEffect;
class KPrPageEffectRunner;
class KPrAnimationData;
class KPrAnimationController;
class KPrShapeAnimation;

class KPrAnimationDirector : public QObject
{
    Q_OBJECT
public:
    enum Navigation
    {
        FirstPage,
        PreviousPage,
        PreviousStep,
        NextStep,
        NextPage,
        LastPage
    };

    KPrAnimationDirector( KoPAView * view, const QList<KoPAPageBase*> & pages );
    virtual ~KPrAnimationDirector();

    void paintEvent( QPaintEvent* event );

    KoViewConverter * viewConverter();

    /**
     * do the next step in the presentation
     */
    bool navigate( Navigation navigation );

    /**
     * Check if the shape is shown
     *
     * A shape is visible when there are no animations on it or when it
     * is animated at the moment even when it is a disappear animation. 
     */
    bool shapeShown( KoShape * shape );

    /**
     * Get the animation for the shape
     *
     * @param shape which should be animated
     * @return pair of the animation and the animation data for the shape or a 0, 0 if there is no animation
     */
    QPair<KPrShapeAnimation *, KPrAnimationData *> shapeAnimation( KoShape * shape );

protected:
    // set the page to be shon and update the UI
    void updateActivePage( KoPAPageBase * page );

    /**
     * change the page
     * @return true when there was a change
     */
    bool changePage( Navigation navigation );

    /**
     * Update to the next step
     */
    void nextStep();

    /**
     * Update to the previous step
     */
    void previousStep();

    // paint the given step to the painter
    void paintStep( QPainter & painter );

    /**
     * Finish the running shape animations
     */
    void finishAnimations();

    /**
     * Start the timeline
     */
    void startTimeLine( int duration );

    void animateShapes( int currentTime );

    // set the animations to the current m_stepIndex
    void updateAnimations();

    // helper method for updateAnimations
    void insertAnimations( KPrAnimationController * controller, KoShapeManager * shapeManager );

    // helper method for freeing the resources of the animations
    void clearAnimations();

    // check if there is a set animation in m_animations
    bool hasAnimation();

protected slots:
    // update the zoom value
    void updateZoom( const QSize & size );
    // acts on the time line event
    void animate();

private:
    KoPAView * m_view;
    KoPACanvas * m_canvas;
    QList<KoPAPageBase*> m_pages;

    KoZoomHandler m_zoomHandler;
    QPoint m_offset;
    QRect m_pageRect;

    KPrPageEffectRunner * m_pageEffectRunner;
    // TODO remove when we read the effect from the page
    QMap<KoShape *, QPair<KPrShapeAnimation *, KPrAnimationData *> > m_animations;
    QTimeLine m_timeLine;
    int m_pageIndex;
    int m_stepIndex;
    QList<int> m_steps;
    int m_maxShapeDuration;
    // true when there is an animtion in this step
    bool m_hasAnimation;
};

#endif /* KPRANIMATIONDIRECTOR_H */
