/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QtCore/QSettings>
#include <QtCore/QStringList>

#include "connectioninfomanager.h"

ConnectionInfo::ConnectionInfo()
{
}

ConnectionInfo::ConnectionInfo( const QString &name )
  : mName( name )
{
}

QString ConnectionInfo::name() const
{
  return mName;
}

bool ConnectionInfo::isValid() const
{
  return ( !mName.isEmpty() && !mHost.isEmpty() && !mBaseDn.isEmpty() &&
           !mNobodyUser.isEmpty() && !mNobodyPassword.isEmpty() );
}

void ConnectionInfo::setHost( const QString &host )
{
  mHost = host;
}

QString ConnectionInfo::host() const
{
  return mHost;
}

void ConnectionInfo::setPort( int port )
{
  mPort = port;
}

int ConnectionInfo::port() const
{
  return mPort;
}

void ConnectionInfo::setBaseDn( const QString &baseDn )
{
  mBaseDn = baseDn;
}

QString ConnectionInfo::baseDn() const
{
  return mBaseDn;
}

void ConnectionInfo::setNobodyUser( const QString &nobodyUser )
{
  mNobodyUser = nobodyUser;
}

QString ConnectionInfo::nobodyUser() const
{
  return mNobodyUser;
}

void ConnectionInfo::setNobodyPassword( const QString &nobodyPassword )
{
  mNobodyPassword = nobodyPassword;
}

QString ConnectionInfo::nobodyPassword() const
{
  return mNobodyPassword;
}



ConnectionInfoManager::ConnectionInfoManager()
{
}

ConnectionInfo::List ConnectionInfoManager::connectionInfos() const
{
  return mConnectionInfos;
}

ConnectionInfo ConnectionInfoManager::connectionInfo( const QString &name ) const
{
  for ( int i = 0; i < mConnectionInfos.count(); ++i )
    if ( mConnectionInfos[ i ].name() == name )
      return mConnectionInfos[ i ];

  return ConnectionInfo();
}

bool ConnectionInfoManager::contains( const QString &name ) const
{
  for ( int i = 0; i < mConnectionInfos.count(); ++i )
    if ( mConnectionInfos[ i ].name() == name )
      return true;

  return false;
}

void ConnectionInfoManager::appendConnectionInfo( const ConnectionInfo &connection )
{
  if ( !contains( connection.name() ) )
    mConnectionInfos.append( connection );
}

void ConnectionInfoManager::removeConnectionInfo( const QString &name )
{
  for ( int i = 0; i < mConnectionInfos.count(); ++i )
    if ( mConnectionInfos[ i ].name() == name ) {
      mConnectionInfos.removeAt( i );
      return;
    }
}

void ConnectionInfoManager::load()
{
  mConnectionInfos.clear();

  QSettings settings( "Kolab", "KolabAdmin" );
  settings.beginGroup( "connections" );

  const QStringList groups = settings.childGroups();
  for ( int i = 0; i < groups.count(); ++i ) {
    settings.beginGroup( groups[ i ] );
    ConnectionInfo connection( settings.value( "name" ).toString() );
    connection.setHost( settings.value( "host" ).toString() );
    connection.setPort( settings.value( "port" ).toInt() );
    connection.setBaseDn( settings.value( "basedn" ).toString() );
    connection.setNobodyUser( settings.value( "nobody_user" ).toString() );
    connection.setNobodyPassword( settings.value( "nobody_password" ).toString() );
    settings.endGroup();

    mConnectionInfos.append( connection );
  }

  settings.endGroup();
}

void ConnectionInfoManager::save()
{
  QSettings settings( "Kolab", "KolabAdmin" );

  // delete old settings
  settings.remove( "connections" );

  // write new settings
  settings.beginGroup( "connections" );

  for ( int i = 0; i < mConnectionInfos.count(); ++i ) {
    settings.beginGroup( QString::number( i ) );
    settings.setValue( "name", mConnectionInfos[ i ].name() );
    settings.setValue( "host", mConnectionInfos[ i ].host() );
    settings.setValue( "port", mConnectionInfos[ i ].port() );
    settings.setValue( "basedn", mConnectionInfos[ i ].baseDn() );
    settings.setValue( "nobody_user", mConnectionInfos[ i ].nobodyUser() );
    settings.setValue( "nobody_password", mConnectionInfos[ i ].nobodyPassword() );
    settings.endGroup();
  }

  settings.endGroup();
  settings.sync();
}
