//                       -*- mode: C++ -*-
//
// Copyright(C) 2005,2006,2007 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>
#include <assert.h>

#include "silcaccount.h"
#include "silcchannelcontact.h"
#include "silcchannelcontactinfowidget.h"
#include "silcbuddycontact.h"
#include "silcmessagemanager.h"

#include <kopetemetacontact.h>
#include <kopetemessage.h>
#include <kopetechatsession.h>
#include <kopetechatsessionmanager.h>
#include <kopeteuiglobal.h>
#include <ui/kopeteview.h>
#include <kdebug.h>
#include <qfileinfo.h>
#include <qcstring.h>
#include <kmimemagic.h>
#include <kfiledialog.h>

#include <klocale.h>
#include <kmessagebox.h>


// sorry for this hack, unfortunately we need it for
// the macros of recent libsilc to work ...
typedef unsigned char SilcUInt8;
typedef SilcTK::SilcUInt32 SilcUInt32;

SilcChannelContact::SilcChannelContact(SilcAccount *account,
				       const QString &channel, 
				       Kopete::MetaContact *meta,
				       const QString &icon)
  : SilcContact(account, QString("#%1").arg(channel), meta, icon),
    _channelEntry(NULL), _allowRichText(false)
{
  modeToBeSet = 0;
  setNickName(channel);

  QObject::connect
    (this, SIGNAL(onlineStatusChanged(Kopete::Contact *,
				      const Kopete::OnlineStatus &, 
				      const Kopete::OnlineStatus &)),
     this, SLOT(slotOnlineStatusChanged(Kopete::Contact *,
					const Kopete::OnlineStatus &,
					const Kopete::OnlineStatus &)));

  if(account->isConnected())
    setOnlineStatus(SilcProtocol::protocol()->statusOnlineChannel);
}
 
SilcChannelContact::~SilcChannelContact() { }

Kopete::ChatSession *
SilcChannelContact::manager(Kopete::Contact::CanCreateFlags flags)
{
  Kopete::ChatSession *session =
    SilcContact::manager(Kopete::Contact::CannotCreate);

  if(! session && (flags & Kopete::Contact::CanCreate)) {
    session = SilcContact::manager(flags);

    if(_channelEntry)
      setNickNameForMode(_channelEntry->mode);

    // update myselves status, this is, show op crown, etc. if we got one
    SilcBuddyContact *b = static_cast<SilcBuddyContact *>(account()->myself());
    updateBuddyOnlineStatus(b);
  }

  return session;
}

/**
 * @brief join the channel
 *
 * We need this wrapper around the more specialized join function in order
 * to overwrite the join method from the SilcContact class.
 */
void
SilcChannelContact::join(void)
{
  join(false, false, QString::null);
}

void
SilcChannelContact::join(bool founder, bool auth, const QString &password)
{
  if(_joined) return;

  SilcAccount *account = static_cast<SilcAccount *>(this->account());

  if(! account->isConnected()) {
    KMessageBox::error(Kopete::UI::Global::mainWidget(),
		       i18n("Sorry, you cannot join channels unless "
			    "you go online"), i18n("SILC Plugin"));
    return;
  }

  QString arg = QString("JOIN %1 %2").arg(nickName()).arg(password);

  if(founder)
    arg.append(" -founder");

  if(auth)
    arg.append(" -auth");

  account->sendSilcCommand(arg);
  setFileCapable(true);
}



void
SilcChannelContact::setChannelEntry(SilcTK::SilcChannelEntry e)
{
  _channelEntry = e;
  if(e) e->context = this;
  
  _joined = e != NULL;

  if(modeToBeSet && e->mode != modeToBeSet) {
    _channelEntry->mode = modeToBeSet;
    commitModeChange();
    modeToBeSet = 0;
  }

  SilcBuddyContact *buddy;
  for (buddy = toInvite.first(); buddy; buddy = toInvite.next() )
    invite(buddy);
  toInvite.clear();
}


void
SilcChannelContact::slotSendMessage(Kopete::Message &msg,
				    Kopete::ChatSession *session)
{
  if(session != manager())
    return;

  SilcAccount *account = static_cast<SilcAccount *>(this->account());

  if(! account->conn()) {
    KMessageBox::queuedMessageBox
      (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry, 
       i18n("Unable to send this message now. The protocol is currently "
	    "offline and does not support offline sending."),
       i18n( "User is Not Reachable"));
    return;
  }

  // get plain text message ...
  SilcTK::SilcMessageFlags flags = SILC_MESSAGE_FLAG_UTF8;
  unsigned char *buf = NULL;
  SilcTK::SilcUInt32 buflen = 0;
  QCString plaintext;

  if(account->signChannelMessages())
    flags |= SILC_MESSAGE_FLAG_SIGNED;

  if(allowRichText()) {
    SilcTK::SilcMime mime = getMessageAsMime(msg);
    buf = SilcTK::silc_mime_encode(mime, &buflen);
    SilcTK::silc_mime_free(mime);

    flags |= SILC_MESSAGE_FLAG_DATA;
  }
  else {
    plaintext = msg.plainBody().utf8();
    buf = (unsigned char *) (const char *) plaintext;
    buflen = plaintext.length();

    // use of rich text is forbidden, reset message to plain
    // (so our channel log doesn't show any markup as well)
    msg.setBody(msg.plainBody());
  }

  prettyPrintMessage(msg, flags);

  // pass message to libsilc ...
  SilcTK::silc_client_send_channel_message
    (account->client(), account->conn(), channelEntry(), NULL, flags,
     account->sha1hash, buf, buflen);
  
  if(allowRichText())
    SilcTK::silc_free(buf);
   
  // append message locally ...
  session->appendMessage(msg);
  session->messageSucceeded();
}

void
SilcChannelContact::slotOnlineStatusChanged(Kopete::Contact *,
					    const Kopete::OnlineStatus &status,
					    const Kopete::OnlineStatus &old)
{
  if(status == SilcProtocol::protocol()->statusOffline
     || status.status() == Kopete::OnlineStatus::Unknown) {
    // assume we're offline, thus left all channels ...
    setChannelEntry(NULL);
    return;
  }

  if(old != SilcProtocol::protocol()->statusOffline
     && old.status() != Kopete::OnlineStatus::Unknown) return;

  if(manager())
    join(); // autojoin the channel, if there's an open window for it ...
}


void
SilcChannelContact::leave(void)
{
  SilcAccount *account = static_cast<SilcAccount *>(this->account());
  account->sendSilcCommand(QString("LEAVE %1").arg(nickName()));
  setChannelEntry(NULL); // libsilc frees the channel entry after leaving
  setFileCapable(false);
}


bool
SilcChannelContact::isJoined(SilcBuddyContact *buddy)
{
  // channelEntry() is not set during the connection phase ...
  if(! channelEntry()) return false;

  return buddy->clientEntry(this) != NULL;
}


/**
 * @todo we shouldn't assume the first available fingerprint is the joined one
 */
int
SilcChannelContact::channelUserMode(SilcBuddyContact *buddy)
{
  // channelEntry() is not set during the connection phase ...
  if(! channelEntry()) return -1;

  SilcTK::SilcChannelUser cu = 
    SilcTK::silc_client_on_channel(channelEntry(), buddy->clientEntry());
  if(! cu) return -1;

  return cu->mode;
}

bool
SilcChannelContact::isOp(SilcBuddyContact *buddy)
{
  int cumode = channelUserMode(buddy);

  if(cumode < 0) return false; // error, how to tell?
  return (cumode & SILC_CHANNEL_UMODE_CHANOP) != 0;
}

bool
SilcChannelContact::isFounder(SilcBuddyContact *buddy)
{
  int cumode = channelUserMode(buddy);

  if(cumode < 0) return false; // error, how to tell?
  return (cumode & SILC_CHANNEL_UMODE_CHANFO) != 0;
}


void
SilcChannelContact::setOp(SilcBuddyContact *buddy, bool status)
{
  SilcTK::SilcChannelUser cu = 
    SilcTK::silc_client_on_channel(channelEntry(), buddy->clientEntry());
  if(! cu) return;

  // get current mode ...
  SilcTK::SilcUInt32 mode = cu->mode;

  // update mode ...
  mode &= ~SILC_CHANNEL_UMODE_CHANOP;
  if(status) mode |= SILC_CHANNEL_UMODE_CHANOP;

  // prepare to send the command finally ...
  SilcTK::SilcBuffer idp_channel = 
    SilcTK::silc_id_payload_encode(&channelEntry()->id, SILC_ID_CHANNEL);
  SilcTK::SilcBuffer idp_client =
    SilcTK::silc_id_payload_encode(&buddy->clientEntry()->id, SILC_ID_CLIENT);

  unsigned char modebuf[4];
  SILC_PUT32_MSB(mode, modebuf);

  // finally send the command ...
  SilcAccount *acc = static_cast<SilcAccount *>(account());
  SilcTK::silc_client_command_send(acc->client(), acc->conn(),
				   SILC_COMMAND_CUMODE, NULL, NULL, 3,
				   1, idp_channel->data, 
                                   silc_buffer_len(idp_channel),
				   2, modebuf, sizeof(modebuf),
				   3, idp_client->data, 
                                   silc_buffer_len(idp_client));
}


void
SilcChannelContact::kick(SilcBuddyContact *buddy)
{
  if(! channelEntry())
    return;

  SilcAccount *acc = static_cast<SilcAccount *>(account());

  for(unsigned int i = 0; i < buddy->clientEntriesCount(); i ++) {
    SilcTK::SilcClientEntry ce = buddy->clientEntry(i);
    if(! SilcTK::silc_client_on_channel(channelEntry(), ce))
      continue; // this ce doesn't participate on this channel

    // prepare to send the command finally ...
    SilcTK::SilcBuffer idp_channel = 
      SilcTK::silc_id_payload_encode(&channelEntry()->id, SILC_ID_CHANNEL);
    SilcTK::SilcBuffer idp_client =
      SilcTK::silc_id_payload_encode(&ce->id, SILC_ID_CLIENT);

    // finally send the command ...
    SilcTK::silc_client_command_send(acc->client(), acc->conn(),
				     SILC_COMMAND_KICK, NULL, NULL, 2,
				     1, idp_channel->data, 
                                     silc_buffer_len(idp_channel),
				     2, idp_client->data, 
                                     silc_buffer_len(idp_client));
  }
}


const Kopete::OnlineStatus &
SilcChannelContact::getBuddyOnlineStatus(SilcBuddyContact *buddy)
{
  if(buddy->onlineStatus() == SilcProtocol::protocol()->statusGone)
    return isOp(buddy)
      ? SilcProtocol::protocol()->statusGoneOp
      : SilcProtocol::protocol()->statusGone;
      
  if(buddy->onlineStatus() == SilcProtocol::protocol()->statusIndisposed)
    return isOp(buddy)
      ? SilcProtocol::protocol()->statusIndisposedOp
      : SilcProtocol::protocol()->statusIndisposed;

  if(buddy->onlineStatus() == SilcProtocol::protocol()->statusBusy)
    return isOp(buddy)
      ? SilcProtocol::protocol()->statusBusyOp
      : SilcProtocol::protocol()->statusBusy;

  if(buddy->onlineStatus() == SilcProtocol::protocol()->statusHyper)
    return isOp(buddy)
      ? SilcProtocol::protocol()->statusHyperOp
      : SilcProtocol::protocol()->statusHyper;
  
  if(buddy->onlineStatus() == SilcProtocol::protocol()->statusOnline)
    return isOp(buddy)
      ? SilcProtocol::protocol()->statusOnlineOp
      : SilcProtocol::protocol()->statusOnline;

  return SilcProtocol::protocol()->statusOffline;
}

void
SilcChannelContact::updateBuddyOnlineStatus(SilcBuddyContact *buddy)
{
  // check whether the assigned status is correct for this channel,
  // i.e. check whether we don't have to use the Op'd variant for example
  const Kopete::OnlineStatus chuser_status = getBuddyOnlineStatus(buddy);

  if(! manager()->members().contains(buddy)) {
    manager()->addContact(buddy, chuser_status);
    return;
  }

  if(manager()->contactOnlineStatus(buddy) != chuser_status)
    manager()->setContactOnlineStatus(buddy, chuser_status);
}

void 
SilcChannelContact::silc_channel_message(SilcTK::SilcClient /* client */, 
					 SilcTK::SilcClientConnection /*conn*/,
					 SilcTK::SilcClientEntry sender, 
					 SilcTK::SilcChannelEntry channel,
					 SilcTK::SilcMessagePayload payload,
					 SilcTK::SilcMessageFlags flags, 
					 const unsigned char *message,
					 SilcTK::SilcUInt32 message_len)
{
  SilcChannelContact *ch = (SilcChannelContact *) channel->context;
  SilcBuddyContact *buddy = (SilcBuddyContact *) sender->context;

  if(! ch) {
    std::cerr << "cannot find SilcChannel structure for "
	      << channel->channel_name << std::endl;
    return;
  }

  // if we haven't heard of the buddy, use the channel name ... hmm.
  // @todo we ought to create a new buddy contact instead
  if(! buddy) {
    return;
  }

  /* If the messages is digitally signed, verify it, if possible. */
  SignatureStatus sigstat = Unknown;
  if (flags & SILC_MESSAGE_FLAG_SIGNED)
    sigstat = buddy->verifySignature(payload);

  // convert Utf8 stuff depending on whether SILC_MESSAGE_FLAG_UTF8 is set
  QString text;
  if(flags & SILC_MESSAGE_FLAG_UTF8)
    text = QString::fromUtf8((const char *) message, message_len);
  else if (flags & SILC_MESSAGE_FLAG_DATA);
  else
    text = QString::fromLatin1((const char *) message, message_len);

    Kopete::Message msg;
  if(flags & SILC_MESSAGE_FLAG_NOTICE)
    msg = Kopete::Message(buddy, ch->manager()->members(), QString("%1 -*- %2")
			  .arg(buddy->nickName()).arg(text),
			  Kopete::Message::Internal,
			  Kopete::Message::PlainText, QString::null,
			  Kopete::Message::TypeAction);
  else if (flags & SILC_MESSAGE_FLAG_DATA) {
    /* SilcMimeMessage */
    QStringList *filenames;
    SilcTK::SilcMime tmp = SilcTK::silc_mime_decode(NULL, message, 
                                                    message_len);
    /* Assemble mime partials */
    SilcTK::SilcMime mime = buddy->mime_asm(tmp);
    if (!mime) return;

    QString type = SilcTK::silc_mime_get_field(mime, "Content-Type");
    if(type.isEmpty()) goto mimeout;

    if (type.left(21).compare("multipart/alternative") == 0) {
        msg = Kopete::Message(buddy, ch->manager()->members(), 
                              QString::null,
                              Kopete::Message::Inbound,
                              Kopete::Message::PlainText, QString::null,
                              Kopete::Message::TypeNormal);
        buddy->mimeAlternateToMsg(msg, mime, ch->allowRichText());
        ch->manager()->appendMessage(msg);
    }
    else {
      filenames = buddy->saveMime(mime);
      for ( QStringList::Iterator it = filenames->begin(); 
            it != filenames->end(); ++it ) {
        msg = Kopete::Message(buddy, ch->manager()->members(), 
                              buddy->mimeDisplayMessage(*it),
                              Kopete::Message::Inbound,
                              Kopete::Message::RichText, QString::null,
                              Kopete::Message::TypeNormal);

        prettyPrintMessage(msg, flags, sigstat);
        ch->manager()->appendMessage(msg);
      }
      delete filenames;
    }
mimeout:
    SilcTK::silc_mime_free(mime);
    return;
  }
  else
    msg = Kopete::Message(buddy, ch->manager()->members(), text, 
			  Kopete::Message::Inbound, 
			  Kopete::Message::PlainText, QString::null,
			  (flags & SILC_MESSAGE_FLAG_ACTION)
			  ? Kopete::Message::TypeAction
			  : Kopete::Message::TypeNormal);

  prettyPrintMessage(msg, flags, sigstat);

  ch->manager()->appendMessage(msg);
}


void 
SilcChannelContact::silc_channel_message(SilcTK::SilcClient client,
					 SilcTK::SilcClientConnection conn,
					 SilcTK::SilcClientEntry sender,
					 SilcTK::SilcChannelEntry channel,
					 SilcTK::SilcMessagePayload payload,
					 SilcTK::SilcChannelPrivateKey,
					 SilcTK::SilcMessageFlags flags, 
					 const unsigned char *msg,
					 SilcTK::SilcUInt32 msg_len)
{
  SilcChannelContact::silc_channel_message(client, conn, sender, channel, 
					   payload, flags, msg, msg_len);
} 


void SilcChannelContact::setNickNameForMode(int mode) {
  QString _mode = "";
  
  if(mode & SILC_CHANNEL_MODE_PRIVATE) {
    _mode.append("p");
  }	

  if(mode & SILC_CHANNEL_MODE_SECRET) {
    _mode.append("s");
  }       

  if(mode & SILC_CHANNEL_MODE_PRIVKEY) {
    _mode.append("K");
  }      
  
  if(mode & SILC_CHANNEL_MODE_INVITE) {
    _mode.append("i");
  }
  
  if(mode & SILC_CHANNEL_MODE_CHANNEL_AUTH) {
    _mode.append("C");
  }
  
  if(_mode != "") {
    _mode = " ["+_mode+"]";
  }
 
 manager()->setDisplayName(nickName().append(_mode));
}



void 
SilcChannelContact::invite(const SilcBuddyContact *buddy)
{
  if(!_joined) {
    toInvite.append(buddy);
    return;
  }
  // prepare to send the command finally ...
  SilcTK::SilcBuffer idp_channel = 
    SilcTK::silc_id_payload_encode(&channelEntry()->id, SILC_ID_CHANNEL);
  SilcTK::SilcBuffer idp_client =
    SilcTK::silc_id_payload_encode(&buddy->clientEntry()->id, SILC_ID_CLIENT);


  SilcAccount *acc = static_cast<SilcAccount *>(account());
  SilcTK::silc_client_command_send(acc->client(), acc->conn(),
				   SILC_COMMAND_INVITE, NULL, NULL, 2,
				   1, idp_channel->data, 
                                   silc_buffer_len(idp_channel),
				   2, idp_client->data, 
                                   silc_buffer_len(idp_client));
}



void 
SilcChannelContact::slotUserInfo(void)
{
  new SilcChannelContactInfoWidget(this);;
}


void
SilcChannelContact::setTopic(QString &newTopic)
{
  QString oldTopic = topic();
  if(!oldTopic.isNull() && newTopic.compare(oldTopic) == 0)
    return;

  account()->sendSilcCommand
    (QString("TOPIC %1 %2").arg(nickName()).arg(newTopic));
}


void
SilcChannelContact::setPrivate(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_PRIVATE;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_PRIVATE;
    return;
  }
  else if (state == isPrivate())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_PRIVATE;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_PRIVATE;

  commitModeChange();
}


void
SilcChannelContact::setSecret(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_SECRET;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_SECRET;
    return;
  }
  else if (state == isSecret())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_SECRET;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_SECRET;

  commitModeChange();
}


void
SilcChannelContact::setInviteOnly(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_INVITE;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_INVITE;
    return;
  }
  else if (state == isInviteOnly())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_INVITE;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_INVITE;

  commitModeChange();
}


void
SilcChannelContact::setSilenceUser(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_SILENCE_USERS;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_SILENCE_USERS;
    return;
  }
  else if (state == isSilenceUser())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_SILENCE_USERS;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_SILENCE_USERS;

  commitModeChange();
}


void
SilcChannelContact::setSilenceOperator(bool state)
{
  if(! channelEntry()) {
    modeToBeSet &= ~SILC_CHANNEL_MODE_SILENCE_OPERS;
    if(state) modeToBeSet |= SILC_CHANNEL_MODE_SILENCE_OPERS;
    return;
  }
  else if (state == isSilenceOperator())
    return;

  _channelEntry->mode &= ~SILC_CHANNEL_MODE_SILENCE_OPERS;
  if(state) _channelEntry->mode |= SILC_CHANNEL_MODE_SILENCE_OPERS;

  commitModeChange();
}


void
SilcChannelContact::commitModeChange(void)
{
  assert(channelEntry());

  SilcTK::SilcBuffer idp_channel = 
    SilcTK::silc_id_payload_encode(&channelEntry()->id, SILC_ID_CHANNEL);

  unsigned char modebuf[4];
  SILC_PUT32_MSB(channelEntry()->mode, modebuf);

  SilcAccount *acc = static_cast<SilcAccount *>(account());
  SilcTK::silc_client_command_send(acc->client(), acc->conn(),
				   SILC_COMMAND_CMODE, NULL, NULL, 2,
				   1, idp_channel->data, 
                                   silc_buffer_len(idp_channel),
				   2, modebuf, sizeof(modebuf));
}

void 
SilcChannelContact::sendFile(const KURL &sourceURL,
			   const QString & /* fileName */, uint /* fileSize */) 
{
  QString filePath;

  if(! sourceURL.isValid())
    filePath = KFileDialog::getOpenFileName(QString::null, "*", 0L,
					    i18n("Kopete File Transfer"));
  else
    filePath = sourceURL.path(-1);

  QFile file(filePath);
  if(! file.exists())
    return;
  sendFileAsMime(filePath);
}

void 
SilcChannelContact::sendFileAsMime(const QString &fileName)
{
  int chunks = 0;
  SilcTK::SilcBuffer buffer;
  QFile file(fileName);

  /* Sending Chunks */
  SilcTK::SilcDList parts = getFileAsMime(fileName);
  SilcTK::silc_dlist_start(parts);
  while ((buffer = (SilcTK::SilcBuffer)SilcTK::silc_dlist_get(parts))
	 != SILC_LIST_END) { 
    chunks ++;
    SilcTK::silc_client_send_channel_message
      (account()->client(), account()->conn(), channelEntry(), NULL, 
       SILC_MESSAGE_FLAG_DATA, account()->sha1hash, 
       (unsigned char*)buffer->data, silc_buffer_len(buffer));
  }
  SilcTK::silc_mime_partial_free(parts);
  
  Kopete::Message msg = 
    Kopete::Message(account()->myself(), manager()->members(), 
		    account()->myself()->mimeDisplayMessage(fileName, chunks),
                    Kopete::Message::Outbound,
                    Kopete::Message::RichText, QString::null,
                    Kopete::Message::TypeNormal);

  manager()->appendMessage(msg);
}

void 
SilcChannelContact::serialize(QMap<QString, QString> &serializedData,
			      QMap<QString, QString>&)
{
  serializedData["allowRichText"] = allowRichText() ? "yes" : "no";
}

#include "silcchannelcontact.moc"

